/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.ccht.service.report.impl;

// Java classes
import java.io.IOException;
import java.io.Serializable;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.report.CompletedReport;
import gov.va.med.ccht.model.report.CompletedReportFieldType;
import gov.va.med.ccht.model.report.ReportParameters;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.report.SimpleCompletedReport;
import gov.va.med.ccht.model.terminology.ReportFormat;
import gov.va.med.ccht.model.terminology.StandardReport;
import gov.va.med.ccht.persistent.ReportDAO;
import gov.va.med.ccht.service.common.BatchProcessTriggerGroups;
import gov.va.med.ccht.service.common.TerminologyService;
import gov.va.med.ccht.service.report.CompletedReportsSearchQueryInfo;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.service.report.ReportFilter;
import gov.va.med.ccht.service.report.ReportFilterSearchQueryInfo;
import gov.va.med.ccht.service.report.ReportParameterConversionService;
import gov.va.med.ccht.service.report.ReportPersistenceService;
import gov.va.med.ccht.service.report.StandardReportCriteria;
import gov.va.med.ccht.service.report.StandardReportService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.NoRecordFoundException;
import gov.va.med.fw.report.ReportConfiguration;
import gov.va.med.fw.report.ReportException;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.report.ReportService;
import gov.va.med.fw.scheduling.AuditableScheduledProcess;
import gov.va.med.fw.scheduling.ScheduledProcessInvocationContext;
import gov.va.med.fw.scheduling.SchedulingService;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.SecurityService;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.InvalidScheduleException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.jms.JMSPayload;
import gov.va.med.fw.service.jms.MessageProducerService;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * Provides services to generate and to query standard reports in ESR
 * 
 * Project: Common</br> Created on: 12:00:00 PM </br>
 * 
 * @author DNS
 */
@Service
public class StandardReportServiceImpl extends AbstractComponent implements StandardReportService,
		AuditableScheduledProcess, ReportConstants {
	
	@Autowired
	private ReportDAO reportDAO = null;
	@Autowired
	private ReportService reportService = null;
	@Autowired
	private TerminologyService terminologyService = null;
	@Autowired
	private ReportParameterConversionService reportParameterConversionService = null;
	@Autowired
	private SchedulingService schedulingService = null;
	@Autowired
	private SecurityService securityService;
	@Autowired
	@Qualifier("common.jms.producer.reportingEvents")
	private MessageProducerService messageProducerService;
	@Autowired
	private ReportPersistenceService reportPersistenceService;

	private String scheduledReportJobName = null;
	private String createReportJMSPayloadName;
	
	@Value("${daysToRetainCompletedReports}")
	private Integer daysToRetainCompletedReports = 365;
	@Value("${daysToWarnBeforeDeletingReports}")
	private Integer daysToWarnBeforeDeletion = 30;


	private static final String PERIOD = ".";

	public static final String SCHEDULED_REPORT = "scheduledReport";
	public static final String DEFAULT_AUDIT_NAME = SCHEDULED_PROCESS_AUDIT_PREFIX + "ScheduledReport";

	private static final long serialVersionUID = 7440299997017245373L;

	public StandardReportServiceImpl() {
		super();
	}


//	public List<Date[]> getSchedulePreview(UserPrincipal user, ReportSetup reportSetup,
//			int numDates, Date startingDate) throws ServiceException {
//		ReportSchedule schedule = reportSetup.getReportSchedule();
//		ReportParameters params = reportSetup.getReportParameters();
//		DateRangeBoundary reportFromDate = params.getScheduledReportFromDate();
//		DateRangeBoundary reportToDate = params.getScheduledReportToDate();
//
//		String cron = SchedulerUtils.getCronExpression(schedule);
//
//		TimeZone timezone = TimeZoneUtils.getTimeZone();
//		if (user != null && user.getCurrentTimeZone() != null)
//			timezone = user.getCurrentTimeZone();
//		/*if (schedule.getDateToGenerate() != null)
//			timezone = schedule.getDateToGenerate().getTimeZone();
//		*/
//
//		final String triggerName = "";
//		final String triggerGroup = "";
//		final String triggerJobName = "";
//		final String triggerJobGroup = "";
//
//		CronTrigger trigger = new CronTriggerImpl(triggerName, triggerGroup, triggerJobName, triggerJobGroup, cron.toString(), timezone);
//		
//		trigger.setStartTime(startingDate);
//
//		Date timeIndex = startingDate;
//		List<Date[]> results = new ArrayList<Date[]>();
//
//		for (int i = 0; i < numDates; i++) {
//			timeIndex = trigger.getFireTimeAfter(timeIndex);
//			if (timeIndex == null)
//				break;
//
//			results.add(new Date[] { timeIndex, reportFromDate.calculateDate(timeIndex),
//					reportToDate.calculateDate(timeIndex) });
//		}
//		return results;
//	}

	/**
	 * @see gov.va.med.ccht.service.report.esr.service.report.StandardReportService#getStandardReports(gov.va.med.fw.security.UserPrincipal)
	 */
	public List<StandardReport> getStandardReports() throws ServiceException {
		List<StandardReport> result = null;
		try {
			result = terminologyService.findAllActive(StandardReport.class);
		} catch (Exception ex) {
			throw new ServiceException(ex);
		}
		return result;
	}

	/**
	 * @see gov.va.med.ccht.service.report.esr.service.report.StandardReportService#getStandardReports(gov.va.med.fw.security.UserPrincipal)
	 */
	public List<StandardReport> getStandardReports(UserPrincipal user) throws ServiceException {
		Validate.notNull(user, "A user principal must not be null");
		List<StandardReport> stdReports = new ArrayList<StandardReport>();
		List<StandardReport> all = this.getStandardReports();
		if (all != null) {
			for (StandardReport stdReport : all)
				if (this.isUserPermittedToScheduleRun(user, stdReport))
					stdReports.add(stdReport);
		}
		return stdReports;
	}

	/**
	 * @see gov.va.med.ccht.service.report.esr.service.report.StandardReportService#getReportSetup
	 */
	public ScheduledReport getScheduledReport(EntityKey<ScheduledReport> identifier) throws ServiceException {
		Validate.notNull(identifier, "A user principal must not be null");
		ScheduledReport result = null;

		try {
			result = reportDAO.getScheduledReport(identifier);
			reportParameterConversionService.convertFromText(result);
		} catch (DAOException e) {
			throw new ServiceException("Failed to get a list of report setup with identifier "
					+ identifier.getKeyValueAsString(), e);
		} catch (Exception e) {
			throw new ServiceException("Failed to get a list of report setup with identifier ", e);
		}
		return result;
	}

	/**
	 * @see gov.va.med.ccht.service.report.esr.service.report.StandardReportService#saveReportSetup
	 */
	public ScheduledReport saveScheduledReport(ScheduledReport scheduledReport, boolean convertToText) throws ServiceException {

		Validate.notNull(scheduledReport, "ScheduledReport to save must not be null");
		try {
			if (convertToText) {
				reportParameterConversionService.convertToText(scheduledReport);
			}
			// CCR 407.  Avoid modifying CISS framework so changes are made 
			// in ReportDAOImpl to return a persistent entity to retrieve an ID
			return reportDAO.saveScheduledReport(scheduledReport);
		} catch (DAOException e) {
			throw new ServiceException("Failed to save report setup with identifier "
					+ scheduledReport.getEntityKey().getKeyValueAsString(), e);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	/**
	 * @see gov.va.med.ccht.service.report.esr.service.report.StandardReportService#getCompletedReport
	 */
	public CompletedReport getCompletedReport(EntityKey<CompletedReport> identifier)
			throws ServiceException {
		Validate.notNull(identifier, "A user principal must not be null");
		CompletedReport result = null;
		try {
			result = reportDAO.getCompletedReport(identifier);
		} catch (DAOException e) {
			throw new ServiceException("Failed to get a list of completed report with identifier "
					+ identifier.getKeyValueAsString(), e);
		}
		return result;
	}
	
	public void saveReportCriteria(ReportSetup reportSetup) throws ServiceException {
		reportPersistenceService.saveReportCriteria(reportSetup);
	}
	

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.esr.service.StandardReportService#buildReport(ReportSetup)
	 */
	public void buildReport(ReportSetup setup) throws ServiceException {
			UserPrincipal user = null;

			if (setup != null) {
				user = this.securityService.getUser(setup.getSetupUser());
				// set default report format
				ReportFormat reportFormat = getTerminologyService().getByCode(ReportFormat.class,
						ReportFormat.SUMMARY);
				if (setup.getReportParameters() == null) {
					setup.setReportParameters(new ReportParameters());
				}
				if (setup.getReportParameters().getReportFormat() == null) {
					setup.getReportParameters().setReportFormat(reportFormat);
				}

				// generateReport(user, setup);
				ReportConfiguration config = null;
				try {
					config = this.getReportConfiguration(user, setup);
					if (config != null) {
						//if (logger.isInfoEnabled()) {
						//	logger.info("Generating Report [" + config.getReportID() + "]");
						//}
						// Generate a report with the specific report data set
						reportService.buildReport(config);
					}
				} catch (IOException ioe) {
					logger.error("Unable to get the Report configuration");
					throw new ServiceException("Unable to get the Report configuration", ioe);
				}
			} else {
				logger.error("Report Setup Object Not found to Generate the Report");
			}
	}

	/**
	 * 
	 * @param setup
	 */

	public void generateReport(ReportSetup setup) throws ServiceException {
		Validate.notNull(setup, "A report setup must not be null");

		try {
			reportParameterConversionService.convertToText(setup);
			Map<String, Object> map = new HashMap<String, Object>();

			map.put(REPORT_PARAMETERS, setup.getReportParametersText());
			map.put(REPORT_SCHEDULE, setup.getReportScheduleText());
			map.put(REPORT_PARAM_EMAIL_REPORT_TO, setup.getEmailAddress());
			map.put(REPORT_PARAM_USER, setup.getSetupUser());
			map.put(REPORT_STANDARD_TYPE, setup.getStandardReport().getCode());
			map.put(REPORT_PARAM_FILE_TYPE, setup.getFileType().getCode());

			JMSPayload payload = (JMSPayload) getApplicationContext().getBean(createReportJMSPayloadName);
			payload.setPayload((Serializable) map);
			messageProducerService.send(payload);

		} catch (Exception e) {
			throw new ServiceException("Failed to generate a report", e);
		}
	}

	@SuppressWarnings("unchecked")
	public void scheduleReport(ScheduledReport scheduledReport) throws ServiceException {

		Exception debug = null;
		try {
			Validate.notNull(scheduledReport, "Scheduled Report id must not be null");
			
			// Step 2 Get a job detail and populate with user and report
			// setup
			StringBuilder group_name = new StringBuilder(BatchProcessTriggerGroups.DYNAMIC_BATCH_JOBS);
			String name = getJobName(scheduledReport);

			JobDetail jobDetail = (JobDetail) this.getComponent(this.scheduledReportJobName);
			jobDetail.getJobDataMap().put("name", name);
			jobDetail.getJobDataMap().put("group", group_name.toString());

			// Populate dynamic arguments to pass to a method for invocation
			Map<String, Object> data = jobDetail.getJobDataMap();
			data.put(REPORT_SETUP_ID, scheduledReport.getId().toString());
			this.schedulingService.getScheduler().addJob(jobDetail, true);

			// Step 4: Get a scheduling service to schedule a job4
			CronTrigger crontrigger = TriggerBuilder.newTrigger()
					.withIdentity(name, group_name.toString())
				    .withSchedule(CronScheduleBuilder.cronSchedule(scheduledReport.getReportSchedule().getCronExpression()).inTimeZone(TimeZoneUtils.getTimeZone()))
				    .startNow()
				    .build();
			
			if (logger.isInfoEnabled()) {
				logger.info("Schedule a report to run at " + crontrigger.getStartTime()
						+ " using cron expression: " + scheduledReport.getReportSchedule().getCronExpression());
			}			
			
		}
		catch (SchedulerException e) {
			debug = e;
			throw new InvalidScheduleException("Invalid schedule. Exception message: " + e.getMessage() );
		} 
		catch (RuntimeException e) {
			debug = e;
			throw new ServiceException("Unexpected exception: " + e.getClass().getName() + " message: " + e.getMessage());
		}
		finally {
			if( debug != null && logger.isDebugEnabled() ) {
				logger.debug( debug );
			}
		}
	}

	/**
	 * Construct the job name from ReportSetup object
	 * 
	 * @param setup
	 * @return
	 */
	private String getJobName(ScheduledReport scheduledReport) {
		StringBuilder name = new StringBuilder(SCHEDULED_REPORT);
		name.append(PERIOD).append(scheduledReport.getUserName()).append(PERIOD).append(
				scheduledReport.getStandardReport().getCode()).append(PERIOD);
		if (scheduledReport.getEntityKey() == null) {
			name.append(DateUtils.format(new Date(), "HHmmss"));
		} else {
			name.append(scheduledReport.getEntityKey().getKeyValueAsString());
		}

		return name.toString();
	}

	public void cancelScheduledReport(EntityKey<ScheduledReport> identifier)
			throws NoRecordFoundException, ServiceException {

		Validate.notNull(identifier, "ScheduledReport identifier must not be null");
		Validate.notNull(identifier.getKeyValue(), "ScheduledReport identifier key can not be null.");

		ScheduledReport setup = null;

		setup = this.getScheduledReport(identifier);

		if (setup == null) {
			throw new NoRecordFoundException("ReportSetup does not exist for identifier: "
					+ identifier.getKeyValueAsString());
		}

		try {
				schedulingService.unschedule(getJobName(setup), BatchProcessTriggerGroups.DYNAMIC_BATCH_JOBS);
				reportDAO.deleteScheduledReport(identifier);
		} catch (SchedulerException e) {
			throw new ServiceException(
					"Failed to unschedule triggers for report setup with identifier "
							+ identifier.getKeyValueAsString(), e);
		} catch (DAOException e) {
			throw new ServiceException("Failed to delete report setup with identifier "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	/**
	 * @param user
	 * @param setup
	 * @return
	 * @throws IOException
	 */
	protected ReportConfiguration getReportConfiguration(UserPrincipal user, ReportSetup setup)
			throws IOException {

		StandardReportCriteria criteria = new StandardReportCriteria();
		criteria.setReportSetup(setup);
		criteria.setReportUser(user);

		return this.getReportConfiguration(user, setup, criteria);
	}

	protected ReportConfiguration getReportConfiguration(UserPrincipal user, ReportSetup setup,
			StandardReportCriteria criteria) throws IOException {

		StandardReport report = setup.getStandardReport();
		String id = report.getCode();
		String reportName = StringUtils.deleteWhitespace(id);

		ReportParameters parameterSet = setup.getReportParameters();
		ReportExportedType type = setup.getFileType();
		ReportExportedType.Code code = type != null ? ReportExportedType.Code.getByCode(type
				.getCode()) : null;

		String formatCode = getLookupName(parameterSet != null ? parameterSet.getReportFormat()
				: null);

		StringBuilder output = new StringBuilder();
		output.append(id).append("-").append(report.getName());
		if (formatCode != null) {
			output.append("-").append(formatCode);
		}
		if(code!=null)
		{
		  output.append(".").append(code.getAlias().toLowerCase());
		}
		ReportConfiguration config = new ReportConfiguration(reportName, id, code, StringUtils
				.deleteWhitespace(output.toString()));

		config.setReportUser(user);
		config.setQueryCriteria(criteria);
		config.setResourceMapping(criteria.getCriteria());

		return config;
	}

	private String getLookupName(Lookup lookup) {
		return lookup != null ? lookup.getName() : null;
	}

	/**
	 * Verifies whether user has run/schedule/archive/delete privilege.
	 * 
	 * @param user
	 * @param report
	 * @return
	 */
	protected boolean isUserPermittedToScheduleRun(UserPrincipal user, StandardReport report) {
		Set<Permission> rptCaps = (report != null) ? report.getPermissions() : null;
		if (rptCaps != null) {
			for (Permission p : rptCaps)
				if (user.isPermissionGranted(p.getName()))
					return true;
		}
		return false;
	}

	/**
	 * Verifies whether user has privilege to inactivate a report.
	 * 
	 * @param user
	 * @param report
	 * @return
	 */
	protected boolean isUserPermittedToInactivate(UserPrincipal user, CompletedReport report) {
		Validate.notNull(user, "UserPrincipal is null");
		Validate.notNull(report, "report is null");

		/*
		 * If the report is generated by same user or National Administrator and
		 * has a capability of Schedule/Run/Delete/Archive, return true.
		 */

		if (report.getUserName().equalsIgnoreCase(user.getUsername())) {
			return true; 
		}
		return false;
	}

	/**
	 * Verifies whether user has privilege to delete a report.
	 * 
	 * @param user
	 * @param report
	 * @return
	 */
	protected boolean isUserPermittedToDelete(UserPrincipal user, CompletedReport report) {

		/*
		 * If the report is generated by same user or System Adminstrator or if
		 * it a inactive report and has a capability of
		 * Schedule/Run/Delete/Archive, return true.
		 */
		if (report.getUserName() != null && user.getUsername().equalsIgnoreCase((report.getUserName()))) {
			return true;
		}
		return false;
	}

	/*
	 * @see
	 * gov.va.med.esr.service.StandardReportService#getSimpleCompletedReport
	 * (EntityKey identifier)
	 */
	public SimpleCompletedReport getSimpleCompletedReport(
			EntityKey<SimpleCompletedReport> identifier) throws ServiceException {
		Validate.notNull(identifier, "SimpleCompletedReport identifier must not be null");
		SimpleCompletedReport result = null;
		try {
			result = getReportDAO().getSimpleCompletedReport(identifier);
		} catch (DAOException e) {
			throw new ServiceException(
					"Failed to get a SimpleCompletedReport report with identifier "
							+ identifier.getKeyValueAsString(), e);
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seegov.va.med.IHTA.service.report.StandardReportService#
	 * getCompletedReportParameters(gov.va.med.fw.model.EntityKey)
	 */
	public ReportSetup getCompletedReportParameters(EntityKey<SimpleCompletedReport> identifier)
			throws ServiceException,ParseException {
		Validate.notNull(identifier, "CompletedReport identifier must not be null");

		try {
			SimpleCompletedReport completedReport = getReportDAO().getSimpleCompletedReport(
					identifier);
			if (completedReport != null) {
				ReportSetup reportSetup = new ReportSetup();
				reportSetup.setStandardReport(completedReport.getStandardReport());
				reportSetup.setReportParametersText(completedReport.getReportParameterText());
				reportSetup.setFileType(completedReport.getFileType());
				reportSetup.setSetupUser(completedReport.getUserName());
				// convert schedule and report parameters
				reportParameterConversionService.convertFromText(reportSetup);
				return reportSetup;
			}
		} catch (DAOException e) {
			throw new ServiceException(
					"Failed to get a SimpleCompletedReport report with identifier "
							+ identifier.getKeyValueAsString(), e);
		}
		return null;
	}

	/*
	 * @see
	 * gov.va.med.esr.service.StandardReportService#deleteCompletedReport(gov
	 * .va.med.fw.model.EntityKey, UserPrincipal user )
	 */
	public void deleteCompletedReport(EntityKey<CompletedReport> identifier, UserPrincipal user)
			throws NoRecordFoundException, ServiceException {

		Validate.notNull(identifier, "CompletedReport identifier must not be null");
		Validate.notNull(identifier.getKeyValue(),
				"CompletedReport identifier key can not be null.");

		CompletedReport completedReport = getCompletedReport(identifier);

		if (completedReport == null) {
			throw new NoRecordFoundException(
					"SimpleCompletedReport does not exist for identifier: "
							+ identifier.getKeyValueAsString());
		}
		try {
			if (isUserPermittedToDelete(user, completedReport)) {
				getReportDAO().deleteCompletedReport(identifier);
			} else
				throw new ServiceException("User doesn't have permissions to delete a report: "
						+ user.getUsername());
		} catch (DAOException e) {
			throw new ServiceException("Failed to delete SimpleCompletedReport with identifier "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	/*
	 * @see
	 * gov.va.med.esr.service.StandardReportService#inactivateCompletedReport
	 * (gov.va.med.fw.model.EntityKey, UserPrincipal user )
	 */
	public void inactivateCompletedReport(EntityKey<CompletedReport> identifier, UserPrincipal user)
			throws NoRecordFoundException, ServiceException, ReportException {

		Validate.notNull(identifier, "CompletedReport identifier must not be null");
		Validate.notNull(identifier.getKeyValue(),
				"CompletedReport identifier key can not be null.");

		CompletedReport completedReport = getCompletedReport(identifier);

		if (completedReport == null) {
			throw new NoRecordFoundException("CompletedReport does not exist for identifier: "
					+ identifier.getKeyValueAsString());
		}
		// If it is an inactive report throw an exception
		if (completedReport.getInactivatedDate() != null)
			throw new ReportException("This report is an inactive report");
		try {
			// If user has permission to inactivate
			// Set inactivate date to sys date.
			if (isUserPermittedToInactivate(user, completedReport)) {
				completedReport.setInactivatedDate(new Date());
				getReportDAO().saveCompletedReport(completedReport);
			} else
				throw new ServiceException("User doesn't have permissions to inactivate a report:"
						+ user.getUsername());
		} catch (DAOException e) {
			throw new ServiceException("Failed to inactivate CompletedReport with identifier "
					+ identifier.getKeyValueAsString(), e);
		}
	}

	/**
	 * Deleted expired completed reports older than 365 days or a configured
	 * value
	 * 
	 * @param purgeQueryString
	 * @param auditInfo
	 * @param systemParameter
	 * @return
	 * @throws ServiceException
	 */
	public Integer purgeCompletedReports() throws ServiceException {
		try {
			return getReportDAO().purgeCompletedReports();
		} catch (DAOException e) {
			throw new ServiceException("Error Purging Completed Reports data " + e);
		}
	}

	public Map<String, String> getExpiringReportIdUserMap() throws ServiceException {
		try {
			return getReportDAO().getExpiringReportIdUserMap(daysToRetainCompletedReports,
					daysToWarnBeforeDeletion);
		} catch (Exception e) {
			throw new ServiceException("getExpiringReportUserIds failed " + e.getMessage(), e);
		}
	}

	public void updateCompletedReport(SimpleCompletedReport simpleCompletedReport)
			throws ServiceException {
		logger.error("$$$$$ fix me.");

//		try {
//			getReportDAO().update(simpleCompletedReport);
//		} catch (DAOException e) {
//			throw new ServiceException("Unable to extend Completed Report expiraion date", e);
//		}
	}

	public void updateScheduledReport(ScheduledReport scheduledReport)
			throws ServiceException {
		logger.error("$$$$$ fix me.");

//		Validate.notNull(scheduledReport, "ScheduledReport to save must not be null");
//		try {
//			ScheduledReport oldScheduledReport = (ScheduledReport) reportDAO.getByKey(scheduledReport.getEntityKey());
//			schedulingService.unschedule(getJobName(oldScheduledReport), BatchProcessTriggerGroups.DYNAMIC_BATCH_JOBS);
//
//			oldScheduledReport.setReportGenerateDate(scheduledReport.getReportGenerateDate());
//			oldScheduledReport.setEmail(scheduledReport.getEmail());
//			oldScheduledReport.setFileType(scheduledReport.getFileType());
//			oldScheduledReport.setReportParameterText(scheduledReport.getReportParameterText());
//			oldScheduledReport.setReportScheduleText(scheduledReport.getReportScheduleText());
//			oldScheduledReport.setUserName(scheduledReport.getUserName());
//			oldScheduledReport.setStandardReport(scheduledReport.getStandardReport());
//
//			scheduleReport(scheduledReport);
//			reportDAO.update(oldScheduledReport);
//		} catch (DAOException e) {
//			throw new ServiceException("Failed to save ScheduledReport with identifier "
//					+ scheduledReport.getEntityKey().getKeyValueAsString(), e);
//		} catch (Exception e) {
//			throw new ServiceException(e.getMessage(), e);
//		}
	}

	/*
	 * @see
	 * gov.va.med.esr.service.StandardReportService#getCompletedReportFilter()
	 */
	public ReportFilter getCompletedReportFilter() throws ServiceException {
		ReportFilter reportFilter = new ReportFilter();
		try {
			reportFilter.setStandardReports(getReportDAO().getDistinctStandardReports());
			reportFilter.setReportFileTypes(getReportDAO().getDistinctReportExportTypes());
		} catch (DAOException e) {
			throw new ServiceException("Error getting the standard reports " + e);
		}
		return reportFilter;
	}

	/**
	 * @return Returns the securityService.
	 */
	public SecurityService getSecurityService() {
		return securityService;
	}

	/**
	 * @param securityService
	 *            The securityService to set.
	 */
	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.scheduling.AuditableScheduledProcess#getAuditInfo(gov.va
	 * .med.fw.scheduling.ScheduledProcessInvocationContext)
	 */
	public String getAuditInfo(ScheduledProcessInvocationContext context) {
		// default implementation
		return DEFAULT_AUDIT_NAME;
	}

	public TerminologyService getTerminologyService() {
		return terminologyService;
	}

	public void setTerminologyService(TerminologyService terminologyService) {
		this.terminologyService = terminologyService;
	}

	public ReportParameterConversionService getReportParameterConversionService() {
		return reportParameterConversionService;
	}

	public void setReportParameterConversionService(
			ReportParameterConversionService reportParameterConversionService) {
		this.reportParameterConversionService = reportParameterConversionService;
	}
	/**
	 * Search Scheduled reports
	 */
	public List<ScheduledReport> searchScheduledReports(ReportFilterSearchQueryInfo searchQueryInfo) throws ServiceException {
		// Initialize the list of return values
		List<ScheduledReport> scheduledReports = new ArrayList<ScheduledReport>();

		try {
			// First filter against the entered search criteria at thd DAO
			// level.
			List<ScheduledReport> rptItems = getReportDAO().searchScheduledReports(searchQueryInfo);

			// Then filter against Trigger.nextFireTime for the next fire time
			// since this is stored
			// in Quartz and not in the database
			if (rptItems != null && !rptItems.isEmpty()) {
				// Iterate through each returned ReportSetupLite object
				for (ScheduledReport scheduledReport : rptItems) {

					// skip over and return record if the setup user is null!
					// TOD Remove the true flag
					if (scheduledReport.getUserName() == null) {
						scheduledReports.add(scheduledReport);
					} else {
						// Get the name of the trigger
						String triggerName = SCHEDULED_REPORT + "." + scheduledReport.getUserName() + "."
								+ scheduledReport.getStandardReport().getCode() + "."
								+ scheduledReport.getEntityKey().getKeyValueAsString();
						try {
							// Get the scheduled trigger
							Trigger trigger = schedulingService.getTrigger(triggerName,
									BatchProcessTriggerGroups.DYNAMIC_BATCH_JOBS);

							// Get the next fire time
							Date nextFireTime = trigger != null ? trigger.getNextFireTime() : null;

							// Only add the report setup to the return list if
							// the date filter passed
							if ((nextFireTime == null)
									|| (searchQueryInfo.getDateFrom() == null || !nextFireTime
											.before(searchQueryInfo.getDateFrom()))
									&& (searchQueryInfo.getDateTo() == null || !nextFireTime
											.after(searchQueryInfo.getDateTo()))) {
								// Set the next fire time on the returned report
								// setup and add it to the returned list
								scheduledReport.setReportGenerateDate(nextFireTime);
								scheduledReports.add(scheduledReport);
							}
						} catch (SchedulerException scEx) {
							throw new ServiceException("Error getting trigger with Id: "
									+ triggerName, scEx);
						}
					}
				}
			}
		} catch (DAOException e) {
			throw new ServiceException("ScheduledReports Search Failed.  searchQueryInfo: "
					+ searchQueryInfo, e);
		}
		return scheduledReports;
	}
	
	public List<SimpleCompletedReport> searchCompletedReports(CompletedReportsSearchQueryInfo searchQueryInfo) throws ServiceException {

		try {
			return getReportDAO().searchCompletedReports(searchQueryInfo);
		} catch (DAOException e) {
			throw new ServiceException("CompletedReports Search Failed.  searchQueryInfo = "
					+ searchQueryInfo, e);
		}
	}

	public LinkedHashMap<String, String> getUniqueFilterValues(CompletedReportFieldType fieldType) {
		return getReportDAO().getUniqueFilterValues(fieldType);
	}
	
	
	private ReportDAO getReportDAO() {
		return reportDAO;
	}
	
	@Override
	public List<Date[]> getSchedulePreview(UserPrincipal user, ReportSetup reportSetup, int numDates, Date startingDate)
			throws ServiceException {
		// TODO Auto-generated method stub
		return null;
	}
}