package gov.va.med.ccht.service.htreports.impl;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ibm.icu.util.Calendar;

import gov.va.med.ccht.model.pssreport.CategoryOfCare;
import gov.va.med.ccht.model.pssreport.DistributionOfSurveysReport;
import gov.va.med.ccht.model.pssreport.SurveyTrendChartResult;
import gov.va.med.ccht.model.pssreport.VendorSubmissionReportData;
import gov.va.med.ccht.model.report.DeviceModality;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.persistent.DeviceModalityDAO;
import gov.va.med.ccht.persistent.PssDistributionReportDAO;
import gov.va.med.ccht.persistent.SatisfactionSurveyReportDAO;
import gov.va.med.ccht.persistent.SurveyTrendChartsDAO;
import gov.va.med.ccht.persistent.VendorSubmissionReportDAO;
import gov.va.med.ccht.service.htreports.PssReportService;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;

@Service
public class PssReportServiceImpl implements PssReportService, ReportConstants{
	private QuestionList bucket;
	@Autowired
	private VendorSubmissionReportDAO vendorSubmissionReportDAO;
	@Autowired
	private PssDistributionReportDAO pssDistributionReportDAO;
	@Autowired
	private SatisfactionSurveyReportDAO satisfactionSurveyReportDAO;
	@Autowired
	private DeviceModalityDAO deviceModalityDao;
	@Autowired
	private SurveyTrendChartsDAO surveyTrendChartsDAO;
	
	SatisfactionSurveyQuestion question = new SatisfactionSurveyQuestion();
	private FiscalData questionList = new FiscalData();
	
	public PssReportServiceImpl() {
	}
	
	public PssReportServiceImpl(VendorSubmissionReportDAO vendorSubmissionReportDAO,
			PssDistributionReportDAO pssDistributionReportDAO,
			SatisfactionSurveyReportDAO satisfactionSurveyReportDAO, DeviceModalityDAO deviceModalityDao,
			SurveyTrendChartsDAO surveyTrendChartsDAO) {
		this.vendorSubmissionReportDAO = vendorSubmissionReportDAO;
		this.pssDistributionReportDAO = pssDistributionReportDAO;
		this.satisfactionSurveyReportDAO = satisfactionSurveyReportDAO;
		this.deviceModalityDao = deviceModalityDao;
		this.surveyTrendChartsDAO = surveyTrendChartsDAO;
	}

	@Override
	public LinkedList<SurveyTrendChartResult> getSurveyTrendChartsReportData(PSSReportForm form) throws ParseException 
	{
		if(form.getDateTypeSelection() == QTR_ID){
			//If report is all visns return results as is
			if(form.getTypeButtonSelection() == QTR_ID && form.getVisnId().equals(ALL_VISN)){
				return surveyTrendChartsDAO.getSurveyTrendChartsReportData(form);
			}else{
				//otherwise group data by quarter
				return surveyTrendsQuarters(form, surveyTrendChartsDAO.getSurveyTrendChartsReportData(form));
			}
		}
		return surveyTrendChartsDAO.getSurveyTrendChartsReportData(form);
	};
	
	@Override
	public List<VendorSubmissionReportData> getVendorSubmissionReport() 
	{
		return vendorSubmissionReportDAO.getVendorSubmissionReport();
	}

	@Override
	public DistributionOfSurveysReport getDistributionNationalReport(PSSReportForm form) throws ParseException 
	{
		return pssDistributionReportDAO.getNationalReport(form);
	}
	
	@Override
	public DistributionOfSurveysReport getDistributionVisnReport(PSSReportForm form) throws ParseException 
	{
		return pssDistributionReportDAO.getVisnReport(form);
	}
	
	@Override
	public DistributionOfSurveysReport getDistributionFacilityReport(PSSReportForm form) throws ParseException 
	{
		return pssDistributionReportDAO.getFacilityReport(form);
	}
	
	@Override
	public List<CategoryOfCare> getCategoriesOfCare() {
		return satisfactionSurveyReportDAO.getCategoriesOfCare();
	}
	
	@Override
	public List<Date> getDatesByMonth() {
		return satisfactionSurveyReportDAO.getDatesByMonth();
	}
	
	@Override
	public ArrayList<String> getDatesByYears() throws ParseException {
		return satisfactionSurveyReportDAO.getDatesByYears();
	}
	
	@Override
	public List<SatisfactionSurveyQuestionResult> getSatisfactionSurveyReport(PSSReportForm form) throws NumberFormatException, DAOException, ServiceException, ParseException {
		if(form.getDateTypeSelection() == QTR_ID) {
			return satSurveyQuarters(form, satAllModalitiesCheck(form, satisfactionSurveyReportDAO.getSatisfactionSurveyReport(form)));
		} else if(form.getDateTypeSelection() == YEAR_ID) {
			return satSurveyYears(form, satAllModalitiesCheck(form, satisfactionSurveyReportDAO.getSatisfactionSurveyReport(form)));
		} else {
			return satAllModalitiesCheck(form, satisfactionSurveyReportDAO.getSatisfactionSurveyReport(form));
		}
	}

	@Override
	public List<DeviceModality> getAllDeviceModalities() {
		return deviceModalityDao.getAllDeviceModalities();
	}
	
	@Override
	public DeviceModality getModalityByName(String name) {
		return deviceModalityDao.getModalityByName(name);
	}
	
	@Override
	public DeviceModality getModalityById(Long id) {
		return deviceModalityDao.getModalityById(id);
	}
	
	//parse the fiscal year from user selection
	public String[] parseFiscalSelected(PSSReportForm form) throws ParseException {
		String[] dateArray = new String[TWO];
		//fiscal from date setup
		Calendar calFromDate = Calendar.getInstance();
		calFromDate.setTime(DateUtils.parseDate(form.getFiscalFromDate(), YEAR));
		calFromDate.set(Integer.parseInt("" + form.getFiscalFromDate()), OCT, ONE);
		calFromDate.roll(Calendar.YEAR, -1);
		dateArray[FROM] = DateUtils.format(calFromDate.getTime(), DateUtils.MMDDYYYY);
		//fiscal to date setup
		Calendar calToDate = Calendar.getInstance();
		calToDate.setTime(DateUtils.parseDate(form.getFiscalFromDate(), YEAR));
		calToDate.set(Integer.parseInt("" + form.getFiscalToDate()), SEPT, THIRTY);
		dateArray[TO] = DateUtils.format(calToDate.getTime(), DateUtils.MMDDYYYY);
		return dateArray;
	}
	
	
	//parse the quarter and year from user selection
	public String[] parseQuarterSelected(PSSReportForm form) throws ParseException {
		String[] dateArray = new String[TWO];
		//quarter from date setup
		String qFrom = "" + form.getQuarterFromDate().charAt(6);
		Calendar calFromDate = Calendar.getInstance();
		String fMonth = null;
		String fMonthYear = null;
		switch (qFrom) {
		case Q1:
			fMonth = Q1_START_STR;
			fMonthYear = fMonth + "/" + form.getQuarterFromDate().substring(ZERO, NUM_QTRS);
			break;
		case Q2: 
			fMonth = Q2_START_STR;
			fMonthYear = fMonth + "/" + form.getQuarterFromDate().substring(ZERO, NUM_QTRS);
			break;
		case Q3:
			fMonth = Q3_START_STR;
			fMonthYear = fMonth + "/" + form.getQuarterFromDate().substring(ZERO, NUM_QTRS);
			break;
		case Q4:
			fMonth = Q4_START_STR;
			fMonthYear = fMonth + "/" + form.getQuarterFromDate().substring(ZERO, NUM_QTRS);
			break;
		default:
			break;
	}
		calFromDate.setTime(DateUtils.parseDate(fMonthYear, "MM/yyyy"));
		if(qFrom.equals(Q1)) calFromDate.roll(Calendar.YEAR, YEAR_ROLLBACK);
		calFromDate.set(Calendar.DAY_OF_MONTH, calFromDate.getActualMinimum(Calendar.DAY_OF_MONTH));
		dateArray[FROM] = DateUtils.format(calFromDate.getTime(), DateUtils.MMDDYYYY);
		//quarter to date setup
		String qTo = "" + form.getQuarterToDate().charAt(6);
		Calendar calToDate = Calendar.getInstance();
		String tMonth = null;
		String tMonthYear = null;
		switch (qTo) {
		case Q1:
			tMonth = Q1_END_STR;
			tMonthYear = tMonth + "/" + form.getQuarterToDate().substring(ZERO, NUM_QTRS);
			break;
		case Q2: 
			tMonth = Q2_END_STR;
			tMonthYear = tMonth + "/" + form.getQuarterToDate().substring(ZERO, NUM_QTRS);
			break;
		case Q3:
			tMonth = Q3_END_STR;
			tMonthYear = tMonth + "/" + form.getQuarterToDate().substring(ZERO, NUM_QTRS);
			break;
		case Q4:
			tMonth = Q4_END_STR;
			tMonthYear = tMonth + "/" + form.getQuarterToDate().substring(ZERO, NUM_QTRS);
			break;
		default:
			break;
	}
		calToDate.setTime(DateUtils.parseDate(tMonthYear, "MM/yyyy"));
		if(qTo.equals(Q1)) calToDate.roll(Calendar.YEAR, YEAR_ROLLBACK);
		calToDate.set(Calendar.DAY_OF_MONTH, calToDate.getActualMaximum(Calendar.DAY_OF_MONTH));
		dateArray[TO] = DateUtils.format(calToDate.getTime(), DateUtils.MMDDYYYY);
		return dateArray;
	}
	
	@SuppressWarnings("deprecation")
	private ArrayList<String> yearParse(PSSReportForm form, List<SatisfactionSurveyQuestionResult> data) {
		ArrayList<String> years = new ArrayList<String>();
		Calendar dateFrom = Calendar.getInstance();
		dateFrom.set(ONE, data.get(0).getResults().get(0).getYear());
		dateFrom.roll(Calendar.YEAR, - ONE);
		Calendar dateTo = Calendar.getInstance();
		dateTo.set(ONE, data.get(0).getResults().get(data.get(0).getResults().size()-ONE).getYear());
		Date dateCnt = new Date();
		for(; dateFrom.equals(dateTo); dateFrom.roll(Calendar.YEAR, YEAR_ROLLUP)) {
			years.add("" + (dateFrom.getTime().getYear()));
		}
		return years;
	}
	
	
	
	private final class QuestionList {
		private int year;
		private double[] qrtValues;
		private double[] qrtStdDevValues;
		private double[] qrtMeanVal;
		private double[][] qrtAllMean;
		private int[][] qrtAllModCtr;
		int[] avgCount = new int[4];
		private Object[][] modalityQrtValues;
		private Object[][] modalityStdrDevQrtValues;
		private String questionText;
		private int questionNum;
		private int[] qrtCntr;
		private int[][] modCntr;
		
		private QuestionList() {
			qrtValues = new double[NUM_QTRS];
			year = ZERO;
			modalityQrtValues = new Object[NUM_QTRS][NUM_QTRS];
			qrtStdDevValues = new double[NUM_QTRS];
			qrtMeanVal = new double[NUM_QTRS];
			qrtAllMean = new double[NUM_QTRS][NUM_QTRS];
			modalityStdrDevQrtValues = new Object[NUM_QTRS][NUM_QTRS];
			qrtCntr = new int[NUM_QTRS];
			modCntr = new int[NUM_QTRS][NUM_QTRS];
			qrtAllModCtr = new int[NUM_QTRS][NUM_QTRS];
			for (int i = ZERO; i < qrtValues.length; i++) {
				qrtValues[i] = 0.0;
				qrtStdDevValues[i] = 0.0;
				qrtMeanVal[i] = 0.0;
				qrtCntr[i] = ZERO;
				avgCount[i] = ZERO;
			}
			for(int i = 0; i < modalityQrtValues.length; i++){
				for(int j = 0; j < modalityQrtValues[i].length; j++){
					modalityQrtValues[i][j]= 0.0;
					modalityStdrDevQrtValues[i][j]= 0.0;
					qrtAllMean[i][j] = 0.0;
					modCntr[i][j] = ZERO;
					qrtAllModCtr[i][j] = 0;
				}
				
			}
		}
	}
	
	/**
	 * always return the correct fiscal year based on given month and year
	 * @param month
	 * @param year
	 * @return
	 */
	private int getFiscalYear(int month, int year) {
		int fiscalYear = 0;
		if(month < FISCAL_START) {
			fiscalYear = year;
		} else {
			fiscalYear = year + YEAR_INCR;
		}
		return fiscalYear;
	}
	
	/**
	 * build a pair to control year counters
	 * @author DNS
	 *
	 * @param <A>
	 * @param <B>
	 */
	private class CntrSet<A, B, C, D> {
		private A cntYear;
		private B endYear;
		private C monthCounter;
		private D divCounter;
		private CntrSet(A first, B second, C third, D fourth) {
		    this.cntYear = first;
		    this.endYear = second;
		    this.monthCounter = third;
		    this.divCounter = fourth;
		}
	}
	
	/**
	 * build a pair to seperate values and mod values in a 
	 * list of objects
	 * @author DNS
	 *
	 * @param <A>
	 * @param <B>
	 */
	private class ValPair<A, B> {
		private A val;
		private B valArr;
		private ValPair(A first, B second) {
		    this.val = first;
		    this.valArr = second;
		}
	}
	
	/**
	 * object that houses the fiscal data
	 * @author DNS
	 *
	 */
	private final class FiscalData {
		private List<Integer> fiscalYearList;
		private List<ValPair<Double, double[]>> fiscalYearValues;
		private List<ValPair<Double, double[]>> fiscalSDValues;
		private List<ValPair<Double, double[]>> fiscalMeanValues;
		private String questionText;
		private int questionNum;
		private List<ValPair<Integer, int[]>> fiscalAnswerCnt;
		private List<int[]> modCntr;
		private int currentMonth;
		
		private FiscalData() {
			this.empty();
		}
		
		private void empty() {
			this.fiscalYearList = new ArrayList<Integer>();
			this.fiscalYearValues = new ArrayList<ValPair<Double, double[]>>();
			this.fiscalMeanValues = new ArrayList<ValPair<Double, double[]>>();
			this.fiscalSDValues = new ArrayList<ValPair<Double, double[]>>();
			this.questionText = "";
			this.fiscalAnswerCnt = new ArrayList<ValPair<Integer, int[]>>();
			this.questionNum = INT_INIT;
			this.modCntr = new ArrayList<int[]>();
			this.currentMonth = INT_INIT;
		}
	}
	
	/**
	 * method to seperate the modalities into buckets 
	 * that correspond to the proper mods
	 * @param form
	 * @param mon
	 * @return
	 */
	private void modDataParse(PSSReportForm form, SatisfactionSurveyQuestion mon, int yr) {
		switch (form.getModalityName()) {
		case BRWSR_ID:
			setFiscalValues(form, yr, BRWSR, mon);
			questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR]++;
			break;
		case DEVICE_ID:
			setFiscalValues(form, yr, H_DEVICE, mon);
			questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE]++;
			break;
		case IVR_ID:
			setFiscalValues(form, yr, I_V_R, mon);
			questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R]++;
			break;
		case PSS_ALL_MOD:
			switch (mon.getModality()) {
			case BRWSR_STR:
				setFiscalValues(form, yr, BRWSR, mon);
				questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR]++;
				break;
			case DEVICE_STR:
				setFiscalValues(form, yr, H_DEVICE, mon);
				questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE]++;
				break;
			case IVR_STR:
				setFiscalValues(form, yr, I_V_R, mon);
				questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R]++;
				break;
			case TOTALS_STR:
				setFiscalValues(form, yr, TOTALS, mon);
				questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[TOTALS]++;
				break;
			}
			break;
		default:
			break;
		}
	}
	/**
	 * during modality cycling used to inser avg and count values for individual
	 * modalities and to reduce code reuse
	 * @param form standard pssReport form
	 * @param yr fiscal year used for indexing
	 * @param mod current mod for proper value placement
	 * @param mon satisfactionSurveyQuestion that contains the values needed
	 */
	private void setFiscalValues(PSSReportForm form, int yr, int mod, SatisfactionSurveyQuestion mon) {
		questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod] += mon.getAvgAnswer();
		questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod] += mon.getStdDeviation();
		questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod] += mon.getMean();
		if(form.getModalityName().equals(PSS_ALL_MOD)) {
			questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod] += mon.getAnswerCount();
		} else {
			questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).val += mon.getAnswerCount();
		}
	}
	
	/**
	 * loop through and build year list to have a full range of the years
	 * for indexing and population of other fields
	 * @param yearCount
	 * @param yrCntr
	 */
	private void buildFiscalData(int yearCount, int yrCntr) {
		Integer cntr = yrCntr;
		for(int i = 0; i < yearCount; i++) {
			questionList.fiscalYearList.add(cntr);
			questionList.questionNum = INT_INIT;
			questionList.questionText = "";
			questionList.fiscalYearValues.add(new ValPair<Double, double[]>(DUB_INIT, new double[FOUR]));
			questionList.fiscalSDValues.add(new ValPair<Double, double[]>(DUB_INIT, new double[FOUR]));
			questionList.fiscalMeanValues.add(new ValPair<Double, double[]>(DUB_INIT, new double[FOUR]));
			questionList.fiscalAnswerCnt.add(new ValPair<Integer, int[]>(INT_INIT, new int[FOUR]));
			questionList.modCntr.add(new int[NUM_MODS]);
			cntr++;
		}
	}
	
	/**
	 * set the averages for fiscal year avgValues based on modality once they are combined
	 * @param yrCount number of years to loop through ensuring all fiscal values are averaged
	 * @param form standard pssReportForm
	 */
	private void setFiscalModAverages(PSSReportForm form, int yr) {
			switch (form.getModalityName()) {
			case PSS_ALL_MOD:
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[TOTALS] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[TOTALS];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[TOTALS] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[TOTALS];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[TOTALS] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[TOTALS];
				break;
			case BRWSR_ID:
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[BRWSR] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR];
				break;
			case DEVICE_ID:
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[H_DEVICE] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE];
				break;
			case IVR_ID:
				questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[I_V_R] /= questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R];
				break;
			default:
				break;
			}
	}
	
	/**
	 * set the averages for fiscal year avgValues once they are combined
	 * @param yrCount number of years to loop through ensuring all fiscal values are averaged
	 */
	private void setFiscalAverages(int yr, int divisor, int divCounter) {
		if(questionList.currentMonth <= FISCAL_END) {
			questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divisor;
			questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divisor;
			questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divisor;
		} else {
			questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divCounter;
			questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divCounter;
			questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).val /= divCounter;
		}
	}
	
	/**
	 * after parsing this method is used to distribute answer counts, modalities, and
	 * avgAnswer to the list of sat surv questions
	 * @param mod value that represents which modality is currently being set up
	 * @param form standard pssReportForm 
	 * @param yr fiscal year used to control indexing
	 */
	private void setQuestionModalityAndAvgs(int mod, PSSReportForm form, Integer yr) {
		switch (mod) {
		case BRWSR:
			if(form.getModalityName().equals(PSS_ALL_MOD)) {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
			} else {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).val);
			}
			setAllModQuestionValues(yr, mod);
			question.setModality(BRWSR_STR);
			break;
		case H_DEVICE:
			if(form.getModalityName().equals(PSS_ALL_MOD)) {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
			} else {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).val);
			}
			setAllModQuestionValues(yr, mod);
			question.setModality(DEVICE_STR);
			break;
		case I_V_R:
			if(form.getModalityName().equals(PSS_ALL_MOD)) {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
			} else {
				question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).val);
			}
			setAllModQuestionValues(yr, mod);
			question.setModality(IVR_STR);
			break;
		case TOTALS:
			question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
			setAllModQuestionValues(yr, mod);
			question.setModality(TOTALS_STR);
			break;
		default:
			break;
		}
	}
	
	private void setAllModQuestionValues(int yr, int mod) {
		question.setStdDeviation(questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
		question.setMean(questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
		question.setAvgAnswer(questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).valArr[mod]);
	}
	
	/**
	 * after data is parsed this method is used to build it into a satisfaction survey question object
	 * to later be added to a list
	 * @param form standard pssReportForm
	 * @param yr fiscal year for indexing
	 * @param qList FiscalData object housing the data needed to build the question
	 * @param i 
	 */
	private void setQuestionUp(PSSReportForm form, int yr, FiscalData qList, Integer mod) {
		question = new SatisfactionSurveyQuestion();
		question.setYear(yr);
		question.setHeaderString("FY" + yr + " Patient Satisfaction Index");
		question.setQuestionNumber(qList.questionNum);
		question.setQuestionText(qList.questionText);
		if(form.getModalityName().equals(PSS_MOD_DNI)) {
			question.setAnswerCount(questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(yr)).val);
			question.setAvgAnswer(questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(yr)).val);
			question.setStdDeviation(questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(yr)).val);
			question.setMean(questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(yr)).val);
			question.setModality(EMPTY_STRING);
		} else {
			setQuestionModalityAndAvgs(mod, form, yr);
		}
	}
	
	private List<SatisfactionSurveyQuestion> addQuestionOnYearUp(SatisfactionSurveyQuestion mon, CntrSet<Integer, Integer, Integer, Integer> counter, int yrCount, PSSReportForm form, List<SatisfactionSurveyQuestion> satSurvQuestList) {
		int yr = counter.cntYear;
		if(form.getModalityName().equals(PSS_MOD_DNI)) {
			setFiscalAverages(yr, counter.monthCounter, counter.divCounter);
    	} else {
    		setFiscalModAverages(form, yr);
    	}
		//if all modalities then create corresponding Satisfaction survey questions
		if(form.getModalityName().equals(PSS_ALL_MOD)) {
			boolean check = false;
			for(int i = 0; i < NUM_MODS; i++) {
				switch(i) {
				case BRWSR:
					if(!(questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[BRWSR] == ZERO)) {
						setQuestionUp(form, yr, questionList, BRWSR);
						check = true;
					}
					break;
				case H_DEVICE:
					if(!(questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[H_DEVICE] == ZERO)) {
						setQuestionUp(form, yr, questionList, H_DEVICE);
						check = true;
					}
					break;
				case I_V_R:
					if(!(questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[I_V_R] == ZERO)) {
						setQuestionUp(form, yr, questionList, I_V_R);
						check = true;
					}
					break;
				case TOTALS:
					if(!(questionList.modCntr.get(questionList.fiscalYearList.indexOf(yr))[TOTALS] == ZERO)) {
						setQuestionUp(form, yr, questionList, TOTALS);
						check = true;
					}
					break;
				default:
					break;
				}
			if (check) satSurvQuestList.add(question);
			check = false;
			}
		} else {
			switch (form.getModalityName()) {
			case BRWSR_ID:
				setQuestionUp(form, yr, questionList, BRWSR);
				break;
			case DEVICE_ID:
				setQuestionUp(form, yr, questionList, H_DEVICE);
				break;
			case IVR_ID:
				setQuestionUp(form, yr, questionList, I_V_R);
				break;
			//default is used for modality do not include
			default:
				setQuestionUp(form, yr, questionList, null);
				break;
			}
			satSurvQuestList.add(question);
		}
		return satSurvQuestList;
	}
	
	/**
	 * method used to parse the fiscal data from the queryResults gotten from the
	 * call to getSatisfactionSurveyReport(PSSReportForm form)
	 * @param form standard pssReportForm
	 * @param data query results
	 * @return a list of SatisfactionSurveyQuestionResults
	 */
	private List<SatisfactionSurveyQuestionResult> yearDataParse(PSSReportForm form, List<SatisfactionSurveyQuestionResult> data) {
		List<SatisfactionSurveyQuestion> satSurvQuestList = new ArrayList<SatisfactionSurveyQuestion>();
		List<SatisfactionSurveyQuestionResult> resultList = new ArrayList<SatisfactionSurveyQuestionResult>();
		SatisfactionSurveyQuestionResult result = null;
		//build and initialize counter pair
		CntrSet<Integer,Integer, Integer, Integer> counter = new CntrSet<Integer, Integer, Integer, Integer>(INT_INIT, INT_INIT, INT_INIT, INT_INIT);
	    
		//loop through given data and get a monthlist for each question in the survey
	    for(SatisfactionSurveyQuestionResult qResult : data) {
	    	//gets month list
	    	List<SatisfactionSurveyQuestion> monthList = qResult.getResults();
	    	result = new SatisfactionSurveyQuestionResult();
	    	result.setQuestionNumber(monthList.get(ZERO).getQuestionNumber());
	    	result.setQuestionText(monthList.get(ZERO).getQuestionText().substring(THREE));
	    	
			//reset year counter to use in the next inner loop
			//set the first fiscal year in the month list and the last
			counter.cntYear = getFiscalYear(monthList.get(ZERO).getMonthAsInt(), monthList.get(ZERO).getYear());
			counter.endYear = getFiscalYear(monthList.get(monthList.size() - YEAR_INCR).getMonthAsInt(), monthList.get(monthList.size() - YEAR_INCR).getYear());
			counter.monthCounter = INT_INIT;
			counter.divCounter = INT_INIT;
			int yrCount = (counter.endYear - counter.cntYear) + YEAR_INCR;
			//rebuild for the new question number
			//if you are beginning a new question
	    	buildFiscalData(yrCount, counter.cntYear);
	    	
	    	//adds question number and text before cycling through each month since they wont change
			questionList.questionNum = monthList.get(INT_INIT).getQuestionNumber();
			
			questionList.questionText = monthList.get(INT_INIT).getQuestionText().substring(THREE);
			Integer totalCount = INT_INIT;
			//loop through each month and assign values to the appropriate fields
			for(SatisfactionSurveyQuestion mon : monthList) {
				questionList.currentMonth = mon.getMonthAsInt();
				totalCount++;
				//on condition where fiscal year rolls over then
				//adds satSurvQuestion to the satSurvQuestion list and iterates counter
				if(getFiscalYear(mon.getMonthAsInt(), mon.getYear()) != counter.cntYear) {
					if(!(form.getModalityName().equals(PSS_ALL_MOD))) {	
						satSurvQuestList = addQuestionOnYearUp(mon, counter, yrCount, form, satSurvQuestList);
						counter.cntYear = getFiscalYear(mon.getMonthAsInt(), mon.getYear());
						counter.divCounter = INT_INIT;
					} else {
						if(mon.getModality().equals(TOTALS_STR)){
							satSurvQuestList = addQuestionOnYearUp(mon, counter, yrCount, form, satSurvQuestList);
							counter.cntYear = getFiscalYear(mon.getMonthAsInt(), mon.getYear());
						}
					}
					counter.monthCounter = 0;
				}
				counter.divCounter++;
				counter.monthCounter++;
				if(form.getModalityName().equals(PSS_MOD_DNI)) {
					questionList.fiscalAnswerCnt.get(questionList.fiscalYearList.indexOf(counter.cntYear)).val += mon.getAnswerCount();
					questionList.fiscalYearValues.get(questionList.fiscalYearList.indexOf(counter.cntYear)).val += mon.getAvgAnswer();
					questionList.fiscalSDValues.get(questionList.fiscalYearList.indexOf(counter.cntYear)).val += mon.getStdDeviation();
					questionList.fiscalMeanValues.get(questionList.fiscalYearList.indexOf(counter.cntYear)).val += mon.getMean();
				} else {
					modDataParse(form, mon, counter.cntYear);
				}
				if((monthList.get(monthList.size() - YEAR_INCR).getMonthAsInt() == mon.getMonthAsInt()) && (totalCount.equals(monthList.size()))) {
					satSurvQuestList = addQuestionOnYearUp(mon, counter, yrCount, form, satSurvQuestList);
					counter.divCounter = INT_INIT;
				}
			}
	    	result.setResults(satSurvQuestList);
	    	satSurvQuestList = new ArrayList<SatisfactionSurveyQuestion>();
	    	resultList.add(result);
	    	questionList.empty();
	    	}
	    
		return resultList;
	}
	
	
	public List<SatisfactionSurveyQuestionResult> satSurveyYears(PSSReportForm form, List<SatisfactionSurveyQuestionResult> results) {
		List<SatisfactionSurveyQuestionResult> yResults = new ArrayList<SatisfactionSurveyQuestionResult>();
		yResults = yearDataParse(form, results);
		if(yResults.isEmpty()){
			return results;
		}
		return yResults;
	}	
	
	private ArrayList<QuestionList> quarterParse(PSSReportForm form, List<SatisfactionSurveyQuestionResult> data) {
		ArrayList<QuestionList> questList = new ArrayList<QuestionList>();
		bucket = new QuestionList();
		for(SatisfactionSurveyQuestionResult qResult : data) {
			List<SatisfactionSurveyQuestion> monthList = qResult.getResults();
			bucket.questionNum = qResult.getQuestionNumber();
			bucket.questionText = qResult.getQuestionText().substring(THREE);
			if(monthList.get(ZERO).getMonthAsInt() < Q1_START) {
				bucket.year = monthList.get(ZERO).getYear();
			} else {
				bucket.year = monthList.get(ZERO).getYear() + ONE;
			}
			int cntYear = ZERO;
			if(monthList.get(ZERO).getMonthAsInt() < Q1_START){
				cntYear = monthList.get(ZERO).getYear() - 1;
			}else{
				cntYear = monthList.get(ZERO).getYear();
			}
			questList.ensureCapacity(monthList.size());
			int countQtr = ZERO;
			for(SatisfactionSurveyQuestion mon : monthList) {
				if(mon.getMonthAsInt() == 10 && mon.getYear() != cntYear) {
					questList.add(bucket);
					bucket = new QuestionList();
					bucket.questionNum = qResult.getQuestionNumber();
					bucket.questionText = qResult.getQuestionText().substring(THREE);
					bucket.year = mon.getYear() + ONE;
					cntYear = mon.getYear();
				}
				if(mon.getMonthAsInt() >= Q1_START){
					setBucketConditional( mon,  form,  ZERO); 
				}
				if(mon.getMonthAsInt() <= Q2_END){
					setBucketConditional( mon,  form,  ONE);
				}
				if(mon.getMonthAsInt() >= Q3_START && mon.getMonthAsInt() <= Q3_END){
					setBucketConditional( mon,  form,  TWO);
				}
				if(mon.getMonthAsInt() >= Q4_START && mon.getMonthAsInt() <= Q4_END){
					setBucketConditional( mon,  form,  THREE);
				}
				countQtr++;
				if(countQtr == monthList.size()) {
					questList.add(bucket);
					bucket = new QuestionList();
					
					
				}
			}
		}
		
		return questList;
	}
	
	public List<SatisfactionSurveyQuestionResult> satSurveyQuarters(PSSReportForm form, List<SatisfactionSurveyQuestionResult> results){
		List<SatisfactionSurveyQuestionResult> qResults = new ArrayList<SatisfactionSurveyQuestionResult>();
		ArrayList<QuestionList> questList = quarterParse(form, results);
		SatisfactionSurveyQuestionResult row = new SatisfactionSurveyQuestionResult();
		List<SatisfactionSurveyQuestion> rowResult = new ArrayList<SatisfactionSurveyQuestion>();
		if(questList.isEmpty()){
			return results;
		}
		int questionCtr = questList.get(ZERO).questionNum;
		int entryCtr = ZERO;
		
		for(QuestionList entry: questList) {
			entryCtr++;
			if(questionCtr != entry.questionNum){
				row.setResults(rowResult);
				qResults.add(row);
				rowResult = new ArrayList<SatisfactionSurveyQuestion>();
				row = new SatisfactionSurveyQuestionResult();
				questionCtr = entry.questionNum;
			}
			
			if(form.getModalityName().equals("-2")){
				row.setQuestionNumber(entry.questionNum);
				row.setQuestionText(entry.questionText);
				for(int i = 0; i < NUM_QTRS; i++){
					SatisfactionSurveyQuestion q = new SatisfactionSurveyQuestion();
					q.setAnswerCount(entry.avgCount[i]);
					
					if(form.getReportVersionButtonSelection() == 1) {
						q.setMean(((double)entry.qrtMeanVal[i]/entry.qrtCntr[i]));
					} else {
						q.setMean(((double)entry.qrtMeanVal[i]/entry.qrtCntr[i]));
					}
					q.setAvgAnswer(entry.qrtValues[i]/entry.qrtCntr[i]);
					q.setStdDeviation(entry.qrtStdDevValues[i]/entry.qrtCntr[i]);
					q.setHeaderString("Q" + (i + ONE) + " FY" + entry.year + " Patient Satisfaction Index");
					q.setQuestionNumber(entry.questionNum);
					q.setQuestionText(entry.questionText);
					q.setQuarterAsInt(i + ONE);
					q.setQuarterAsString("Q" +  (i + ONE) + " FY" + entry.year);
					q.setModality("");
					q.setYear(entry.year);
					if(q.getAnswerCount() != ZERO) rowResult.add(q);
				}
			} else if(form.getModalityName().equals("-1")){
				row.setQuestionNumber(entry.questionNum);
				row.setQuestionText(entry.questionText);
				for(int i = 0; i < NUM_QTRS; i++){
					for(int j = 0; j < NUM_QTRS; j++){
						SatisfactionSurveyQuestion q = new SatisfactionSurveyQuestion();
						q.setAnswerCount(entry.modCntr[i][j]);
						q.setStdDeviation((double)entry.modalityStdrDevQrtValues[i][j]/entry.qrtAllModCtr[i][j]);
						q.setAvgAnswer((double)entry.modalityQrtValues[i][j]/entry.qrtAllModCtr[i][j]);
						if(form.getReportVersionButtonSelection() == 1) {
							q.setMean(((double)entry.qrtAllMean[i][j]/entry.qrtAllModCtr[i][j]));
						} else {
							q.setMean(((double)entry.qrtAllMean[i][j]/entry.qrtAllModCtr[i][j]));
						}
						q.setHeaderString("Q" + (i +1) + " FY" + entry.year + " Patient Satisfaction Index");
						q.setQuestionNumber(entry.questionNum);
						q.setQuestionText(entry.questionText);
						q.setQuarterAsInt(i+1);
						q.setQuarterAsString("Q" +  (i +1) + " FY" + entry.year);
						switch (j) {
							case BRWSR:  
								q.setModality("Browser");
								break;
							case H_DEVICE: 
								q.setModality("Home Device");
								break;
							case I_V_R:
								q.setModality("IVR");
								break;
							default:
								q.setModality("Total");
								break;
						}
						q.setYear(entry.year);
						if(q.getAnswerCount() != ZERO)	rowResult.add(q);
					}
				}
			} else {
				row.setQuestionNumber(entry.questionNum);
				row.setQuestionText(entry.questionText);
				for(int i = 0; i < FOUR; i++){
					SatisfactionSurveyQuestion q = new SatisfactionSurveyQuestion();
					q.setAnswerCount(entry.avgCount[i]);
					int pos = 0;
					switch (form.getModalityName()) {
					case BRWSR_ID:  
						pos = BRWSR;
						q.setModality(BRWSR_STR);
						break;
					case DEVICE_ID: 
						pos = H_DEVICE;
						q.setModality(DEVICE_STR);
						break;
					case IVR_ID:
						pos = I_V_R;
						q.setModality(IVR_STR);
						break;
					default:
						break;
					}
					q.setStdDeviation((double)entry.modalityStdrDevQrtValues[i][pos]/entry.qrtCntr[i]);
					q.setAvgAnswer((double)entry.modalityQrtValues[i][pos]/entry.qrtCntr[i]);
					if(form.getReportVersionButtonSelection() == 1) {
						q.setMean(((double)entry.qrtMeanVal[i]/entry.qrtCntr[i]));
					} else {
						q.setMean(((double)entry.qrtMeanVal[i]/entry.qrtCntr[i]));
					}
					q.setHeaderString("Q" + (i +1) + " FY" + entry.year + " Patient Satisfaction Index");
					q.setQuestionNumber(entry.questionNum);
					q.setQuestionText(entry.questionText);
					q.setQuarterAsInt(i + ONE); //one added to correct indexing
					q.setQuarterAsString("Q" +  (i + ONE) + " FY" + entry.year);
					q.setYear(entry.year);
					if(q.getAnswerCount() != ZERO) rowResult.add(q);
				}
			}
			if(entryCtr == questList.size()){
				row.setResults(rowResult);
				qResults.add(row);
			}
		}
		return qResults;
	}

	private LinkedList<SurveyTrendChartResult> surveyTrendsQuarters(PSSReportForm form, LinkedList<SurveyTrendChartResult> results){
		LinkedList<SurveyTrendChartResult> quarterResults = new LinkedList<SurveyTrendChartResult>();
		Object[][] temp = new Object[NUM_QTRS][TWO];
		for(Object[] entry: temp){
			entry[ZERO] ="";
			entry[ONE] = 0;
		}
		int month = 0;
		for(SurveyTrendChartResult entry: results){
			month = entry.getMonthNum();
			if(month >= Q1_START){
				temp[ZERO][ZERO] = "Q1";
				temp[ZERO][ONE] = (int)temp[ZERO][ONE] + entry.getNumOfSurveys();
			}
			if(month <= Q2_END){
				temp[ONE][ZERO] = "Q2";
				temp[ONE][ONE] = (int)temp[ONE][ONE] + entry.getNumOfSurveys();
			}
			if(month >= Q3_START && month <= Q3_END){
				temp[TWO][ZERO] = "Q3";
				temp[TWO][ONE] = (int)temp[TWO][ONE] + entry.getNumOfSurveys();
			}
			if(month >= Q4_START && month <= Q4_END){
				temp[THREE][ZERO] = "Q4";
				temp[THREE][ONE] = (int)temp[THREE][ONE] + entry.getNumOfSurveys();
			}
		}
		
		for(Object[] entry: temp){
			SurveyTrendChartResult row = new SurveyTrendChartResult();
			row.setQuarter(entry[ZERO].toString());
			row.setNumOfSurveys(Integer.parseInt(entry[ONE].toString()));
			if(row.getNumOfSurveys() != ZERO) quarterResults.add(row);
		}
		return quarterResults;
	}
	
	private void setBucket(int num, int mod, SatisfactionSurveyQuestion mon, PSSReportForm form) {
		bucket.modalityQrtValues[num][mod] = (double)bucket.modalityQrtValues[num][mod] + mon.getAvgAnswer();
		bucket.modalityStdrDevQrtValues[num][mod] = (double)bucket.modalityStdrDevQrtValues[num][mod] + mon.getStdDeviation();
		if(form.getModalityName().equals("-1")) {
			bucket.modCntr[num][mod] = bucket.modCntr[num][mod] + mon.getAnswerCount();
			bucket.qrtAllMean[num][mod] = (double)bucket.qrtAllMean[num][mod] + mon.getMean();
			bucket.qrtAllModCtr[num][mod]++;
		} else {
			bucket.avgCount[num] += mon.getAnswerCount();
			bucket.qrtMeanVal[num] += mon.getMean();
			bucket.qrtCntr[num]++;
		}
	}
	
	private void setBucketConditional(SatisfactionSurveyQuestion mon, PSSReportForm form, int num) {
		if (mon.getModality().equals("")){
			bucket.qrtValues[num] = bucket.qrtValues[num] + mon.getAvgAnswer();
			bucket.qrtStdDevValues[num] = bucket.qrtStdDevValues[num] + mon.getStdDeviation();
			bucket.qrtMeanVal[num] = bucket.qrtMeanVal[num] + mon.getMean();
			bucket.avgCount[num] = bucket.avgCount[num] + mon.getAnswerCount();
			bucket.qrtCntr[num]++;
		}else{
			switch (mon.getModality()) {
			case "Browser":  
				setBucket(num, BRWSR, mon, form);
				break;
			case "Home Device": 
				setBucket(num, H_DEVICE, mon, form);
				break;
			case "IVR":
				setBucket(num, I_V_R, mon, form);
				break;
			case "Total":
				setBucket(num, TOTALS, mon, form);
				break;
			default:
				break;
			}
		}
	}
	

	private List<SatisfactionSurveyQuestionResult> satAllModalitiesCheck(PSSReportForm form, List<SatisfactionSurveyQuestionResult> data) {
		if(!form.getModalityName().equals(PSS_ALL_MOD)){
			return data;
		}
		List<SatisfactionSurveyQuestionResult> resultList = new ArrayList<SatisfactionSurveyQuestionResult>();
		SatisfactionSurveyQuestionResult result = null;
	    
		//loop through given data and get a monthlist for each question in the survey
	    for(SatisfactionSurveyQuestionResult qResult : data) {
	    	//gets month list
	    	List<SatisfactionSurveyQuestion> monthList = qResult.getResults();
	    	result = new SatisfactionSurveyQuestionResult();
	    	result.setQuestionNumber(monthList.get(ZERO).getQuestionNumber());
	    	result.setQuestionText(monthList.get(ZERO).getQuestionText().substring(THREE));
	    		    	
	    	int currentMonth = monthList.get(ZERO).getMonthAsInt();
	    	SatisfactionSurveyQuestion totalRow = new SatisfactionSurveyQuestion();
			totalRow.setMonthAsInt(monthList.get(ZERO).getMonthAsInt());
			totalRow.setYear(monthList.get(ZERO).getYear());
			totalRow.setModality(TOTALS_STR);
			totalRow.setMonthAsString(monthList.get(ZERO).getMonthAsString());
	    	int answerCount = ZERO;
	    	int loopCounter = ZERO;
	    	
	    	List<Integer> surveysPerModPerMon = new ArrayList<Integer>();
	    	List<Double> avgsPerModPerMon = new ArrayList<Double>();
	    	List<Double> stdDevPerModPerMon = new ArrayList<Double>();
	    	List<Double> meanPerModPerMon = new ArrayList<Double>();
	    	
			for(SatisfactionSurveyQuestion mon : monthList) {
				loopCounter++;
				if(currentMonth != mon.getMonthAsInt()){
					
					Double avg = 0.0;
			    	Double mean = 0.0;
			    	Double stdDev = 0.0;
					for(int i = 0; i < surveysPerModPerMon.size(); i++) {

						// Calculate a weighted average
						avg += avgsPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
						stdDev += stdDevPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
						mean += meanPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
					}
					
					totalRow.setAnswerCount(answerCount);
					totalRow.setAvgAnswer(avg);
					totalRow.setMean(mean);
					totalRow.setStdDeviation(stdDev);
					
					result.addToResultsList(totalRow);
			    	surveysPerModPerMon = new ArrayList<Integer>();
			    	avgsPerModPerMon = new ArrayList<Double>();
			    	stdDevPerModPerMon = new ArrayList<Double>();
			    	meanPerModPerMon = new ArrayList<Double>();
			    	answerCount = ZERO;
					totalRow = new SatisfactionSurveyQuestion();
					totalRow.setMonthAsInt(mon.getMonthAsInt());
					totalRow.setYear(mon.getYear());
					totalRow.setModality(TOTALS_STR);
					totalRow.setMonthAsString(mon.getMonthAsString());
					currentMonth = mon.getMonthAsInt();
					
				}
				result.addToResultsList(mon);
				answerCount += mon.getAnswerCount();
				
				surveysPerModPerMon.add(mon.getAnswerCount());
				avgsPerModPerMon.add(mon.getAvgAnswer());
				stdDevPerModPerMon.add(mon.getStdDeviation());
				meanPerModPerMon.add(mon.getMean());
				
				if(loopCounter == monthList.size()){
					
					Double avg = 0.0;
			    	Double mean = 0.0;
			    	Double stdDev = 0.0;
					for(int i = 0; i < surveysPerModPerMon.size(); i++) {

						// Calculate a weighted average
						avg += avgsPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
						stdDev += stdDevPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
						mean += meanPerModPerMon.get(i)*(surveysPerModPerMon.get(i)/(double)answerCount);
					}
					
					totalRow.setAnswerCount(answerCount);
					totalRow.setAvgAnswer(avg);
					totalRow.setMean(mean);
					totalRow.setStdDeviation(stdDev);
					
					result.addToResultsList(totalRow);
				}	
	    	}
			resultList.add(result);
	    }
		return resultList;
	}
	

	public List<SatisfactionSurveyQuestionResult> getAllModalitiesGraphData(List<SatisfactionSurveyQuestionResult> data) {
		List<SatisfactionSurveyQuestionResult> graphData = new ArrayList<SatisfactionSurveyQuestionResult>();
		SatisfactionSurveyQuestionResult result = null;
	    
		//loop through given data and get a monthlist for each question in the survey
	    for(SatisfactionSurveyQuestionResult qResult : data) {
	    	List<SatisfactionSurveyQuestion> monthList = qResult.getResults();
	    	result = new SatisfactionSurveyQuestionResult();
	    	result.setQuestionNumber(monthList.get(ZERO).getQuestionNumber());
	    	result.setQuestionText(monthList.get(ZERO).getQuestionText().substring(THREE));
	    	
			for(SatisfactionSurveyQuestion mon : monthList) {
				if(mon.getModality().equals(TOTALS_STR)) {
					result.addToResultsList(mon);
				}
			}
			graphData.add(result);
	    }
		return graphData;
	}
	
}
