package gov.va.med.ccht.persistent.hibernate;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.pssreport.SurveyTrendChartResult;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.persistent.SurveyTrendChartsDAO;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.service.htreports.PssReportService;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.util.DateUtils;

/**
 * 
 * @author DNS
 *
 */

@Repository
@Transactional
public class SurveyTrendChartsDAOImpl implements SurveyTrendChartsDAO, ReportConstants,
	CensusActivityReportsConstants {

	private static final int MONTH = 0;
	private static final int VISN = 0;
	private static final int NUM_OF_SURVEYS = 1;
	private static final int MONTH_NUM = 2;
	
	public static final int NATIONAL_SELECTION = 1;
	public static final int VISN_SELECTION = 2;
	public static final int FACILITY_SELECTION = 3;
	
	@Autowired
	private ReportsDAO reportsDao;
	
	@Autowired
	private PssReportService pssReportService;
	
	@Autowired
	private SessionFactory sessionFactory;

	public SurveyTrendChartsDAOImpl() {
	}
	
	public SurveyTrendChartsDAOImpl(SessionFactory factory, ReportsDAO reportsDao) {
		sessionFactory = factory;
		this.reportsDao = reportsDao;
	}

	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@Override
	public LinkedList<SurveyTrendChartResult> getSurveyTrendChartsReportData(PSSReportForm form) throws ParseException {
		
		Query query = null;
		
		switch (form.getTypeButtonSelection()) {
		
		case  NATIONAL_SELECTION:
			if((form.getDateTypeSelection() == ONE) || (form.getDateTypeSelection() == TWO)) {
				query = getNationalReportQuery(form);
			} else {
				query = getNationalReportQueryByYear(form);
			}
			break;
		case VISN_SELECTION:
			if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) {
				query = getAllVisnReportQuery(form);
			}
			else {
				if((form.getDateTypeSelection() == ONE) || (form.getDateTypeSelection() == TWO)) {
					query = getVisnReportQuery(form);
				} else {
					query = getVisnReportQuerybyYear(form);
				}
				// if query is for a single visn sets query parameter for that visn
				query.setParameter("visnId", form.getVisnId());
			}
			break;
		case FACILITY_SELECTION:
			if(form.getFacilityId().equalsIgnoreCase(ALL_FACILITIES)) {
				if((form.getDateTypeSelection() == ONE) || (form.getDateTypeSelection() == TWO)) {
					query = getAllFacilityReport(form);
				} else {
					query = getNationalReportQueryByYear(form);
				}
			}
			else {
				if((form.getDateTypeSelection() == ONE) || (form.getDateTypeSelection() == TWO)) {
					query = getFacilityReportQuery(form);
				} else {
					query = getFacilityReportQueryByYear(form);
				}
				// if query is for a single facility sets query parameter for that facility
				query.setParameter("facilityId", form.getFacilityId());
			}
			break;
		default:
			return new LinkedList<SurveyTrendChartResult>();
		}
		
		//fortify Fix, Null Dereference
		if(query == null) {
			return new LinkedList<SurveyTrendChartResult>();
		}
		
		//fortify Fix, Null Dereference
		if((form.getShortFormatReportFromDate() != null && form.getShortFormatReportToDate() != null) ||
				(form.getQuarterFromDate() != null && form.getQuarterToDate() != null) ||
				(form.getFiscalFromDate() != null && form.getFiscalToDate() != null)) {
			addDateParams(query, form);
		} 
		else {
			// return empty list to avoid errors, will result in empty report.
			return new LinkedList<SurveyTrendChartResult>();
		}
		
		List<Object[]> reportData = query.getResultList();
		
		LinkedList <SurveyTrendChartResult> results = new LinkedList<SurveyTrendChartResult>();
		
		if(form.getDateTypeSelection() == THREE && (!(form.getTypeButtonSelection() == VISN_SELECTION && form.getVisnId().equals(ALL_VISNS)))) {
			//if the report is run by fiscal year & the report is not being run for all visns parse the data by fiscal year
			results = yearParse(form, reportData); 
		}else {
			//If report is being run for All visns the data returned is each visn 
			// & the number of surveys over the selected time period
			if(form.getTypeButtonSelection() == VISN_SELECTION && form.getVisnId().equals(ALL_VISNS)) {
				for(Object[] entry : reportData) {
					SurveyTrendChartResult row = new SurveyTrendChartResult();
					row.setVisn(entry[VISN].toString());
					row.setNumOfSurveys(Integer.parseInt(entry[NUM_OF_SURVEYS].toString()));
					results.add(row);
				}
			}
			else{
				for(Object[] entry : reportData) {
					SurveyTrendChartResult row = new SurveyTrendChartResult();
					row.setMonth(entry[MONTH].toString());
					row.setNumOfSurveys(Integer.parseInt(entry[NUM_OF_SURVEYS].toString()));
					row.setMonthNum((int)entry[MONTH_NUM]);
					results.add(row);
				}
			}
		}
		return results;
	}
	
	private LinkedList <SurveyTrendChartResult> yearParse(PSSReportForm form, List<Object[]> data) {
		LinkedList <SurveyTrendChartResult> temp = new LinkedList<SurveyTrendChartResult>();
		Calendar dateFrom = Calendar.getInstance();
		dateFrom.setTime(form.getShortFormatReportFromDate());
		Calendar dateTo = Calendar.getInstance();
		dateTo.setTime(form.getShortFormatReportToDate());
		//added to days so that the last year in the list will be included as it compares the from date (10/01/yyyy) to (10/02/yyyy)
		dateTo.add(Calendar.DATE, 2);
		List<String> years = new ArrayList<String>();
		Date dateCounter = new Date(); dateCounter = form.getShortFormatReportFromDate();
		for(dateCounter = dateFrom.getTime(); dateFrom.before(dateTo); dateFrom.add(Calendar.YEAR, 1), dateCounter = dateFrom.getTime()) {
			years.add(DateUtils.format(dateCounter, DateUtils.YYYY));
			}
		Calendar divider = Calendar.getInstance();
		Calendar current = Calendar.getInstance();
		int mm;
		String yyyy;
		int surveyCount=0;
		Object[][] surveyCountBucket = new Object[years.size()][TWO];
		for(Object[] entry: surveyCountBucket){
			entry[ZERO] ="";
			entry[ONE] = 0;
		}
		for(Object[] entry : data) {
			mm= (int) entry[ZERO];
			yyyy = entry[ONE].toString();
			surveyCount = (int) entry[TWO];
			int pos = 0;
			for(int i=0;i<years.size();i++){
		      if(years.get(i).indexOf(yyyy)!=-1){
		    	  pos = i;
		    	  break;
		      }
			}
			if (mm < 10){
				surveyCountBucket[pos-1][ONE] = (int)surveyCountBucket[pos-1][ONE] + surveyCount;
				surveyCountBucket[pos-1][ZERO] = yyyy;
			}
			if (mm >= 10){
				surveyCountBucket[pos][ONE] = (int)surveyCountBucket[pos][ONE] + surveyCount;
				surveyCountBucket[pos][ZERO] = (Integer.parseInt(yyyy)+1 +"");
			}

			
		}
		for(Object[] entry : surveyCountBucket) {
			SurveyTrendChartResult row = new SurveyTrendChartResult();
			String year = entry[ZERO].toString();
			if(year.equals(""))break;
			row.setYear("FY" + year);
			row.setNumOfSurveys(Integer.parseInt(entry[ONE].toString()));
			temp.add(row);
		}
		return temp;
	}
	
	
	
	
	private String getCommonCOCJoin(){
		StringBuilder sql = new StringBuilder();
		sql.append("INNER JOIN (select cp.patient_id, cp.level_of_care from censuspatient cp ");
		sql.append("inner join (select distinct MAX(census_id) as censusid, patient_id, disenrollment_date ");
		sql.append("from censuspatient where disenrollment_date is null group by patient_id, disenrollment_date) as lcpe ");
		sql.append("on cp.census_id = lcpe.censusid and cp.patient_id = lcpe.patient_id) f ");
		sql.append("on b.patient_id = f.patient_id ");
		return sql.toString();
	}
	
	/**
	* Creates query for National totals
	 * @param form
	 * @return query for national totals
	 */
	private Query getNationalReportQuery(PSSReportForm form) {

		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date) as [MonthNum] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include VR-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by convert(char(3), received_date),datepart(month, received_date) ");
		sql.append("order by MonthNum");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	* Creates query for National totals by year query
	* @param form
	* @return query for national totals
	*/
	private Query getNationalReportQueryByYear(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT DATEPART(mm, received_date) AS Mon, DATEPART(yy, received_date) AS Yr, ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include VR-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by DATEPART(mm, received_date), DATEPART(yy, received_date) ");
		sql.append("order by Yr, Mon");
		return getSession().createNativeQuery(sql.toString());
	}
	
	
	
	 /**
	 * Creates query for All VISNS
     * which returns results by visn instead of by month
	 * @param form
	 * @return query for all visns by visn instead of month
	 */
	private Query getAllVisnReportQuery(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		
		sql.append("SELECT 'Visn' + convert(varchar(2), b.visn_id) ");
		sql.append("as [VISN], ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by b.visn_id  ");
		sql.append("order by b.visn_id");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single visn   
	 * @param form
	 * @return query for single visn by month
	 */
	private Query getVisnReportQuery(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();	
		
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date) as [MonthNum] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and b.visn_id= :visnId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by convert(char(3), received_date),datepart(month, received_date), b.visn_id ");
		sql.append("order by MonthNum, b.visn_id ");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single visn by year  
	 * @param form
	 * @return query for single visn by month
	 */
	private Query getVisnReportQuerybyYear(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();		
		sql.append("SELECT DATEPART(mm, received_date) AS Mon, DATEPART(yy, received_date) AS Yr, ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and b.visn_id= :visnId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by DATEPART(mm, received_date), DATEPART(yy, received_date), b.visn_id ");
		sql.append("order by Yr, Mon, b.visn_id");		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for All Facilities
	 * @param form
	 * @return query for all facilities by month
	 */
	private Query getAllFacilityReport(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date)  as [MonthNum] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("GROUP BY CONVERT(char(3), a.Received_Date),DATEPART(month, a.Received_Date) ");
		sql.append("order by MonthNum");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single facility   
	 * @param form
	 * @return query for single facility by month
	 */
	private Query getFacilityReportQuery(PSSReportForm form) {

		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date) as [MonthNum] ");
		sql.append("from Surveyed_Activity_Response a, facilities c, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and c.id = b.Facility_id and b.Facility_id= :facilityId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("GROUP BY CONVERT(char(3), a.Received_Date),");
		sql.append("DATEPART(month, a.Received_Date), c.Facility_Name ");
		sql.append("order by MonthNum");	
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single facility by year  
	 * @param form
	 * @return query for single facility by month
	 */
	private Query getFacilityReportQueryByYear(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT DATEPART(mm, received_date) AS Mon, DATEPART(yy, received_date) AS Yr, ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, facilities c, surveyed_activity b ");
		sql.append(getCommonCOCJoin());
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and c.id = b.Facility_id and b.Facility_id= :facilityId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append(reportsDao.getCategoryOfCare(form));
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by DATEPART(mm, received_date), DATEPART(yy, received_date) , c.Facility_Name ");
		sql.append("order by Yr, Mon");
		return getSession().createNativeQuery(sql.toString());
	}
	
	private Query addDateParams(Query query, PSSReportForm form) throws ParseException{
		switch (form.getDateTypeSelection()) {
		case 2:
			String[] quarterDate = pssReportService.parseQuarterSelected(form);
			form.setShortFormatReportFromDate(DateUtils.parseDate(quarterDate[0], DateUtils.MMDDYYYY));
			form.setShortFormatReportToDate(DateUtils.parseDate(quarterDate[1], DateUtils.MMDDYYYY));
			break;
		case 3:
			String[] fiscalDate = pssReportService.parseFiscalSelected(form);
			form.setShortFormatReportFromDate(DateUtils.parseDate(fiscalDate[0], DateUtils.MMDDYYYY));
			form.setShortFormatReportToDate(DateUtils.parseDate(fiscalDate[1], DateUtils.MMDDYYYY));
			break;
		default:
			break;
		}
		query.setParameter("fromDate", form.getShortFormatReportFromDate());
		
		//Calendar defaults to the 1st of the month so adding an additional month minus 1 day
		//to include the entire month in the report and setting the time to the last millisecond
		//of the last day in the month
		Calendar toDate = Calendar.getInstance();
		toDate.setTime(form.getShortFormatReportToDate());

		toDate.set(Calendar.MILLISECOND, 999);
		toDate.set(Calendar.SECOND, 59);
		toDate.set(Calendar.MINUTE, 59);
		toDate.set(Calendar.HOUR_OF_DAY, 23);
		if(form.getDateTypeSelection() == 1){
			toDate.add(Calendar.MONTH, 1);
			toDate.add(Calendar.DATE, -1);
		}
		
		query.setParameter("toDate", toDate.getTime());
		
		return query;
	}
}
