package gov.va.med.ccht.persistent.hibernate;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.IsoFields;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.pssreport.CategoryOfCare;
import gov.va.med.ccht.model.report.DeviceModality;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.persistent.DeviceModalityDAO;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.persistent.SatisfactionSurveyReportDAO;
import gov.va.med.ccht.persistent.VendorDAO;
import gov.va.med.ccht.persistent.utils.PSSReportUtils;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.service.htreports.PssReportService;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;

@Repository
@Transactional
public class SatisfactionSurveyReportDAOImpl  implements SatisfactionSurveyReportDAO, 
	CensusActivityReportsConstants {
	
	private String selectedModality = null;
	
	@Autowired
	private ReportsDAO reportsDao;
	
	@Autowired
	private PssReportService pssReportService;
	
	@Autowired
	private VendorDAO vendorDao;
	
	@Autowired
	private DeviceModalityDAO deviceModalityDao;
	
	@Autowired
	private SessionFactory sessionFactory;

	public SatisfactionSurveyReportDAOImpl() {
	}
	
	public SatisfactionSurveyReportDAOImpl(SessionFactory factory, VendorDAO vd, DeviceModalityDAO dm, ReportsDAO reportsDao) {
		sessionFactory = factory;
		vendorDao = vd;
		deviceModalityDao = dm;
		this.reportsDao = reportsDao;
	}

	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@Override
	public List<CategoryOfCare> getCategoriesOfCare() {
		String sql = "SELECT * FROM LevelOfCare WHERE Category_Name != 'Unknown'";
		return getSession()
				.createNativeQuery(sql, CategoryOfCare.class)
				.getResultList();
	}
	
	@SuppressWarnings("deprecation")
	@Override
	public List<Date> getDatesByMonth() {
		// TODO: Connect to the database for the dates if necessary.
		// For now, the business logic always starts in January of 2006.

		Calendar lastDate = Calendar.getInstance();
		lastDate.set(2006, Calendar.JANUARY, 1);
		List<Date> datesList = new ArrayList<Date>();
		Calendar current = Calendar.getInstance();
		Date today = Calendar.getInstance().getTime();
		current.set(today.getYear() + 1900, today.getMonth(), 1);
		while(current.getTime().after(lastDate.getTime())) {
			datesList.add(current.getTime());
			current.add(Calendar.MONTH, -1);
		}
		datesList.add(current.getTime()); // add last month
		return datesList;
	}
	
	@Override
	public ArrayList<String> getDatesByYears() throws ParseException {
		
		List<Date> months = getDatesByMonth();
		ArrayList<String> yearList = new ArrayList<String>();
		SimpleDateFormat df = new SimpleDateFormat("yyyy");
		for(Date byMonth : months) {
			String iter = df.format(byMonth);
			if(!yearList.contains(iter)) {
				yearList.add(iter);
			}
		}
		return yearList;
	}
	

	@Override
	public List<SatisfactionSurveyQuestionResult> getSatisfactionSurveyReport(
			PSSReportForm form) throws DAOException, NumberFormatException, ServiceException, ParseException {

		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		Query query = null;
		selectedModality = form.getModalityName();
		
		switch (form.getTypeButtonSelection())
		{
			case 1:
				query = getNationalSatisfactionSurveyReport(form);
				break;
			case 2:
				if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) {
					// Same query as national
					query = getNationalSatisfactionSurveyReport(form);
				}
				else {
					query = getSatisfactionSurveyReportByVisn(form);
				}
				break;
			case 3:
				if(form.getFacilityId().equalsIgnoreCase(ALL_FACILITIES)) {
					// Same query as national
					query = getNationalSatisfactionSurveyReport(form);
				}
				else {
					query = getSatisfactionSurveyReportByFacilities(form);
				}
				break;
		}

		try
		{
			if(query == null) { // if null, expected value was not selected, return empty list.
				return new ArrayList<SatisfactionSurveyQuestionResult>();
			}
			
			List<?> report = query.getResultList();

			int numIterations = 0;
			if(report != null) {
				numIterations = report.size();
			}

			satisfactionSurveyQuestionResults = new ArrayList<SatisfactionSurveyQuestionResult>();

			Map<Integer, SatisfactionSurveyQuestionResult> resultsByQuestion = new HashMap<Integer, SatisfactionSurveyQuestionResult>();
			
			// for loop will not execute if numIterations is 0;
			for (int j = 0; j < numIterations; j++) 
			{
				Object[] record = (Object[]) report.get(j);
				if (record != null) 
				{
					Integer questionNumber = (Integer) record[QUESTION_NUMBER_COLUMN];
					String questionText = (String) record[QUESTION_TEXT_COLUMN];
					int month = (Integer) record[MONTH_COLUMN];
					int year = (Integer) record[YEAR_COLUMN];
					Integer answerCount = (Integer) record[ANS_COUNT_COLUMN];
					
					double stdDeviation = (double) record[STD_DEVIATION_COLUMN];
					double mean = (double) record[MEAN_COLUMN];
					
					double avgAnswer = (Double) record[AVG_ANSWER_COLUMN];
					String mod = (String) record[MODALITY_COLUMN];
					
					SatisfactionSurveyQuestion ssq = new SatisfactionSurveyQuestion();
					ssq.setQuestionNumber(questionNumber);
					ssq.setQuestionText(questionText);
					ssq.setModality(mod);
					ssq.setAvgAnswer(avgAnswer);
					ssq.setMonthAsInt(month);
					ssq.setYear(year);
					ssq.setAnswerCount(answerCount);
					ssq.setMonthAsString();
					ssq.setHeaderString(false);
					ssq.setTheDate();
					ssq.setStdDeviation(stdDeviation);
					ssq.setMean(mean);
					if (resultsByQuestion.containsKey(questionNumber)) 
					{
						SatisfactionSurveyQuestionResult tempResult = resultsByQuestion
								.remove(questionNumber);
						tempResult.addToResultsKeyedByMonthMap(ssq);

						resultsByQuestion.put(questionNumber, tempResult);
					} 
					else 
					{
						SatisfactionSurveyQuestionResult newResult = new SatisfactionSurveyQuestionResult();
						newResult.setQuestionNumber(questionNumber);
						newResult.setQuestionText(questionText);
						newResult.setModality(selectedModality);

						newResult.addToResultsKeyedByMonthMap(ssq);
						resultsByQuestion.put(questionNumber, newResult);
					}
				}

			}
			
			satisfactionSurveyQuestionResults.addAll(resultsByQuestion.values());
		}
		catch(Exception e)
		{
			throw new DAOException(e.getMessage(), e);
		}

		return satisfactionSurveyQuestionResults;
	}

	private Query getNationalSatisfactionSurveyReport(
			PSSReportForm form) throws DAOException, NumberFormatException, ServiceException, ParseException {
		
		StringBuilder sql = new StringBuilder();
		
		// BUILD SELECT SECTION OF SQL
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));
		sql.append(getMeanClause(form.getDisplayTypeSelection()));
		sql.append(getStandardDeviationClause(form.getDisplayTypeSelection()));
		
		// BUILD FROM SECTION OF SQL
		sql.append(getCommonFromSQL());
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}

		// BUILD WHERE SECTION OF SQL
		sql.append(getCommonWhereSQL(form, vendor));
		
		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);
		sql.append(" GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append(" ORDER BY Yr, Mon, c.Question_Number, Modality");
		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));
		
		addDateParams(query, form);
		
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;
	}

	private Query getSatisfactionSurveyReportByVisn(final PSSReportForm form) throws DAOException, NumberFormatException, ServiceException, ParseException {
		
		StringBuilder sql = new StringBuilder();
			
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));
		sql.append(getMeanClause(form.getDisplayTypeSelection()));
		sql.append(getStandardDeviationClause(form.getDisplayTypeSelection()));

		sql.append(getCommonFromSQL());
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}
		
		sql.append(getCommonWhereSQL(form, vendor));
		sql.append("AND d.visn_id = :visnId ");

		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);

		sql.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));

		addDateParams(query, form);

		query.setParameter("visnId", form.getVisnId());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;
	}

	private Query getSatisfactionSurveyReportByFacilities(final PSSReportForm form) throws DAOException, NumberFormatException, ServiceException, ParseException {
		
		StringBuilder sql = new StringBuilder();
			
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));
		sql.append(getMeanClause(form.getDisplayTypeSelection()));
		sql.append(getStandardDeviationClause(form.getDisplayTypeSelection()));

		sql.append(getCommonFromSQL());
		sql.append("INNER JOIN facilities as g ON d.Facility_ID = g.ID ");
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}
		
		sql.append(getCommonWhereSQL(form, vendor));
		sql.append("and d.facility_id = :facilityId ");

		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);

		sql.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));

		addDateParams(query, form);

		query.setParameter("facilityId", form.getFacilityId());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;

	}

	private String getCommonSelectSQL() {
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT c.Question_Number, c.Question_Text_Short, ");
		sb.append("DATEPART(mm, b.Completed_Date) AS Mon, DATEPART(yy, b.Completed_Date) AS Yr, COUNT(*) AS AnsCount, ");
		
		return sb.toString();
	}

	private String getCommonFromSQL() {
		StringBuilder sb = new StringBuilder();
		sb.append("FROM Surveyed_Activity AS d ");
		sb.append("INNER JOIN Surveyed_Activity_Response AS b ON d.Surveyed_Activity_ID = b.Surveyed_Activity_ID ");
		sb.append("INNER JOIN Survey_Response AS a ON b.Surveyed_Activity_Response_ID = a.Surveyed_Activity_Response_ID ");
		sb.append("INNER JOIN Survey_Question AS c ON a.Survey_Question_ID = c.Survey_Question_ID ");
		sb.append("INNER JOIN Device ON d.Device_ID = Device.Device_ID ");
		sb.append("INNER JOIN (select cp.patient_id, cp.level_of_care from censuspatient cp ");
		sb.append("inner join (select distinct MAX(census_id) as censusid, patient_id, disenrollment_date ");
		sb.append("from censuspatient where disenrollment_date is null group by patient_id, disenrollment_date) as lcpe ");
		sb.append("on cp.census_id = lcpe.censusid and cp.patient_id = lcpe.patient_id) f ");
		sb.append("on d.patient_id = f.patient_id ");

		return sb.toString();
	}

	private String getCommonWhereSQL(final PSSReportForm form, final Vendor vendor) {
		StringBuilder sb = new StringBuilder();

		sb.append(" WHERE ");

		// If both dates are not null append dates to the query
		if ((form.getShortFormatReportFromDate() != null && form.getShortFormatReportToDate() != null) ||
			(form.getQuarterFromDate() != null && form.getQuarterToDate() != null) ||
			(form.getFiscalFromDate() != null && form.getFiscalToDate() != null)) {
			sb.append("(b.Completed_Date between :fromDate AND :toDate ) ");
		}

		sb.append(PSSReportUtils.getSurveyIdClause(form.getReportVersionButtonSelection()));

		if (vendor != null && vendor.getName() != null) {
			if (!vendor.getName().equalsIgnoreCase("All Vendors")) {
				sb.append("AND d.Vendor_ID = " + vendor.getId() + " ");
			}
		}
		
		sb.append(reportsDao.getCategoryOfCare(form));

		sb.append("AND c.question_number in ");
		sb.append("( :selectedQuestionList ) ");

		return sb.toString();
	}
	
	private String getAverageClause(final int patientSatisfactionVersion) {
		
		StringBuilder qString = new StringBuilder();
		// Ignore answer #6 for PS Ver. 1.0. This is "No experience" 
	    // since there are only 4 choices for PS Ver. 2.0 the same condition can be used.
		// More "versions" can be used if necessary. AvgAnswer returns as percentage
		switch(patientSatisfactionVersion) {
		// Pat. Sat. Version 1
		case 1:
			qString
			.append(" ROUND(AVG(CAST((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/5.0 as float)), 3) AS AvgAnswer, ");
			break;
		// Pat. Sat. Version 2
		case 2:
			qString
			.append(" ROUND(AVG(CAST((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/4.0 as float)), 3) AS AvgAnswer, ");
			break;
		}
		return qString.toString();
	}
	
	private String getMeanClause(int displayTypeSelection) {

		StringBuilder qString = new StringBuilder();
		
		// Ignore answer #6, This is "No experience" 
	    // AvgAnswer returns as mean
		switch(displayTypeSelection) {
			case DISPLAY_MEAN_STD_DEVIATION_SECLECTION:
				qString.append(" ROUND(AVG(CAST(CASE a.answer WHEN 6 THEN NULL ELSE a.answer END as float)), 3) AS Mean, ");
				break;
			default:
				qString.append(" CAST(0 as FLOAT) AS Mean, ");
				break;
		}
		return qString.toString();
	}
	
	private String getStandardDeviationClause(int displayTypeSelection) {

		StringBuilder qString = new StringBuilder();
		
		// Ignore answer #6, This is "No experience" 
		switch(displayTypeSelection) {
		case DISPLAY_MEAN_STD_DEVIATION_SECLECTION:
			qString.append(" ROUND(STDEVP((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)), 3) AS Stdevp ");
			break;
		default: // percentage case
			qString.append(" CAST(0 as FLOAT) AS Stdevp ");
			break;
		}
		return qString.toString();
	}

	private List<Long> getModalityIds(String modalityString)
	{
		List<DeviceModality> modalities = deviceModalityDao.getAllDeviceModalities();
		List<Long> modalityIds = new ArrayList<Long>(modalities.size()); 
		
		modalityIds.add(new Long(modalityString));
		
		return modalityIds;
	}
	
	private Query addDateParams(Query query, PSSReportForm form) throws ParseException{
		switch (form.getDateTypeSelection()) {
		case 2:
			String[] quarterDate = pssReportService.parseQuarterSelected(form);
			form.setShortFormatReportFromDate(DateUtils.parseDate(quarterDate[0], DateUtils.MMDDYYYY));
			form.setShortFormatReportToDate(DateUtils.parseDate(quarterDate[1], DateUtils.MMDDYYYY));
			break;
		case 3:
			String[] fiscalDate = pssReportService.parseFiscalSelected(form);
			form.setShortFormatReportFromDate(DateUtils.parseDate(fiscalDate[0], DateUtils.MMDDYYYY));
			form.setShortFormatReportToDate(DateUtils.parseDate(fiscalDate[1], DateUtils.MMDDYYYY));
			break;
		default:
			break;
		}
		query.setParameter("fromDate", form.getShortFormatReportFromDate());
		
		//Calendar defaults to the 1st of the month so adding an additional month minus 1 day
		//to include the entire month in the report and setting the time to the last millisecond
		//of the last day in the month
		Calendar toDate = Calendar.getInstance();
		toDate.setTime(form.getShortFormatReportToDate());

		toDate.set(Calendar.MILLISECOND, 999);
		toDate.set(Calendar.SECOND, 59);
		toDate.set(Calendar.MINUTE, 59);
		toDate.set(Calendar.HOUR_OF_DAY, 23);
		if(form.getDateTypeSelection() == 1){
			toDate.add(Calendar.MONTH, 1);
			toDate.add(Calendar.DATE, -1);
		}
		
		query.setParameter("toDate", toDate.getTime());
		
		return query;
	}
}
