/**

 * 
 */
package gov.va.med.ccht.persistent.hibernate;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSearchResult;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.persistent.QIRDAO;
import gov.va.med.fw.util.CollectionUtils;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;

/**
 * @author DNS
 *
 */
@Repository
@Transactional
public class QIRDAOImpl implements QIRDAO { 
	
	static final private Long NEW_STATUS = (long) 20301;
	static final private Long WITHDRAWN_STATUS = (long) 20303;

	public QIRDAOImpl() {
	}

	public QIRDAOImpl(SessionFactory factory) {
		sessionFactory = factory;
	}

	@Autowired
	private SessionFactory sessionFactory;
	
	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}

	public QIR saveQIR(QIR qir) {
		getSession().saveOrUpdate(qir);	
		return qir;
	}
	
	@Override
	public List<String> getVendorResponseDueQIRIds() {
		
		final StringBuilder sql = new StringBuilder();
		sql.append("SELECT \n");
		sql.append("	cast(q.QIR_ID as varchar) \n");
		sql.append("FROM \n");
		sql.append("	qir.QIR q  \n");
		sql.append("	join qir.qir_status_types qst  \n");
		sql.append("	on (qst.id = q.qir_status_type_id)   \n");
		sql.append("WHERE \n");
		sql.append("	q.VENDOR_RESP_DUE_DATE is not null  \n");
		sql.append("	and q.VENDOR_RESP_DUE_DATE < current_timestamp \n");
		sql.append("	and q.date_email_reminder_sent is null \n");
		sql.append("	and qst.name in ('Approved','Replied','Agreed') \n");
		
		@SuppressWarnings("unchecked")
		final List<String> qirIds = (List<String>)getSession()
				.createNativeQuery(sql.toString())
				.list();
		
		return qirIds;

	}

	private Query buildQuerySearchByGeneric(QIRSearchParameters qirSearchParameters) {
		
		Map<String, Object> parameters = new HashMap<String, Object>();
		String sql = "SELECT ";
		if (qirSearchParameters.getMaxRecords() != null && qirSearchParameters.getMaxRecords() > 0) {
			sql += " top " + qirSearchParameters.getMaxRecords().intValue() + " ";
		}
		
		sql += "q.QIR_ID id, qt.NAME qirType, qst.NAME qirStatusType, "
				+ "cast(v.visn_name as varchar) visn, f.facility_id facilityCode, f.facility_name facility, "
				+ "mdmt.NAME deviceType, ven.VENDOR_NAME vendor, "
				+ "q.QIR_COMPLAINT complaint, q.QIR_SUMMARY headline, "
				+ "q.DATE_SUBMITTED submittedDate, q.SUBMITTED_BY submittedBy, q.RECORD_MODIFIED_COUNT, "
				+ "isnull(u.last_name,'') + ', ' + isnull(u.first_name,'')+' ' + isnull(u.middle_name,'') submittedByName, "
				+ "count(qda.QIR_DOC_ATTCHMNT_ID) attchmnt_cnt " + "FROM qir.QIR q "
				+ "join qir.QIR_STATUS_TYPES qst on q.QIR_STATUS_TYPE_ID = qst.ID "
				+ "join ht.app_user u on q.SUBMITTED_BY = u.user_name "
				+ "left join qir.QIR_TYPES qt on q.QIR_TYPE_ID = qt.id "
				+ "left join dbo.MEDICAL_DEVICE_MODEL_TYPE mdmt on q.MED_DVC_MODEL_TYPE_ID = mdmt.id "
				+ "left join dbo.VENDOR ven on q.VENDOR_ID = ven.Vendor_ID "
				+ "left join dbo.facilities f on q.FACILITIES_ID = f.ID "
				+ "left join dbo.visn v on q.visn_id = v.visn_id "
				+ "left outer join qir.QIR_DOC_ATTACHMENTS qda on q.QIR_ID = qda.QIR_ID " + "where 1=1 ";

		// qir id and statuses are exclusive
		if (StringUtils.isEmpty(qirSearchParameters.getId())) {
			if (CollectionUtils.isNotEmpty(qirSearchParameters.getQirStatuses())) {
				parameters.put("qirStatuses", qirSearchParameters.getQirStatuses());
				sql += "and qst.code in (:qirStatuses) ";
			}

			// Additional search conditions
			if (qirSearchParameters.getDeviceType() != null) {
				parameters.put("deviceType", qirSearchParameters.getDeviceType().getValue());
				sql += "and mdmt.code = :deviceType ";
			}

			if (qirSearchParameters.getVendor() != null) {
				parameters.put("vendor_id", qirSearchParameters.getVendor().getValue());
				sql += "and ven.vendor_id = :vendor_id ";
			}

			if (qirSearchParameters.getQirType() != null) {
				parameters.put("qirType", qirSearchParameters.getQirType().getValue());
				sql += "and qt.code = :qirType ";
			}

			if (qirSearchParameters.getFacility() != null) {
				parameters.put("station", qirSearchParameters.getFacility().getValue());
				sql += "and f.facility_id = :station ";
			}

			if (qirSearchParameters.getVisn() != null) {
				parameters.put("visnName", qirSearchParameters.getVisn().getValue());
				sql += "and v.visn_name = :visnName ";
			}
			if (StringUtils.isNotEmpty(qirSearchParameters.getSubmittedBy())) {
				parameters.put("submittedBy", qirSearchParameters.getSubmittedBy());
				sql += "and q.submitted_by = :submittedBy ";
			}
			if (qirSearchParameters.getSubmittedFromDate() != null) {
				parameters.put("submittedFromDate", qirSearchParameters.getSubmittedFromDate());
				sql += "and DATE_SUBMITTED >= :submittedFromDate ";
			}
			if (qirSearchParameters.getSubmittedToDate() != null) {
				// Because our dates have timestamps in the database,
				// we have to go forward one day then use a less than to ensure
				// we are grabbing everything up to the end of the toDate.
				Calendar cal = Calendar.getInstance();
				cal.setTime(qirSearchParameters.getSubmittedToDate());
				cal.add(Calendar.DATE, 1);
				parameters.put("submittedToDate", cal.getTime());
				sql += "and DATE_SUBMITTED < :submittedToDate ";
			}
			if (qirSearchParameters.getIsVendor() == true) { // vendor should not be able to view new or withdrawn QIRs
				parameters.put("NewStatus", NEW_STATUS);
				parameters.put("WithdrawnStatus", WITHDRAWN_STATUS);
				sql += "and QIR_STATUS_TYPE_ID != :NewStatus ";
				sql += "and QIR_STATUS_TYPE_ID != :WithdrawnStatus ";
			}
		} else {
			parameters.put("id", qirSearchParameters.getId());
			sql += "and q.qir_id = :id ";
		}

		sql += "group by q.QIR_ID, qt.NAME, qst.NAME, v.visn_name, f.facility_id, f.facility_name, mdmt.NAME, ven.VENDOR_NAME, q.QIR_COMPLAINT, q.QIR_SUMMARY, q.DATE_SUBMITTED, "
				+ "q.SUBMITTED_BY, q.RECORD_MODIFIED_COUNT, u.last_name, u.first_name, u.middle_name order by q.QIR_ID DESC";

		final Query query = getSession().createNativeQuery(sql);
		for(String key: parameters.keySet()) {
			query.setParameter(key, parameters.get(key));
		}
		
		return query;
	}

	public Date getVendorResponseDueDate(long id) {

		final String sql = "SELECT VENDOR_RESP_DUE_DATE FROM qir.QIR WHERE qir_id = :id";
		
		return (Date) sessionFactory
				.getCurrentSession()
				.createNativeQuery(sql)
				.setParameter("id", id)
				.uniqueResult();

	}

	@Override
	public List<QIR> findByIds(Collection<Long> ids) { 

		final String sql = "SELECT * FROM qir.QIR WHERE qir_id in :ids";
		
		return getSession()
				.createNativeQuery(sql, QIR.class)
				.setParameter("ids", ids)
				.getResultList();

	}
 
	@Override
	public QIR findById(Long id) {
		
		final String sql = "SELECT * FROM qir.QIR WHERE qir_id = :id";
		
		return getSession()
				.createNativeQuery(sql, QIR.class)
				.setParameter("id", id)
				.getSingleResult();
	}

	@Override
	public List<QIRSearchResult> searchQIR(QIRSearchParameters qirSearchParameters) { 
		
		Query query = buildQuerySearchByGeneric(qirSearchParameters);
		
		@SuppressWarnings("unchecked")
		List<Object[]> qirs = query.getResultList();
		final List<QIRSearchResult> results = new ArrayList<QIRSearchResult>();
		DateFormat df = new SimpleDateFormat("MM/dd/yyyy");

		for(Object[] objs: qirs) {
			QIRSearchResult record = new QIRSearchResult();
			record.id = ESAPIValidator.validateStringInput(String.valueOf(objs[0]), ESAPIValidationType.Numbers_WhiteList);
			record.qirType = (String) objs[1];
			record.qirStatusType = (String) objs[2];
			record.visn = (String) objs[3];
			record.facilityCode = (String) objs[4];
			record.facility = (String) objs[5];
			record.deviceType = (String) objs[6];
			record.vendor = (String) objs[7];
			record.complaint = (String) objs[8];
			record.headline = (String) objs[9];
			record.submittedDate = (Date) objs[10];
			record.submittedDateAsString = df.format((Date) objs[10]);
			record.submittedBy = (String) objs[11];
			record.submittedByName = (String) objs[13];
			record.attachmentCount = ((Integer) objs[14]).toString();
			if ((Integer) objs[14] > 0) {
				record.hasAttachments = "true";
			}

			results.add(record);
		}
		
		return results;
		
	}


	@Override
	public void updateAttachment(QIRAttachment attachment) {
		getSession().saveOrUpdate(attachment);
	}
	
	@Override
	public void deleteAttachment(Long attachmentId) {
		getSession().delete(findAttachmentById(attachmentId));
	}


	@Override
	public QIRAttachment findAttachmentById(Long attachmentId) {
		final String sql = "SELECT * FROM qir.qir_doc_attachments "
				+ "WHERE qir_doc_attchmnt_id = :id";
		
		return getSession()
				.createNativeQuery(sql, QIRAttachment.class)
				.setParameter("id", attachmentId)
				.getSingleResult();
	}


	@Override
	public void insertQIRStatusType(QIRStatusType type) {
		getSession().save(type);
		
	}


	@Override
	public void deleteQIRStatusType(QIRStatusType type) {
		getSession().delete(type);
	}


	@Override
	public void deleteQIR(QIR qir) {
		getSession().delete(qir);
	}


	@Override
	public List<QIRStatusType> findAllQIRStatusTypes() {

		final String sql = "SELECT * FROM qir.QIR_STATUS_TYPES";
		
		return getSession()
				.createNativeQuery(sql, QIRStatusType.class)
				.getResultList();

	}
	
	@Override
	public QIRStatusType findQIRStatusTypeByName(String name) {
		final String sql = "SELECT * FROM qir.QIR_STATUS_TYPES WHERE name = :name";
		
		return getSession()
				.createNativeQuery(sql, QIRStatusType.class)
				.setParameter("name", name)
				.getSingleResult();
	}


	@Override
	public QIRStatusType findQIRStatusType(String code) {
		
		final String sql = "SELECT * FROM qir.QIR_STATUS_TYPES WHERE code = :code";
		
		return getSession()
				.createNativeQuery(sql, QIRStatusType.class)
				.setParameter("code", code)
				.getSingleResult();
	}

}
