package gov.va.med.ccht.persistent.hibernate;

import java.lang.annotation.Documented;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.ccht.model.car.CensusActivityRecord;
import gov.va.med.ccht.model.car.CensusActivitySubTotalRecord;
import gov.va.med.ccht.model.car.CensusActivityVisnRecord;
import gov.va.med.ccht.model.car.VendorCensusReportStatus;
import gov.va.med.ccht.model.report.AggregateModality;
import gov.va.med.ccht.persistent.CensusActivityReportDAO;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.ccht.util.ReportUtils;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.util.DateUtils;

@Repository
@Transactional
public class CensusActivityReportDAOImpl implements CensusActivityReportDAO {
	public final static String TOTAL_PATIENTS = "Total Patients";
	
	public final static String COMPLETED = "completed";
	 
	private final static String MESSAGE_BY_DEVICE_CODE = "6";

	@Autowired
	private ReportsDAO reportsDao;
	
	@Autowired
	private SessionFactory sessionFactory;
	
	public CensusActivityReportDAOImpl() {
	}

	public CensusActivityReportDAOImpl(SessionFactory factory, ReportsDAO reportDao) {
		sessionFactory = factory;
		reportsDao = reportDao;
	}
	
	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	/**
	 * this method is used to find the latest census report end date used in the census report vendor compliance table
	 * to display when the last report was submitted by vendors
	 * @param vendorName
	 * @return single report date
	 */
    private Date getLatestEndDate(String vendorName) {
        String sql = "select max(report_end_date) from dbo.census where "
                + "vendor_id = (select vendor_id from dbo.vendor where vendor_name = :vendorName)";
        Date testDate = (Date) getSession()
                .createNativeQuery(sql.toString())
                .setParameter("vendorName", vendorName)
                .getSingleResult();
        
        return testDate;
    }

    @Transactional
    public CensusActivityRecord getAllVendorsReportStatus()
            throws DAOException {
        // Execute the stored procedure and extract results from it
        String sql = "EXEC GetWeeklyCensusReportTotals :weekEndDate";
        
        Calendar cal = getReportEndDateAsCal();
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        Date weekEndDate = cal.getTime();
        
        List<?> results = getSession()
                .createNativeQuery(sql)
                .setParameter("weekEndDate", weekEndDate)
                .getResultList();
        CensusActivityRecord retStatus = new CensusActivityRecord();
        retStatus.setLastDayOfReportWeek(DateUtils.format(weekEndDate, DateUtils.EEEEMMMMDYYYY));
        List<VendorCensusReportStatus> vendorsNotCompleted = new ArrayList<>();
        cal.add(Calendar.DAY_OF_WEEK, -6);
        for(int i = 0; i < results.size(); i++) {
            Object[] record = (Object[]) results.get(i);
            String vendorName = (String) record[0];
            Date lastReportDate = (Date) getLatestEndDate(vendorName);
            VendorCensusReportStatus status = new VendorCensusReportStatus();
            if((lastReportDate != null) && (lastReportDate.compareTo(weekEndDate) == 0)) {
                continue;
            }
            status.setVendorName(vendorName);
            status.setEndDate(weekEndDate);
            status.setStartDate(cal.getTime());
            if((lastReportDate != null) && (lastReportDate.before(weekEndDate))) {
                status.setLastReportDate(lastReportDate);
            }
            vendorsNotCompleted.add(status);
        }
        retStatus.setVendorCensusReportStatus(vendorsNotCompleted);
        return retStatus;
    }
    
	
	/**
	 * 
	 * @param reportEndDate
	 * @param reportStartDate
	 * @return
	 * @throws DAOException
	 */
	public Map<String, VendorCensusReportStatus> getActiveVendorsStatusForReportWeek(String reportEndDate, String reportStartDate) throws DAOException
	{	

		Map<String, VendorCensusReportStatus> vendors = new HashMap<String, VendorCensusReportStatus>();
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT Vendor.Vendor_Name, Vendor.Date_active,Vendor.Date_Inactive, COUNT(*) AS [PatientCount], ");
		sql.append("CASE WHEN HL7Activity.AckType IS NULL THEN '??' ELSE HL7Activity.AckType END AS [AckType], ");
		sql.append("CASE WHEN (HL7Activity.AckType = 'AA') AND (COUNT(*) > 0) THEN 'Completed' WHEN (HL7Activity.AckType is null) AND (COUNT(*) > 0) THEN 'In progress' ");
		sql.append("WHEN (HL7Activity.AckType <> 'AA') AND (COUNT(*) = 0) THEN 'Errors' ELSE 'Unknown' END AS [Status] ");
		sql.append("FROM Census INNER JOIN HL7Activity ON HL7Activity.ID = Census.HL7ActivityID ");
		sql.append("INNER JOIN CensusPatient ON Census.Census_ID = CensusPatient.Census_ID INNER JOIN Vendor ON Census.Vendor_ID = Vendor.Vendor_ID ");
		sql.append("WHERE (Census.Report_Start_Date >= :reportStartDate AND Census.Report_End_Date <= :reportEndDate) ");
		sql.append("AND (Vendor.Date_Active is not null OR Vendor.Date_Active <= :reportStartDate) ");
		sql.append("AND (Vendor.Date_Inactive is NULL OR (Vendor.Date_InActive >= :reportStartDate AND Vendor.Date_InActive <= :reportEndDate)) ");
		sql.append("GROUP BY Vendor.Vendor_Name, Vendor.Date_active, Vendor.Date_Inactive, HL7Activity.AckType ORDER BY Vendor.vendor_name ");
		
		List<?> report = getSession()
				.createNativeQuery(sql.toString())
				.setParameter("reportStartDate", reportStartDate)
				.setParameter("reportEndDate", reportEndDate)
				.getResultList();
		
		int numIterations = report.size();
		
		for (int j = 0; j < numIterations; j++) 
		{
			Object[] record = (Object[]) report.get(j);
			String vendorName = (String)record[0];
			Integer patientCount = (Integer)record[3];
			String ackType = (String)record[4];
			String status = (String)record[5];
		
			VendorCensusReportStatus currentRecord = new VendorCensusReportStatus();
			currentRecord.setVendorName(vendorName);
			currentRecord.setPatientCount(patientCount);
			currentRecord.setAckType(ackType);
			currentRecord.setEndDateAsString(reportEndDate);
			currentRecord.setStartDateAsString(reportStartDate);			
			currentRecord.setStatus(status);
			
			vendors.put(vendorName, currentRecord);
		}
		
		return vendors;
	}
	
	public Map<String, VendorCensusReportStatus> getActiveVendorsForReportWeek(String reportEndDate, String reportStartDate) throws DAOException
	{	

		Map<String, VendorCensusReportStatus> activeVendors = new HashMap<String, VendorCensusReportStatus>();
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT vendor_name, date_active, date_inactive, -1 as PatientCount, '??' as AckType, 'No report' as Status FROM Vendor ");
		sql.append("WHERE (Date_Active is not null OR Date_Active <= :reportStartDate) ");
		sql.append("AND (Date_Inactive is NULL OR (Date_InActive >= :reportStartDate AND Date_InActive <= :reportEndDate)) ");
		sql.append("ORDER BY vendor_name");
	
		List<?> report = getSession()
				.createNativeQuery(sql.toString())
				.setParameter("reportStartDate", reportStartDate)
				.setParameter("reportEndDate", reportEndDate)
				.getResultList();
		int numIterations = report.size();
		
		
		for (int j = 0; j < numIterations; j++) 
		{
			Object[] record = (Object[]) report.get(j);
			String vendorName = (String)record[0];
			Integer patientCount = (Integer)record[3];
			String ackType = (String)record[4];
			String status = (String)record[5];
		
			VendorCensusReportStatus currentRecord = new VendorCensusReportStatus();
			currentRecord.setVendorName(vendorName);
			currentRecord.setPatientCount(patientCount);
			currentRecord.setAckType(ackType);
			currentRecord.setEndDateAsString(reportEndDate);
			currentRecord.setStartDateAsString(reportStartDate);
			currentRecord.setStatus(status);
			
			activeVendors.put(vendorName, currentRecord);
		}
		return activeVendors;
	}
	
	


	private String getVisnReportSQL(CensusActivityReportForm form, List<AggregateModality> modalities) {
		StringBuilder sql = new StringBuilder();

		sql.append("select VISN.VISN_ID, cast(VISN.VISN_Name As VARCHAR) As visn_name, ");
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Patient_Totals] ");
		
		sql.append(getModalitiesSumSql(modalities));
		
		sql.append(" from Facilities, Facility_Activity, VISN ");
		sql.append("where Facility_Activity.Facility_ID=Facilities.ID and ");
		sql.append("Facility_Activity.Rpt_Wk= :weekEndingDate and ");
		sql.append("Facilities.VISN_ID=VISN.VISN_ID ");
		if(!form.getVisnId().equals("-1")) {
			sql.append("and VISN.VISN_ID= :visnId ");
		}
		if(!form.getVendorId().equals("-1")) {
			sql.append("and Facility_Activity.VENDOR_ID = :vendorId ");
		}
		sql.append("group by VISN.VISN_ID, visn_name  order by VISN.VISN_ID ASC");

		return sql.toString();
	}
	
	private String getSubtotaledVisnReportSQL(CensusActivityReportForm form, List<AggregateModality> modalities) {
		StringBuilder sql = new StringBuilder();

		sql.append("select VISN.VISN_ID, ");
		sql.append("cast(VISN.VISN_Name As VARCHAR) As visn_name, ");
		sql.append("Vendor.VENDOR_NAME as [Vendor_Name], ");
		
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Patient_Totals]");
		
		sql.append(getModalitiesSumSql(modalities));
		
		sql.append(" from Facilities, Facility_Activity, VISN, Vendor ");
		sql.append("where Facility_Activity.Facility_ID=Facilities.ID and ");
		sql.append("Facility_Activity.Rpt_Wk= :weekEndingDate ");
		sql.append("and Facilities.VISN_ID=VISN.VISN_ID ");
		sql.append("and Vendor_Name <> 'None' ");
		sql.append("and Facility_activity.VENDOR_ID = Vendor.VENDOR_ID ");
		if(!form.getVisnId().equals("-1")) {
			sql.append("and VISN.VISN_ID= :visnId ");
		}
		
		sql.append("group by VISN.VISN_ID, visn_name, Vendor_Name  order by VISN.VISN_ID, Vendor_Name ASC");

		return sql.toString();
	}
	
	private String getNationalSQL() {
		StringBuilder sql = new StringBuilder();

		sql.append("select 'National Totals' AS [National Totals], ");
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Total Patients]");
		// Don't hard code modalities since we have a table that ties modalities to
		// column names.
		sql.append(getModalitiesSumSql(reportsDao.getAggregateModalities()));
		
		sql.append(" from Facility_Activity ");
		sql.append("where Facility_Activity.Rpt_Wk= :weekEndingDate ");
		sql.append("order by 1 ASC");

		return sql.toString();
	}

	private String getNationalSQLByModality(String columnName, String modalityDescription) {

		StringBuilder sql = new StringBuilder();
		sql.append("select 'National Totals', sum(Facility_Activity.Patient_Totals) AS [Total Patients], ");
		
		sql.append("sum(Facility_Activity.");
		sql.append(columnName);
		sql.append(") AS [" + modalityDescription + "] ");
		
		sql.append("from Facility_Activity ");
		sql.append("where Facility_Activity.Rpt_Wk=:weekEndingDate ");

		return sql.toString();
	}

	private String getNationalSQLByVendor(String vendorName, List<AggregateModality> modalities) {

		StringBuilder sql = new StringBuilder();
		sql.append("select 'National Totals for ");
		sql.append(vendorName);
		sql.append("', sum(Facility_Activity.Patient_Totals) AS [Total Patients] ");
		
		sql.append(getModalitiesSumSql(modalities));
		
		sql.append(" from Vendor ");
		sql.append("INNER JOIN Facility_Activity ON Facility_Activity.Vendor_ID = Vendor.Vendor_ID ");
		sql.append("where Facility_Activity.Rpt_Wk = :weekEndingDate ");
		sql.append("and Vendor.Vendor_Id = :vendorId ");
		sql.append("GROUP BY Vendor.Vendor_Name order by 1 ASC ");

		return sql.toString();
	}

	private String getNationalSQLByModalityAndVendor(CensusActivityReportForm form) {

		StringBuilder sql = new StringBuilder();
		sql.append("select 'National Totals for ");
		sql.append(form.getVendorColumnName());
		sql.append("', sum(Facility_Activity.Patient_Totals) AS [Total Patients], ");

		// Determine which column to sum for Messaging Total according to
		// modality chosen.
		sql.append("sum(Facility_Activity.");
		sql.append(form.getModalityColumnName());
		sql.append(") AS [" + form.getModalityDescription() + "] ");
		sql.append("from Vendor ");
		sql.append("INNER JOIN Facility_Activity ON Facility_Activity.Vendor_ID = Vendor.Vendor_ID ");
		sql.append("where Facility_Activity.Rpt_Wk = :weekEndingDate ");
		sql.append("and Vendor.Vendor_ID = :vendorId ");
		sql.append("GROUP BY Vendor.Vendor_Name order by 1 ASC ");

		return sql.toString();
	}
	
	
	private String getSubTotaledNationalReportSQL(List<AggregateModality> modalities) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT Vendor.Vendor_Name as [Vendor], ");
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Total Patients]");
		
		sql.append(getModalitiesSumSql(modalities));
		
		sql.append(" FROM Facility_Activity,Vendor ");
		sql.append("WHERE Facility_Activity.Rpt_Wk= :weekEndingDate ");
		sql.append("and Vendor.Vendor_Name <> 'None' and Facility_Activity.Vendor_ID=Vendor.Vendor_ID ");
		sql.append("GROUP BY Vendor.Vendor_Name order by 1 ASC ");
		return sql.toString();
	}
	
	private String getVisnDrilldownDataSQL(CensusActivityReportForm form, 
			List<AggregateModality> modalities) {

		StringBuilder sql = new StringBuilder();
		sql.append("select CONVERT(VARCHAR(10), Facilities.Facility_ID), ");
		sql.append("Facilities.Facility_Name AS [Facility],VISN.VISN_ID AS [VISN], ");
		sql.append("cast(VISN.VISN_Name As VARCHAR) As visn_name, sum(Facility_Activity.Patient_Totals) AS [Total Patients]"); 
		sql.append(getModalitiesSumSql(modalities));
		sql.append(" from Facilities, Facility_Activity, VISN ");
		sql.append("where Facility_Activity.Facility_ID=Facilities.ID ");
		sql.append("and Facility_Activity.Rpt_Wk= :weekEndingDate ");
		if(form.getVisnId() != null && !form.getVisnId().equals("-1")) {
			sql.append("and VISN.VISN_ID = :visnId ");
		}
		sql.append("and Facilities.VISN_ID=VISN.VISN_ID ");
		if(form.getVendorId() != null && !form.getVendorId().equals("-1")) {
			sql.append("and Facility_Activity.VENDOR_ID = :vendorId ");
		}
		if(form.getFacilityId() != null && !form.getFacilityId().equals("-1")) {
			sql.append("and Facilities.FACILITY_ID = :facilityId ");
		}
		sql.append("group by Facilities.facility_Name,VISN.VISN_ID, VISN.VISN_NAME, Facilities.facility_id ");
		sql.append("order by Facilities.Facility_Name DESC");
		return sql.toString();
	}

	private String getSubtotaledFacilityReportSQL(CensusActivityReportForm form, 
			List<AggregateModality> modalities) {
		StringBuilder sql = new StringBuilder();

		sql.append("select Facilities.Facility_Id, ");
		sql.append("Facilities.Facility_Name, ");
		sql.append("VISN.VISN_ID, ");
		sql.append("cast(VISN.VISN_Name As VARCHAR) As visn_name, ");
		sql.append("Vendor.VENDOR_NAME as [Vendor_Name], ");
		
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Patient_Totals]");
		
		sql.append(getModalitiesSumSql(modalities));
		
		sql.append(" from Facilities, Facility_Activity, VISN, Vendor ");
		sql.append("where Facility_Activity.Facility_ID=Facilities.ID and ");
		sql.append("Facility_Activity.Rpt_Wk= :weekEndingDate ");
		sql.append("and Facilities.VISN_ID=VISN.VISN_ID ");
		sql.append("and Vendor_Name <> 'None' ");
		sql.append("and Facility_activity.VENDOR_ID = Vendor.VENDOR_ID ");
		if(!form.getFacilityId().equals("-1")) {
			sql.append("and Facilities.Facility_Id= :facilityId ");
		}
		
		sql.append("group by Facilities.Facility_Id, Facilities.Facility_Name, VISN.VISN_ID, visn_name, Vendor_Name order by Facilities.Facility_Name, Vendor_Name ASC");

		return sql.toString();
	}
	
	private String getModalitiesSumSql(List<AggregateModality> modalities) {
		StringBuilder sql = new StringBuilder();
		for(AggregateModality mod: modalities) {
			if(!mod.getCode().equals(MESSAGE_BY_DEVICE_CODE)){   
			sql.append(
					String.format(", sum(Facility_Activity.%s) AS [%s]",
							mod.getName(), mod.getDescription()));
			}
			else {
				sql.append(
						String.format(", sum(Facility_Activity.%s)-sum(IsNull(Facility_Activity.Browser,0)) AS [%s]",
								mod.getName(), mod.getDescription()));
			}
		}
		return sql.toString();
	}

	private String getSubTotaledNationalReportByModalitySQL(
			CensusActivityReportForm form) {
		StringBuilder sql = new StringBuilder();

		sql.append("select Vendor.Vendor_Name as [Vendor],");
		sql.append("sum(Facility_Activity.Patient_Totals) AS [Total Patients],");

		// Determine which column to sum for Messaging Total according to
		// modality chosen.
		sql.append("sum(Facility_Activity.");
		sql.append(form.getModalityColumnName());
		sql.append(") AS ["+ form.getModalityDescription() + "]");

		sql.append("from Facility_Activity,Vendor ");
		sql.append("where Facility_Activity.Rpt_Wk= :weekEndingDate ");
		sql.append("and Vendor.Vendor_Name <> 'None' ");
		sql.append("and Facility_Activity.Vendor_ID=Vendor.Vendor_ID ");
		sql.append("GROUP BY Vendor.Vendor_Name order by 1 ASC");
		return sql.toString();
	}

	/**
	 * 
	 * @param rows the result of the SQL query
	 * @param modalities list of modalities to include in the subtotals
	 * @return
	 */
	private List<CensusActivitySubTotalRecord> getNationalSubTotalPivotedList(
			List<?> rows, List<AggregateModality> modalities) {
		List<HashMap<String,Integer>> totalsList = new ArrayList<HashMap<String,Integer>>();
		// Include a + 1 for total patients.
		for(int i = 0; i < modalities.size() + 1; i++) {
			totalsList.add(new HashMap<String,Integer>());
		}
		
		int numRows = rows.size();
		for (int row = 0; row < numRows; row++) {
			
			Object[] record = (Object[]) rows.get(row);

			String vendorName = (String) record[0];
			for (int col = 1; col < record.length; col++) {
				Integer val = (Integer) record[col];

				if (val == null) {
					val = 0;
				}

				totalsList.get(col - 1).put(vendorName, val);
			}
		}
	
		List<CensusActivitySubTotalRecord> returnVal = new ArrayList<CensusActivitySubTotalRecord>();
		
		returnVal.add(getCensusActivitySubTotalRecord(
				totalsList.get(0), TOTAL_PATIENTS));
		// Loop through modalities and add records as we go.
		for(int i = 0; i < modalities.size(); i++) {
			returnVal.add(getCensusActivitySubTotalRecord(
					totalsList.get(i + 1), modalities.get(i).getDescription()));
		}
		
		return returnVal;
	}

	private CensusActivitySubTotalRecord getCensusActivitySubTotalRecord(
			Map<String, Integer> row, String modalityDescription) {
		CensusActivitySubTotalRecord retVal = new CensusActivitySubTotalRecord();

		retVal.setModalityName(modalityDescription);
		Set<String> keys = row.keySet();
		for (String key: keys) { 
			retVal.setVendorTotal(key, row.get(key));
		}

		return retVal;
	}

	private Calendar getReportEndDateAsCal()
	{
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.DAY_OF_WEEK, Calendar.SATURDAY);
		cal.add(Calendar.DAY_OF_WEEK, -7);
		return cal;
	}
	
	@Override
	public List<CensusActivityRecord> getNationalReport(CensusActivityReportForm form) throws DAOException {
		
		List<CensusActivityRecord> returnResults = new ArrayList<CensusActivityRecord>();
		String sql = getNationalSQL();
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		List<?> report = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.getResultList();
		
		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);

			CensusActivityRecord tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName("Total Patients");
			tempRecord.setTotal((Integer) record[1]);
			tempRecord.setNationalTotal((Integer) record[1]);
			returnResults.add(tempRecord);

			// Get list of modalities and iterate through them here.
			List<AggregateModality> modalities = reportsDao.getAggregateModalities();
			for(int i = 0; i < modalities.size(); i++) {
				tempRecord = new CensusActivityRecord();
				tempRecord.setModalityName(modalities.get(i).getDescription());
				tempRecord.setTotal((Integer) record[i+2]);
				tempRecord.setNationalTotal((Integer) record[i+2]);
				returnResults.add(tempRecord);
			}
		}
		return returnResults;
	}
	
	@Override
	public List<CensusActivitySubTotalRecord> getSubTotaledNationalReport(
			CensusActivityReportForm form) throws DAOException, ParseException {
		
		List<CensusActivitySubTotalRecord> retVal = null;
		String sql = getSubTotaledNationalReportSQL(reportsDao.getAggregateModalities());
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);

		List<?> report = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.getResultList();
		
		if (report != null && report.size() > 0) {
			retVal = getNationalSubTotalPivotedList(report, reportsDao.getAggregateModalities());
		}

		return retVal;
	}

	public List<CensusActivityRecord> getNationalReportByVendor(
			CensusActivityReportForm form)throws DAOException, ParseException {
		
		List<CensusActivityRecord> returnResults = new ArrayList<CensusActivityRecord>();
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		String sql = getNationalSQLByVendor(form.getVendorColumnName(), reportsDao.getAggregateModalities());
		
		List<?> report = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.setParameter("vendorId", form.getVendorId())
				.getResultList();

		for (int j = 0; j < report.size(); j++) {
			Object[] record = (Object[]) report.get(j);

			CensusActivityRecord tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName("Total Patients");
			tempRecord.setTotal((Integer) record[1]);
			tempRecord.setNationalTotal((Integer) record[1]);
			returnResults.add(tempRecord);

			// Get list of modalities and iterate through them here.
			List<AggregateModality> modalities = reportsDao.getAggregateModalities();
			for(int i = 0; i < modalities.size(); i++) {
				tempRecord = new CensusActivityRecord();
				tempRecord.setModalityName(modalities.get(i).getDescription());
				tempRecord.setTotal((Integer) record[i+2]);
				tempRecord.setNationalTotal((Integer) record[i+2]);
				returnResults.add(tempRecord);
			}
		}
		return returnResults;
	}
	
	public List<CensusActivityRecord> getNationalReportByModality(CensusActivityReportForm form) throws DAOException {
		List<CensusActivityRecord> returnResults = new ArrayList<CensusActivityRecord>();

		String sql = getNationalSQLByModality(form.getModalityColumnName(), form.getModalityDescription());
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		
		List<?> report = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.getResultList();

		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);

			CensusActivityRecord tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName("Total Patients");
			tempRecord.setTotal((Integer) record[1]);
			tempRecord.setNationalTotal((Integer) record[1]);
			returnResults.add(tempRecord);

			tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName(form.getModalityDescription());
			tempRecord.setTotal((Integer) record[2]);
			tempRecord.setNationalTotal((Integer) record[2]);
			returnResults.add(tempRecord);
		}

		return returnResults;
	}
	
	@Override
	public List<CensusActivitySubTotalRecord> getSubtotaledNationalReportByModality(CensusActivityReportForm form)throws DAOException {
		List<CensusActivitySubTotalRecord> retVal = null;
		String sql = getSubTotaledNationalReportByModalitySQL(form);

		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		
		List<?> rows = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.getResultList();

		if (rows != null && rows.size() > 0) {
			List<AggregateModality> modalities = new ArrayList<AggregateModality>();
			AggregateModality mod = new AggregateModality();
			mod.setName(form.getModalityColumnName());
			mod.setDescription(form.getModalityDescription());
			modalities.add(mod);
			retVal = getNationalSubTotalPivotedList(rows, modalities);
		}

		return retVal;
	}
	public List<CensusActivityRecord> getNationalReportByModalityAndVendor(CensusActivityReportForm form) throws DAOException {
		List<CensusActivityRecord> returnResults = new ArrayList<CensusActivityRecord>();

		String sql = getNationalSQLByModalityAndVendor(form);
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		
		List<?> report = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate)
				.setParameter("vendorId", form.getVendorId())
				.getResultList();
		
		int numIterations = report.size();

		for (int j = 0; j < numIterations; j++) {
			Object[] record = (Object[]) report.get(j);

			CensusActivityRecord tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName("Total Patients");
			tempRecord.setTotal((Integer) record[1]);
			tempRecord.setNationalTotal((Integer) record[1]);
			returnResults.add(tempRecord);

			tempRecord = new CensusActivityRecord();
			tempRecord.setModalityName(
					ReportUtils.getModalityDisplayNameFromColumnName(form.getModalityColumnName()));
			tempRecord.setTotal((Integer) record[2]);
			tempRecord.setNationalTotal((Integer) record[2]);
			returnResults.add(tempRecord);
		}

		return returnResults;
	}
	
	@Override
	public List<CensusActivityVisnRecord> getVisnReport(
			CensusActivityReportForm form, List<AggregateModality> modalities) throws DAOException {

		List<CensusActivityVisnRecord> returnResults = new ArrayList<CensusActivityVisnRecord>();
		
		String sql = getVisnReportSQL(form, modalities);
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		Query query = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate);
		if(!form.getVisnId().equals("-1"))
		{
			query.setParameter("visnId", form.getVisnId());
		}
		if(!form.getVendorId().equals("-1")) {
			query.setParameter("vendorId", form.getVendorId());
		}
		List<?> report = query.getResultList();
		
		int numIterations = report.size();
		CensusActivityVisnRecord totalRecord = new CensusActivityVisnRecord();
		totalRecord.setVisnName("Total");
		// Initialize the total record so it can be added to later.
		for(int i = 0; i < modalities.size(); i++) {
			totalRecord.setModalityTotal(modalities.get(i).getDescription(), 0);
		}
		
		for (int i = 0; i < numIterations; i++) {
			Object[] record = (Object[]) report.get(i);
			CensusActivityVisnRecord tempRecord = new CensusActivityVisnRecord();

			tempRecord.setVisnId((Integer) record[0]);
			tempRecord.setVisnName((String) record[1]);
			tempRecord.setPatientTotals(record[2] == null ? 0 : (Integer) record[2]);
			
			for(int j = 0; j < modalities.size(); j++) {
				Integer amount;
				if(record[j+3] == null) {
					amount = 0;
				}
				else {
					amount = (Integer) record[j+3];
				}
				tempRecord.setModalityTotal(
						modalities.get(j).getDescription(), amount);
			}

			// Add the values of the current record to a
			// record that will represent a totals row
			totalRecord.add(tempRecord);

			returnResults.add(tempRecord);
		}

		returnResults.add(totalRecord);

		return returnResults;
	}
	
	@Override
	public List<CensusActivityVisnRecord> getSubtotaledVisnReport(
			CensusActivityReportForm form, List<AggregateModality> modalities) {
		List<CensusActivityVisnRecord> returnResults = new ArrayList<CensusActivityVisnRecord>();
		
		String sql = getSubtotaledVisnReportSQL(form, modalities);
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		Query query = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate);
		if(!form.getVisnId().equals("-1"))
		{
			query.setParameter("visnId", form.getVisnId());
		}
		
		List<?> report = query.getResultList();
		
		int numIterations = report.size();
		CensusActivityVisnRecord totalRecord = new CensusActivityVisnRecord();
		totalRecord.setVisnName("Total");
		// Initialize the total record so it can be added to later.
		for(int i = 0; i < modalities.size(); i++) {
			totalRecord.setModalityTotal(modalities.get(i).getDescription(), 0);
		}
		
		for (int i = 0; i < numIterations; i++) {
			Object[] record = (Object[]) report.get(i);
			CensusActivityVisnRecord tempRecord = new CensusActivityVisnRecord();

			tempRecord.setVisnId((Integer) record[0]);
			tempRecord.setVisnName((String) record[1]);
			tempRecord.setVendorName((String) record[2]);
			tempRecord.setPatientTotals(record[3] == null ? 0 : (Integer) record[3]);
			
			for(int j = 0; j < modalities.size(); j++) {
				Integer amount;
				if(record[j+4] == null) {
					amount = 0;
				}
				else {
					amount = (Integer) record[j+4];
				}
				tempRecord.setModalityTotal(
						modalities.get(j).getDescription(), amount);
			}

			// Add the values of the current record to a
			// record that will represent a totals row
			totalRecord.add(tempRecord);

			returnResults.add(tempRecord);
		}

		returnResults.add(totalRecord);

		return returnResults;
	}
	
	@Override
	public List<CensusActivityVisnRecord> getVisnDrilldownData(CensusActivityReportForm form,
			List<AggregateModality> modalities) throws DAOException {
		List<CensusActivityVisnRecord> returnResults = new ArrayList<CensusActivityVisnRecord>();
		String sql = getVisnDrilldownDataSQL(form, modalities);
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		Query query = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate);
		
		if(!form.getVisnId().equals("-1")) {
			query.setParameter("visnId", form.getVisnId());
		}
		if(!form.getVendorId().equals("-1")) {
			query.setParameter("vendorId", form.getVendorId());
		}
		if(!form.getFacilityId().equals("-1")) {
			query.setParameter("facilityId", form.getFacilityId());
		}
		
		List<?> results = query.getResultList();
		if(results == null || results.size() == 0) {
			// If we didn't get any results, no need to continue.
			return new ArrayList<CensusActivityVisnRecord>();
		}
		CensusActivityVisnRecord totalRecord = new CensusActivityVisnRecord();
		totalRecord.setFacilityName("Totals");
		// Initialize all modalities to 0 for the total record
		for(int i = 0; i < modalities.size(); i++) {
			totalRecord.setModalityTotal(modalities.get(i).getDescription(), 0);
		}
		for(int i = 0; i < results.size(); i++) {
			Object[] record = (Object[]) results.get(i);
			CensusActivityVisnRecord tempRecord = new CensusActivityVisnRecord();
			
			tempRecord.setFacilityId((String) record[0]);
			tempRecord.setFacilityName((String) record[1]);
			tempRecord.setVisnId((Integer) record[2]);
			tempRecord.setVisnName((String) record[3]);
			if( record[4] == null) {
				tempRecord.setPatientTotals(0);
			}
			else {
				tempRecord.setPatientTotals((Integer) record[4]);
			}
			for(int j = 0; j < modalities.size(); j++) {
				if( record[j + 5] == null) {
					tempRecord.setModalityTotal(modalities.get(j).getDescription(), 0);
				}
				else {
					tempRecord.setModalityTotal(modalities.get(j).getDescription(),
						(Integer) record[j + 5]);
				}
			}
			totalRecord.add(tempRecord);
			returnResults.add(tempRecord);
		}
		returnResults.add(totalRecord);
		
		return returnResults;
	}
	
	@Override
	public List<CensusActivityVisnRecord> getSubtotaledFacilityReport(CensusActivityReportForm form,
			List<AggregateModality> modalities) throws DAOException {
		List<CensusActivityVisnRecord> returnResults = new ArrayList<CensusActivityVisnRecord>();
		
		String sql = getSubtotaledFacilityReportSQL(form, modalities);
		String weekEndingDate = DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY);
		Query query = getSession()
				.createNativeQuery(sql)
				.setParameter("weekEndingDate", weekEndingDate);
		if(!form.getFacilityId().equals("-1"))
		{
			query.setParameter("facilityId", form.getFacilityId());
		}
		
		List<?> report = query.getResultList();
		
		int numIterations = report.size();
		CensusActivityVisnRecord totalRecord = new CensusActivityVisnRecord();
		totalRecord.setFacilityName("Totals");
		// Initialize the total record so it can be added to later.
		for(int i = 0; i < modalities.size(); i++) {
			totalRecord.setModalityTotal(modalities.get(i).getDescription(), 0);
		}
		
		for (int i = 0; i < numIterations; i++) {
			Object[] record = (Object[]) report.get(i);
			CensusActivityVisnRecord tempRecord = new CensusActivityVisnRecord();

			tempRecord.setFacilityId((String) record[0]);
			tempRecord.setFacilityName((String) record[1]);
			tempRecord.setVisnId((Integer) record[2]);
			tempRecord.setVisnName((String) record[3]);
			tempRecord.setVendorName((String) record[4]);
			tempRecord.setPatientTotals(record[5] == null ? 0 : (Integer) record[5]);
			final int modalityOffset = 6;
			for(int j = 0; j < modalities.size(); j++) {
				Integer amount;
				if(record[j + modalityOffset] == null) {
					amount = 0;
				}
				else {
					amount = (Integer) record[j + modalityOffset];
				}
				tempRecord.setModalityTotal(
						modalities.get(j).getDescription(), amount);
			}

			// Add the values of the current record to a
			// record that will represent a totals row
			totalRecord.add(tempRecord);

			returnResults.add(tempRecord);
		}

		returnResults.add(totalRecord);

		return returnResults;
	}

	@Override
	public Date getMinStartDate() throws DAOException {
		
		// Report Story(s) has hard specific date for earlest date which is before min date in db:
		SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
		String minDate = "09/10/2005";
		try {
			return df.parse(minDate);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		return null;
		
//		String sql = "SELECT MIN(rpt_wk) FROM Facility_Activity";
//		return (Date) getSession()
//				.createNativeQuery(sql)
//				.getSingleResult();
	}
	
}
