package gov.va.med.ccht.model.report.scheduled;

import static gov.va.med.ccht.model.report.scheduled.FixedDateType.FIFTEENTH;
import static gov.va.med.ccht.model.report.scheduled.FixedDateType.FIRST;
import static gov.va.med.ccht.model.report.scheduled.FixedDateType.LAST;
import static java.util.Calendar.APRIL;
import static java.util.Calendar.DAY_OF_MONTH;
import static java.util.Calendar.DAY_OF_WEEK;
import static java.util.Calendar.DAY_OF_YEAR;
import static java.util.Calendar.DECEMBER;
import static java.util.Calendar.JANUARY;
import static java.util.Calendar.JULY;
import static java.util.Calendar.JUNE;
import static java.util.Calendar.MARCH;
import static java.util.Calendar.OCTOBER;
import static java.util.Calendar.SEPTEMBER;

import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.util.DateUtils;

/**
 * These types represent units of time that contain days. Given a specific day,
 * we can deduce the start and end of the unit containing that day (and the
 * previous or next unit as well).
 * 
 * @author DNS
 * 
 */
public enum MultiDayTimeUnitType implements Lookup {
	WEEK("Week", new DateCalculator() {
		public Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous) {
			if (fixedDateType == FixedDateType.FIFTEENTH)
				throw new IllegalArgumentException("Can't calculate the fifteenth of a week!");

			Date adjustedStartDate = startDate;
			if (previous)
				adjustedStartDate = DateUtils.add(startDate, Calendar.WEEK_OF_YEAR, -1);

			return calculateDateGivenField(fixedDateType, adjustedStartDate, DAY_OF_WEEK);
		}
	}),
	MONTH("Month", new DateCalculator() {
		public Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous) {
			Date adjustedStartDate = startDate;
			if (previous)
				adjustedStartDate = DateUtils.add(startDate, Calendar.MONTH, -1);

			return calculateDateGivenField(fixedDateType, adjustedStartDate, DAY_OF_MONTH);
		}
	}),
	QUARTER("Quarter", new DateCalculator() {
		public Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous) {
			Calendar c = Calendar.getInstance(Locale.US);
			c.setTime(startDate);

			/* Determine current quarter */
			int month = c.get(Calendar.MONTH);

			int quarter = 1;
			if (month >= APRIL && month <= JUNE) {
				quarter = 2;
			} else if (month >= JULY && month <= SEPTEMBER) {
				quarter = 3;
			} else if (month >= OCTOBER && month <= DECEMBER) {
				quarter = 4;
			}

			/* Calculate new quarter (and roll year back if necessary) */
			int finalQuarter = quarter;
			if (previous) {
				finalQuarter = quarter - 1;
				if (finalQuarter == 0) {
					finalQuarter = 4;
					c.add(Calendar.YEAR, -1);
				}
			}

			/* Month setting */
			if (fixedDateType == FIRST || fixedDateType == FixedDateType.SECOND
					|| fixedDateType == FIFTEENTH) {
				if (finalQuarter == 1) {
					c.set(Calendar.MONTH, JANUARY);
				} else if (finalQuarter == 2) {
					c.set(Calendar.MONTH, APRIL);
				} else if (finalQuarter == 3) {
					c.set(Calendar.MONTH, JULY);
				} else {
					c.set(Calendar.MONTH, OCTOBER);
				}
			} else if (fixedDateType == LAST) {
				if (finalQuarter == 1) {
					c.set(Calendar.MONTH, MARCH);
				} else if (finalQuarter == 2) {
					c.set(Calendar.MONTH, JUNE);
				} else if (finalQuarter == 3) {
					c.set(Calendar.MONTH, SEPTEMBER);
				} else {
					c.set(Calendar.MONTH, DECEMBER);
				}
			}

			/* Day setting */
			fixedDateType.setCalendar(c, DAY_OF_MONTH);
			return c.getTime();
		}
	}),
	YEAR("Year", new DateCalculator() {
		public Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous) {
			Date adjustedStartDate = startDate;
			if (previous)
				adjustedStartDate = DateUtils.add(startDate, Calendar.YEAR, -1);

			return calculateDateGivenField(fixedDateType, adjustedStartDate, DAY_OF_YEAR);
		}
	});

	private String name;
	private DateCalculator dateCalculator;

	private MultiDayTimeUnitType(String name, DateCalculator dateCalculator) {
		this.name = name;
		this.dateCalculator = dateCalculator;
	}

	public String getName() {
		return name;
	}

	public String getCode() {
		return name();
	}

	public String getDescription() {
		return getName();
	}

	/**
	 * Calculates a date given a starting date, FixedDateType and whether to use
	 * the previous or current version of the FixedDateType.
	 * 
	 * @param startDate
	 *            The start date
	 * @param fixedDateType
	 *            The FixedDateType to use inside this specified range
	 * @param previous
	 *            True to use the previous version of the FixedDateType; false
	 *            to use the current.
	 * @return The newly calculated Date
	 */
	public Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous) {
		return dateCalculator.calculateDate(startDate, fixedDateType, previous);
	}

	private interface DateCalculator {
		Date calculateDate(Date startDate, FixedDateType fixedDateType, boolean previous);
	}

	private static Date calculateDateGivenField(FixedDateType fixedDateType,
			Date adjustedStartDate, int field) throws AssertionError {
		Calendar c = Calendar.getInstance(Locale.US);
		c.setTime(adjustedStartDate);
		fixedDateType.setCalendar(c, field);
		return c.getTime();
	}

}
