package gov.va.med.ccht.model.qir;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.SqlResultSetMapping;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import org.hibernate.annotations.Formula;

import gov.va.med.ccht.model.CCHTAuditFields;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.DeviceType;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.fw.util.DateUtils;

@Entity
@Table(schema="qir", name="QIR")
public class QIR extends CCHTAuditFields{
	
	private Long id;
	@Valid
	private DmpQirType dmpQirType = null;
	@Valid
	private QIRType qirType = null;
	@Valid
	private QIRStatusType qirStatusType = null;
	@Valid
	private SimpleVisn visn;
	@Valid
	@NotNull
	private SimpleFacility facility;
	@Valid
	private DeviceType deviceType = null;
	@Valid
	@NotNull
	private Vendor vendor = null;
	private String serialNumber = null;
	private String purchaseOrderNumber = null;
	private Date submittedDate = new Date();

	private Date vendorResponseDueDate = DateUtils.getDateTodayPlusDays(14);
	private Date emailReminderSentDate;
	private Date statusChangeDate = null;
	@NotNull
	@Size(min = 0, max = 2000)
	private String complaint = null;

	@NotNull
	@Size(min = 0, max = 100)
	private String summary = null;

	private String submittedBy = null;
	private String submittedByName = null;
	private String emailSubmittedByName = null;
	private String modifiedByName = null;
	private String isDmpIssue = null;

	@Valid
	@NotNull
	private Set<QIRRemarks> internalQIRRemarks = null;
	private Set<QIRVendorAction> internalQIRVendorActions = null;
	private Set<QIRSimpleAttachment> internalQIRAttachments = null;
	
	private Integer version = null;
	
	@Id 
	@GeneratedValue(strategy = javax.persistence.GenerationType.IDENTITY)
	@Column(name = "QIR_ID")
	public Long getId() {
		return id;
	}
	
	public void setId(Long id) {
		this.id = id;
	}

	// ---------------------------------------- Business Methods
	@Transient
	public Integer getVersion() {
		return version;
	}

	public void setVersion(Integer version) {
		this.version = version;
	}

	@ManyToOne
    @JoinColumn(name="QIR_STATUS_TYPE_ID")
	public QIRStatusType getQirStatusType() {
		return qirStatusType;
	}
    
    public void setQirStatusType(QIRStatusType qirStatusType) {
    	this.qirStatusType = qirStatusType;
    }

	@Transient
	public boolean isNew() {
		return qirStatusType.isNew();
	}

	@Transient
	public boolean isApproved() {
		return qirStatusType.isApproved();
	}

	@Transient
	public boolean isWithdrawn() {
		return qirStatusType.isWithdrawn();
	}

	@Transient
	public boolean isReplied() {
		return qirStatusType.isReplied();
	}

	@Transient
	public boolean isAgreed() {
		return qirStatusType.isAgreed();
	}

	@Transient
	public boolean isClosed() {
		return qirStatusType.isClosed();
	}

	public boolean isSameVendor(User user) {
		return getVendor() != null && user.getVendor() != null
				&& user.getVendor().getCode().equals(getVendor().getCode());
	}

	// ---------------------------------------- Accessor Methods

	@ManyToOne
	@JoinColumn(name="QIR_TYPE_ID")
	public QIRType getQirType() {
		return qirType;
	}

	public void setQirType(QIRType qirType) {
		this.qirType = qirType;
	}

	@ManyToOne
    @JoinColumn(name="VISN_ID")
	public SimpleVisn getVisn() {
		return visn;
	}

	public void setVisn(SimpleVisn visn) {
		this.visn = visn;
	}

	@OneToOne
    @JoinColumn(name="FACILITIES_ID", referencedColumnName="ID")
	public SimpleFacility getFacility() {
		return facility;
	}

	public void setFacility(SimpleFacility facility) {
		this.facility = facility;
	}

	@ManyToOne
	@JoinColumn(name="MED_DVC_MODEL_TYPE_ID")
	public DeviceType getDeviceType() {
		return deviceType;
	}

	public void setDeviceType(DeviceType deviceType) {
		this.deviceType = deviceType;
	}

	@ManyToOne
    @JoinColumn(name="VENDOR_ID")
	public Vendor getVendor() {
		return vendor;
	}

	public void setVendor(Vendor vendor) {
		this.vendor = vendor;
	}

	@Column(name = "MED_DVC_SERIAL_NBR", length = 100)
	public String getSerialNumber() {
		return serialNumber;
	}

	public void setSerialNumber(String serialNumber) {
		this.serialNumber = serialNumber;
	}

	@Column(name = "PURCHASE_ORDER_NBR", length = 30)
	public String getPurchaseOrderNumber() {
		return purchaseOrderNumber;
	}

	public void setPurchaseOrderNumber(String purchaseOrderNumber) {
		this.purchaseOrderNumber = purchaseOrderNumber;
	}

	@Column(name = "DATE_SUBMITTED")
	public Date getSubmittedDate() {
		return submittedDate;
	}

	public void setSubmittedDate(Date submittedDate) {
		this.submittedDate = submittedDate;
	}

	@Column(name = "DATE_LAST_STATUS")
	public Date getStatusChangeDate() {
		return statusChangeDate;
	}

	public void setStatusChangeDate(Date statusChangeDate) {
		this.statusChangeDate = statusChangeDate;
	}

	@Column(name = "QIR_COMPLAINT", length = 2000)
	public String getComplaint() {
		return complaint;
	}

	public void setComplaint(String complaint) {
		this.complaint = complaint;
	}

	@Column(name = "QIR_SUMMARY", length = 400)
	public String getSummary() {
		return summary;
	}

	public void setSummary(String summary) {
		this.summary = summary; 
	}

	@Column(name = "SUBMITTED_BY", length = 30)
	public String getSubmittedBy() {
		return submittedBy;
	}

	public void setSubmittedBy(String submittedBy) {
		this.submittedBy = submittedBy;
	}

	// Commented out for now as @Formula can throw a NullPointerException due to a documented bug in Hibernate.
	// Possible work-around is to use SqlResultSetMapping
	//@Formula(value = "(select isnull(u.last_name,'') + ', ' + isnull(u.first_name,'')+' ' + isnull(u.middle_name,'') from ht.app_user u where u.user_name = SUBMITTED_BY)")
	@Transient
	public String getSubmittedByName() {
		return (submittedByName == null) ? submittedBy : submittedByName;
	}

	public void setSubmittedByName(String submittedByName) {
		this.submittedByName = submittedByName;
	}

	@Column(name = "VENDOR_RESP_DUE_DATE")
	public Date getVendorResponseDueDate() {
		return vendorResponseDueDate;
	}

	public void setVendorResponseDueDate(Date vendorResponseDueDate) {
		this.vendorResponseDueDate = vendorResponseDueDate;
	}

	@Transient
	public String getModifiedByName() {
		return modifiedByName;
	}

	public void setModifiedByName(String modifiedByName) {
		this.modifiedByName = modifiedByName;
	}

	@Column(name="DMP_RELATED")
	public String getIsDmpIssue() {
		return isDmpIssue;
	}

	public void setIsDmpIssue(String isDmpIssue) {
		this.isDmpIssue = isDmpIssue;
	}

	// get, add, remove methods for internal sets
	@Transient
	public Set<QIRSimpleAttachment> getAttachments() {
		return Collections.unmodifiableSet(getInternalQIRAttachments());
	}

	public void addAttachment(QIRSimpleAttachment attachment) {
		if (attachment != null) {
			attachment.setQir(this);
			getInternalQIRAttachments().add(attachment);
		}
	}

	public void removeAttachment(QIRSimpleAttachment attachment) {
		getInternalQIRAttachments().remove(attachment);
	}

	@Transient
	public Set<QIRRemarks> getRemarks() {
		return Collections.unmodifiableSet(getInternalQIRRemarks());
	}

	public void addRemarks(QIRRemarks remarks) {
		if (remarks != null) {
			remarks.setQir(this);
			getInternalQIRRemarks().add(remarks);
		}
	}

	public void removeRemarks(QIRRemarks remarks) {
		getInternalQIRRemarks().remove(remarks);
	}

	@Transient
	public Set<QIRVendorAction> getVendorActions() {
		return Collections.unmodifiableSet(getInternalQIRVendorActions());
	}

	public void addVendorActions(QIRVendorAction vendorAction) {
		if (vendorAction != null) {
			vendorAction.setQir(this);
			getInternalQIRVendorActions().add(vendorAction);
		}
	}

	public void removeVendorActions(QIRVendorAction vendorAction) {
		getInternalQIRVendorActions().remove(vendorAction);
	}

	@Column(name = "DATE_EMAIL_REMINDER_SENT")
	public Date getEmailReminderSentDate() {
		return emailReminderSentDate;
	}

	public void setEmailReminderSentDate(Date emailReminderSentDate) {
		this.emailReminderSentDate = emailReminderSentDate;
	}

	@OneToMany(targetEntity=QIRRemarks.class,
			mappedBy="qir",
			fetch=FetchType.EAGER,
			cascade=CascadeType.ALL)
	private Set<QIRRemarks> getInternalQIRRemarks() {
		if (internalQIRRemarks == null)
			internalQIRRemarks = new HashSet<QIRRemarks>();
		return internalQIRRemarks;
	}

	@SuppressWarnings("unused")
	private void setInternalQIRRemarks(Set<QIRRemarks> internalQIRRemarks) {
		this.internalQIRRemarks = internalQIRRemarks;
	}

	@ManyToOne
	@JoinColumn(name="DMP_QIR_TYPE_ID")
	public DmpQirType getDmpQirType() {
		return dmpQirType;
	}

	public void setDmpQirType(DmpQirType dmpQirType) {
		this.dmpQirType = dmpQirType;
	}

	@OneToMany(targetEntity=QIRVendorAction.class,
			mappedBy="qir",
			fetch=FetchType.EAGER,
			cascade=CascadeType.ALL)
	private Set<QIRVendorAction> getInternalQIRVendorActions() {
		if (internalQIRVendorActions == null)
			internalQIRVendorActions = new HashSet<QIRVendorAction>();
		return internalQIRVendorActions;
	}

	@SuppressWarnings("unused")
	private void setInternalQIRVendorActions(Set<QIRVendorAction> internalQIRVendorActions) {
		this.internalQIRVendorActions = internalQIRVendorActions;
	}

	@OneToMany(targetEntity=QIRSimpleAttachment.class,
			mappedBy="qir",
			fetch=FetchType.EAGER,
			cascade=CascadeType.ALL)
	private Set<QIRSimpleAttachment> getInternalQIRAttachments() {
		if (internalQIRAttachments == null)
			internalQIRAttachments = new HashSet<QIRSimpleAttachment>();
		return internalQIRAttachments;
	}

	@SuppressWarnings("unused")
	private void setInternalQIRAttachments(Set<QIRSimpleAttachment> internalQIRAttachments) {
		this.internalQIRAttachments = internalQIRAttachments;
	}

	@Transient//@Formula(value = "(select isnull(u.first_name,'') + ' ' + isnull(u.last_name,'') from ht.app_user u where u.user_name = SUBMITTED_BY)")
	public String getEmailSubmittedByName() {
		return emailSubmittedByName;
	}

	public void setEmailSubmittedByName(String emailSubmittedByName) {
		this.emailSubmittedByName = emailSubmittedByName;
	}

}