package gov.va.med.ccht.controller.validators;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.fw.util.StringUtils;

@Component
public class QIRFormValidator implements Validator {
	
	// Matches on any non-whitespace, non-keyboard characters.
	// May need to add other specific characters in later.
	private String blacklistRegex = "[^!-~\\s]+";
	private Pattern blackPattern = Pattern.compile(blacklistRegex);

	@Override
	public boolean supports(Class<?> arg0) {

		return QIRForm.class.isAssignableFrom(arg0);
	}

	@Override
	public void validate(Object target, Errors errors) {

		QIRForm form = (QIRForm) target;
		// Validate headline
		if (form.getHeadline() == null || form.getHeadline().length() == 0) {
			errors.rejectValue("headline", "NotEmpty.command.headline", "  - Headline is required.");
		}
		else {
			validateCharacters(form.getHeadline(), "headline", "invalidCharacters.command.headline", errors);
		}

		// Validate facility
		if (form.getSelectedFacilityId() == null || form.getSelectedFacilityId().length() == 0
				|| form.getSelectedFacilityId().equalsIgnoreCase("-1")) {
			errors.rejectValue("selectedFacilityId", "Min.command.selectedFacilityId", " - Facility is required.");
		}

		if (form.getSelectedVendorId() < 1) { // check
			errors.rejectValue("selectedVendorId", "Min.command.selectedVendorId", " - Vendor is required.");
		}

		if (form.getComplaint() == null || form.getComplaint().length() == 0) {
			errors.rejectValue("complaint", "NotEmpty.command.complaint", " - Cause of Complaint is required.");
		}
		else {
			validateCharacters(form.getComplaint(), "complaint", "invalidCharacters.command.complaint", errors);
		}

		if (form.getRemarks() == null || form.getRemarks().length() == 0) {
			errors.rejectValue("remarks", "NotEmpty.command.remarks", " - Recommendations and Remarks is required.");
		}
		else {
			validateCharacters(form.getRemarks(), "remarks", "invalidCharacters.command.remarks", errors);
		}

	}

	public void validateEditQir(QIRForm form, Errors errors) {
		// Validate is dmp issue selection
		if (form.getIsDmpIssue() != null && !form.getIsDmpIssue().equalsIgnoreCase("N")) {
			if (form.getIsDmpIssue().length() == 0 || form.getIsDmpIssue().equalsIgnoreCase("S")) {
				errors.rejectValue("isDmpIssue", "NotEmpty.command.isDmpIssue", "Please select DMP Issue.");
			}
			if (form.getIsDmpIssue().equalsIgnoreCase("Y") && form.getSelectedDmpQirTypeId().equalsIgnoreCase("-1")) {
				errors.rejectValue("selectedDmpQirTypeId", "NotEmpty.command.dmpQirType", "Please select DMP Issue Type.");
			}
		}

		// Validate complaints
		if (form.getComplaint() == null || form.getComplaint().length() == 0) {
			errors.rejectValue("complaint", "NotEmpty.command.complaint", " - Cause of Complaint is required.");
		}
		else {
			validateCharacters(form.getComplaint(), "complaint", "invalidCharacters.command.complaint", errors);
		}
		
		if(form.getVendorActions() != null && form.getVendorActions().length() > 0) {
			validateCharacters(form.getVendorActions(), "vendorActions", "invalidCharacters.command.vendorActions", errors);
		}
		
		if (form.getRemarks() != null && form.getRemarks().length() > 0) {
			validateCharacters(form.getRemarks(), "remarks", "invalidCharacters.command.remarks", errors);
		}
	}

	public void validateApproveQir(QIRForm form, Errors errors) {
		String dmpIssue = form.getIsDmpIssue();
		if (!"Y".equals(dmpIssue) && !"N".equals(dmpIssue)) {
			errors.rejectValue("isDmpIssue", "", "Please select DMP Issue.");
		}

		if ("Y".equals(dmpIssue) && "-1".equals(form.getSelectedDmpQirTypeId())) {
			errors.rejectValue("selectedDmpQirTypeId", "", "Please select DMP Issue Type.");
		}
	}

	public void validateReplyQir(QIRForm form, Errors errors) {
		if (StringUtils.isBlank(form.getVendorActions())) {
			errors.rejectValue("vendorActions", "", "Vendor Actions is required.");
		}
	}

	public void validateAgreeQir(QIRForm command, BindingResult result) {
		// TODO Auto-generated method stub

	}
	
	private void validateCharacters(String input, String field, String errorCode, Errors errors) {
		Matcher matcher = blackPattern.matcher(input);
		if(matcher.find()) {
			errors.rejectValue(field, errorCode);
		}
	}
}
