package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.AppUserGroup;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserPreference;
import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.terminology.InactiveReason;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.ui.common.AbstractConversionService;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.ui.model.UserRegistrationForm;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.ui.conversion.ConversionServiceException;
import gov.va.med.fw.ui.model.TermType;

@Service
public final class SecurityConversionService extends AbstractConversionService {

	@Autowired
	private SecurityDAO securityDao;
	/**
	 * Convert a User object to a UserForm object.
	 * @param user
	 * @param userForm
	 * @throws ConversionServiceException
	 */
	public void convert(User user, UserForm userForm) throws ConversionServiceException {
		userForm.setId(user.getId() == null ? null : String.valueOf(user.getId()));
		userForm.setUserName(user.getUsername());
		userForm.setGivenName(user.getFirstName());
		userForm.setMiddleName(user.getMiddleName());
		userForm.setFamilyName(user.getLastName());
		userForm.setTitle(user.getTitle());
		userForm.setDepartment(user.getDepartment());
		userForm.setDescription(user.getDescription());
		userForm.setOffice(user.getOffice());
		userForm.setTelephoneNumber(user.getTelephoneNumber());
		userForm.setEmail(user.getEmail());

		userForm.setRegistrationStatus(toTermType(user.getRegistrationStatus()));
		userForm.setRegistrationDeniedReason(toTermType(user.getRegistrationDeniedReason()));
		userForm.setVisn(toTermType(user.getVisn()));
		
		if(userForm.getVisn() !=  null) {
			userForm.setAssignedVisnId(userForm.getVisn().getLabel().trim());
		}
		userForm.setFacility(toTermType(user.getFacility()));
		userForm.setVendor(toTermType(user.getVendor()));
		userForm.setSecondaryFacility(toTermType(user.getSecondaryFacility()));
		userForm.setRegistrationReason(toTermType(user.getRegistrationReason()));

		userForm.setAccountExpireDate(user.getAccountExpireDate());
		// Lock
		userForm.setLocked(user.isAccountLocked());
		userForm.setAccountLockDate(user.getAccountLockDate());
		// Inactive
		userForm.setInactive(user.isInactive());
		userForm.setInactiveReason(toTermType(user.getInactiveReason()));
		userForm.setInactiveDate(user.getInactiveDate());

		userForm.setLoginFailedDate(user.getLoginFailedDate());
		userForm.setLoginFailedCount(user.getLoginFailedCount());

		// Roles
		List<TermType> assignedRoles = new ArrayList<TermType>();
		for (UserRole userRole : user.getRoles()) {
			assignedRoles.add(new TermType(userRole.getRole().getName(), userRole.getRole().getName()));
		}
		userForm.setRoles(assignedRoles);

		// Only one group is supported
		TermType userGroup = null;
		for (AppUserGroup appUserGroup : user.getUserGroups()) {
			userGroup = new TermType(appUserGroup.getUserGroup().getName(), appUserGroup.getUserGroup().getCode());
			userForm.setUserGroup(userGroup);
			break;
		}

		// User Preferences
		String value = user.getPreferenceValue(UserPreference.MULTIPLE_WINDOWS);
		userForm.setMultipleWindows(value != null ? Boolean.valueOf(value) : null);
		userForm.setMultipleWindows(value != null ? Boolean.valueOf(value) : null);
	}

	/**
	 * Convert a UserForm object to a User object
	 * @param userForm
	 * @param user
	 * @throws ConversionServiceException
	 */
	public void convert(UserForm userForm, User user) throws ConversionServiceException {
		try {
			if(userForm.getRegistrationStatus() != null) {
				user.setRegistrationStatus(getLookup(RegistrationStatus.class, userForm.getRegistrationStatus()));
			}
			user.setVisn(getVisn(userForm.getVisn()));
			user.setFacility(getFacility(userForm.getFacility()));
			user.setSecondaryFacility(getFacility(userForm.getSecondaryFacility()));
			if (user.getFacility() != null) {
				user.setVisn(user.getFacility().getVisn());
			}
			user.setVendor(getVendor(userForm.getVendor()));
			user.setAccountExpireDate(userForm.getAccountExpireDate());

			if (userForm.getLocked() != null && userForm.getLocked()) {
				if (user.getAccountLockDate() == null) {
					user.setAccountLockDate(new Date());
				}
			} else {
				user.setAccountLockDate(null);
				user.setLoginFailedCount((short) 0);
			}

			if (userForm.getInactive() != null && userForm.getInactive()) {
				user.setInactiveDate(userForm.getInactiveDate());
				user.setInactiveReason(getLookup(InactiveReason.class, userForm.getInactiveReason()));
			} else {
				user.setInactiveDate(null);
				user.setInactiveReason(null);
			}

			String username = SecurityContextHelper.getCurrentUser().getUsername();
			Date currentDate = Calendar.getInstance().getTime();
			
			// roles
			String[] selectedRoles = userForm.getSelectedRoles();
			List<UserRole> deletedRoles = new ArrayList<UserRole>(user.getRoles());
			List<Role> newRoles = new ArrayList<Role>();
			
			// convert from array of names to List of Roles
			for(String role : selectedRoles) {
				Role newRole = securityDao.getRoleByName(role);
				newRoles.add(newRole);
			}
			
			// delete roles
			for (UserRole userRole : deletedRoles) {
				user.removeRole(userRole);
			}
			
			// add roles
			for (Role newRole : newRoles) {
				UserRole userRole = new UserRole();
				userRole.setRole(newRole);
				userRole.setRecordModifiedDate(currentDate);
				userRole.setRecordModifiedBy(username);
				userRole.setRecordCreatedBy(username);
				userRole.setRecordCreatedDate(currentDate);
				userRole.setUser(user);
				userRole.setRecordModifiedCount((short) 0); // Modifying role does not apply. roles are only added/removed from user
				user.addRole(userRole);
			}
			
			// group - read only no update
			// address - no update not yet supported

			// User Preferences
			user.addOrUpdatePreference(UserPreference.MULTIPLE_WINDOWS, String.valueOf(userForm.getMultipleWindows()));

		} catch (TerminologyException e) {
			throw new ConversionServiceException("UserForm to User conversion failed ", e);
		}
	}

	public void convert(UserRegistrationForm urf, User user) throws ConversionServiceException {
		try {
			user.setFirstName(urf.getGivenName());
			user.setMiddleName(urf.getMiddleName());
			user.setLastName(urf.getFamilyName());
			user.setTitle(urf.getTitle());
			user.setDepartment(urf.getDepartment());
			user.setDescription(urf.getDescription());
			user.setOffice(urf.getOffice());
			user.setTelephoneNumber(urf.getPhone());
			user.setEmail(urf.getEmail());

			user.setRegistrationStatus(urf.getSelectedRegistrationStatus());
			user.setRegistrationDeniedReason(urf.getRegistrationDeniedReason());

			user.setVisn(urf.getVisn());
			user.setFacility(urf.getFacility());
			user.setSecondaryFacility(null);

			if (user.getFacility() != null) {
				user.setVisn(user.getFacility().getVisn());
			}

			user.setVendor(urf.getVendor());
			user.setRegistrationReason(urf.getRegistrationReason());
			user.setAccountExpireDate(urf.getAccountExpireDate());

			if (urf.isLocked()) {
				if (user.getAccountLockDate() == null) {
					user.setAccountLockDate(new Date());
				}
			} else {
				user.setAccountLockDate(null);
				user.setLoginFailedCount((short) 0);
			}

			if (urf.isInactive()) {
				user.setInactiveDate(urf.getInactiveDate());
				user.setInactiveReason(urf.getInactiveReason());
			} else {
				user.setInactiveDate(null);
				user.setInactiveReason(null);
			}

			// roles
			List<SimpleRole> newRoles = urf.getAssignedUserRoles();

			for (SimpleRole role : newRoles) {

				UserRole newRole = new UserRole();
				TermType termType = new TermType();
				termType.setLabel(role.getName());
				termType.setValue(role.getId().toString());
				newRole.setRole(getRole(termType));
				user.addRole(newRole);
			}

		} catch (Exception e) {
			throw new ConversionServiceException("UserRegistrationForm to User conversion failed ", e);
		}
	}
}
