package gov.va.med.ccht.controller;

import java.io.OutputStream;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.persistence.EntityManager;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.log4j.Logger;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
//import org.jfree.util.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import gov.va.med.ccht.model.CCHTPermissions;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRRemarks;
import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSearchResult;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.qir.QIRStatusTypeEnum;
import gov.va.med.ccht.model.qir.QIRVendorAction;
import gov.va.med.ccht.persistent.QIRDAO;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.persistent.hibernate.AbstractDAOAction;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.util.date.TimeZoneUtils;

@Controller
public class SearchAllQIRSController extends CchtController {

	@Autowired
	private QIRDAO qirDao;
	
	private Logger logger = Logger.getLogger(SearchAllQIRSController.class);
	
	//Add @InitBinder to fix Fortify issue
	@InitBinder
    public void initBinder(WebDataBinder binder)
    {   
    	String[] allowedFields = {"selectedFacilityId", "selectedVendorId", 
    							  "selectedComplaintTypeId", "selectedStatusTypeId", 
    							  "fromDateAsString", "toDateAsString", "id"};
    	binder.setAllowedFields(allowedFields);    	
    }

	@RequestMapping(value = "/searchAllQirs.html", method = RequestMethod.GET)
	public String showSearchAllQirs(Model model, HttpSession session) throws Exception {
		
		User currentUser = CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class);
		QIRSearchParameters qirSearchParameters = new QIRSearchParameters();

		Set<String> uniqueIds = new HashSet<String>();

		List<QIRSearchResult> results = qirService.searchQIR(qirSearchParameters);
		
		results.forEach(result -> {
			uniqueIds.add(result.getId());
		});

		addReferenceData(model);
		model.addAttribute("qirList", results);
		model.addAttribute("command", qirSearchParameters);

		session.setAttribute(ViewMyQIRSController.SESSION_ATTR_LAST_QIR_SEARCH_RESULT_ID_SET, uniqueIds);

		return "searchAllQirs";
	}

	@RequestMapping(value = "/searchAllQirs.html", method = { RequestMethod.POST, RequestMethod.GET }, params = {
			"Search" })
	public String showSearchAllQirs(@ModelAttribute("command") QIRSearchParameters command, Model model,
			BindingResult bindingResult, HttpSession session) throws Exception {
		if(hasErrors(command, bindingResult)) {
			Set<String> uniqueIds = new HashSet<String>();
			command.setId("");

			addReferenceData(model);
			model.addAttribute("qirList", new ArrayList<QIRSearchResult>());
			model.addAttribute("command", command);

			session.setAttribute(ViewMyQIRSController.SESSION_ATTR_LAST_QIR_SEARCH_RESULT_ID_SET, uniqueIds);
			return "searchAllQirs";
		}
		
		String selectedId = command.getId();
		if (selectedId != null && selectedId.length() > 0
				&& !selectedId.equalsIgnoreCase(NEGATIVE_ONE)) {
			command.setId(selectedId);
		}		
		
		String selectedFacilityId = command.getSelectedFacilityId();
		if (selectedFacilityId != null && selectedFacilityId.length() > 0
				&& !selectedFacilityId.equalsIgnoreCase(NEGATIVE_ONE)) {
			command.setFacility(new TermType(selectedFacilityId, selectedFacilityId));
		}

		String selectedVendorId = command.getSelectedVendorId();
		if (selectedVendorId != null && selectedVendorId.length() > 0
				&& !selectedVendorId.equalsIgnoreCase(NEGATIVE_ONE)) {
			command.setVendor(new TermType(selectedVendorId, selectedVendorId));
		}

		String getSelectedComplaintTypeId = command.getSelectedComplaintTypeId();
		if (getSelectedComplaintTypeId != null && getSelectedComplaintTypeId.length() > 0
				&& !getSelectedComplaintTypeId.equalsIgnoreCase(NEGATIVE_ONE)) {
			command.setQirType(new TermType(getSelectedComplaintTypeId, getSelectedComplaintTypeId));
		}

		String selectedStatusTypeId = command.getSelectedStatusTypeId();
		if (selectedStatusTypeId != null && selectedStatusTypeId.length() > 0
				&& !selectedStatusTypeId.equalsIgnoreCase(NEGATIVE_ONE)) {
			command.setQirStatuses(Arrays.asList(selectedStatusTypeId));
		}

		String fromDateAsString = command.getFromDateAsString();

		if (fromDateAsString != null && fromDateAsString.length() > 1) {
			Date fromDate = DateUtils.parseDate(fromDateAsString, DateUtils.MMDDYYYY);
			command.setSubmittedFromDate(fromDate);
		}

		String toDateAsString = command.getToDateAsString();

		if (toDateAsString != null && toDateAsString.length() > 1) {
			Date toDate = DateUtils.parseDate(toDateAsString, DateUtils.MMDDYYYY);
			command.setSubmittedToDate(toDate);
		}
		List<QIRSearchResult> results = getQirs(command, session);

		addReferenceData(model);

		model.addAttribute("qirList", results);

		model.addAttribute("command", command);
		return "searchAllQirs";
	}

	private void addReferenceData(Model model) throws Exception, TerminologyException {
		model.addAttribute("vendors", getVendors());
		model.addAttribute("facilities", getFacilities());
		model.addAttribute("dmpQirTypes", getTerms("gov.va.med.ccht.model.qir.QIRType"));

		Map<String, String> statusTypes = getTerms("gov.va.med.ccht.model.qir.QIRStatusType");

		List<String> validTypes = null;
		if (getCurrentUser().isPermissionGranted(CCHTPermissions.VENDOR)) {
			
			validTypes = Arrays.asList(
				QIRStatusTypeEnum.Approved.toString(),
				QIRStatusTypeEnum.Replied.toString(),
				QIRStatusTypeEnum.Agreed.toString(),
				QIRStatusTypeEnum.Closed.toString());

		} 

		if (validTypes != null) {
			for (Iterator<String> it = statusTypes.values().iterator(); it.hasNext();)
				if (!validTypes.contains(it.next()))
					it.remove();
		}
		
		boolean isVendor = false;
		if(getCurrentUser().isPermissionGranted(CCHTPermissions.VENDOR)) {
			isVendor = true;
		}
		
		model.addAttribute("userIsAdmin", isNationalAdmin());
		model.addAttribute("isVendor", isVendor);
		model.addAttribute("qirStatusTypes", statusTypes);
	}

	private List<QIRSearchResult> getQirs(QIRSearchParameters searchParams, HttpSession session) throws Exception {

		List<QIRSearchResult> results = qirService.searchQIR(searchParams);

		Set<String> uniqueIds = new HashSet<String>();
		
		results.forEach(result -> {
			uniqueIds.add(result.getId());
		});
		
		session.setAttribute(ViewMyQIRSController.SESSION_ATTR_LAST_QIR_SEARCH_RESULT_ID_SET, uniqueIds);

		return results;
	}

	@RequestMapping(value = "/qirExcelExport.html")
	public void excelExport(@RequestParam(required = false) Long qirId, HttpSession session, OutputStream os,
			HttpServletResponse response) throws Exception {
		String filename = "IHTA QIR " + DateUtils.format(new Date(), "MMddyyyy");
		response.setContentType("application/vnd.ms-excel");
		response.setHeader("Content-disposition", "attachment; filename=" + filename + ".xlsx");

		@SuppressWarnings("unchecked")
		final Set<Long> ids = qirId != null ? new HashSet<Long>(Arrays.asList(qirId))
				: (Set<Long>) session.getAttribute(ViewMyQIRSController.SESSION_ATTR_LAST_QIR_SEARCH_RESULT_ID_SET);

		try {
			Workbook wb = new XSSFWorkbook();
			
			Sheet sheet = wb.createSheet(filename);

			if (ids != null && !ids.isEmpty()) {
				
				List<QIR> qirs = qirDao.findByIds(ids);
				
				buildExcelFile(sheet, qirs);
			}
			wb.write(response.getOutputStream());
		} catch (Exception e) {
			Logger.getLogger(getClass()).error("Error creating Excel file", e);
			throw e;
		}
	}

	private void setCellWhiteBorder(CellStyle style) {
		style.setBorderBottom(CellStyle.BORDER_THIN);
		style.setBottomBorderColor(IndexedColors.WHITE.getIndex());
		style.setBorderLeft(CellStyle.BORDER_THIN);
		style.setLeftBorderColor(IndexedColors.WHITE.getIndex());
		style.setBorderRight(CellStyle.BORDER_THIN);
		style.setRightBorderColor(IndexedColors.WHITE.getIndex());
		style.setBorderTop(CellStyle.BORDER_MEDIUM_DASHED);
		style.setTopBorderColor(IndexedColors.WHITE.getIndex());
	}

	private void buildExcelFile(Sheet sheet, List<QIR> qirs) {
		Workbook workbook = sheet.getWorkbook();
		CreationHelper createHelper = workbook.getCreationHelper();

		Row row;
		Cell cell;
		Font font;

		// --------------------- Styles

		Font defaultFont = workbook.getFontAt((short) 0);
		defaultFont.setFontName("Calibri");
		defaultFont.setFontHeightInPoints((short) 10);

		CellStyle boldItalic = workbook.createCellStyle();
		font = workbook.createFont();
		font.setBoldweight(Font.BOLDWEIGHT_BOLD);
		font.setItalic(true);
		font.setFontName("Calibri");
		font.setFontHeightInPoints((short) 12);
		boldItalic.setFont(font);
		boldItalic.setAlignment(CellStyle.ALIGN_CENTER);
		boldItalic.setVerticalAlignment(CellStyle.VERTICAL_TOP);
		boldItalic.setWrapText(true);
		setCellWhiteBorder(boldItalic);

		CellStyle boldSmall = workbook.createCellStyle();
		font = workbook.createFont();
		font.setBoldweight(Font.BOLDWEIGHT_BOLD);
		font.setFontName("Calibri");
		font.setFontHeight((short) 150); // 7.5 pt
		boldSmall.setFont(font);
		boldSmall.setAlignment(CellStyle.ALIGN_CENTER);
		boldSmall.setVerticalAlignment(CellStyle.VERTICAL_TOP);
		boldSmall.setWrapText(true);
		setCellWhiteBorder(boldSmall);

		CellStyle bold = workbook.createCellStyle();
		font = workbook.createFont();
		font.setBoldweight(Font.BOLDWEIGHT_BOLD);
		font.setFontName("Calibri");
		font.setFontHeightInPoints((short) 10);
		bold.setFont(font);
		bold.setVerticalAlignment(CellStyle.VERTICAL_BOTTOM);
		bold.setBorderBottom(CellStyle.BORDER_MEDIUM);
		bold.setFillBackgroundColor(IndexedColors.GREY_50_PERCENT.getIndex());

		CellStyle small = workbook.createCellStyle();
		font = workbook.createFont();
		font.setFontName("Calibri");
		font.setFontHeight((short) 150); // 7.5 pt
		small.setFont(font);
		small.setVerticalAlignment(CellStyle.VERTICAL_BOTTOM);
		small.setWrapText(true);
		setCellWhiteBorder(small);

		CellStyle wrappedDefault = workbook.createCellStyle();
		wrappedDefault.setAlignment(CellStyle.ALIGN_LEFT);
		wrappedDefault.setWrapText(true);

		// ---------------------- Row 0

		row = sheet.createRow(0);

		cell = row.createCell(0);
		cell.setCellValue("Date of Export: " + DateUtils.format(new Date(), DateUtils.MMDDYYYY));
		cell.setCellStyle(boldItalic);

		cell = row.createCell(1);
		cell.setCellValue("Total count of QIRs contained in this export: " + qirs.size());
		cell.setCellStyle(boldItalic);

		// ---------------------- Row 1

		row = sheet.createRow(1);

		cell = row.createCell(0);
		cell.setCellValue("* - 'Vendor Met Due Date' calculation logic:");
		cell.setCellStyle(boldSmall);
		sheet.addMergedRegion(new CellRangeAddress(1, // first row (0-based)
				4, // last row (0-based)
				0, // first column (0-based)
				0 // last column (0-based)
		));

		cell = row.createCell(1);
		cell.setCellValue(
				"If the Vendor has not yet clicked the Reply button and the Vendor Response Due Date has not yet occurred, 'N/A' is present in the 'Vendor Met Due Date?' column.");
		cell.setCellStyle(small);

		cell = row.createCell(2);
		cell.setCellValue("** - '# of Days Open' calculation logic:");
		cell.setCellStyle(boldSmall);
		sheet.addMergedRegion(new CellRangeAddress(1, // first row (0-based)
				4, // last row (0-based)
				2, // first column (0-based)
				2 // last column (0-based)
		));

		cell = row.createCell(3);
		cell.setCellValue(
				"If the QIR does not have a status of Withdrawn or Closed,  'Active' is present in the 'Status' column.");
		cell.setCellStyle(small);

		// ---------------------- Row 2

		row = sheet.createRow(2);

		cell = row.createCell(1);
		cell.setCellValue(
				"If the Vendor has not yet clicked the Reply button and the Vendor Response Due Date has occurred, 'No' is present in the 'Vendor Met Due Date?' column.");
		cell.setCellStyle(small);

		cell = row.createCell(3);
		cell.setCellValue("If the QIR has a status of Withdrawn, 'Withdrawn' is present in the 'Status' column.");
		cell.setCellStyle(small);

		// ---------------------- Row 3

		row = sheet.createRow(3);

		cell = row.createCell(1);
		cell.setCellValue(
				"If the Vendor clicks the Reply button prior to or on the same day as the Vendor Response Due Date,  'Yes' is present in the 'Vendor Met Due Date?' column.");
		cell.setCellStyle(small);

		cell = row.createCell(3);
		cell.setCellValue(
				"If the QIR is Closed, the Submitted Date was subtracted from the Closed Date and the resulting # of business days the QIR was open is present in the '# of Days Open' column.");
		cell.setCellStyle(small);

		// ---------------------- Row 4

		row = sheet.createRow(4);

		cell = row.createCell(1);
		cell.setCellValue(
				"If the Vendor clicks the Reply button after the Vendor Response Due Date, then 'No' is present in the 'Vendor Met Due Date?' column.");
		cell.setCellStyle(small);

		// ---------------------- Row 5

		row = sheet.createRow(5);

		String[] columnHeaders = new String[] { "TN", "Status", "Headline", "Complaint Cause", "DMP Issue",
				"Type of DMP", "VISN", "Facility", "Submitted By", "Submitted Date", "Closed Date", "# of Days Open**",
				"Vendor", "Device Model", "Type", "Has Attachment", "Vendor Response Due Date", "Date Vendor Replied",
				"Vendor Met Due Date?*", "Remarks", "Vendor Comments" };

		for (int i = 0; i < columnHeaders.length; i++) {
			cell = row.createCell(i);
			cell.setCellValue(columnHeaders[i]);
			cell.setCellStyle(bold);
		}

		// ---------------------- QIR data rows

		int firstDataRowIndex = 6;
		try {
			appendData(sheet, qirs, createHelper, wrappedDefault, firstDataRowIndex);
		} catch (Exception e) {
			try {
				logger.error("There was an error writing the QIRs to the spreadsheet", e);
				row = sheet.createRow(sheet.getLastRowNum() + 1);
				row.createCell(0).setCellValue(ExceptionUtils.getStackTrace(e));
			} catch (Exception e2) {
				logger.error("There was an error writing debug info into the spreadsheet", e2);
			}
		}

		// ---------------------- Add filters

		sheet.setAutoFilter(new CellRangeAddress(firstDataRowIndex - 1, firstDataRowIndex + qirs.size() - 1, 0, 20));

		// ---------------------- Column Widths

		int[] widths = new int[] { 256, 256, 256, 256 };

		int maxFontCharWidth = 32;
		for (int i = 0; i < widths.length; i++)
			sheet.setColumnWidth(i, maxFontCharWidth * widths[i]);

		Row firstDataRow = sheet.getRow(firstDataRowIndex);
		if (firstDataRow != null) {
			int noOfColumns = firstDataRow.getLastCellNum();
			for (int i = widths.length; i < noOfColumns; i++)
				sheet.autoSizeColumn(i);
		}

	}

	private void appendData(Sheet sheet, List<QIR> qirs, CreationHelper createHelper, CellStyle wrappedDefault,
			int firstDataRowIndex) {
		Row row;
		Cell cell;
		int rowIndex = firstDataRowIndex;
		for (Iterator<QIR> it = qirs.iterator(); it.hasNext(); rowIndex++) {
			QIR qir = it.next();
			QIRStatusTypeEnum statusEnum = QIRStatusTypeEnum.toEnum(qir.getQirStatusType());

			row = sheet.createRow(rowIndex);

			row.createCell(0).setCellValue(qir.getId());
			row.createCell(1).setCellValue(statusEnum == QIRStatusTypeEnum.Null ? "N/A" : statusEnum.toString());

			cell = row.createCell(2);
			cell.setCellStyle(wrappedDefault);
			cell.setCellValue(createHelper.createRichTextString(qir.getSummary()));

			cell = row.createCell(3);
			cell.setCellStyle(wrappedDefault);
			cell.setCellValue(createHelper.createRichTextString(qir.getComplaint()));

			row.createCell(4).setCellValue(StringUtils.defaultString(qir.getIsDmpIssue(), "N/A"));
			row.createCell(5).setCellValue(
					qir.getDmpQirType() == null ? "N/A" : "Y".equals(qir.getDmpQirType().getName()) ? "Yes" : "No");
			row.createCell(6).setCellValue(qir.getVisn() == null ? "N/A" : qir.getVisn().getName());

			SimpleFacility facility = qir.getFacility();
			if (facility != null) {
				String num = facility.getStationNumber();
				String name = facility.getName();
				String finalName = StringUtils.defaultString(num)
						+ (StringUtils.isNotBlank(num) && StringUtils.isNotBlank(name) ? " " : "")
						+ StringUtils.defaultString(name);
				row.createCell(7).setCellValue(finalName);
			} else {
				row.createCell(7).setCellValue("N/A");
			}

			row.createCell(8).setCellValue(qir.getSubmittedByName());
			row.createCell(9).setCellValue(
					StringUtils.defaultString(DateUtils.format(qir.getSubmittedDate(), "M/d/yyyy"), "N/A"));
			row.createCell(10)
					.setCellValue(statusEnum == QIRStatusTypeEnum.Closed
							? StringUtils.defaultString(DateUtils.format(qir.getStatusChangeDate(), "M/d/yyyy"), "N/A")
							: "N/A");

			if (statusEnum == QIRStatusTypeEnum.Null) {
				row.createCell(11).setCellValue("N/A");
			} else if (statusEnum != QIRStatusTypeEnum.Closed
					&& statusEnum != QIRStatusTypeEnum.Withdrawn) {
				row.createCell(11).setCellValue("Active");
			} else if (statusEnum == QIRStatusTypeEnum.Withdrawn) {
				row.createCell(11).setCellValue("Withdrawn");
			} else {
				row.createCell(11).setCellValue(String
						.valueOf(DateUtils.getElapsedBusinessDays(qir.getSubmittedDate(), qir.getStatusChangeDate())));
			}

			row.createCell(12).setCellValue(
					qir.getVendor() == null ? "N/A" : StringUtils.defaultString(qir.getVendor().getName(), "N/A"));
			row.createCell(13).setCellValue(qir.getDeviceType() == null ? "N/A"
					: StringUtils.defaultString(qir.getDeviceType().getName(), "N/A"));
			row.createCell(14).setCellValue(
					qir.getQirType() == null ? "N/A" : StringUtils.defaultString(qir.getQirType().getName(), "N/A"));
			row.createCell(15).setCellValue(qir.getAttachments().isEmpty() ? "No" : "Yes");

			Date responseDueDate = qir.getVendorResponseDueDate();
			row.createCell(16)
					.setCellValue(StringUtils.defaultString(DateUtils.format(responseDueDate, "M/d/yyyy"), "N/A"));

			List<QIRVendorAction> vendorActionsList = new ArrayList<QIRVendorAction>(qir.getVendorActions());
			Collections.sort(vendorActionsList, Collections.reverseOrder());
			Date lastRepliedDate = null;
			for (QIRVendorAction vendorActions : vendorActionsList) {
				if (lastRepliedDate == null || vendorActions.getDate().after(lastRepliedDate))
					lastRepliedDate = vendorActions.getDate();
			}
			String repliedDate = statusEnum == QIRStatusTypeEnum.Replied 
					? StringUtils.defaultString(DateUtils.format(lastRepliedDate, "M/d/yyyy"), "N/A") : "N/A";

			row.createCell(17).setCellValue(repliedDate);

			String metDueDate = "N/A";
			Date now = new Date();
			if (responseDueDate != null) {
				if ("N/A".equals(repliedDate) && responseDueDate.before(now)) {
					metDueDate = "No";
				} else if (lastRepliedDate != null) {
					metDueDate = lastRepliedDate.after(responseDueDate) ? "No" : "Yes";
				}
			}
			row.createCell(18).setCellValue(metDueDate);

			String remarksHistory = "";
			List<QIRRemarks> remarksList = new ArrayList<QIRRemarks>(qir.getRemarks());
			Collections.sort(remarksList, Collections.reverseOrder());
			for (QIRRemarks remarks : remarksList) {
				remarksHistory += TimeZoneUtils.convertDateToTimeZone(remarks.getDate(), TimeZone.getDefault()) + ", "
						+ remarks.getSubmittedByName() + ": " + remarks.getRemarks() + "\n";
			}
			cell = row.createCell(19);
			cell.setCellStyle(wrappedDefault);
			cell.setCellValue("".equals(remarksHistory) ? "N/A" : remarksHistory);

			String vendorActionsHistory = "";
			for (QIRVendorAction vendorActions : vendorActionsList) {
				vendorActionsHistory += TimeZoneUtils.convertDateToTimeZone(vendorActions.getDate(),
						TimeZone.getDefault()) + ", " + vendorActions.getSubmittedByName() + ": "
						+ vendorActions.getAction() + "\n";
			}
			cell = row.createCell(20);
			cell.setCellStyle(wrappedDefault);
			cell.setCellValue("".equals(vendorActionsHistory) ? "N/A" : vendorActionsHistory);
		}
	}
	
	private boolean hasErrors(QIRSearchParameters command, BindingResult result) {
		boolean hasErrors = false;
		if(command.getId() != null && command.getId().length() > 0) {
			// If there are any non-numeric characters the tracking number is invalid.
			Pattern regex = Pattern.compile("[^0-9]+");
			Matcher matcher = regex.matcher(command.getId());
			if(matcher.find()) {
				result.rejectValue("id", "tracking.number.invalid", "The tracking number provided is invalid.");
				hasErrors = true;
			}
		}
		Pattern pattern = Pattern.compile("[0-9]{2}\\/[0-9]{2}\\/[0-9]{4}");
		if(command.getFromDateAsString() != null && command.getFromDateAsString().length() > 0) {
			Matcher matcher = pattern.matcher(command.getFromDateAsString());
			if(command.getFromDateAsString().length() == 10 && matcher.find()) {
				try {
					DateUtils.parseDateStrictly(command.getFromDateAsString(), DateUtils.MMDDYYYY);
				}
				catch(ParseException e) {
					result.rejectValue("fromDateAsString", "from.date.invalid", "The from date is invalid.");
					hasErrors = true;
				}
			}
			else {
				result.rejectValue("fromDateAsString", "from.date.invalid", "The from date is invalid.");
				hasErrors = true;
			}
		}
		if(command.getToDateAsString() != null && command.getToDateAsString().length() > 0) {
			Matcher matcher = pattern.matcher(command.getToDateAsString());
			if(command.getToDateAsString().length() == 10 && matcher.find()) {
				try {
					DateUtils.parseDateStrictly(command.getToDateAsString(), DateUtils.MMDDYYYY);
				}
				catch(ParseException e) {
					result.rejectValue("toDateAsString", "to.date.invalid", "The to date is invalid.");
					hasErrors = true;
				}
			}
			else {
				result.rejectValue("toDateAsString", "to.date.invalid", "The to date is invalid.");
				hasErrors = true;
			}
		}
		return hasErrors;
	}
}
