package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.naming.SizeLimitExceededException;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.ServletRequestUtils;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.support.SessionStatus;
import org.springframework.web.multipart.MultipartFile;

import gov.va.med.ccht.controller.validators.QIRFormValidator;
import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.DeviceType;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRSimpleAttachment;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.qir.QIRType;
import gov.va.med.ccht.model.terminology.DocumentType;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;

@Controller
public class NewQIRController extends CchtController {
	
	@Autowired
	private QIRConversionService qirConversionService;
	@Autowired
	private QIRFormValidator validator;
	
	@Value("${qir.maxCombinedAttachmentBytes}")
	private int maxCombinedAttachmentLength;
	
	
    @InitBinder
    public void initBinder(WebDataBinder binder)
    {   
    	String[] allowedFields = {"headline", "selectedFacilityId", "selectedQirType", 
    							  "purchaseOrderNumber", "selectedDeviceId", 
    							  "selectedVendorId", "serialNumber", 
    							  "complaint", "remarks", "attachmentFiles"};
    	binder.setAllowedFields(allowedFields);    	
    }
    
	@RequestMapping(value = "/newQir.html", method = RequestMethod.GET)
	public String showNewQIRForm(Model model) throws Exception {
		QIRForm qirForm = new QIRForm();

		populateFormDevicesAndVendors(qirForm);

		model.addAttribute("command", qirForm);
		return "newQir";
	}

	@RequestMapping(value = "/newQir.html", params = { "Submit" }, method = RequestMethod.POST)
	public String saveNewQIR(@ModelAttribute("command") @Valid QIRForm command, BindingResult result,
			SessionStatus status, HttpServletRequest request, Model model) throws Exception {
		validator.validate(command, result);
		if (result.hasErrors()) {
			populateFormDevicesAndVendors(command);
			return "newQir";
		}
		QIR qir = new QIR();
		populateFormDevicesAndVendors(command);
		populateQIRForm(command);
		qirConversionService.convertToQIR(command, qir,
				CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class));

		/*
		 * Following two methods should be within a single transaction, but they
		 * want to preserve the original semantics - CPB
		 */
		qir.setQirStatusType(qirService.findQIRStatusTypeByName("New"));
		qirService.updateQIR(qir);
		addAttachments(command, qir, terminologyCache, qirService, request, maxCombinedAttachmentLength);

		userNotifier.notifyUserOnceWithMessage(request, "New QIR created successfully.");
		status.setComplete();
		return "redirect:" + basePath + "newQir.html";
	}

	public static void addAttachments(QIRForm command, QIR qir, TerminologyCache terminologyCache,
			QIRService qirService, HttpServletRequest request, int maxCombinedAttachmentLength) throws SizeLimitExceededException, Exception {
		DocumentType unknownDocType = null;

		List<QIRAttachment> filesToAdd = new ArrayList<QIRAttachment>();
		List<QIRAttachment> filesToDelete = new ArrayList<QIRAttachment>();

		for (MultipartFile file : command.getAttachmentFiles()) {
			String fileName = file.getOriginalFilename();
			String fileSuffix = fileName == null ? "" : StringUtils.substringAfterLast(fileName.toUpperCase(), ".");

			DocumentType docType = terminologyCache.getTermByName(DocumentType.class, fileSuffix);
			if (docType == null) {
				if (unknownDocType == null)
					unknownDocType = terminologyCache.getTermByCode(DocumentType.class, "10" /* "object" */);
				docType = unknownDocType;
			}

			QIRAttachment qirAttachment = new QIRAttachment(qir.getId(), file.getBytes(), (int) file.getSize(),
					new Date(), docType, fileName);
			qirAttachment.setRecordModifiedCount((short) 1);
			qirAttachment.setRecordCreatedBy(getCurrentUser().getUsername());
			qirAttachment.setRecordModifiedBy(getCurrentUser().getUsername());
			final Date currentDate = DateUtils.getCurrentDateTime();
			qirAttachment.setRecordCreatedDate(currentDate);
			qirAttachment.setRecordModifiedDate(currentDate);
			filesToAdd.add(qirAttachment);
		}

		long[] deletedAttachments = ServletRequestUtils.getLongParameters(request, "deletedAttachments");
		for (long id : deletedAttachments) {
			QIRAttachment a = qirService.getQIRAttachment(id);
			filesToDelete.add(a);
		}

		int length = 0;
		for (QIRSimpleAttachment a : qir.getAttachments())
			length += a.getLength();
		for (QIRAttachment a : filesToAdd)
			length += a.getLength();
		for (QIRAttachment a : filesToDelete)
			length -= a.getLength();

		if (length > maxCombinedAttachmentLength)
			throw new SizeLimitExceededException("Exceeded max file size of " + (maxCombinedAttachmentLength / 1024.0 / 1024.0) + "Mb.");

		for (QIRAttachment a : filesToAdd)
			qirService.updateQIRAttachment(a);
		for (long id : deletedAttachments)
			qirService.deleteQIRAttachment(id);
	}

	private void populateQIRForm(QIRForm qirForm) {
		User theUser = (User) getCurrentUser();
		SimpleVisn visn = theUser.getVisn();

		if (visn != null) {
			TermType theVisn = new TermType();
			theVisn.setLabel(visn.getDescription());
			theVisn.setValue(visn.getDescription());
			qirForm.setVisn(theVisn);
		}

		qirForm.setQirStatusType(qirService.findQIRStatusTypeByName("New"));
		qirForm.setSubmittedByName(theUser.getUserCredentials().getUserID());
		qirForm.setSubmittedBy(theUser.getUserCredentials().getUserID());
		qirForm.setSubmittedDate(new Date());
		qirForm.setStatusChangeDate(qirForm.getSubmittedDate());

		int selectedDeviceId = qirForm.getSelectedDeviceId();

		if (selectedDeviceId > 0) {
			DeviceType deviceType = qirForm.getDevices().get(Long.valueOf(selectedDeviceId));
			TermType device = new TermType();
			device.setValue(deviceType.getCode());
			qirForm.setDeviceType(device);
		}

		Long selectedQirTypeId = qirForm.getSelectedQirType();
		String typeName = qirForm.getQirTypes().get(selectedQirTypeId);
		if (selectedQirTypeId != null && selectedQirTypeId > 0) {
			TermType qirType = new TermType();
			qirType.setLabel(typeName);
			qirType.setValue(selectedQirTypeId.toString());
			qirForm.setQirType(qirType);
		}

		int selectedVendorId = qirForm.getSelectedVendorId();
		Vendor selectedVendor = qirForm.getVendors().get(selectedVendorId);
		if (selectedVendor != null) {
			TermType vendor = new TermType();
			vendor.setValue(selectedVendor.getCode());
			vendor.setLabel(selectedVendor.getName());
			qirForm.setVendor(vendor);
		}

		String selectedFacilityStationNumber = qirForm.getSelectedFacilityId();
		TermType facility = new TermType();
		facility.setValue(selectedFacilityStationNumber);
		qirForm.setFacility(facility);
	}

	private SortedSet<SimpleFacility> getUserFacilities() throws Exception {
		User currentUser = (User) getCurrentUser();
		SortedSet<SimpleFacility> terms = new TreeSet<SimpleFacility>();
		if (currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN)
				|| currentUser.isPermissionGranted(CCHTRoles.APPLICATION_ADMIN)
				|| currentUser.isPermissionGranted(CCHTRoles.MANAGEMENT)) {
			terms = facilityService.getSimpleFacilities();

		} else if (currentUser.isPermissionGranted(CCHTRoles.VISN_ADMIN)
				|| currentUser.isPermissionGranted(CCHTRoles.QIR_ORIGINATOR)) {
			if (currentUser.getVisn() != null) {
				terms = facilityService.getSimpleFacilities(currentUser.getVisn());
			} else if (currentUser.getFacility() != null) {
				terms.addAll(facilityService.getSimpleFacilities(currentUser.getFacility().getVisn()));
			}
		} else {
			if (currentUser.getFacility() != null) {
				terms.add(currentUser.getFacility());
			}
			if (currentUser.getSecondaryFacility() != null) {
				terms.add(currentUser.getSecondaryFacility());
			}
		}
		return terms;
	}
	
	private void populateFormDevicesAndVendors(QIRForm qirForm) throws Exception {
		List<DeviceType> terms = terminologyCache.getTerms(DeviceType.class);
		Map<Integer, Vendor> venList = getVendorsLookUpMap();
		qirForm.setVendors(venList);
		Map<Long, DeviceType> devices = getDevicesLookUpMap(terms, venList);

		Map<Long, Integer> deviceVendorMap = new HashMap<Long, Integer>();
		for (Map.Entry<Long, DeviceType> device : devices.entrySet()) {
			Integer vendorId = device.getValue().getVendor().getId();
			Long deviceId = device.getValue().getId();
			deviceVendorMap.put(deviceId, vendorId);
		}
		qirForm.setDevices(devices);
		qirForm.setDeviceVendorMap(deviceVendorMap);

		List<QIRType> qirTypes = terminologyCache.getTerms(QIRType.class);
		Map<String, String> qirTypeMap = new LinkedHashMap<String, String>();
		for (Lookup t : qirTypes) {
			qirTypeMap.put(t.getCode(), t.getName());
		}
		qirForm.setQirTypes(qirTypeMap);

		Map<String, SimpleFacility> userFacilities = null;
		SortedSet<SimpleFacility> list = new TreeSet<SimpleFacility>(getUserFacilities());
		userFacilities = new LinkedHashMap<String, SimpleFacility>();
		for (SimpleFacility facility : list) {
			userFacilities.put(facility.getCode(), facility);
		}
		qirForm.setFacilities(userFacilities);
	}
}