package gov.va.med.ccht.controller;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import gov.va.med.ccht.controller.validators.ManageUserEditFormValidator;
import gov.va.med.ccht.model.CCHTPermissions;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.service.common.RegistrationService;
import gov.va.med.ccht.ui.common.UserNotifier;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;

@Controller
public class ManageUsersEditController extends CchtController {

	@Autowired
	RegistrationService registrationService;
	@Autowired
	ManageUserEditFormValidator editFormValidator;
	@Autowired
	protected UserNotifier userNotifier;

	@InitBinder
	public void initBinder(WebDataBinder binder) {
		String[] allowedFields = { "assignedRegistrationStatusId", "jobDescriptionId", "assignedUserGroup",
				"assignedFacilityId", "assignedVendorId", "assignedVisnId", "inactiveReasonId",
				"inactivationDateAsString", "registrationDeniedReasonId", "selectedRoles", "locked", "inactive",
				"userName", "registrationReason", "fromPage", "returnPage"};
		binder.setAllowedFields(allowedFields);
	}

	@RequestMapping(value = "/ShowManageUsersEdit.html", method = { RequestMethod.GET })
	public String buildForm(ModelMap model, @RequestParam String userName, @RequestParam String fromPage)
			throws Exception {

		User currentUser = CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class);
		
		User user = securityService.getUser(userName);

		// Build userForm based on acquired user to pass
		// to form to edit
		UserForm userForm = getUser(userName);
		securityConversionService.convert(user, userForm);
		completeUserForm(userForm);

		userForm.setFromPage(fromPage);
		// Get available options to assign to user.
		model.addAttribute("userGroups", getUserGroups());
		model.addAttribute("registrationStatuses", getRegistrationStatuses());
		model.addAttribute("registrationDeniedReasons", getRegistrationDeniedReasons(true));
		model.addAttribute("jobDescriptions", getJobDescriptions());
		model.addAttribute("facilities", getFacilities(true));
		model.addAttribute("vendors", getVendors(true));
		model.addAttribute("visns", getVisns(true));
		model.addAttribute("inactiveReasons", getInactiveReasons(true));

		Integer assignedFacilitiesVisnId = getUsersAssignedFacilitiesAssignedVisn(user);
		Integer assignedVisnId = getUsersAssignedVisnId(user);
		model.addAttribute("assignedFacilitiesVisnId", assignedFacilitiesVisnId);
		model.addAttribute("assignedVisnId", assignedVisnId);
		
		boolean isCurrentUserAppAdmin = isLoggedInUserAppAdmin(currentUser);
		model.addAttribute("isCurrentUserAppAdmin", isCurrentUserAppAdmin);

		// Will have to be reworked to more elegant solution when time allows
		setRoles(model, userForm);

		model.addAttribute("canEdit", canEdit(currentUser, user));
		model.addAttribute("isCurrentUserAppAdmin", isApplicationAdmin(currentUser));

		if (fromPage.equalsIgnoreCase("userSearch")) {
			userForm.setReturnPage("manageUsersSearch.html");
		} else {

			TermType registrationStatus = new TermType();

			if (fromPage.equalsIgnoreCase("approveReg")) {
				model.addAttribute("pageTitle", "Approve Registration");
				userForm.setAssignedRegistrationStatusId("2");
				registrationStatus.setValue("2");
				registrationStatus.setLabel("Approved");
				userForm.setRegistrationStatus(registrationStatus);
				userForm.setRegistrationDeniedReason(null);
				model.addAttribute("showRegistrationDenied", false);
				model.addAttribute("isDeny", false);
			} else {
				model.addAttribute("showRegistrationDenied", true);
				model.addAttribute("pageTitle", "Deny Registration");
				model.addAttribute("isDeny", true);
				model.addAttribute("disableRegDeniedDropdown", false);
				userForm.setAssignedRegistrationStatusId("3");
				registrationStatus.setValue("3");
				registrationStatus.setLabel("Denied");
				userForm.setRegistrationStatus(registrationStatus);
			}
			userForm.setReturnPage("manageRegistrations.html");
		}
		model.addAttribute("returnPage", userForm.getReturnPage());

		TermType regDeniedReason = userForm.getRegistrationDeniedReason();

		if (regDeniedReason != null) {
			model.addAttribute("showRegistrationDenied", true);
			model.addAttribute("disableRegDeniedDropdown", true);
			userForm.setRegistrationDeniedReasonId(regDeniedReason.getValue());
		}
		
		if(currentUser.isRoleGranted(VISN_ADMIN) && !(user.getVisn() != null && user.getVisn().getName().equals(currentUser.getVisn().getName()))) {
			
			model.addAttribute("showRoles", false); // display no roles to edit.
			model.addAttribute("canEdit", false);
		}
		if(currentUser.isRoleGranted(FACILITY_ADMIN) &&  !(user.getFacility() != null && user.getFacility().getName().equals(currentUser.getFacility().getName()))) {
			
			model.addAttribute("canEdit", false);
			model.addAttribute("showRoles", false); // display no roles to edit.
		}
		
		Map<String, String> availRoles = getAvailableRoles();
		
		if (availRoles != null || availRoles.isEmpty()) {

			// Get logged in users available roles
			String[] allRoles = new String[availRoles.size()];
			int i = 0;
			Iterator iter = availRoles.keySet().iterator();
			while (iter.hasNext()) {
				allRoles[i] = (String) iter.next();
				i++;
			}
			
			// Get roles of user being edited
			List<String> assignedRoles = userForm.getAssignedRoles();
			String[] selectedRoles = new String[assignedRoles.size()];
			i = 0;
			for (String role : assignedRoles) {
				selectedRoles[i] = role;
				i++;
			}
			userForm.setSelectedRoles(selectedRoles);

			Map<String, Boolean> roles = new LinkedHashMap<String, Boolean>();
			
			for(String role : allRoles) {
				if(assignedRoles.contains(role)) {
					roles.put(role, true); // is checked

				}
				else {
					roles.put(role, false); // not checked
				}
			}
			
			model.put("rolesMap", roles);
		}

		model.addAttribute("command", userForm);

		return "manageUsersEdit";

	}
	// submit form and update user
	@RequestMapping(value = "/manageUsersEdit.html", method = { RequestMethod.POST, RequestMethod.GET })
	public String submit(@ModelAttribute("command") @Valid UserForm command, 
			BindingResult result, Model model, HttpServletRequest request)
			throws Exception {

		// Step one, ensure all user entries are
		// represented in a UserForm object.
		buildValidatedUserForm(command);

		User user = securityService.getUser(command.getUserName());
		
		// Check if user is App Admin. 
		// App Admin can only add/remove the App Admin role.
		// Need to enforce this.
		User currentUser = CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class);
		if(isLoggedInUserAppAdmin(currentUser)) {
			
			// Get the users current roles as they exist in the DB
			Set<UserRole> roles = user.getRoles();
			
			// Get the currently selected roles to determine if App Admin is selected on the form or not
			boolean appAdminRoleSelected = false;
			List<TermType> currentAvailableRoles = command.getRoles();
			for(TermType type : currentAvailableRoles) {
				
				String name = type.getValue();
				if(name.equalsIgnoreCase("Application Administrator")) {
					appAdminRoleSelected = true;
					break;
				}
			}
			
			// Iterate through current roles in DB. 
			// Only add appAdmin back in IF it was selected on the form.
			for(UserRole role : roles ) {				
				String name = role.getRole().getName();
				if(!name.equalsIgnoreCase("Application Administrator") || appAdminRoleSelected) {
					TermType newRole = new TermType(name, name);
					currentAvailableRoles.add(newRole);
				}
			}			
			command.setRoles(currentAvailableRoles);
			// Add the user's previous roles into the selected role since an App Admin
			// can only see or assign App Admin
			final String[] selectedRoleNames = new String[currentAvailableRoles.size()];
			int i = 0;
			for(TermType role: currentAvailableRoles) {
				selectedRoleNames[i] = role.getLabel();
				i++;
			}
			command.setSelectedRoles(selectedRoleNames);
		}
		
		TermType registrationStatus = new TermType();
		if (command.getFromPage().equalsIgnoreCase("approveReg")) {
			command.setAssignedRegistrationStatusId("2");
			registrationStatus.setValue("2");
			registrationStatus.setLabel("Approved");
			command.setRegistrationStatus(registrationStatus);
			command.setRegistrationDeniedReason(null);
		} else if(command.getFromPage().equalsIgnoreCase("denyReg")) {
			command.setAssignedRegistrationStatusId("3");
			registrationStatus.setValue("3");
			registrationStatus.setLabel("Denied");
			command.setRegistrationStatus(registrationStatus);
		}
		
		String oldRegStatusCode = user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();

		// Get all roles that the current user can modify
		Map<String, String> availRoles = getAvailableRoles();
		// Only modify the selected roles versus the available roles.
		// That way pre-existing roles that aren't able to be modified are not lost.
		List<String> allRoles = new ArrayList<String>();
		for(UserRole userRole: user.getRoles()) {
			// If existing user role is not available to be selected, add it to the selected roles
			// because we don't want to remove it
			if(availRoles.get(userRole.getRole().getName()) == null) {
				allRoles.add(userRole.getRole().getName());
			}
		}
		for(String selectedRole : command.getSelectedRoles()) {
			allRoles.add(selectedRole);
		}
		command.setSelectedRoles(allRoles.toArray(new String[0]));
		
		// Step two, convert UserForm to User by merging
		// new info into User object
		securityConversionService.convert(command, user);
		
		// Step three, save.
		securityService.updateUser(user);
		
        //Fix Fortify Issues
		ESAPIValidator.validateStringInput(user.getRegistrationStatus().getCode(), ESAPIValidationType.CROSS_SITE_SCRIPTING_PERSISTENT);
		ESAPIValidator.validateStringInput(user.getRegistrationStatus().getCode(), ESAPIValidationType.CROSS_SITE_SCRIPTING_REFLECTED);
		
		String newRegStatusCode = user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();
		if (!oldRegStatusCode.equals(newRegStatusCode)) {
			if (RegistrationStatus.APPROVED.equals(newRegStatusCode)) {
				registrationService.sendWelcomeMail(user);
			} else if (RegistrationStatus.REJECTED.equals(newRegStatusCode)) {
				registrationService.sendDeniedMail(user);
			} else {
				// submitted nothing to do moved to pending status
			}
		}

		// First validate userName against very restrictive and minimal white list.
		String returnPage = ESAPIValidator.validateStringInput(command.getReturnPage(), ESAPIValidationType.FromPage_WhiteList);
		// redirectStr does not need to be validated because it is defined by predefined/hardcoded basePath and other hardcoded values.
		// The returnPage portion of the redirectStr is validated above.
		String redirectStr = "redirect:" + basePath + returnPage;
		ESAPIValidator.validateStringInput(redirectStr, ESAPIValidationType.RedirectStr_WhiteList);

		userNotifier.notifyUserOnceWithMessage(request, "User updated successfully.");

		return redirectStr;

	}

	@RequestMapping(value = "/manageUsersEdit.html", method = { RequestMethod.GET, RequestMethod.POST },
			params = { "Reset" })
	public String refresh(@ModelAttribute("command") @Valid UserForm command, BindingResult result, ModelMap model)
			throws Exception {

		User currentUser = CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class);
		String userName = command.getUserName();
		String fromPage = command.getFromPage();
		String returnPage = command.getReturnPage();
		User selectedUser = securityService.getUser(userName);

		// Build userForm based on aquired user to pass
		// to form to edit
		UserForm userForm = getUser(userName);
		securityConversionService.convert(selectedUser, userForm);
		completeUserForm(userForm);
		userForm.setReturnPage(returnPage);
		userForm.setFromPage(fromPage);

		model.addAttribute("userGroups", getUserGroups());
		model.addAttribute("registrationStatuses", getRegistrationStatuses());
		model.addAttribute("registrationDeniedReasons", getRegistrationDeniedReasons(true));
		model.addAttribute("jobDescriptions", getJobDescriptions());
		model.addAttribute("facilities", getFacilities(true));
		model.addAttribute("vendors", getVendors(true));
		model.addAttribute("visns", getVisns(true));
		model.addAttribute("inactiveReasons", getInactiveReasons(true));

		setRoles(model, userForm);

		model.addAttribute("canEdit", canEdit(currentUser, selectedUser));
		model.addAttribute("isCurrentUserAppAdmin", isApplicationAdmin(currentUser));

		// Set up values for checkbox
		Map<String, String> availRoles = getAvailableRoles();
		if (availRoles != null) {
			
			setModelsRolesMap(model, availRoles, selectedUser.getRoles());
			setUserFormsSelectedRoles(userForm);
		}

		if (fromPage.equalsIgnoreCase("userSearch")) {
			userForm.setReturnPage("manageUsersSearch.html");
		} else {

			TermType registrationStatus = new TermType();

			if (fromPage.equalsIgnoreCase("approveReg")) {
				model.addAttribute("pageTitle", "Approve Registration");
				userForm.setAssignedRegistrationStatusId("2");
				registrationStatus.setValue("2");
				registrationStatus.setLabel("Approved");
				userForm.setRegistrationStatus(registrationStatus);
				userForm.setRegistrationDeniedReason(null);
				model.addAttribute("showRegistrationDenied", false);
				model.addAttribute("isDeny", false);
			} else {
				model.addAttribute("showRegistrationDenied", true);
				model.addAttribute("pageTitle", "Deny Registration");
				model.addAttribute("isDeny", true);
				model.addAttribute("disableRegDeniedDropdown", false);
				userForm.setAssignedRegistrationStatusId("3");
				registrationStatus.setValue("3");
				registrationStatus.setLabel("Denied");
				userForm.setRegistrationStatus(registrationStatus);
			}
			userForm.setReturnPage("manageRegistrations.html");
		}

		TermType regDeniedReason = userForm.getRegistrationDeniedReason();

		if (regDeniedReason != null) {
			model.addAttribute("showRegistrationDenied", true);
			model.addAttribute("disableRegDeniedDropdown", true);
			userForm.setRegistrationDeniedReasonId(regDeniedReason.getValue());
		}

		model.addAttribute("command", userForm);

		return "manageUsersEdit";
	}

	private void setModelsRolesMap(ModelMap model, Map<String, String> availRoles, Set<UserRole> currentRoles) {

		final Map<String, Boolean> roles = new LinkedHashMap<String,Boolean>();
		Set<String> roleNames = new TreeSet<String>();
		Iterator roleIter = currentRoles.iterator();
		while (roleIter.hasNext()) {
			final UserRole role = (UserRole) roleIter.next();
			roleNames.add(role.getRole().getName());
		}
		
		Iterator iter = availRoles.keySet().iterator();
		while (iter.hasNext()) {
			final String role = (String) iter.next();
			if(roleNames.contains(role)) {
				roles.put(role, true);
			}
			else {
				roles.put(role, false);
			}
		}
		model.addAttribute("rolesMap", roles);

	}

	private void setUserFormsSelectedRoles(UserForm userForm) {

		List<String> assignedRoles = userForm.getAssignedRoles();
		String[] selectedRoles = new String[assignedRoles.size()];
		int i = 0;
		for (String role : assignedRoles) {
			selectedRoles[i] = role;
			i++;
		}
		userForm.setSelectedRoles(selectedRoles);

	}

	/**
	 * Builds a validated user form. This form has had edits in it to update a
	 * user.
	 * 
	 * @param userForm
	 * @return
	 * @throws Exception
	 */
	private void buildValidatedUserForm(UserForm userForm) throws Exception {

		// If the user selected a VISN, Create a TermType
		// and set it on the form.
		String assignedVisnId = userForm.getAssignedVisnId();
		if (assignedVisnId != null) {

			if (assignedVisnId.equalsIgnoreCase(NEGATIVE_ONE)) {
				userForm.setVisn(null);
			} else {
				TermType newVisn = new TermType();
				newVisn.setValue(assignedVisnId);
				newVisn.setLabel(assignedVisnId);
				userForm.setVisn(newVisn);
			}
		}

		// If the user selected a Facility, Create a TermType
		// and set it on the form.
		String assignedFacilityId = userForm.getAssignedFacilityId();
		if (assignedFacilityId != null) {

			if (assignedFacilityId.equalsIgnoreCase(NEGATIVE_ONE)) {
				userForm.setFacility(null);
			} else {
				TermType newFacility = new TermType();
				newFacility.setValue(assignedFacilityId);
				userForm.setFacility(newFacility);
			}
		}

		// If the user selected a VENDOR, Create a TermType
		// and set it on the form.
		String assignedVendorId = userForm.getAssignedVendorId();
		if (assignedVendorId != null) {

			if (assignedVendorId.equalsIgnoreCase(NEGATIVE_ONE)) {
				userForm.setVendor(null);
			} else {
				TermType newVendor = new TermType();
				newVendor.setLabel(assignedVendorId);
				userForm.setVendor(newVendor);
			}
		}

		String[] assignedRoles = userForm.getSelectedRoles();
		if (assignedRoles != null) {
			List<TermType> updatedRoleAssignments = new ArrayList<TermType>();
			for (String name : assignedRoles) {
				TermType role = new TermType(name, name);
				updatedRoleAssignments.add(role);
			}

			userForm.setRoles(updatedRoleAssignments);
		} else {
			userForm.setRoles(new ArrayList<TermType>());
		}

		// Set locked status.
		Boolean isLocked = userForm.getLocked();
		if (isLocked != null && isLocked) {
			userForm.setAccountLockDate(new Date());
		}

		// Set inactive status
		Boolean isInactive = userForm.getInactive();
		if (isInactive != null && isInactive) {

			if (userForm.getInactivationDateAsString() != null) {
				Date inactiveDate = DateUtils.parseDate(userForm.getInactivationDateAsString(), DateUtils.MMDDYYYY);
				userForm.setInactiveDate(inactiveDate);
			}

			String inactiveReasonId = userForm.getInactiveReasonId();
			if (inactiveReasonId != null) {
				TermType inactiveReason = new TermType();
				inactiveReason.setValue(inactiveReasonId);
				userForm.setInactiveReason(inactiveReason);
			}
		}

		String registrationDeniedReasonid = userForm.getRegistrationDeniedReasonId();
		if (registrationDeniedReasonid != null) {
			TermType registrationDeniedReason = new TermType();
			registrationDeniedReason.setValue(registrationDeniedReasonid);
			userForm.setRegistrationDeniedReason(registrationDeniedReason);
		}
	}

	/**
	 * Build User Form when the edit screen initially loads.
	 * 
	 * @param userForm
	 * @return
	 * @throws Exception
	 */
	private void completeUserForm(UserForm userForm) throws Exception {

		// Set userGroup id.
		TermType usersGroup = userForm.getUserGroup();
		if (usersGroup != null) {
			userForm.setAssignedUserGroup(usersGroup.getValue());
		}

		// Set registration status id.
		TermType regStatus = userForm.getRegistrationStatus();
		if (regStatus != null) {
			userForm.setAssignedRegistrationStatusId(regStatus.getValue());
		}

		// Set Job Description Id
		TermType regReason = userForm.getRegistrationReason();
		if (regReason != null) {
			userForm.setJobDescriptionId(regReason.getValue());
		}

		TermType visn = userForm.getVisn();
		if (visn != null) {
			userForm.setAssignedVisnId(visn.getValue().trim());
		}

		TermType facility = userForm.getFacility();
		if (facility != null) {
			userForm.setAssignedFacilityId(facility.getValue().trim());
		}

		TermType vendor = userForm.getVendor();
		if (vendor != null) {
			userForm.setAssignedVendorId(vendor.getLabel());
		}

		List<TermType> userRoles = userForm.getRoles();

		if (userRoles != null) {

			if (userForm.getAssignedRoles() == null) {
				userForm.setAssignedRoles(new ArrayList<String>());
			}
			for (TermType role : userRoles) {
				String name = role.getValue();
				userForm.getAssignedRoles().add(name);
			}
		}

		Date accountLockedDate = userForm.getAccountLockDate();
		if (accountLockedDate != null) {
			userForm.setLocked(true);
		} else {
			userForm.setLocked(false);
		}

		Date accountInactiveDate = userForm.getInactiveDate();
		if (accountInactiveDate != null) {
			userForm.setInactive(true);
			SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
			userForm.setInactivationDateAsString(dateFormat.format(accountInactiveDate));

			TermType inactiveReason = userForm.getInactiveReason();
			if (inactiveReason != null) {
				userForm.setInactiveReasonId(inactiveReason.getValue());
			}
		} else {
			userForm.setInactive(false);
		}
	}

	/**
	 * 
	 * Set up the rules that are needed to determine which page to return to
	 * when the cancel button is pressed on the Manage User / Approve-Deny
	 * registration page.
	 * 
	 * @param userForm
	 * @param model
	 */
	private void setFromPageRules(UserForm userForm, Model model) {

		String fromPage = userForm.getReturnPage();

		if (fromPage.equalsIgnoreCase("userSearch")) {
			userForm.setReturnPage("manageUsersSearch.html");
		} else {

			TermType registrationStatus = new TermType();

			if (fromPage.equalsIgnoreCase("approveReg")) {
				model.addAttribute("pageTitle", "Approve Registration");
				userForm.setAssignedRegistrationStatusId("2");
				registrationStatus.setValue("2");
				registrationStatus.setLabel("Approved");
				userForm.setRegistrationStatus(registrationStatus);
			} else {
				model.addAttribute("pageTitle", "Deny Registration");
				userForm.setAssignedRegistrationStatusId("3");
				registrationStatus.setValue("3");
				registrationStatus.setLabel("Denied");
				userForm.setRegistrationStatus(registrationStatus);
			}
			userForm.setReturnPage("manageRegistrations.html");
		}
	}

	/**
	 * Brute force method to enforce roles based rules of view widgets this
	 * needs to be changed when time allows.
	 * 
	 * @param model
	 * @param userForm
	 */
	private void setRoles(ModelMap model, UserForm userForm) {

		model.addAttribute("isNationalAdmin", userForm.getAssignedRoles().contains(CCHTPermissions.NATIONAL_ADMIN));
		model.addAttribute("isApplicationAdmin",
				userForm.getAssignedRoles().contains(CCHTPermissions.APPLICATION_ADMIN));
		model.addAttribute("isCareCoordinator", userForm.getAssignedRoles().contains(CCHTPermissions.CARE_COORDINATOR));
		model.addAttribute("isFacilityAdmin", userForm.getAssignedRoles().contains(CCHTPermissions.FACILITY_ADMIN));
		model.addAttribute("isHT", userForm.getAssignedRoles().contains(CCHTPermissions.CONTRACT_MANAGER));
		model.addAttribute("isPSA", userForm.getAssignedRoles().contains("Program Support Assistant"));
		model.addAttribute("isQIR", userForm.getAssignedRoles().contains("QIR Originator"));
		model.addAttribute("isReportsOnly", userForm.getAssignedRoles().contains("Reports Only"));
		model.addAttribute("management", userForm.getAssignedRoles().contains(CCHTPermissions.MANAGEMENT));
		model.addAttribute("isVendor", userForm.getAssignedRoles().contains(CCHTPermissions.VENDOR));
		model.addAttribute("isVisnAdmin", userForm.getAssignedRoles().contains(CCHTPermissions.VISN_ADMIN));

	}

	private boolean canEdit(User currentUser, User editUser) {

		boolean isNationalAdmin = currentUser.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN);
		if (isNationalAdmin) {
			return true;
		}

		boolean isVisnAdmin = currentUser.isPermissionGranted(CCHTPermissions.VISN_ADMIN);
		boolean isFacilityAdmin = currentUser.isPermissionGranted(CCHTPermissions.FACILITY_ADMIN);

		// Get current logged in users information
		SimpleVisn currentUserVisn = currentUser.getVisn();
		SimpleFacility currentUserFacility = currentUser.getFacility();
		Integer currentUserVisnId = -1;
		Integer currentUserFacilityId = -1;

		if (currentUserVisn != null) {
			currentUserVisnId = currentUserVisn.getId();
		}

		if (currentUserFacility != null) {
			currentUserFacilityId = currentUserFacility.getId();
		}

		// Get the user to be edited information
		SimpleVisn editUserVisn = editUser.getVisn();
		SimpleFacility editUserFacility = editUser.getFacility();
		Integer editUserVisnId = -1;
		Integer editUserFacilityId = -1;

		if (editUserVisn != null) {
			editUserVisnId = editUserVisn.getId();
		}

		if (editUserFacility != null) {
			editUserFacilityId = editUserFacility.getId();
		}

		// Determine if the logged in user can edit the desired user.
		if (isVisnAdmin) {
			return currentUserVisnId.intValue() == editUserVisnId.intValue();
		}

		if (isFacilityAdmin && !isVisnAdmin) {
			return currentUserFacilityId.intValue() == editUserFacilityId.intValue();
		}

		return true;

	}

	private Integer getUsersAssignedFacilitiesAssignedVisn(User user) {

		Integer assignedVisnId = -1;

		SimpleFacility fac = user.getFacility();

		if (fac != null) {

			SimpleVisn visn = fac.getVisn();

			if (visn != null) {
				assignedVisnId = visn.getId();
			}

		}

		return assignedVisnId;
	}

	private Integer getUsersAssignedVisnId(User user) {

		Integer assignedVisnId = -1;

		SimpleVisn visn = user.getVisn();

		if (visn != null) {
			assignedVisnId = visn.getId();
		}

		return assignedVisnId;
	}
	
	private boolean isLoggedInUserAppAdmin(User loggedInUser) {
		
		Set<UserRole> userRoles = loggedInUser.getRoles();
		
		for(UserRole role : userRoles) {
			
			String roleDescription = role.getRole().getName();	
			if(roleDescription.equalsIgnoreCase("Application Administrator")) {
				return true;
			}
		}
		
		return false;
	}
}
