package gov.va.med.ccht.controller;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.Trigger.TriggerState;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.quartz.impl.matchers.GroupMatcher;
import org.springframework.batch.core.BatchStatus;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.ibm.icu.util.Calendar;

import gov.va.med.ccht.jobs.CchtJob;
import gov.va.med.ccht.jobs.JobScheduler;
import gov.va.med.ccht.model.Schedule;
import gov.va.med.ccht.model.terminology.RunFrequency;
import gov.va.med.ccht.persistent.JobDAO;
import gov.va.med.ccht.persistent.QuartzDAO;
import gov.va.med.ccht.service.report.StandardReportService;
import gov.va.med.ccht.service.scheduleJob.ScheduleJobService;
import gov.va.med.ccht.ui.model.BatchProcessDetailForm;
import gov.va.med.ccht.ui.model.ReportScheduleForm;
import gov.va.med.fw.batchprocess.ProcessStatistics;
import gov.va.med.fw.model.batchprocess.JobConfig;
import gov.va.med.fw.model.batchprocess.JobResult;
import gov.va.med.fw.model.batchprocess.JobStatus;
import gov.va.med.fw.scheduling.TriggerStatus;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.UserPrincipal;

@Controller
public class ManageScheduledJobsController extends CchtController implements ApplicationContextAware {

	private Logger logger = Logger.getLogger(getClass());

	public static final String JOB_NAME = "jobName";
	public static final String JOB_PARAMETERS = "jobParameters";
	public static final String USER_ID = "userId";
	public static final int MAX_USER_ID_LENGTH = 30;

	@Autowired
	private JobScheduler jobScheduler;
	
	@Autowired
	private ScheduleJobService scheduleJobService;

	@Autowired
	private JobDAO jobDao;

	@Autowired
	private QuartzDAO quartzDao;
	
	@Autowired
	private StandardReportService standardReportService;

	private ApplicationContext applicationContext;

	@RequestMapping("/scheduledJobs")
	public @ResponseBody Map<String, Object> getScheduledJobs() throws Exception {

		UserPrincipal user = getCurrentUser();
		List<BatchProcessDetailForm> jobAndTriggerStatus = new ArrayList<BatchProcessDetailForm>();

		for (String groupName : jobScheduler.getScheduler().getJobGroupNames()) {

			for (JobKey jobKey : jobScheduler.getScheduler().getJobKeys(GroupMatcher.jobGroupEquals(groupName))) {

				JobDetail detail = jobScheduler.getScheduler().getJobDetail(jobKey);

				BatchProcessDetailForm form = new BatchProcessDetailForm();

				final TriggerKey triggerKey = new TriggerKey(JobScheduler.getTriggerName(jobKey.getName()),
						JobScheduler.TRIGGERS_GROUP);

				form.setId(jobKey.getName());
				boolean canExecuteOrView = true;
				// set batch process details name, group ..
				form.setJobName(jobKey.getName());
				form.setGroupName(jobKey.getGroup());
				form.setJobDescription(detail.getDescription());
				form.setTriggerName(triggerKey.getName());
				form.setTriggerGroup(triggerKey.getGroup());
				detail.getDescription();
				if(jobScheduler.getScheduler().getTrigger(triggerKey) != null) {
					form.setJobScheduleDescription(jobScheduler.getScheduler().getTrigger(triggerKey).getDescription());
					form.setNextFireTime(jobScheduler.getScheduler().getTrigger(triggerKey).getNextFireTime());
					form.setStatus(getStatus(jobScheduler.getScheduler().getTriggerState(triggerKey)));
				}
				else {
					form.setJobScheduleDescription("Schedule no longer valid.");
					form.setStatus("Invalid schedule.");
				}

				JobConfig jobConfig = batchProcessService.getJobConfig(detail.getKey().getName(),
						detail.getKey().getGroup());
				if (jobConfig != null) {
					if (jobConfig.getJobSchedule() != null) {
						form.setJobSchedule(jobConfig.getJobSchedule());
						form.setJobScheduleDescription(jobConfig.getJobScheduleDescription());
						if (StringUtils.isNotEmpty(jobConfig.getJobScheduleText())) {
							Schedule schedule = scheduleConversionService.toSchedule(jobConfig.getJobScheduleText());
							ReportScheduleForm rsForm = new ReportScheduleForm();
							reportConversionService.convert(schedule, rsForm);
							form.setReportSchedule(rsForm);
						}
					}
					form.setEmailDistributionList(jobConfig.getEmailDistributionList());
					canExecuteOrView = isAllowedToExecuteOrViewBatchJob(user, jobConfig.getPermissions());
				}
				if (canExecuteOrView) {
					jobAndTriggerStatus.add(form);
				}
			}
		}
		Collections.sort(jobAndTriggerStatus);

		Map<String, Object> results = new HashMap<String, Object>();
		results.put("jobs", jobAndTriggerStatus);

		return results;

	}

	private String getStatus(TriggerState triggerState) {

		if (triggerState == TriggerState.NORMAL) {
			return TriggerStatus.NORMAL_STATUS;
		} else {
			return triggerState.toString();
		}

	}

	@RequestMapping("/scheduledJobs/active")
	public @ResponseBody Map<String, Object> getActiveJobs() throws Exception {

		Map<String, Object> results = new HashMap<String, Object>();
		@SuppressWarnings("rawtypes")
		List activeJobs = batchProcessService.getJobResults(BatchStatus.STARTED);
		results.put("jobs", activeJobs);
		return results;

	}

	@RequestMapping("/scheduledJobs/configUpdate")
	public @ResponseBody boolean submitConfigUpdate(@RequestParam String jobName, @RequestParam String groupName,
			@RequestParam String emails, @RequestParam String jobStatus, @RequestParam String previousJobStatus)
			throws Exception {
		JobConfig jobConfig = batchProcessService.getJobConfig(jobName, groupName);
		if (jobConfig == null) {
			jobConfig = new JobConfig();
			jobConfig.setName(jobName);
			jobConfig.setGroup(groupName);
		}

		jobConfig.setEmailDistributionList(emails);
		batchProcessService.saveJobConfig(jobConfig);

		if (StringUtils.isNotEmpty(jobStatus) && !jobStatus.equals(previousJobStatus)) {
			if (TriggerStatus.NORMAL_STATUS.equals(jobStatus)) {
				for (JobKey jobKey : jobScheduler.getScheduler().getJobKeys(GroupMatcher.jobGroupEquals(groupName))) {
					if (jobKey.getName().equalsIgnoreCase(jobName)) {
						jobScheduler.getScheduler().resumeJob(jobKey);
					}
				}
			} else if (TriggerStatus.PAUSED_STATUS.equals(jobStatus)) {
				for (JobKey jobKey : jobScheduler.getScheduler().getJobKeys(GroupMatcher.jobGroupEquals(groupName))) {
					if (jobKey.getName().equalsIgnoreCase(jobName)) {
						jobScheduler.getScheduler().pauseJob(jobKey);
					}
				}
			}
			quartzDao.updateTriggerStatus(JobScheduler.getTriggerName(jobName), JobScheduler.TRIGGERS_GROUP, jobStatus);
		}

		return true;
	}

	@RequestMapping("/scheduledJobs/execute")
	public @ResponseBody boolean submitExecuteJob(@RequestParam String jobName) throws Exception {

		if(JobScheduler.PURGE_COMPLETED_JOBS_JOB_NAME.equals(jobName)) {
			runPurgeCompletedReports(jobName);
		}
		else if(JobScheduler.QIR_VENDOR_RESPONSE_DUE_JOB_NAME.equals(jobName)) {
			runQIRVendorResponseDue(jobName);
		}
		else {
			// Unrecognized job, as these are the only two jobs that we will ever need.
			return false;
		}

		return true;
	}

	@RequestMapping("/scheduledJobs/scheduleUpdate")
	public @ResponseBody boolean submitJobScheduleUpdate(@RequestParam String jobName,
			@RequestParam String cronExpression, @RequestParam String cronDescription) throws Exception {

		CronExpression expression = new CronExpression(cronExpression);
		Calendar cal = Calendar.getInstance();
		// Add one minute to account for rollover or timing mismatch between submitting to the database and scheduling.
		cal.add(Calendar.MINUTE, 1);
		if(expression.getNextValidTimeAfter(cal.getTime()) != null) {
			final String triggerName = JobScheduler.getTriggerName(jobName);
			quartzDao.updateCronExpression(triggerName, JobScheduler.TRIGGERS_GROUP, cronExpression);
			quartzDao.updateCronDescription(triggerName, JobScheduler.TRIGGERS_GROUP, cronDescription);
			
			Trigger newTrigger = TriggerBuilder.newTrigger().withIdentity(triggerName, JobScheduler.TRIGGERS_GROUP)
					.withDescription(cronDescription).withSchedule(CronScheduleBuilder.cronSchedule(cronExpression))
					.build();
	
			TriggerKey triggerKey = new TriggerKey(triggerName, JobScheduler.TRIGGERS_GROUP);
			if(jobScheduler.getScheduler().getTrigger(triggerKey) != null) {
				jobScheduler.getScheduler().rescheduleJob(triggerKey, newTrigger);
			}
			else {
				// If the previous trigger is no longer valid.
				JobKey jobKey = new JobKey(jobName, JobScheduler.JOBS_GROUP);
				JobDetail jobDetail = jobScheduler.getScheduler().getJobDetail(jobKey);
				jobScheduler.getScheduler().deleteJob(jobKey);
				jobScheduler.getScheduler().scheduleJob(jobDetail, newTrigger);
			}
			return true;
		}
		else {
			return false;
		}
	}

	@RequestMapping("/scheduledJobs/resumeAll")
	public @ResponseBody boolean submitResumeAll() throws SchedulerException {
		jobScheduler.getScheduler().resumeAll();
		quartzDao.updateAllTriggerStatus(TriggerStatus.NORMAL_STATUS);
		return true;
	}

	@RequestMapping("/scheduledJobs/pauseAll")
	public @ResponseBody boolean submitPauseAll() throws SchedulerException {
		jobScheduler.getScheduler().pauseAll();
		quartzDao.updateAllTriggerStatus(TriggerStatus.PAUSED_STATUS);
		return true;
	}

	@RequestMapping(value = "/manageScheduledJobs.html", method = RequestMethod.GET)
	public String showManageScheduledJobs(Model model) throws Exception {
		model.addAttribute("allJobStatuses", Arrays.asList(TriggerStatus.NORMAL_STATUS, TriggerStatus.PAUSED_STATUS));
		model.addAttribute("allRunFrequencies", RunFrequency.getFrequencies());

		return "manageScheduledJobs";
	}

	private boolean isAllowedToExecuteOrViewBatchJob(UserPrincipal user, Set<Permission> permissions) {
		if (permissions != null && permissions.size() > 0) {
			for (Permission permission : permissions) {
				String code = permission.getName();
				if (user.isPermissionGranted(code)) {
					return true;
				}
			}
		} else {
			return true;
		}
		return false;
	}

	@RequestMapping("/scheduledJobs/jobHistory")
	// @PreAuthorize("hasAuthority('" + Permission.VOLUNTEER_VIEW + "')")
	public @ResponseBody Map<String, Object> getJobHistory(@RequestParam String jobName, @RequestParam String jobGroup)
			throws Exception {

		Map<String, Object> results = new HashMap<String, Object>();

		results.put("history", jobDao.findFinishedJobResults(jobName, jobGroup));

		return results;
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}
	
	private void runPurgeCompletedReports(String jobName) throws Exception {
		logger.info("Alerting of purge completed reports start.");
		ProcessStatistics stats = new ProcessStatistics();
		stats.setProcessingStartDate(Calendar.getInstance().getTime());
		stats.setExecutedOnServer(InetAddress.getLocalHost().getHostName());
		stats.setProcessName(jobName);
		CchtJob job = new CchtJob();
		
		JobResult result = job.saveJobStartingInDatabase(jobDao, JobScheduler.JOBS_GROUP, jobName,
				getCurrentUser().getUsername());
		try {
			int successfulRecords = standardReportService.purgeCompletedReports();
			// Total is equal to the records returned.
			stats.setNumberOfTotalRecords(successfulRecords);
			stats.setNumberOfSuccessfulRecords(successfulRecords);
			// This should be 0, since an error would constitute an exception.
			stats.setNumberOfErrorRecords(stats.getNumberOfTotalRecords() - stats.getNumberOfSuccessfulRecords());
			stats.setProcessingEndDate(Calendar.getInstance().getTime());
			result.setStatistics(stats.toString());
			job.saveJobCompletedInDatabase(jobDao, result, JobStatus.COMPLETE);

		} catch (final Exception e) {
			logger.warn("Issue performing job, ManageScheduleJobsController1");
			e.printStackTrace();
			job.saveJobCompletedInDatabase(jobDao, result, JobStatus.COMPLETE_WITH_ERROR);
		}
		finally {
			scheduleJobService.sendExecuteNotificationMail(jobName);
		}
	}
	
	private void runQIRVendorResponseDue(String jobName) throws Exception {
		logger.info("Alerting of QIR vendor responses due start.");
		ProcessStatistics stats = new ProcessStatistics();
		stats.setProcessingStartDate(Calendar.getInstance().getTime());
		stats.setExecutedOnServer(InetAddress.getLocalHost().getHostName());
		stats.setProcessName(jobName);
		CchtJob job = new CchtJob();
		
		JobResult result = job.saveJobStartingInDatabase(jobDao, JobScheduler.JOBS_GROUP, jobName,
				getCurrentUser().getUsername());
		try {
			int successfulRecords = qirService.generateVendorResponseDueNotifications();
			// Total is equal to the records returned.
			stats.setNumberOfTotalRecords(successfulRecords);
			stats.setNumberOfSuccessfulRecords(successfulRecords);
			// This should be 0, since an error would constitute an exception.
			stats.setNumberOfErrorRecords(stats.getNumberOfTotalRecords() - stats.getNumberOfSuccessfulRecords());
			stats.setProcessingEndDate(Calendar.getInstance().getTime());
			result.setStatistics(stats.toString());
			job.saveJobCompletedInDatabase(jobDao, result, JobStatus.COMPLETE);
		}
		catch (final Exception e) {
			logger.warn("Issue performing job, ManageScheduleJobsController2");
			e.printStackTrace();
			job.saveJobCompletedInDatabase(jobDao, result, JobStatus.ERROR);
		}
		finally {
			scheduleJobService.sendExecuteNotificationMail(jobName);
		}

		logger.info("Finished  alerting of QIR vendor responses due.");
	}
}
