package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.support.SessionStatus;

import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.service.common.JobDescription;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.ui.model.UserRegistrationForm;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.ui.model.TermType;

@Controller
public class ManageRegistrationsController {

	@Autowired
	protected SecurityService securityService;
	@Autowired
	protected SecurityConversionService securityConversionService;
	
	@Value("${ccht.web.url}")
	protected String basePath;
	
	@Autowired
	ManageUsersEditController manageUsersEditController;
	
    @InitBinder
    public void initBinder(WebDataBinder binder)
    {   
    	String[] allowedFields = {"id", "userName", "lastName", "firstName"};
    	binder.setAllowedFields(allowedFields);    	
    }

	@RequestMapping(value = "/manageRegistrations.html", method = RequestMethod.GET)
	public String showManageRegistrations(Model model) throws Exception {

		UserPrincipal currentUser = CommonReferenceDataInterceptor.getCurrentUser();
		User user = securityService.getUser(currentUser.getUserCredentials().getUserID());
		UserForm userForm = new UserForm();
		securityConversionService.convert(user, userForm);

		List<UserForm> users = new ArrayList<UserForm>();
		Integer visnId = null;
		Integer facilityId = null;

		List<User> registrations = new ArrayList<User>();
		if (currentUser.isPermissionGranted(SecurityController.NATIONAL_ADMIN)) {
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
			// all visns and sites
		} else if (currentUser.isPermissionGranted(SecurityController.VISN_ADMIN)) {
			visnId = ((User) currentUser).getVisn().getId();
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
			if(!registrations.isEmpty()) {
				List<User> notApplicableReg = new ArrayList<User>();
				for(User reg : registrations) {
					if(reg.getRegistrationReason().getCode().equals(JobDescription.VISN_ADMIN.getValue())) {
						notApplicableReg.add(reg);
					}
				}
				if(!notApplicableReg.isEmpty()) {
					registrations.removeAll(notApplicableReg);
				}
			}
		} else if (currentUser.isPermissionGranted(SecurityController.FACILITY_ADMIN)) {
			facilityId = ((User) currentUser).getFacility().getId();
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
		}
		// convert
		for (User registration : registrations) {
			userForm = new UserForm();
			userForm.setUserName(registration.getName());
			userForm.setGivenName(registration.getFirstName());
			userForm.setFamilyName(registration.getLastName());
			userForm.setMiddleName(registration.getMiddleName());
			userForm.setTelephoneNumber(registration.getTelephoneNumber());
			RegistrationStatus registrationStatus = registration.getRegistrationStatus();
			if (registrationStatus != null) {
				userForm.setRegistrationStatus(
						new TermType(registrationStatus.getName(), registrationStatus.getCode()));
			}
			SimpleVisn visn = registration.getVisn();
			if (visn != null) {
				userForm.setVisn(new TermType(visn.getName(), String.valueOf(visn.getId())));
			}
			SimpleFacility facility = registration.getFacility();
			if (facility != null) {
				userForm.setFacility(new TermType(facility.getName(), facility.getStationNumber()));
			}
			RegistrationReason registrationReason = registration.getRegistrationReason();
			if (registrationReason != null) {
				userForm.setRegistrationReason(
						new TermType(registrationReason.getName(), registrationReason.getCode()));
			}
			RegistrationDeniedReason registrationDeniedReason = registration.getRegistrationDeniedReason();
			if (registrationDeniedReason != null) {
				userForm.setRegistrationDeniedReason(
						new TermType(registrationDeniedReason.getName(), registrationDeniedReason.getCode()));
			}
			users.add(userForm);
		}

		model.addAttribute("users", users);

		return "manageRegistrations";
	}

	@RequestMapping(value = "/manageRegistrations.html", params = { "Cancel" }, method = RequestMethod.POST)
	public String doCancelRequest(Model model) throws Exception {
		return "redirect:" + basePath + "manageRegistrations.html";
	}

	@RequestMapping(value = "/manageRegistrations.html", params = { "Reset" }, method = RequestMethod.POST)
	public String doResetRequest(Model model) throws Exception {
		return "approveRegistration";
	}

	@RequestMapping(value = "/manageRegistrations.html", params = { "denyReset" }, method = RequestMethod.POST)
	public String doDenyResetRequest(Model model) throws Exception {
		return "denyRegistration";
	}

	@RequestMapping(value = "/manageRegistrations.html", method = RequestMethod.POST)
	public String doApproveRegistration(@ModelAttribute("command") UserRegistrationForm userRegistrationForm,
			BindingResult result, Model model, SessionStatus status) throws Exception {
		User user = userRegistrationForm.getUser();
		securityConversionService.convert(userRegistrationForm, user);
		securityService.updateUser(user);

		return "redirect:" + basePath + "manageRegistrations.html";
	}

	// Approve Registration
	@RequestMapping(value = "/approveRegistration.html", method = RequestMethod.GET)
	public String showApproveRegistration(@RequestParam String userName, ModelMap model) throws Exception {

		return manageUsersEditController.buildForm(model, userName, "approveReg");
	}
	
	@RequestMapping(value = "/approveRegistration.html", params = { "Deny" }, method = RequestMethod.GET)
	public String showDenyRegistration(@RequestParam String userName, ModelMap model) throws Exception {

		return manageUsersEditController.buildForm(model, userName, "denyReg");
	} 
}
