package gov.va.med.ccht.controller;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import com.ibm.icu.util.Calendar;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.car.CensusActivityRecord;
import gov.va.med.ccht.model.car.CensusActivitySubTotalRecord;
import gov.va.med.ccht.model.car.CensusActivityVisnRecord;
import gov.va.med.ccht.model.coc.CategoryOfCareRecord;
import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.common.Visn;
import gov.va.med.ccht.model.pssreport.CategoryOfCare;
import gov.va.med.ccht.model.report.AggregateModality;
import gov.va.med.ccht.model.responserate.ResponseRateResults;
import gov.va.med.ccht.model.responserate.ResponseRateRow;
import gov.va.med.ccht.service.common.facility.FacilityService;
import gov.va.med.ccht.service.common.reports.ReportService;
import gov.va.med.ccht.service.common.vendor.VendorService;
import gov.va.med.ccht.service.common.visn.VisnService;
import gov.va.med.ccht.service.htreports.CategoryOfCareReportService;
import gov.va.med.ccht.service.htreports.CensusActivityReportService;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.service.htreports.PssReportService;
import gov.va.med.ccht.service.htreports.ResponseRateReportService;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;

@Controller
public class CensusActivityReportController implements CensusActivityReportsConstants, ReportConstants{
	
	/**
	 * Spring uses a comma to separate multiple values coming to a single path.
	 */
	private String INPUT_SEPARATOR = ",";
	private static final String ALL_EXCEPT_L2_VALUE = "-2";
	private static final int CENSUS_REPORT = 0;
	private static final int ZERO = 0;
	private static final int COC_REPORT = 1;
	private static final int RR_REPORT = 2;
	private static final String MESSAGE_COC_SELECTION_REQUIRED = "categoryOfCare.selection.required";
	
	@Autowired
	private FacilityService facilityService;
	@Autowired
	private VendorService vendorService;
	@Autowired
	private VisnService visnService;
	@Autowired
	private CensusActivityReportService censusActivityReportService;
	@Autowired
	private CategoryOfCareReportService categoryOfCareReportService;
	@Autowired
	private ReportService reportService;
	@Autowired
	private ResponseRateReportService responseRateReportService;
	@Autowired
	private PssReportService pssReportService;

	@InitBinder
	public void initBinder(WebDataBinder binder) {
		String[] allowedFields = { "buttonSelection", "categoriesOfCare", "visnId", "facilityId", "reportEndDate", "modalityCode",
				"vendorId", "showVendorSubtotals", "drillDownVisn", "singleFacility", "originalReportVisnId", "reportID" };
		binder.setAllowedFields(allowedFields);
	}

	@RequestMapping(value = "/censusActivityReports.html", method = RequestMethod.GET)
	public String showCensusReports(Model model) throws ServiceException {
		CensusActivityReportForm form = new CensusActivityReportForm();
		addCommonAttributes(model);
		form.setReportID(CENSUS_REPORT);
		model.addAttribute("reportID", CENSUS_REPORT);
		model.addAttribute("form", form);
		return "censusActivityReports";
	}
	
	@RequestMapping(value = "/categoryOfCareReports.html", method = RequestMethod.GET)
	public String showCategoryOfCareReports(Model model) throws ServiceException {
		CensusActivityReportForm form = new CensusActivityReportForm();
		addCommonAttributes(model);
		form.setReportID(COC_REPORT);
		model.addAttribute("reportID", COC_REPORT);
		model.addAttribute("form", form);
		model.addAttribute("isCategoryOfCareReport", true);
		
		return "categoryOfCareReports";
	}
	
	@RequestMapping(value = "/responseRateReports.html", method = RequestMethod.GET)
	public String showResponseRateReports(Model model) throws ServiceException {
		CensusActivityReportForm form = new CensusActivityReportForm();
		model.addAttribute("isResponseRateReport", true);
		model.addAttribute("facilities", facilityService.getFacilities());
		model.addAttribute("vendors", vendorService.getAllVendors());
		model.addAttribute("visns", visnService.getVisns());
		model.addAttribute("lastDaysOfWeek", getLastDaysOfWeek());
		model.addAttribute("vendorsReportStatus", censusActivityReportService.getAllVendorsReportStatus());
		model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
		model.addAttribute("categoriesOfCare", pssReportService.getCategoriesOfCare());
		//set default COC value to all except L2 option
		setCoCDefault(form, model);
		form.setReportID(RR_REPORT);
		model.addAttribute("reportID", RR_REPORT);
		model.addAttribute("form", form);
		return "responseRateReports";
	}

	@RequestMapping(value = {"/reports.html"}, method = RequestMethod.POST)
	public String generateReports(@ModelAttribute("form") CensusActivityReportForm form, 
			BindingResult result, Model model)
			throws ServiceException, ParseException {		
		addCommonAttributes(model);
		
		model.addAttribute("backButton", false);// used to determine if drill down occurred & back button needs to be available
		switch (form.getButtonSelection()) {

		case NATIONAL_TOTAL:
			form.setVisnId(ALL_VISNS);
			form.setFacilityId(ALL_VISNS);
			// National totals
			model.addAttribute("reportTitle", buildReportTitle(form, NATIONAL_TITLE));
			if (form.isShowVendorSubtotals()) {
				model.addAttribute("reportMenuUrl", "censusActivityReports.html");
				List<CensusActivitySubTotalRecord> reports = censusActivityReportService
						.getSubTotaledNationalReport(form);
				model.addAttribute("nationalSubtotaledReport", reports);
				// Need to update to only include vendors that come back with
				// information.
				model.addAttribute("vendors", getVendorsWithData(vendorService.getAllVendors(), reports));
				return "nationalSubtotalsReport";
			}
			if(form.getReportID() == COC_REPORT) {
				model.addAttribute("firstRowLabel", "Vendor");
				model.addAttribute("reportTitle", buildCategoryOfCareReportTitle(form, NATIONAL_TITLE));
				List<CategoryOfCareRecord> reports = categoryOfCareReportService.getNationalReport(form);
				
				model.addAttribute("categoryOfCareReport", reports);
				addCommonCategoryOfCareData(model, reports);
				
				return "categoryOfCareReport";
			}
			else if (form.getReportID() == RR_REPORT) {
				// Validate the coc selection
				validateCocSelection(form, result, model);
				if(result.hasErrors()) {
					if (form.getReportID() == RR_REPORT) {
						model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
						model.addAttribute("modalityCode", form.getModalityCode());
						model.addAttribute("vendorId", form.getVendorId());
						model.addAttribute("reportEndDate", form.getReportEndDate());
					    setCoCDefault(form, model);
						form.setFacilityId("");
						model.addAttribute("facilityId", form.getFacilityId());
						model.addAttribute("isResponseRateReport", true);
						return "responseRateReports";
					}
				}
				model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
				model.addAttribute("modalityCode", form.getModalityCode());
				model.addAttribute("reportTitle", buildResponseRateTitle(form, NATIONAL_TITLE));
				ResponseRateResults results = responseRateReportService.getTopLevelReport(form);
				if (results.getResponseRateRows() == null || results.getResponseRateRows().size() < 3) {
					model.addAttribute("responseRateRows", new ArrayList<ResponseRateRow>());
				} else {
					model.addAttribute("responseRateRows", results.getResponseRateRows());
				}
				return "responseRateNationalReport";
			} else {
				model.addAttribute("reportMenuUrl", "censusActivityReports.html");
				List<CensusActivityRecord> reports = censusActivityReportService.getNationalReport(form);
				model.addAttribute("nationalTotalReports", reports);
				// This is adding onto the "National Totals" column if a vendor
				// is selected
				String vendorName = "";
				if (!form.getVendorId().equals(ALL_VENDORS)) {
					vendorName += " for " + vendorService.getVendorById(new Long(form.getVendorId())).getName();
				}
				model.addAttribute("vendorName", vendorName);
				return "nationalTotalsReport";
			}
		case VISN_REPORT:
			
			form.setFacilityId(ALL_VISNS);
			model.addAttribute("reportTitle", buildReportTitle(form, VISN_TITLE));
			model.addAttribute("modalityCode", form.getModalityCode());
			model.addAttribute("vendorId", form.getVendorId());
			model.addAttribute("reportEndDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
			
			model.addAttribute("buttonSelection", form.getButtonSelection());
			model.addAttribute("showVendorSubtotals", form.isShowVendorSubtotals());
			//In order to return to previous visn report after drill down
			//Set originalReportVisnId to what was originally run from report menu
			if(form.getOriginalReportVisnId() == null){
				form.setOriginalReportVisnId(form.getVisnId());
				model.addAttribute("originalReportVisnId", form.getOriginalReportVisnId());
			}
			//Set the visn Id from the original Visn Id will either be the same value or resetting
			//the value from the visn selected in drill down 
			form.setVisnId(form.getOriginalReportVisnId());
			
			if (form.getReportID() == COC_REPORT) {
				model.addAttribute("visnReport", true); // used to determine when Drill down is active
				model.addAttribute("firstRowLabel", "VISN       "); // extra space to allow visn name and number to not be superated to multi lines
				model.addAttribute("reportTitle", buildCategoryOfCareReportTitle(form, VISN_TITLE));
				List<CategoryOfCareRecord> reports = categoryOfCareReportService.getVisnReport(form);
				
				if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) { // all vendors and all visns
					model.addAttribute("showPercentColumns", true);
				}
				
				model.addAttribute("categoryOfCareReport", reports);
				addCommonCategoryOfCareData(model, reports);
				
				return "categoryOfCareReport";
			}
			else if (form.getReportID() == RR_REPORT) {
				
				// Validate the coc selection
				validateCocSelection(form, result, model);
				if(result.hasErrors()) {
					if (form.getReportID() == RR_REPORT) {
						model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
						model.addAttribute("modalityCode", form.getModalityCode());
						model.addAttribute("vendorId", form.getVendorId());
						model.addAttribute("reportEndDate", form.getReportEndDate());
						model.addAttribute("visnId", form.getVisnId());
						setCoCDefault(form, model);
						form.setFacilityId("");
						model.addAttribute("facilityId", form.getFacilityId());
						model.addAttribute("isResponseRateReport", true);
						return "responseRateReports";
					}
				}
				//Response rate report only has option of all vendors
				form.setVendorId(ALL_VENDORS);
				model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
				model.addAttribute("modalityCode", form.getModalityCode());
				model.addAttribute("vendorId", form.getVendorId());
				model.addAttribute("reportTitle", buildResponseRateTitle(form, VISN_TITLE));
				model.addAttribute("categoriesOfCare", form.getCategoriesOfCare());
				ResponseRateResults results = responseRateReportService.getTopLevelReport(form);
				if (results.getResponseRateRows() == null || results.getResponseRateRows().size() < 3) {
					model.addAttribute("responseRateRows", new ArrayList<ResponseRateRow>());
				} else {
					model.addAttribute("responseRateRows", results.getResponseRateRows());
				}
				return "responseRateVisnReport";
			}
			else {
				model.addAttribute("reportMenuUrl", "censusActivityReports.html");
				// census Activity Reports
				if (form.isShowVendorSubtotals()) {
					//Subtotaled reports are the same, but the vendor name is included.
					List<CensusActivityVisnRecord> subtotaledRecords = censusActivityReportService.getSubtotaledVisnReport(form);
					if(subtotaledRecords.size() < 2) {
						subtotaledRecords = new ArrayList<CensusActivityVisnRecord>();
					}
					model.addAttribute("visnTotalReports", subtotaledRecords);
					return "visnSubtotalsReport";
				} else {
					if (!form.getModalityCode().equals(ALL_MODALITIES)) {
						List<AggregateModality> modalities = new ArrayList<AggregateModality>();
						modalities.add(censusActivityReportService.getModalityByCode(form.getModalityCode()));
						model.addAttribute("modalities", modalities);
					}
					List<CensusActivityVisnRecord> records = censusActivityReportService.getVisnReport(form);
					// If there are no records, there will be a single total record.
					// Pass in empty list to let datatables handle the no data
					// message.
					if (records.size() < 2) {
						records = new ArrayList<CensusActivityVisnRecord>();
					}
					model.addAttribute("visnTotalReports", records);
					return "visnTotalsReport";
				}
			}
		case FACILITY_REPORT:
			// Validate the user defined facility.
			Facility facility = null;
			if(form.getFacilityId() != null) {
				int lastComma = form.getFacilityId().lastIndexOf(INPUT_SEPARATOR);
				if(lastComma != 0) {
					String userStationNumber = form.getFacilityId().substring(0,lastComma);
					// Avoid adding the comma separator back into the facility id text field.
					form.setFacilityId(userStationNumber);
					try {
						facility = facilityService.getFacility(userStationNumber);
						form.setFacilityId(facility.getStationNumber() + "");
					}
					catch(ServiceException e) {
						result.rejectValue("facilityId","bad.station.number");
						addCommonAttributes(model);
						model.addAttribute("modalityCode", form.getModalityCode());
						model.addAttribute("vendorId", form.getVendorId());
						model.addAttribute("reportEndDate", form.getReportEndDate());
						
						form.setFacilityId("");
						model.addAttribute("facilityId", form.getFacilityId());
						model.addAttribute("buttonSelection", form.getButtonSelection());
						model.addAttribute("form", form);
						if(form.getReportID() == COC_REPORT) {
							model.addAttribute("isCategoryOfCareReport", true);
							return "categoryOfCareReports";
						}
						else if (form.getReportID() == RR_REPORT) {
							model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
							setCoCDefault(form, model);
							model.addAttribute("isResponseRateReport", true);
							return "responseRateReports";
						}
						else {
							return "censusActivityReports";
						}
					}
				}
				else {
					//Remove the input separator
					form.setFacilityId(form.getFacilityId().substring(1));
					if(!form.getFacilityId().equals(ALL_FACILITIES)) {
						//Remove the input separator
						facility = facilityService.getFacility(form.getFacilityId());
						form.setFacilityId(facility.getStationNumber() + "");
					}
				}
			}

			if(form.getReportID() == COC_REPORT) {
				model.addAttribute("firstRowLabel", "Facility");
				model.addAttribute("reportTitle", buildCategoryOfCareReportTitle(form, FACILITY_TITLE));
				List<CategoryOfCareRecord> reports = categoryOfCareReportService.getFacilityReport(form);
				if(form.getFacilityId().equalsIgnoreCase(ALL_FACILITIES)) { // all vendors and all visns
					model.addAttribute("showPercentColumns", true);
				}
				model.addAttribute("categoryOfCareReport", reports);
				addCommonCategoryOfCareData(model, reports);
				
				return "categoryOfCareReport";
			} 
			else if (form.getReportID() == RR_REPORT) {
				// Validate the coc selection
				validateCocSelection(form, result, model);
				if(result.hasErrors()) {
					if (form.getReportID() == RR_REPORT) {
						model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
						model.addAttribute("modalityCode", form.getModalityCode());
						model.addAttribute("vendorId", form.getVendorId());
						model.addAttribute("reportEndDate", form.getReportEndDate());
						setCoCDefault(form, model);
						form.setFacilityId("");
						model.addAttribute("facilityId", form.getFacilityId());
						model.addAttribute("isResponseRateReport", true);
						return "responseRateReports";
					}
				}
				model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
				model.addAttribute("modalityCode", form.getModalityCode());
				model.addAttribute("reportTitle", buildResponseRateTitle(form, FACILITY_TITLE));
				model.addAttribute("categoriesOfCare", form.getCategoriesOfCare());
				ResponseRateResults results = responseRateReportService.getTopLevelReport(form);
				if (results.getResponseRateRows() == null || results.getResponseRateRows().size() < 3) {
					model.addAttribute("responseRateRows", new ArrayList<ResponseRateRow>());
				} else {
					model.addAttribute("responseRateRows", results.getResponseRateRows());
				}
				if (!form.getFacilityId().equals(ALL_FACILITIES)) {
					// A single facility only displays the vendor totals for that facility
					return "responseRateNationalReport";
				}
				return "responseRateFacilityReport";
			} else {
				form.setVisnId(ALL_VISNS);
				model.addAttribute("reportMenuUrl", "censusActivityReports.html");
				model.addAttribute("modalityCode", form.getModalityCode());
				model.addAttribute("vendorId", form.getVendorId());
				model.addAttribute("reportEndDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
				model.addAttribute("reportTitle", buildReportTitle(form, FACILITY_TITLE));
				if (form.isShowVendorSubtotals()) {
					if (!form.getModalityCode().equals(ALL_MODALITIES)) {
						List<AggregateModality> modalities = new ArrayList<AggregateModality>();
						modalities.add(censusActivityReportService.getModalityByCode(form.getModalityCode()));
						model.addAttribute("modalities", modalities);
					}
					// Subtotaled reports are the same, but the vendor name is
					// included.
					List<CensusActivityVisnRecord> subtotaledRecords = censusActivityReportService
							.getSubtotaledFacilityReport(form);
					if (subtotaledRecords.size() < 2) {
						subtotaledRecords = new ArrayList<CensusActivityVisnRecord>();
					}
					model.addAttribute("facilityTotalReports", subtotaledRecords);
					return "facilitySubtotalsReport";
				}
				// Should be the same code, just with visnId of -1 (all visns)
				// instead of a chosen value.
				return drillDownIntoVisn(form, true, model);
			}
		}

		return "censusActivityReports";
	}

	@RequestMapping(value = "/reports.html", method = RequestMethod.POST, params = { "drillDownVisn" })
	public String drillDownIntoVisn(@ModelAttribute("form") CensusActivityReportForm form,
			@RequestParam(name = "drillDownVisn", required = true) boolean drillDownVisn, Model model)
			throws NumberFormatException, ServiceException {
		addCommonAttributes(model);
		model.addAttribute("reportID", CENSUS_REPORT);
		model.addAttribute("reportMenuUrl", "censusActivityReports.html");
		if (!form.getModalityCode().equals(ALL_MODALITIES)) {
			List<AggregateModality> modalities = new ArrayList<AggregateModality>();
			modalities.add(censusActivityReportService.getModalityByCode(form.getModalityCode()));
			model.addAttribute("modalities", modalities);
		}
		model.addAttribute("reportTitle", buildReportTitle(form, FACILITY_TITLE));
		model.addAttribute("modalityCode", form.getModalityCode());
		model.addAttribute("vendorId", form.getVendorId());
		model.addAttribute("reportEndDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		//The back button is only available if the facility is set to all facilities(used when running top level report for visns)
		//& a single visn is selected by drill down
		if(!form.getVisnId().equals(ALL_VISNS) && form.getFacilityId().equals(ALL_FACILITIES)){
			model.addAttribute("backButton", true);// used to determine if drill down occurred & back button needs to be available
		}
		model.addAttribute("buttonSelection", form.getButtonSelection());
		model.addAttribute("originalReportVisnId", form.getOriginalReportVisnId());
        model.addAttribute("showVendorSubtotals", form.isShowVendorSubtotals());
		
		List<CensusActivityVisnRecord> reports = censusActivityReportService.getVisnDrilldownData(form);
		// Pull out totals report (the last in the list) before sorting
		if (reports.size() > 1) {
			CensusActivityVisnRecord totalsRecord = reports.remove(reports.size() - 1);
			Collections.sort(reports);
			reports.add(totalsRecord);
		}
		model.addAttribute("facilityTotalReports", reports);
		return "facilityTotalsReport";
	}

	@RequestMapping(value = "/reports.html", method = RequestMethod.POST, params = { "cocDrillDownVisn" })
	public String cocDrillDownIntoVisn(@ModelAttribute("form") CensusActivityReportForm form,
			@RequestParam(name = "cocDrillDownVisn", required = true) boolean cocDrillDownVisn, Model model)
			throws NumberFormatException, ServiceException {
		addCommonAttributes(model);
		model.addAttribute("reportID", COC_REPORT);
		model.addAttribute("reportTitle", buildCategoryOfCareDrillDowntitle(form));
		model.addAttribute("backButton", true);// used to determine if drill down occurred & back button needs to be available
		model.addAttribute("vendorId", form.getVendorId());
		model.addAttribute("firstRowLabel", "Facility");
		model.addAttribute("reportEndDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		
		model.addAttribute("buttonSelection", form.getButtonSelection());
		model.addAttribute("showVendorSubtotals", form.isShowVendorSubtotals());
		model.addAttribute("originalReportVisnId", form.getOriginalReportVisnId());
		model.addAttribute("modalityCode", form.getModalityCode());
		
		List<CategoryOfCareRecord> reports = new ArrayList<CategoryOfCareRecord>();
		reports = categoryOfCareReportService.getFacilityReportByVisn(form);
		model.addAttribute("categoryOfCareReport", reports);
		addCommonCategoryOfCareData(model, reports);

		return "categoryOfCareReport";
	}

	@RequestMapping(value = "/responseRateReports.html", method = RequestMethod.POST,
			params = {"rrDrillDownVisn"}) 
	public String responseRateDrillDownToVisn(@ModelAttribute("form") CensusActivityReportForm form,
			@RequestParam(name = "rrDrillDownVisn", required = true) boolean rrDrillDownFacility, Model model) throws NumberFormatException, ServiceException {
		//Response rate report only has option of all vendors
		form.setVendorId(ALL_VENDORS);
		model.addAttribute("vendorId", form.getVendorId());
		model.addAttribute("reportID", RR_REPORT);
		model.addAttribute("categoriesOfCare", form.getCategoriesOfCare());
		model.addAttribute("reportTitle", buildResponseRateTitle(form, VISN_TITLE));
		model.addAttribute("reportMenuUrl", "responseRateReports.html");
		
		model.addAttribute("backButton", true);// used to determine if drill down occurred & back button needs to be available
		model.addAttribute("buttonSelection", form.getButtonSelection());
		model.addAttribute("showVendorSubtotals", form.isShowVendorSubtotals());
		model.addAttribute("originalReportVisnId", form.getOriginalReportVisnId());
		model.addAttribute("modalityCode", form.getModalityCode());
		
		ResponseRateResults results = responseRateReportService.getVisnDrillDown(form);
		if (results.getResponseRateRows() == null || results.getResponseRateRows().size() < 3) {
			model.addAttribute("responseRateRows", new ArrayList<ResponseRateRow>());
		} else {
			model.addAttribute("responseRateRows", results.getResponseRateRows());
		}
		// Give the same drill down options as when all facilities are selected.
		addCommonAttributes(model);
		
		return "responseRateFacilityReport";
	}
	
	@RequestMapping(value = "/responseRateReports.html", method = RequestMethod.POST,
			params = {"rrDrillDownFacility"})
	public String responseRateDrillDownToFacility(@ModelAttribute("form") CensusActivityReportForm form,
			@RequestParam(name = "rrDrillDownFacility", required = true) boolean rrDrillDownFacility, Model model) throws NumberFormatException, ServiceException {
		form.setReportID(RR_REPORT);
		model.addAttribute("reportTitle", buildResponseRateTitle(form, FACILITY_TITLE));
		model.addAttribute("reportMenuUrl", "responseRateReports.html");
		ResponseRateResults results = responseRateReportService.getTopLevelReport(form);
		if (results.getResponseRateRows() == null || results.getResponseRateRows().size() < 3) {
			model.addAttribute("responseRateRows", new ArrayList<ResponseRateRow>());
		} else {
			model.addAttribute("responseRateRows", results.getResponseRateRows());
		}
		// Since we are only dealing with a single facility, the vendors are displayed the same way
		// as on a national report.
		addCommonAttributes(model);
		model.addAttribute("categoriesOfCare", form.getCategoriesOfCare());
		model.addAttribute("modalityCode", form.getModalityCode());
		return "responseRateNationalReport";
	}	
	
	private void addCommonAttributes(Model model) throws ServiceException {
		// Add facilities, vendors, days of week, modalities to the model
		model.addAttribute("facilities", facilityService.getFacilities());
		model.addAttribute("vendors", vendorService.getAllVendors());
		model.addAttribute("visns", visnService.getVisns());
		model.addAttribute("lastDaysOfWeek", getLastDaysOfWeek());
		model.addAttribute("modalities", reportService.getAggregateModalities());
		model.addAttribute("vendorsReportStatus", censusActivityReportService.getAllVendorsReportStatus());
	}

	private String buildReportTitle(CensusActivityReportForm form, String reportType)
			throws NumberFormatException, ServiceException {
		if(form.getReportID() == RR_REPORT) {
			String[] cocList = new String[form.getCategoriesOfCare().length];
			cocList = form.getCategoriesOfCare();
			for (int i = 0; i < cocList.length; i++) {
				if(cocList[i].contains("{")) {
					String replacement = cocList[i].replaceAll("\\{", "");
					cocList[i] = replacement;
				}
				if(cocList[i].contains("}")) {
					String replacement = cocList[i].replaceAll("\\}", "");
					cocList[i] = replacement;
				}
			}
			form.setCategoriesOfCare(cocList);
		}
		String result = "Telehealth Activity " + reportType + " Report";
		if (form.getVendorId() != null && !form.getVendorId().equals(ALL_VENDORS)) {
			result += " for " + vendorService.getVendorById(new Long(form.getVendorId())).getName();
		}
		if (form.getFacilityId() != null && !form.getFacilityId().equals(ALL_FACILITIES)) {
			Facility facility = facilityService.getFacility(form.getFacilityId());
			result += String.format(" (Facility %s - %s)", facility.getStationNumber(), facility.getName());
		}
		if(form.getReportID() == RR_REPORT) {
			result += modalitySelectionTitle(form);
			result += categoryOfCareSelectionTitle(form);
		}
		result += " - Week ending " + DateUtils.format(form.getReportEndDate(), DateUtils.MMMDYYYY);
		
		// Test final result for XSS
		ESAPIValidator.validateStringInput(result, ESAPIValidationType.ReportTitle_WhiteList);
		return result;
	}
	
	private String buildCategoryOfCareReportTitle(CensusActivityReportForm form, String reportType)
			throws NumberFormatException, ServiceException {
		String result = "Telehealth Activity Category of Care ";
		
		switch(reportType) {
			
			case NATIONAL_TITLE:
				result += "National Report for ";
				break;
			case VISN_TITLE:
				if(form.getVisnId().equals("-1")) {
					result += "VISN Report (All VISNs) for ";
				}
				else {
					result += "VISN Report (VISN " + form.getVisnId() + ") for ";
				}
				break;
			case FACILITY_TITLE:
				if(form.getFacilityId().equals("-1")) {
					result += "Facility Report for All Facilities for ";
				}
				else {
					Facility facility = facilityService.getFacility(form.getFacilityId());
					if(facility != null)
						result += "Facility Report for Facility " + facility.getStationNumber() + " " + facility.getName() + " for ";
				}
				break; 
		}
		 // check all vendors or one vendor
		if(form.getVendorId().equals("-1")) {
			result += "All Vendors ";
		}
		else {
			result += "Vendor " + vendorService.getVendorById(Long.parseLong(form.getVendorId())).getName() + " ";
		}

		result += " - Week ending " + DateUtils.format(form.getReportEndDate(), DateUtils.MMMDYYYY);
		
		// Test final result for XSS
		ESAPIValidator.validateStringInput(result, ESAPIValidationType.ReportTitle_WhiteList);
		return result;
	}
	
	private String buildCategoryOfCareDrillDowntitle(CensusActivityReportForm form)
			throws NumberFormatException, ServiceException {
		
		// Visn Drill down is always for all facilites into one visn
		String result = "Telehealth Activity Category of Care Facility Report (All Facilities) ";
		result += "for VISN " + form.getVisnId() + " ";
		
		 // check all vendors or one vendor
		if(form.getVendorId().equals("-1")) {
			result += "All Vendors ";
		}
		else {
			result += "Vendor " + vendorService.getVendorById(Long.parseLong(form.getVendorId())).getName() + " ";
		}

		result += " - Week ending " + DateUtils.format(form.getReportEndDate(), DateUtils.MMMDYYYY);
		
		// Test final result for XSS
		ESAPIValidator.validateStringInput(result, ESAPIValidationType.ReportTitle_WhiteList);
		return result;
	}
	
	private String buildResponseRateTitle(CensusActivityReportForm form, String reportType) throws NumberFormatException, ServiceException {
		if(form.getReportID() == RR_REPORT) {
			String[] cocList = new String[form.getCategoriesOfCare().length];
			cocList = form.getCategoriesOfCare();
			for (int i = 0; i < cocList.length; i++) {
				if(cocList[i].contains("{")) {
					String replacement = cocList[i].replaceAll("\\{", "");
					cocList[i] = replacement;
				}
				if(cocList[i].contains("}")) {
					String replacement = cocList[i].replaceAll("\\}", "");
					cocList[i] = replacement;
				}
			}
			form.setCategoriesOfCare(cocList);
		}
		String result = "Telehealth Activity Response Rate " + reportType + " Report";
		if (form.getVisnId() != null && !form.getVisnId().equals(ALL_VISNS)) {
			Visn visn = visnService.getVisnById(new Long(form.getVisnId()));
			// For some reason some visns have trailing whitespace
			result += String.format(" (%s)", visn.getName().trim());
		}
		else if (form.getFacilityId() != null && !form.getFacilityId().equals(ALL_FACILITIES)) {
			Facility facility = facilityService.getFacility(form.getFacilityId());
			result += String.format(" (Facility %s - %s)", facility.getStationNumber(), facility.getName());
		}
		if(form.getReportID() == RR_REPORT) {
			result += modalitySelectionTitle(form);
			result += categoryOfCareSelectionTitle(form);
		}
		result += " - Week ending " + DateUtils.format(form.getReportEndDate(), DateUtils.MMMDYYYY);
		// Test final result for XSS
		ESAPIValidator.validateStringInput(result, ESAPIValidationType.ReportTitle_WhiteList);
		return result;
	}

	private List<Date> getLastDaysOfWeek() throws ServiceException {
		Calendar cal = Calendar.getInstance();
		// Sets the day to the next Saturday, if this is not Saturday.
		cal.set(Calendar.DAY_OF_WEEK, Calendar.SATURDAY);
		// Walk back a week since reports will always have end dates in the past.
		cal.add(Calendar.DAY_OF_YEAR, -7);
		final Date minDate = censusActivityReportService.getMinStartDate();
		List<Date> lastDaysOfWeek = new ArrayList<Date>();
		while (cal.getTime().compareTo(minDate) > 0) {
			lastDaysOfWeek.add(cal.getTime());
			cal.add(Calendar.DAY_OF_YEAR, -7);
		}
		return lastDaysOfWeek;
	}

	private List<Vendor> getVendorsWithData(List<Vendor> allVendors, List<CensusActivitySubTotalRecord> reports) {
		List<Vendor> vendorsWithData = new ArrayList<Vendor>();
		if (reports != null && reports.size() > 0) {
			for (Vendor vendor : allVendors) {
				// If there is no data for "Total Patients" then the vendor had
				// no data for that week.
				if (reports.get(0).getTotalsMap().get(vendor.getName()) != null) {
					vendorsWithData.add(vendor);
				}
			}
		}
		return vendorsWithData;
	}
	
	private void addCommonCategoryOfCareData(Model model, List<CategoryOfCareRecord> reports) {
		
		// main reports menu url
		model.addAttribute("reportMenuUrl", "categoryOfCareReports.html");
		
		// if 1 there are no reports, only two default reports.
		if(reports.size() > 2) {
			// specify number of time to loop over reports, last two "reports" are totals rows.
			// and need to be printed differently
			model.addAttribute("vendorReportsSize", reports.size() - 3);
			model.addAttribute("dataPresent", true);
		}
		else {
			model.addAttribute("dataPresent", false);
		}
	}
	
	private String modalitySelectionTitle( CensusActivityReportForm form){
		StringBuilder sb = new StringBuilder();
		String modalityCode = form.getModalityCode();
			switch(modalityCode) {
			case BROWSER:
				sb.append(", Modality - Web Browser ");
	            break;
	        case IVR:
	        	sb.append(", Modality - IVR ");
	            break;
	        case HOME_DEVICE:
	        	sb.append(", Modality - Home Device ");
	            break;
	        case DO_NOT_INCLUDE:
	            break;
	        case ALL_MODALITIES:
	        	sb.append(", All Modalities ");
	            break;
			}
			return sb.toString();
	}
	
	private String categoryOfCareSelectionTitle( CensusActivityReportForm form){
		if(form.getCategoriesOfCare() == null || form.getCategoriesOfCare().length == ZERO) return "";
		
		StringBuilder sb = new StringBuilder();
		if(form.getCategoriesOfCare()[0].equals(PSS_ALL_COC)) {
			sb.append(", All Categories of Care");
		}
		else if(form.getCategoriesOfCare()[0].equals(PSS_ALL_COC_EXCEPT_L2)){
			sb.append(", All Categories of Care Except L2");
		}
		else {
			for(String categoryString: form.getCategoriesOfCare()) {
				// No great way to do the lookup, so use the least DB calls because this report title
				// method is not called often
				for (CategoryOfCare category: pssReportService.getCategoriesOfCare()) {
					if(category.getId() == Integer.parseInt(categoryString)) {
						sb.append(", " + category.getName());
					}
				}
			}
		}
		return sb.toString();
	}
	
	private void validateCocSelection(CensusActivityReportForm form, BindingResult result, Model model) throws ServiceException{
		if((form.getCategoriesOfCare() == null) || (form.getCategoriesOfCare().length == 0)) {
			result.rejectValue("categoriesOfCare", MESSAGE_COC_SELECTION_REQUIRED, "Category of Care selection is a required field.");
			addCommonAttributes(model);
			model.addAttribute("reportEndDate", form.getReportEndDate());
			model.addAttribute("buttonSelection", form.getButtonSelection());
			model.addAttribute("form", form);
			
		}
	}
	private void setCoCDefault(CensusActivityReportForm form, Model model) {
		String[] categoriesOfCare = new String[8];
		categoriesOfCare[1] = ALL_EXCEPT_L2_VALUE;
		form.setCategoriesOfCare(categoriesOfCare);
	}
}
