/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.controller;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.springframework.batch.core.BatchStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.CronTriggerBean;
import org.springframework.scheduling.quartz.JobDetailBean;
import org.springframework.stereotype.Controller;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.Schedule;
import gov.va.med.ccht.service.common.BatchProcessTriggerGroups;
import gov.va.med.ccht.service.report.ScheduleConversionService;
import gov.va.med.ccht.ui.model.BatchProcessDetailForm;
import gov.va.med.ccht.ui.model.FormattedDateForm;
import gov.va.med.ccht.ui.model.ReportScheduleForm;
import gov.va.med.fw.batchprocess.BatchProcessDetail;
import gov.va.med.fw.batchprocess.BatchProcessDetailStatus;
import gov.va.med.fw.batchprocess.BatchProcessInvoker;
import gov.va.med.fw.batchprocess.BatchProcessService;
import gov.va.med.fw.model.batchprocess.JobConfig;
import gov.va.med.fw.scheduling.SchedulingService;
import gov.va.med.fw.scheduling.TriggerStatus;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.jms.JMSPayload;
import gov.va.med.fw.service.jms.MessageProducerException;
import gov.va.med.fw.service.jms.MessageProducerService;
import gov.va.med.fw.service.jms.SynchronousMessageProducerServiceImpl;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * 
 * 
 * IHTA ohrs_web Nov 12, 2010
 * 
 * @author DNS
 */

@Controller
public class BatchProcessController extends AbstractComponent implements CCHTRoles {

	public static final int MAX_USER_ID_LENGTH = 30;
	public static final String JOB_NAME = "jobName";
	public static final String JOB_PARAMETERS = "jobParameters";
	public static final String USER_ID = "userId";

	private BatchProcessInvoker batchProcessInvoker;
	@Autowired
	private BatchProcessService batchProcessService;
	@Autowired
	private SchedulingService schedulingService;
//	@Autowired
//	private JobLauncher jobLauncherx;
//	@Autowired
//	private JobLocator jobLocator;
	private MessageProducerService messageProducerService;

	// bean name for batch payload defined in cchtJms.xml
	private String runBatchJobJMSPayloadName = "runBatchJobJMSPayload";
	@Autowired
	private ScheduleConversionService scheduleConversionService;
	@Autowired
	private ReportConversionService reportConversionService;
	
	public BatchProcessController() {
		batchProcessInvoker = new BatchProcessInvoker();
		messageProducerService = new SynchronousMessageProducerServiceImpl();
	}

	/**
	 * Get Active jobs
	 * 
	 * @return
	 * @throws Exception
	 */
	public List getActiveJobs() throws Exception {
		List activeJobs = batchProcessService.getJobResults(BatchStatus.STARTED);
		return activeJobs;
	}

	public void cancelActiveJob(long jobResultId) throws Exception {
		batchProcessService.cancelActiveJob(jobResultId);
	}

	public void markActiveJobAsError(long jobResultId) throws Exception {
		batchProcessService.errActiveJob(jobResultId);
	}

	@SuppressWarnings("unchecked")
	public List<BatchProcessDetailForm> getAllJobs() throws Exception {
		UserPrincipal user = SecurityContextHelper.getSecurityContext().getUserPrincipal();
		Collection<BatchProcessDetail> jobs = getBatchProcessDetails();
		List<BatchProcessDetailForm> jobAndTriggerStatus = new ArrayList<BatchProcessDetailForm>(jobs.size());
		// now iterate through and determine the dynamic pieces from the
		// Scheduler....next Fire time, etc
		Iterator itr = jobs != null ? jobs.iterator() : null;
		BatchProcessDetail detail = null;
		BatchProcessDetailForm form = null;
		TriggerStatus triggerStatus = null;
		while (itr != null && itr.hasNext()) {
			detail = (BatchProcessDetail) itr.next();
			form = new BatchProcessDetailForm();
			boolean canExecuteOrView = true;
			// set batch process details name, group ..
			form.setJobName(detail.getJobName());
			form.setGroupName(detail.getGroupName());
			form.setJobDescription(detail.getJobDescription());
			form.setTriggerName(detail.getTriggerName());
			form.setTriggerGroup(detail.getTriggerGroup());

			// set trigger status and next fire time
			if (org.apache.commons.lang3.StringUtils.isNotEmpty(detail.getTriggerName())) {
				triggerStatus = schedulingService.getTriggerStatus(detail.getTriggerName(), detail.getTriggerGroup());
				if (triggerStatus != null) {
					form.setStatus(triggerStatus.getStatus());
					form.setNextFireTime(triggerStatus.getNextFireTime());
				}
			}

			JobConfig jobConfig = batchProcessService.getJobConfig(detail.getJobName(), detail.getGroupName());
			if (jobConfig != null) {
				if (jobConfig.getJobSchedule() != null) {
					form.setJobSchedule(jobConfig.getJobSchedule());
					form.setJobScheduleDescription(jobConfig.getJobScheduleDescription());
					if (org.apache.commons.lang3.StringUtils.isNotEmpty(jobConfig.getJobScheduleText())) {
						Schedule schedule = scheduleConversionService.toSchedule(jobConfig.getJobScheduleText());
						ReportScheduleForm rsForm = new ReportScheduleForm();
						reportConversionService.convert(schedule, rsForm);  
						form.setReportSchedule(rsForm);
					}
				}
				form.setEmailDistributionList(jobConfig.getEmailDistributionList());
				canExecuteOrView = isAllowedToExecuteOrViewBatchJob(user, jobConfig.getPermissions());
			}
			if (canExecuteOrView) {
				jobAndTriggerStatus.add(form);
			}
		}
		Collections.sort(jobAndTriggerStatus);
		return jobAndTriggerStatus;
	}

	public void executeBatchProcess(String jobDetailsName, String invocationArgsStrArray) throws Exception {
		sendMessageForexecution(jobDetailsName, invocationArgsStrArray);
	}

	public void reschedule(BatchProcessDetailForm batchProcessDetailForm) throws Exception {
		String jobName = batchProcessDetailForm.getJobName();
		String groupName = batchProcessDetailForm.getGroupName();
		String newSchedule = batchProcessDetailForm.getJobSchedule();
		String jobScheduleDescription = batchProcessDetailForm.getJobScheduleDescription();
		ReportScheduleForm reportScheduleForm = batchProcessDetailForm.getReportSchedule();
		String triggerName = batchProcessDetailForm.getTriggerName();
		String triggerGroup = batchProcessDetailForm.getTriggerGroup();

		JobConfig jobConfig = batchProcessService.getJobConfig(jobName, groupName);
		if (jobConfig == null) {
			jobConfig = new JobConfig();
			jobConfig.setName(jobName);
			jobConfig.setGroup(groupName);
		}
		// New schedule is empty
		if (org.apache.commons.lang3.StringUtils.isEmpty(newSchedule)) {
			return;
		}

		jobConfig.setJobSchedule(newSchedule);
		jobConfig.setJobScheduleDescription(jobScheduleDescription);
		// convert the schedule object
		Schedule schedule = new Schedule();
		reportConversionService.convert(schedule, reportScheduleForm);
		jobConfig.setJobScheduleText(scheduleConversionService.toText(schedule));
		batchProcessService.saveJobConfig(jobConfig);
		Trigger trigger = schedulingService.getTrigger(triggerName, triggerGroup);

		if (trigger != null && trigger instanceof CronTrigger) {
			
			CronTrigger cronTrigger = TriggerBuilder
					.newTrigger()
					.withSchedule(CronScheduleBuilder.cronSchedule(jobConfig.getJobSchedule()).inTimeZone(TimeZoneUtils.getTimeZone()))
					.build();
			
			schedulingService.reschedule(triggerName, triggerGroup, cronTrigger);
		} else {

			// Construct the name by configuration standards
			if (org.apache.commons.lang3.StringUtils.isEmpty(triggerName)) {
				triggerName = (jobName.substring(0, 1)).toLowerCase() + jobName.substring(1) + "CronTrigger";
			}
			try {
				CronTriggerBean cronTriggerBean = (CronTriggerBean) getComponent(triggerName);
				if (cronTriggerBean != null) {
					
					CronTrigger cronTrigger = TriggerBuilder
							.newTrigger()
							.withSchedule(CronScheduleBuilder.cronSchedule(jobConfig.getJobSchedule()).inTimeZone(TimeZoneUtils.getTimeZone()))
							.build();
					
					JobDetail jobDetail = cronTriggerBean.getJobDetail();
					if (jobDetail != null)
						schedulingService.schedule(jobDetail, cronTrigger);
					else
						schedulingService.schedule(cronTrigger);
				}
			} catch (Exception e) {
				throw new Exception(
						"Missing or invalid configuration, could not schedule the batch job for " + jobName);
			}
		}
	}

	public void updateConfig(BatchProcessDetailForm batchProcessDetailForm) throws Exception {
		JobConfig jobConfig = batchProcessService.getJobConfig(batchProcessDetailForm.getJobName(),
				batchProcessDetailForm.getGroupName());
		if (jobConfig == null) {
			jobConfig = new JobConfig();
			jobConfig.setName(batchProcessDetailForm.getJobName());
			jobConfig.setGroup(batchProcessDetailForm.getGroupName());
		}

		jobConfig.setEmailDistributionList(batchProcessDetailForm.getEmailDistributionList());
		batchProcessService.saveJobConfig(jobConfig);

		String triggerStatus = batchProcessDetailForm.getStatus();
		String previousTriggerStatus = batchProcessDetailForm.getPreviousTriggerStatus();
		if (org.apache.commons.lang3.StringUtils.isNotEmpty(triggerStatus) && !triggerStatus.equals(previousTriggerStatus)) {
			if (TriggerStatus.NORMAL_STATUS.equals(triggerStatus)) {
				schedulingService.resumeJob(batchProcessDetailForm.getJobName(), batchProcessDetailForm.getGroupName());
			} else if (TriggerStatus.PAUSED_STATUS.equals(triggerStatus)) {
				schedulingService.pauseJob(batchProcessDetailForm.getJobName(), batchProcessDetailForm.getGroupName());
			}
		}
	}

	public void pauseAll() throws Exception {
		schedulingService.pauseTriggerGroup(BatchProcessTriggerGroups.STATIC_BATCH_JOBS);
	}

	public void resumeAll() throws Exception {
		schedulingService.resumeTriggerGroup(BatchProcessTriggerGroups.STATIC_BATCH_JOBS);
	}

	@SuppressWarnings("unchecked")
	public List getJobHistory(String jobName) throws Exception {
		BatchProcessDetailStatus job = getBatchProcessDetailStatus(jobName);
		List jobResults = batchProcessService.getFinishedJobResults(job.getBatchProcessDetail().getJobName(),
				job.getBatchProcessDetail().getJobGroup());
		return jobResults;
	}

	public BatchProcessDetailForm viewExecuteWithArgs(String jobName) throws Exception {
		BatchProcessDetailForm form = new BatchProcessDetailForm();
		BatchProcessDetailStatus job = getBatchProcessDetailStatus(jobName);
		convert(job, form);
		return form;
	}

	public BatchProcessService getBatchProcessService() {
		return batchProcessService;
	}

	public void setBatchProcessService(BatchProcessService batchProcessService) {
		this.batchProcessService = batchProcessService;
	}

	public BatchProcessInvoker getBatchProcessInvoker() {
		return batchProcessInvoker;
	}

	public void setBatchProcessInvoker(BatchProcessInvoker batchProcessInvoker) {
		this.batchProcessInvoker = batchProcessInvoker;
	}

	public SchedulingService getSchedulingService() {
		return schedulingService;
	}

	public void setSchedulingService(SchedulingService schedulingService) {
		this.schedulingService = schedulingService;
	}

//	public JobLauncher getJobLauncher() {
//		return jobLauncher;
//	}
//
//	public void setJobLauncher(JobLauncher jobLauncher) {
//		this.jobLauncher = jobLauncher;
//	}
//
//	public JobLocator getJobLocator() {
//		return jobLocator;
//	}
//
//	public void setJobLocator(JobLocator jobLocator) {
//		this.jobLocator = jobLocator;
//	}

	public MessageProducerService getMessageProducerService() {
		return messageProducerService;
	}

	public void setMessageProducerService(MessageProducerService messageProducerService) {
		this.messageProducerService = messageProducerService;
	}

	public String getRunBatchJobJMSPayloadName() {
		return runBatchJobJMSPayloadName;
	}

	public void setRunBatchJobJMSPayloadName(String runBatchJobJMSPayloadName) {
		this.runBatchJobJMSPayloadName = runBatchJobJMSPayloadName;
	}

	public ScheduleConversionService getScheduleConversionService() {
		return scheduleConversionService;
	}

	public void setScheduleConversionService(ScheduleConversionService scheduleConversionService) {
		this.scheduleConversionService = scheduleConversionService;
	}

	public ReportConversionService getReportConversionService() {
		return reportConversionService;
	}

	public void setReportConversionService(ReportConversionService reportConversionService) {
		this.reportConversionService = reportConversionService;
	}

	
	
	/**
	 * Get the job name and send a message to execute the job async
	 * 
	 * @param jobDetailsName
	 * @param invocationArgsStrArray
	 * @throws Exception
	 */
	private void sendMessageForexecution(String jobDetailsName, String invocationArgsStrArray)
			throws MessageProducerException {

		JobDetailBean jobDetailsBean = (JobDetailBean) getApplicationContext().getBean(jobDetailsName);
		String jobName = (String) jobDetailsBean.getJobDataMap().get(JOB_NAME);

		Map<String, Object> map = new HashMap<String, Object>();
		map.put(JOB_NAME, jobName);
		map.put(JOB_PARAMETERS, invocationArgsStrArray);
		map.put(USER_ID, SecurityContextHelper.getSecurityContext().getUserPrincipal().getUserCredentials().getUserID());

		JMSPayload jmsPayLoad = (JMSPayload) getApplicationContext().getBean(getRunBatchJobJMSPayloadName());
		jmsPayLoad.setPayload((Serializable) map);

		// set header properties
		Map<String, Serializable> props = new HashMap<String, Serializable>();
		String logicalUserName = jobName;
		if (logicalUserName.length() > MAX_USER_ID_LENGTH) {
			logicalUserName = logicalUserName.substring(0, MAX_USER_ID_LENGTH);
		}
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_TYPE, jmsPayLoad.getTargetServiceDescriptor());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_INITIATER, logicalUserName);
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_ORIGINATING_TIMEZONE, TimeZoneUtils.getTimeZone().getID());

		messageProducerService.send((Serializable) jmsPayLoad.getPayload(), props);
	}

	private BatchProcessDetailStatus getBatchProcessDetailStatus(String jobName)
			throws SchedulerException, ServiceException {

		BatchProcessDetailStatus status = new BatchProcessDetailStatus();
		BatchProcessDetail detail = getBatchProcessDetail(jobName);
		status.setBatchProcessDetail(detail);
		// call Scheduler to get trigger status
		TriggerStatus triggerStatus = schedulingService.getTriggerStatus(detail.getTriggerName(),
				detail.getTriggerGroup());
		status.setTriggerStatus(triggerStatus);
		JobConfig jobConfig = batchProcessService.getJobConfig(detail.getJobName(), detail.getGroupName());
		if (jobConfig != null) {
			status.setEmailDistributionList(jobConfig.getEmailDistributionList());
		}
		return status;
	}

	@SuppressWarnings("unchecked")
	private Collection getBatchProcessDetails() {
		return batchProcessInvoker.getBatchProcessDetails();
	}

	private BatchProcessDetail getBatchProcessDetail(String jobName) {
		return batchProcessInvoker.getBatchProcessDetail(jobName);
	}

	private boolean isAllowedToExecuteOrViewBatchJob(UserPrincipal user, Set<Permission> permissions) {
		if (permissions != null && permissions.size() > 0) {
			for (Permission permission : permissions) {
				String code = permission.getName();
				if (user.isPermissionGranted(code)) {
					return true;
				}
			}
		} else {
			return true;
		}
		return false;
	}

	private void convert(BatchProcessDetailStatus batchProcessDetailStatus, BatchProcessDetailForm form) {
		if (batchProcessDetailStatus == null)
			return;
		form.setEmailDistributionList(batchProcessDetailStatus.getEmailDistributionList());

		BatchProcessDetail detail = batchProcessDetailStatus.getBatchProcessDetail();
		if (detail != null) {
			form.setJobName(detail.getJobName());
			form.setJobDescription(detail.getJobDescription());
			form.setTriggerName(detail.getTriggerName());
			form.setTriggerGroup(detail.getTriggerGroup());
		}

		TriggerStatus triggerStatus = batchProcessDetailStatus.getTriggerStatus();
		if (triggerStatus != null) {
			form.setStatus(triggerStatus.getStatus());
			form.setNextFireTime(triggerStatus.getNextFireTime());
		}
	}
}
