package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import gov.va.med.ccht.model.pssreport.SurveyTrendChartResult;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.persistent.SurveyTrendChartsDAO;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.persistent.DAOException;

public class SurveyTrendChartsDAOImplTest {
	private static final int MONTH = 0;
	private static final int VISN = 0;
	private static final int NUM_OF_SURVEYS_ALL_VISNS = 1;
	private static final int NUM_OF_SURVEYS_FACILITIES = 1;
	private static final int NUM_OF_SURVEYS = 2;
	
	public static final int NATIONAL_SELECTION = 1;
	public static final int VISN_SELECTION = 2;
	public static final int FACILITY_SELECTION = 3;
	
	static final private String ALL = "-1";
	static final private String TESTID = "1";
	static final private String VISN_NAME = "Visn";
	static final private String MONTH_NAME = "Month";
	
	static final private Date TEST_DATE = new Date();
	
	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	@Mock
	ReportsDAO reportsDao;
	
	private SurveyTrendChartsDAO surveyReport;
	
	private PSSReportForm form;
	private List<Object[]> emptyTestRows;
	private LinkedList<SurveyTrendChartResult> result;
	private Calendar testCalDate;
	
	@Before
	public void setup() {
		MockitoAnnotations.initMocks(this);
		
		Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
		Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
		
		surveyReport = new SurveyTrendChartsDAOImpl(sessionFactory);
		emptyTestRows = new ArrayList<>();
		testCalDate = Calendar.getInstance();
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNational() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllVisnsShouldNotSetVisnIdParam() throws DAOException{
		form = new PSSReportForm();
		form.setVisnId(ALL);
		form.setTypeButtonSelection(VISN_SELECTION);
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
		Mockito.verify(nQuery, Mockito.never()).setParameter("visnId", form.getVisnId());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithSingleVisnShouldSetVisnIdParam() throws DAOException{
		form = new PSSReportForm();
		form.setVisnId(TESTID);
		form.setTypeButtonSelection(VISN_SELECTION);
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", form.getVisnId());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllFacilitiesShouldNotSetFacilityIdParam() throws DAOException{
		form = new PSSReportForm();
		form.setFacilityId(ALL);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
		Mockito.verify(nQuery, Mockito.never()).setParameter("facilityId", form.getFacilityId());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithSingleFacilityShouldSetFacilityIdParam() throws DAOException{
		form = new PSSReportForm();
		form.setFacilityId(TESTID);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", form.getFacilityId());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNoButtonTypeSelectionDefaultCaseShouldReturnEmptyList() throws DAOException{
		form = new PSSReportForm();
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNullQueryShouldReturnEmptyList() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		Mockito.doReturn(null).when(session).createNativeQuery(Mockito.anyString());
		assertTrue(surveyReport.getSurveyTrendChartsReportData(form).isEmpty());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNationalSelection6MonthsOfResultRowsShouldReturn6Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		
		Mockito.doReturn(createTestData(form)).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 6);	
		
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("fromDate", form.getShortFormatReportFromDate());
		
		testCalDate.setTime(form.getShortFormatReportToDate());
		testCalDate.add(Calendar.MONTH, 1);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("toDate", testCalDate.getTime());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllVisnsSelection6VisnsOfResultRowsShouldReturn6Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setVisnId(ALL);
		
		Mockito.doReturn(createTestData(form)).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 6);	
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllFacilitiesSelection6MonthsOfResultRowsShouldReturn6Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setFacilityId(ALL);
		
		Mockito.doReturn(createTestData(form)).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 6);	
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNationalSelection0MonthsOfResultRowsShouldReturn0Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 0);	
		
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("fromDate", form.getShortFormatReportFromDate());
		
		testCalDate.setTime(form.getShortFormatReportToDate());
		testCalDate.add(Calendar.MONTH, 1);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("toDate", testCalDate.getTime());
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllVisnsSelection0VisnsOfResultRowsShouldReturn0Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setVisnId(ALL);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 0);	
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithAllFacilitiesSelection0MonthsOfResultRowsShouldReturn0Results() throws DAOException{
		form = new PSSReportForm();
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setFacilityId(ALL);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSurveyTrendChartsReportData(form);
		assertTrue(result.size() == 0);	
	}
	
	@Test
	public void getDatesByMonthShouldReturnAListOfMonthYearPairings(){
		List<Date> dateList = surveyReport.getDatesByMonth();
		assertNotNull(dateList);
		
		testCalDate.set(Calendar.getInstance().getTime().getYear() + 1900, Calendar.getInstance().getTime().getMonth(), 1);
		assertTrue(dateList.get(0).getMonth() == testCalDate.getTime().getMonth());
		assertTrue(dateList.get(0).getYear() == testCalDate.getTime().getYear());
		
		testCalDate = Calendar.getInstance();
		testCalDate.set(2006, Calendar.JANUARY, 1);
		assertTrue(dateList.get(dateList.size() - 1).getMonth() == testCalDate.getTime().getMonth());
		assertTrue(dateList.get(dateList.size() - 1).getYear() == testCalDate.getTime().getYear());
	}
	
	private List<Object[]> createTestData(PSSReportForm form){
		List<Object[]> testRows= new ArrayList<>();
		for(int i = 0; i < 6; i++){
			Object[] row = new Object[3];
			if(form.getTypeButtonSelection() == VISN_SELECTION && form.getVisnId().equals(ALL)) {
				row[VISN] = VISN_NAME + i;
				row[NUM_OF_SURVEYS_ALL_VISNS] = NUM_OF_SURVEYS_ALL_VISNS + i;
			}else if(form.getTypeButtonSelection() == FACILITY_SELECTION){
				row[MONTH] = MONTH_NAME + i;
				row[NUM_OF_SURVEYS_FACILITIES] = NUM_OF_SURVEYS_FACILITIES + i;
			}else{
				row[MONTH] = MONTH_NAME + i;
				row[NUM_OF_SURVEYS]= NUM_OF_SURVEYS + i;
			}
			testRows.add(row);
		}
		return testRows;
	}
}
