package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.med.ccht.model.SimpleUser;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.RolePermission;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class SecurityDAOImplTest {
    
    private static final String CCHT_USER = "CCHT_USER";
    private static final String TEST_ID_KNOWN = "test";
    private static final int TEST_OBJECT_ARRAY_SIZE = 4;
    private static final String TEST_USER_NAME = "testUser";
    private static final int TEST_ID = 1;
    static final private String TEST_ID_NAME = "1";
    static final private String TEST_FN_KNOWN = "testerFirstName";
    static final private String TEST_LN_KNOWN = "testerLastName";
    
    @Mock
    private SessionFactory sessionFactory;
    @Mock
    private Session session;
    @Mock
    private NativeQuery<Session> nQuery;
    @Mock
    private NativeQuery<Session> pQuery;
    @Mock
    private NativeQuery<Session> qQuery;
    @Mock
    private NativeQuery<Session> nativeQuery;
    private SecurityDAOImpl securityDao;
    List<Role> testRole;
    Role test;

    @Before
    public void setup() {
       MockitoAnnotations.initMocks(this);
       securityDao = new SecurityDAOImpl(sessionFactory);
       Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
       Mockito.doReturn(nQuery).when(session).createQuery(Mockito.anyString());
       Mockito.doReturn(pQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
       Mockito.doReturn(pQuery).when(nativeQuery).setParameter(Mockito.anyString(), Mockito.any());
       Mockito.doReturn(qQuery).when(pQuery).setParameter(Mockito.anyString(), Mockito.any());
       Mockito.doReturn(nativeQuery).when(session).createNativeQuery(Mockito.anyString());
       testRole = new ArrayList<Role>();
       test = new Role();
       test.setName(TEST_ID_NAME);
          testRole.add(test);
    }
    
    @After
    public void cleanUp() {
        test = null;
        testRole = null;
    }
  
    /**
     * 
     */
    @Test
    public void getSubmittedRegistrationsByVisnIdShouldAddVisnIdToWhereClause() {
        //test for submitted registration by visn
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code "
                + "!= :registrationStatus and l.visn.id = :visnId";
        assertNotNull(securityDao.getSubmittedRegistrations(TEST_ID, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", TEST_ID);
    }
    
    @Test
    public void getSubmittedRegistrationsByFacilityIdShouldAddFacilityIdToWhereClause() {
        //test for submitted registration by facility
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code "
                + "!= :registrationStatus and l.facility.id = :facilityId";
        assertNotNull(securityDao.getSubmittedRegistrations(null, TEST_ID));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID);
    }
    
    @Test
    public void getSubmittedRegistrationsWithallShouldSubmitQueryWithBaseWhereClause() {
        //test for all submitted registrations
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code "
                + "!= :registrationStatus";
        assertNotNull(securityDao.getSubmittedRegistrations(null, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
    }

    @Test
    public void getNewRegisbtrationsByVisnIdShouldAddVisnIdToWhereClause() {
        //test for submitted registration by visn
        String sqlStr = "select l from gov.va.med.ccht.model.User "
                + "l where l.registrationStatus.code = :registrationStatus and l.visn.id = :visnId";
        assertNotNull(securityDao.getNewRegistrations(TEST_ID, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", TEST_ID);
    }
    
    @Test
    public void getNewRegistrationsByFacilityIdShouldAddFacilityIdToWhereClause() {
        //test for submitted registration by facility
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code = :registrationStatus and l.facility.id = :facilityId";
        assertNotNull(securityDao.getNewRegistrations(null, TEST_ID));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID);
    }
    
    @Test
    public void getNewRegistrationsWithallShouldSubmitQueryWithBaseWhereClause() {
        //test for all submitted registrations
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code "
                + "= :registrationStatus";
        assertNotNull(securityDao.getNewRegistrations(null, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
    }
    
    @Test
    public void getSubmittedRegistrationsForDmpByVisnIdShouldAddVisnIdToWhereClause() throws DAOException {
        //test for all submitted registrations
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code != :registrationStatus and (l.registrationReason.name "
                + "like '%DMP%' or l.registrationReason.name = 'Training Center Staff') and l.visn.id = :visnId";
        assertNotNull(securityDao.getSubmittedRegistrationsForDmp(TEST_ID, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", TEST_ID);
    }
    
    @Test
    public void getSubmittedRegistrationsForDmpByFacilityIdShouldAddFacilityIdToWhereClause() throws DAOException {
        //test for submitted registration by facility
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code != :registrationStatus and (l.registrationReason.name "
                + "like '%DMP%' or l.registrationReason.name = 'Training Center Staff') and l.facility.id = :facilityId";
        assertNotNull(securityDao.getSubmittedRegistrationsForDmp(null, TEST_ID));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID);
    }
    
    @Test
    public void getSubmittedRegistrationsForDmpByAnnShouldSubmitQueryWithBaseWhereClause() throws DAOException {
        //test for submitted registration by facility
        String sqlStr = "select l from gov.va.med.ccht.model.User l "
                + "where l.registrationStatus.code != :registrationStatus and (l.registrationReason.name "
                + "like '%DMP%' or l.registrationReason.name = 'Training Center Staff')";
        assertNotNull(securityDao.getSubmittedRegistrationsForDmp(null, null));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sqlStr);
    }    
    
    @Test
    public void getRoleByNameShouldReturnARoleWithGivenName() {
        String sql = "select l from gov.va.med.fw.security.Role l where UPPER(l.name) = :roleName";
        Mockito.doReturn(testRole).when(nQuery).getResultList();
        assertTrue((securityDao.getRoleByName(TEST_ID_NAME).getName()).contains(testRole.get(0).getName()));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sql);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("roleName", TEST_ID_NAME);
        
    }
    
    @Test
    public void getRoleByNameWithNullListShouldReturnNull() {
        String sql = "select l from gov.va.med.fw.security.Role l where UPPER(l.name) = :roleName";
        Mockito.doReturn(null).when(nQuery).getResultList();
        assertNull(securityDao.getRoleByName(TEST_ID_NAME));
        Mockito.verify(session, Mockito.atLeast(1)).createQuery(sql);
        Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("roleName", TEST_ID_NAME);
        
    }
    
    @Test
    public void getAuditUserGivenValidUserNameShouldReturnUserPrincipalWithCorrectName() throws DAOException {
        final String userName = "Testee";
        final String sql = "select USER_NAME, FIRST_NAME, MIDDLE_NAME, LAST_NAME"
                + " from ht.app_user"
                + " where upper(USER_NAME) = :userName";
        Object[] testOne = new Object[TEST_OBJECT_ARRAY_SIZE];
        Object[] testTwo = new Object[TEST_OBJECT_ARRAY_SIZE];
        Object[] testThree = new Object[TEST_OBJECT_ARRAY_SIZE];
        testOne[1] = "testFirst"; testOne[2] = "testMiddle"; testOne[3] = "testLast";
        testTwo[1] = "testFirst"; testTwo[2] = "testMiddle"; testTwo[3] = "testLast";
        testThree[1] = "testFirst"; testThree[2] = "testMiddle"; testThree[3] = "testLast";
        List<Object[]> rows = new ArrayList<>();
        rows.add(testOne);
        rows.add(testTwo);
        rows.add(testThree);
        Mockito.doReturn(rows).when(pQuery).getResultList();
        assertTrue(securityDao.getAuditUser(userName).getFirstName().contains("testFirst"));
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(sql);
        Mockito.verify(nativeQuery, Mockito.atLeastOnce()).setParameter("userName", userName);
    }
    
    @Test
    public void getAuditUserGivenUserNameThatYieldsNullResultsShouldReturnNewUserPrincipal() throws DAOException {
        final String userName = "Testee";
        final String sql = "select USER_NAME, FIRST_NAME, MIDDLE_NAME, LAST_NAME"
                + " from ht.app_user"
                + " where upper(USER_NAME) = :userName";
        Mockito.doReturn(null).when(pQuery).getResultList();
        assertTrue(securityDao.getAuditUser(userName).getUserCredentials().getUserID().contains("Testee"));
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(sql);
        Mockito.verify(nativeQuery, Mockito.atLeastOnce()).setParameter("userName", userName);
        
    }
    
    @Test
    public void findAppUsersGivenEmptySearchCriteriaShouldThrowException() throws DAOException {
        SearchCriteria sc = new SearchCriteria();
        List<SimpleUser> testList = null;
        try {
            testList = securityDao.findAppUsers(sc);
        } catch (Exception e) {
            assertTrue(e.getCause().getMessage().contains("findAppUsers failed: Invalid Search criteria"));
        }
        assertNull(testList);
        Mockito.verifyZeroInteractions(session);
    }
    
    @Test
    public void findAppUsersGivenSAMAccountNameSearchCriteriaShouldQueryWithUserID() throws DAOException {
        SearchCriteria sc = new SearchCriteria();
        sc.setSAMAccountName(TEST_ID_KNOWN);
        List<SimpleUser> testList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            testList.add(new SimpleUser(TEST_ID_KNOWN));
        }
        final String sql = "select l from " + SimpleUser.class.getName()
                + " l where upper(l.userCredentials.userID) = :userId";
        Mockito.doReturn(testList).when(pQuery).getResultList();
        assertTrue(securityDao.findAppUsers(sc).get(TEST_ID).getUserCredentials()
                .getUserID().contains(TEST_ID_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(sql);
    }
    
    @Test
    public void findAppUsersGivenFirstAndLastNameSearchCriteriaShouldQueryWithFirstAndLastName() throws DAOException {
        SearchCriteria sc = new SearchCriteria();
        sc.setLastName(TEST_LN_KNOWN);
        sc.setFirstName(TEST_FN_KNOWN);
        List<SimpleUser> testList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            testList.add(new SimpleUser(TEST_ID_KNOWN));
        }
        final String sql = "select l from " + SimpleUser.class.getName()
                + " l where l.lastName = :lastName and l.firstName like :firstName";        
        Mockito.doReturn(testList).when(nQuery).getResultList();
        assertTrue(securityDao.findAppUsers(sc).get(TEST_ID).getUserCredentials()
                .getUserID().contains(TEST_ID_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(sql);
    }
    
    @Test
    public void findAppUsersGivenLastNameSearchCriteriaShouldQueryWithLastName() throws DAOException {
        SearchCriteria sc = new SearchCriteria();
        sc.setLastName(TEST_LN_KNOWN);
        List<SimpleUser> testList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            testList.add(new SimpleUser(TEST_ID_KNOWN));
        }
        final String sql = "select l from " + SimpleUser.class.getName()
                + " l where l.lastName = :lastName";
        Mockito.doReturn(testList).when(nQuery).getResultList();
        assertTrue(securityDao.findAppUsers(sc).get(TEST_ID).getUserCredentials()
                .getUserID().contains(TEST_ID_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(sql);
    }
    
    @Test
    public void findAppUsersGivenRolesSearchCriteriaShouldQueryWithRoles() throws DAOException {
        List<SimpleUser> testList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            testList.add(new SimpleUser(TEST_ID_KNOWN));
        }
        List<String> stringList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            stringList.add(TEST_ID_KNOWN);
        }
        SearchCriteria sc = new SearchCriteria();
        sc.setRoles(stringList);
        final String sql = "select l from " + SimpleUser.class.getName()
                + " l where l.id in (select distinct ur.user.id from "
                + UserRole.class.getName() + " ur where ur.role.name in (:roles))";
        
        Mockito.doReturn(testList).when(nQuery).getResultList();
        assertTrue(securityDao.findAppUsers(sc).get(TEST_ID).getUserCredentials()
                .getUserID().contains(TEST_ID_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(sql);
        Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.anyString(), Mockito.any());
    }
    
    @Test
    public void findAppUsersGivenRolesAndAdditionalFieldsSearchCriteriaShouldQueryWithRolesAndAdditionalFields() throws DAOException {
        List<SimpleUser> testList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            testList.add(new SimpleUser(TEST_ID_KNOWN));
        }
        List<String> stringList = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            stringList.add(TEST_ID_KNOWN);
        }
        SearchCriteria sc = new SearchCriteria();
        sc.setRoles(stringList); sc.setVisns(stringList);
        sc.setStations(stringList); sc.setVendors(stringList);
        final String sql = "select l from " + SimpleUser.class.getName()
                + " l where l.id in (select distinct ur.user.id from "
                + UserRole.class.getName() + " ur where ur.role.name in"
                + " (:roles)) and l.visn.id in (select v.id from " + SimpleVisn.class.getName()
                + " v where v.name in (:visns)) and l.facility.id"
                + " in (select f.id from " + SimpleFacility.class.getName()
                + " f where f.stationNumber in (:stations)) and l.vendor.id"
                + " in (select ven.id from " + Vendor.class.getName()
                + " ven where ven.name in (:vendors))";
        
        Mockito.doReturn(testList).when(nQuery).getResultList();
        assertTrue(securityDao.findAppUsers(sc).get(TEST_ID).getUserCredentials()
                .getUserID().contains(TEST_ID_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(sql);
        Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.anyString(), Mockito.any());
    }
    
    @Test (expected = RuntimeException.class)
    public void loadUserByUsernameGivenNullUsernameShould() {
         String un = "";
         UserDetails user = null;
         user = securityDao.loadUserByUsername(un);
    }
    
    @Test
    public void loadUserByUsernameGivenValidNameShouldReturnUser() {
        String un = TEST_USER_NAME;
        User testUser = new User(TEST_USER_NAME);
        SimpleRole testRole = new SimpleRole();
        Role testRoleTwo = new Role();
        testRole.setName(TEST_ID_KNOWN);
        testRoleTwo.setName(TEST_ID_KNOWN);
        testRoleTwo.setName(TEST_FN_KNOWN);
        testRoleTwo.setId((long)TEST_ID);
        Permission pTest = new Permission();
        pTest.setId((long)TEST_ID);
        pTest.setName(TEST_ID_KNOWN);
        Set<Permission> spTest = new HashSet<>();
        for (int i = 0; i < 3; i++) {
            spTest.add(pTest);
        }
        List<SimpleRole> sRoleList = new ArrayList<>();
        List<Role> roleList = new ArrayList<>();
        for (int i = 0; i < 2; i++) {
            sRoleList.add(testRole);
            roleList.add(testRoleTwo);
            testRoleTwo.setPermissions(spTest, TEST_ID_KNOWN);
        }
        Mockito.doReturn(nativeQuery).when(session).createNativeQuery("select * "
                + "from ht.app_user where user_name='" + TEST_USER_NAME +"'", User.class);
        Mockito.doReturn(testUser).when(nativeQuery).getSingleResult();
        SecurityDAO spyedDAO = Mockito.spy(securityDao);
        Mockito.when(spyedDAO.getRolesByUserName(un)).thenReturn(sRoleList);
        Mockito.doReturn(roleList).when(nQuery).getResultList();
        
        assertTrue(securityDao.loadUserByUsername(un).getUsername().contains(TEST_ID_KNOWN));
    }
    
    @Test
    public void loadUserByUsernameGivenNameShouldReturnUser() {
        String un = TEST_USER_NAME;
        User testUser = new User(TEST_USER_NAME);
        SimpleRole testRole = new SimpleRole();
        Role testRoleTwo = new Role();
        testRole.setName(CCHT_USER);
        testRoleTwo.setName(CCHT_USER);
        testRoleTwo.setId((long)TEST_ID);
        Permission pTest = new Permission();
        pTest.setId((long)TEST_ID);
        pTest.setName(CCHT_USER);
        Set<Permission> spTest = new HashSet<>();
        for (int i = 0; i < 3; i++) {
            spTest.add(pTest);
        }
        List<SimpleRole> sRoleList = new ArrayList<>();
        List<Role> roleList = new ArrayList<>();
        for (int i = 0; i < 2; i++) {
            sRoleList.add(testRole);
            roleList.add(testRoleTwo);
            testRoleTwo.setPermissions(spTest, CCHT_USER);
        }
        Mockito.doReturn(nativeQuery).when(session).createNativeQuery("select * "
                + "from ht.app_user where user_name='" + TEST_USER_NAME +"'", User.class);
        Mockito.doReturn(testUser).when(nativeQuery).getSingleResult();
        SecurityDAO spyedDAO = Mockito.spy(securityDao);
        Mockito.when(spyedDAO.getRolesByUserName(un)).thenReturn(sRoleList);
        Mockito.when(spyedDAO.getRoleByName(un)).thenReturn(null);
        //Mockito.doReturn(roleList).when(nQuery).getResultList();
        
        assertTrue(securityDao.loadUserByUsername(un).getUsername().contains(TEST_ID_KNOWN));
    }
    
    @Test
    public void findUsersWithAnyRoleGivenAnEmptyListShouldReturnEmptyMap() {
        List<String> roleNames = new ArrayList<>();
        assertTrue(securityDao.findUsersWithAnyRole(roleNames).isEmpty());
        roleNames = null;
        assertTrue(securityDao.findUsersWithAnyRole(roleNames).isEmpty());
    }
    
    @SuppressWarnings("null")
    @Test
    public void findUsersWithAnyRoleGivenNonEmptyListShouldReturnNonEmptyMap() {
        List<String> roleNames = new ArrayList<>();
        for(int i = 0; i < 3; i++) {
            roleNames.add(TEST_ID_KNOWN);
        }
        Object[] testOOne = new Object[TEST_OBJECT_ARRAY_SIZE];
        Object[] testOTwo = new Object[TEST_OBJECT_ARRAY_SIZE];
        UserPrincipal<?> testUP = null;
        testOOne[0] = "testString"; testOOne[1] = testUP;
        testOTwo[0] = "testString"; testOTwo[1] = testUP;
        List<Object[]> rows = new ArrayList<>();
        rows.add(testOOne);
        rows.add(testOTwo);
        Mockito.doReturn(pQuery).when(nQuery).setParameterList("roleNames", roleNames);
        Mockito.doReturn(rows).when(pQuery).getResultList();
        assertTrue(!securityDao.findUsersWithAnyRole(roleNames).isEmpty());
        Mockito.verify(session, Mockito.atLeastOnce()).createQuery(Mockito.anyString());
    }
    
    @Test
    public void deleteRolePermissionsFromRoleWhenNullShouldNotInteractWithSession() {
        Role role = new Role();
        role.setInternalPermissions(null);
        securityDao.deleteRolePermissionsFromRole(role);
        Mockito.verifyZeroInteractions(session);
    }
    
    @Test
    public void deleteRolePermissionsFromRoleWhenNotNullShouldRunDeleteFunction() {
        Set<RolePermission> testSet = new HashSet<>();
        for (int i = 0; i < 3; i++) {
            RolePermission test = new RolePermission();
            testSet.add(test);
        }
        Role role = new Role();
        role.setInternalPermissions(testSet);
        securityDao.deleteRolePermissionsFromRole(role);
        Mockito.verify(session, Mockito.atLeastOnce()).delete(Mockito.isNotNull());
    }
}
