package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.report.DeviceModality;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.persistent.DeviceModalityDAO;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.persistent.SatisfactionSurveyReportDAO;
import gov.va.med.ccht.persistent.VendorDAO;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;

public class SatisfactionSurveyReportDAOImplTest {
	private static final Integer TEST_INTEGER = 0;
	private static final String TEST_TEXT= "Test";
	private static final int TEST_NUM = 0;
	private static final long TEST_LONG = 1;	
	private static final double TEST_DOUB = 1.0;
	private static final BigDecimal TEST_BD = new BigDecimal(TEST_DOUB);
	
	private static final int NATIONAL_SELECTION = 1;
	private static final int VISN_SELECTION = 2;
	private static final int FACILITY_SELECTION = 3;
	
	private static final int VERSION1 = 1;
	private static final int VERSION2 = 2;
	
	static final private String ALL = "-1";
	static final private String DO_NOT_INCLUDE = "-2";
	static final private String TESTID = "1";
	
	
	static final private Date TEST_DATE = new Date();
	
	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	@Mock
	private NativeQuery<Session> modalityQuery;
	@Mock
	private NativeQuery<Session> vendorQuery;
	@Mock
	private NativeQuery<Session> vendorParamQuery;
	@Mock
	ReportsDAO reportsDao;
	
	private SatisfactionSurveyReportDAO surveyReport;
	private VendorDAO vd;
	private DeviceModalityDAO dm;
	
	private PSSReportForm form;
	private List<Object[]> emptyTestRows;
	private List<SatisfactionSurveyQuestionResult> result;
	private Calendar testCalDate;
	private String[] testQuesNumbs;
	private List<DeviceModality> deviceModalities;
	private String[] categoriesOfCare;
	
	@Before
	public void setup() {
		MockitoAnnotations.initMocks(this);
		
		Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
		Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
		vd = new VendorDAOImpl(sessionFactory);
		dm = new DeviceModalityDAOImpl(sessionFactory);
		surveyReport = new SatisfactionSurveyReportDAOImpl(sessionFactory, vd, dm);
		
		emptyTestRows = new ArrayList<>();
		testCalDate = Calendar.getInstance();
		categoriesOfCare = new String[1];
		categoriesOfCare[0] = ALL;
		testQuesNumbs = new String[4];
		for(int i = 0; i < testQuesNumbs.length; i ++){
			testQuesNumbs[i] = TEST_TEXT + i;
		}
		
		deviceModalities  = new ArrayList<>();
		DeviceModality testDM = new DeviceModality();
		testDM.setId(TEST_LONG);
		testDM.setName(TESTID);
		deviceModalities.add(testDM);
		DeviceModality testDM2 =  new DeviceModality();
		testDM2.setName(TESTID + TESTID);
		testDM2.setId(TEST_LONG + TEST_LONG);
		deviceModalities.add(testDM2);
		Mockito.doReturn(modalityQuery).when(session).createNativeQuery(Mockito.anyString(), Mockito.eq(DeviceModality.class));
		Mockito.doReturn(deviceModalities).when(modalityQuery).getResultList();
		
		Vendor testVendor = new Vendor();
		testVendor.setId(TEST_INTEGER);
		testVendor.setName(TESTID);
		Mockito.doReturn(vendorQuery).when(session).createNativeQuery(Mockito.anyString(), Mockito.eq(Vendor.class));
		Mockito.doReturn(vendorParamQuery).when(vendorQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(testVendor).when(vendorParamQuery).getSingleResult();
	}
	
	@Test
	public void getSurveyTrendChartsReportDataWithNationalSingleModalityAndSingleVendorShouldAppendVendorIdAndModalityListAndNotVisnOrFacility() throws DAOException, NumberFormatException, ServiceException{
		String visnSql = "AND d.visn_id = :visnId ";
		String facilitySql = "and d.facility_id = :facilityId ";
		form = new PSSReportForm();
		form.setModalityName(TESTID);
		form.setVendorId(TESTID);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.never()).createNativeQuery(Mockito.contains(visnSql));
		Mockito.verify(session, Mockito.never()).createNativeQuery(Mockito.contains(facilitySql));
		Mockito.verify(nQuery, Mockito.never()).setParameter("visnId", form.getVisnId());
		Mockito.verify(nQuery, Mockito.never()).setParameter("facilityId", form.getFacilityId());
		
		testCalDate.setTime(TEST_DATE);
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		List<Long> modIds = new ArrayList<>();
		modIds.add(deviceModalities.get(0).getId());
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("fromDate"), Mockito.eq(TEST_DATE));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("toDate"), Mockito.eq(testCalDate.getTime()));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("modalityIdList"), Mockito.eq(modIds));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalAllParamsShouldNotSetVisnFacilityOrVendorParam() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(ALL);
		form.setVendorId(ALL);
		form.setCategoriesOfCare(categoriesOfCare);
		form.setQuestionNumbers(testQuesNumbs);
		form.setVisnId(ALL);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(nQuery, Mockito.never()).setParameter("visnId", form.getVisnId());
		Mockito.verify(nQuery, Mockito.never()).setParameter("faciltyId", form.getFacilityId());
		Mockito.verify(nQuery, Mockito.never()).setParameter("vendor", form.getVendorId());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalAllModalitiesSingleVendorParamsShouldNotSetVisnOrFacilityParam() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(ALL);
		form.setVendorId(TESTID);
		form.setCategoriesOfCare(categoriesOfCare);
		form.setQuestionNumbers(testQuesNumbs);
		form.setVisnId(ALL);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		testCalDate.setTime(TEST_DATE);
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		Mockito.verify(nQuery, Mockito.never()).setParameter("visnId", form.getVisnId());
		Mockito.verify(nQuery, Mockito.never()).setParameter("faciltyId", form.getFacilityId());
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("fromDate"), Mockito.eq(TEST_DATE));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("toDate"), Mockito.eq(testCalDate.getTime()));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalPatSatVersion1ShouldAppendAvgDivideBy5() throws DAOException, NumberFormatException, ServiceException{
		String sql = " AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/5.0) AS AvgAnswer, ";
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setReportVersionButtonSelection(VERSION1);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalPatSatVersion2ShouldAppendAvgDivideBy4() throws DAOException, NumberFormatException, ServiceException{
		String sql = " AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/4.0) AS AvgAnswer, ";
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setReportVersionButtonSelection(VERSION2);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalCOCIncludeAllExceptL2ShouldAppendQuestionNumbers() throws DAOException, NumberFormatException, ServiceException{
		String sql = "AND f.level_of_care in ";
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		categoriesOfCare[0] = DO_NOT_INCLUDE;
		form.setCategoriesOfCare(categoriesOfCare);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setReportVersionButtonSelection(VERSION2);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllVisnsShouldNotSetVisnParam() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(ALL);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setVisnId(ALL);
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(nQuery, Mockito.never()).setParameter("visnId", form.getVisnId());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithSingleVISNAndAllModalitiesShouldSetVISNIdParam() throws DAOException, NumberFormatException, ServiceException{
		String sql = "AND d.visn_id = :visnId ";
		form = new PSSReportForm();
		form.setModalityName(ALL);
		form.setVendorId(TESTID);
		form.setQuestionNumbers(testQuesNumbs);
		form.setVisnId(TESTID);
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());		
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("visnId"), Mockito.eq(form.getVisnId()));	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithSingleVisnSingleVendorAndSingleModalityShouldSetEachParam() throws DAOException, NumberFormatException, ServiceException{
		String sql = "AND d.visn_id = :visnId ";
		form = new PSSReportForm();
		form.setModalityName(TESTID);
		form.setVendorId(TESTID);
		form.setQuestionNumbers(testQuesNumbs);
		form.setVisnId(TESTID);
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
		
		testCalDate.setTime(TEST_DATE);
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		List<Long> modIds = new ArrayList<>();
		modIds.add(deviceModalities.get(0).getId());
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("fromDate"), Mockito.eq(TEST_DATE));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("toDate"), Mockito.eq(testCalDate.getTime()));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("visnId"), Mockito.eq(form.getVisnId()));	
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("modalityIdList"), Mockito.eq(modIds));
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllFacilitiesShouldNotSetFacilityIdParam() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setFacilityId(ALL);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(nQuery, Mockito.never()).setParameter("facilityId", form.getFacilityId());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithSingleFacilityAndAllModalitiesShouldSetFacilityIdParam() throws DAOException, NumberFormatException, ServiceException{
		String sql = "and d.facility_id = :facilityId ";
		form = new PSSReportForm();
		form.setModalityName(ALL);
		form.setVendorId(TESTID);
		form.setQuestionNumbers(testQuesNumbs);
		form.setFacilityId(TESTID);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());		
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("facilityId"), Mockito.eq(form.getFacilityId()));	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithSingleFacilitySingleVendorAndSingleModalityShouldSetEachParam() throws DAOException, NumberFormatException, ServiceException{
		String sql = "and d.facility_id = :facilityId ";
		form = new PSSReportForm();
		form.setModalityName(TESTID);
		form.setVendorId(TESTID);
		form.setQuestionNumbers(testQuesNumbs);
		form.setFacilityId(TESTID);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
		Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(Mockito.contains(sql));
		
		testCalDate.setTime(TEST_DATE);
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		List<Long> modIds = new ArrayList<>();
		modIds.add(deviceModalities.get(0).getId());
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("fromDate"), Mockito.eq(TEST_DATE));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("toDate"), Mockito.eq(testCalDate.getTime()));
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("facilityId"), Mockito.eq(form.getFacilityId()));	
		Mockito.verify(nQuery, Mockito.atLeastOnce()).setParameter(Mockito.eq("modalityIdList"), Mockito.eq(modIds));
	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNoButtonTypeSelectionDefaultCaseShouldReturnEmptyList() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		assertTrue(surveyReport.getSatisfactionSurveyReport(form).isEmpty());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalSelection6MonthsOfResultRowsShouldReturn6Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		
		Mockito.doReturn(createTestData()).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 6);	
		
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("fromDate", form.getShortFormatReportFromDate());
		
		testCalDate.setTime(form.getShortFormatReportToDate());
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("toDate", testCalDate.getTime());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllVisnsSelection6VisnsOfResultRowsShouldReturn6Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setVisnId(ALL);
		
		Mockito.doReturn(createTestData()).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 6);	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllFacilitiesSelection6MonthsOfResultRowsShouldReturn6Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setFacilityId(ALL);
		
		Mockito.doReturn(createTestData()).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 6);	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithNationalSelection0MonthsOfResultRowsShouldReturn0Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 0);	
		
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("fromDate", form.getShortFormatReportFromDate());
		
		testCalDate.setTime(form.getShortFormatReportToDate());
		testCalDate.setTime(TEST_DATE);
		testCalDate.set(Calendar.MILLISECOND, 999);
		testCalDate.set(Calendar.SECOND, 59);
		testCalDate.set(Calendar.MINUTE, 59);
		testCalDate.set(Calendar.HOUR_OF_DAY, 23);
		testCalDate.add(Calendar.MONTH, 1);
		testCalDate.add(Calendar.DATE, -1);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("toDate", testCalDate.getTime());
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllVisnsSelection0VisnsOfResultRowsShouldReturn0Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(VISN_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setVisnId(ALL);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 0);	
	}
	
	@Test
	public void getSatisfactionSurveyReportWithAllFacilitiesSelection0MonthsOfResultRowsShouldReturn0Results() throws DAOException, NumberFormatException, ServiceException{
		form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setVendorId(ALL);
		form.setQuestionNumbers(testQuesNumbs);
		form.setTypeButtonSelection(FACILITY_SELECTION);
		form.setShortFormatReportFromDate(TEST_DATE);
		form.setShortFormatReportToDate(TEST_DATE);
		form.setFacilityId(ALL);
		
		Mockito.doReturn(emptyTestRows).when(nQuery).getResultList();
		
		result = surveyReport.getSatisfactionSurveyReport(form);
		assertTrue(result.size() == 0);	
	}
	
	@Test
	public void getDatesByMonthShouldReturnAListOfMonthYearPairings(){
		List<Date> dateList = surveyReport.getDatesByMonth();
		assertNotNull(dateList);
		
		testCalDate.set(Calendar.getInstance().getTime().getYear() + 1900, Calendar.getInstance().getTime().getMonth(), 1);
		assertTrue(dateList.get(0).getMonth() == testCalDate.getTime().getMonth());
		assertTrue(dateList.get(0).getYear() == testCalDate.getTime().getYear());
		
		testCalDate = Calendar.getInstance();
		testCalDate.set(2006, Calendar.JANUARY, 1);
		assertTrue(dateList.get(dateList.size() - 1).getMonth() == testCalDate.getTime().getMonth());
		assertTrue(dateList.get(dateList.size() - 1).getYear() == testCalDate.getTime().getYear());
	}
	
	private List<Object[]> createTestData(){
		List<Object[]> testRows= new ArrayList<>();
		for(int i = 0; i < 6; i++){
			Object[] row = new Object[7];
			row[0] = TEST_INTEGER + i;
			row[1] = TEST_TEXT  + i;
			row[2] = TEST_NUM  + i;
			row[3] = TEST_NUM  + i;
			row[4] = TEST_INTEGER  + i;
			row[5] = TEST_BD.add(new BigDecimal(1.0 + i));
			row[6] = TEST_TEXT  + i;
			
			testRows.add(row);
		}
		//Row of Duplicate Data
		Object[] row = new Object[7];
		row[0] = TEST_INTEGER;
		row[1] = TEST_TEXT;
		row[2] = TEST_NUM;
		row[3] = TEST_NUM;
		row[4] = TEST_INTEGER;
		row[5] = TEST_BD;
		row[6] = TEST_TEXT;
		testRows.add(row);
		return testRows;
	}
}
