package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.common.Visn;
import gov.va.med.ccht.model.responserate.ResponseRateResults;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.fw.persistent.DAOException;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class ResponseRateReportDAOImplTest {
    private static final String testSqlbyDate = "EXEC GetResponseReportData :mode, :reportEndDate";
    private static final String testSqlBySN = "select * from Facilities where facility_id = :stationNumber";
    private static final String testSqlByAscId = "select * from Facilities order by Facility_Id asc";
    private static final String ALL_VISN = "-1";
    private static final String ALL_FACS = "-1";
    private static final String VIS_ID = "visnId";
    private static final String FAC_ID = "facility_id";
    private static final int ZERO = 0;
    private static final String VISN_PATH_VAR = ", :visnId";
    private static final String FAC_PATH_VAR = ", :visnId, :facility_id";
    private static final String TEST_ID_STRING = "1";
    private static final int TEST_BUTTON_SELECTION_NAT = 1;
    private static final int TEST_BUTTON_SELECTION_VISN = 2;
    private static final int TEST_BUTTON_SELECTION_FAC = 3;
    private static final int TEST_ID = 1;
    private static final String TEST_FN_KNOWN = "testerFirstName";
    private static final double TEST_DUB = 1.0;
    final BigDecimal TEST_BIG_DEC = new BigDecimal(TEST_DUB);
    private ResponseRateReportDAOImpl responseRateReportDao;
    private VendorDAOImpl vendorDao;
    private FacilityDAOImpl facDao;
    private VisnDAOImpl visnDao;
    private Visn testVisn;
    private CensusActivityReportForm form;
    private List<Object[]> testQueryResList;
    private List<Vendor> testVenList;
    private Set<Facility> testFacilities;

    @Mock
    private SessionFactory sessionFactory;
    @Mock
    private SessionFactory sessionFactoryForVisnQuery;
    @Mock
    private Session session;
    @Mock
    private Session sessionForVisnQuery;
    @Mock
    private Session venSession;
    @Mock
    private NativeQuery<Session> nQuery;
    @Mock
    private NativeQuery<Session> pQuery;
    @Mock
    private NativeQuery<Session> qQuery;
    @Mock
    private NativeQuery<Session> yetAnotherQuery;
    @Mock
    private NativeQuery<Session> facAnotherQuery;
    @Mock
    private NativeQuery<Session> getFacQuery;
    @Mock
    private NativeQuery<Session> ascFacQuery;
    @Mock
    private NativeQuery<Session> getVisnByIdQuery;
    @Mock
    private NativeQuery<Session> getVisnQuery;
    @Mock
    private NativeQuery<Session> nativeQuery;
    @Mock
    private NativeQuery<Session> venQuery;
    @Before
    public void setup() {
        MockitoAnnotations.initMocks(this);
        vendorDao = new VendorDAOImpl(sessionFactory);
        visnDao = new VisnDAOImpl(sessionFactory);
        facDao = new FacilityDAOImpl(sessionFactory);
        responseRateReportDao = new ResponseRateReportDAOImpl(sessionFactory, vendorDao, visnDao, facDao);
        Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
        Mockito.doReturn(sessionForVisnQuery).when(sessionFactoryForVisnQuery).getCurrentSession();
        Mockito.doReturn(nQuery).when(session).createQuery(Mockito.anyString());
        Mockito.doReturn(nativeQuery).when(session).createNativeQuery(Mockito.anyString());
        Mockito.doReturn(pQuery).when(nativeQuery).setParameter(Mockito.anyString(), Mockito.any());
        Mockito.doReturn(pQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
        Mockito.doReturn(qQuery).when(pQuery).setParameter(Mockito.anyString(), Mockito.any());
        form = new CensusActivityReportForm();
        form.setVisnId(TEST_ID_STRING);
        form.setFacilityId(TEST_ID_STRING);
        testQueryResList = setUpQueryResults(ZERO);
        testFacilities = setUpFacilitySet();
        testVenList = setUpVendorList();
        testVisn = buildVisn();
        
    }
    
    @After
    public void cleanUp() {
        testVenList = null;
        testQueryResList = null;
        form = null;
    }
    
    private Visn buildVisn() {
        Visn testVisnBuilder = new Visn();
        testVisnBuilder.setId(TEST_ID);
        testVisnBuilder.setName(TEST_FN_KNOWN);
        testVisnBuilder.setFacilities(setUpFacilitySet());
        
        return testVisnBuilder;
    }
    
    private List<Vendor> setUpVendorList() {
        List<Vendor> testVenList = new ArrayList<>();
        Vendor testVen = new Vendor();
        testVen.setName(TEST_FN_KNOWN);
        
        for (int i = 0; i < TEST_BUTTON_SELECTION_FAC; i++) {
            testVenList.add(testVen);
        }
        return testVenList;
    }
    
    private Set<Facility> setUpFacilitySet() {
       
        Set<Facility> testFacSet = new HashSet<Facility>();
        Facility testFac = new Facility();
        testFac.setName(TEST_FN_KNOWN);
        testFac.setId(TEST_ID);
        testFac.setStationNumber(FAC_ID);
        testFac.setVisn(testVisn);
        
        for (int i = 0; i < TEST_BUTTON_SELECTION_FAC; i++) {
            testFacSet.add(testFac);
        }
        return testFacSet;
    }
    
    private List<Object[]> setUpQueryResults(int testNum) {
        List<Object[]> testRec = new ArrayList<>();
        Object[] testArr;
        if(testNum == 0) {
            testArr = new Object[6];
            testArr[0] = new Integer(TEST_BUTTON_SELECTION_NAT);
            testArr[1] = new Integer(TEST_BUTTON_SELECTION_NAT);
            testArr[2] = new String(TEST_ID_STRING);
            testArr[3] = new Integer(TEST_BUTTON_SELECTION_NAT);
            testArr[4] = new Integer(TEST_BUTTON_SELECTION_NAT);
            testArr[5] = TEST_BIG_DEC;
        
            for (int i = 0; i < TEST_BUTTON_SELECTION_FAC; i++) {
                testRec.add(testArr);
            }
            return testRec;
        } else {
            return Collections.<Object[]>emptyList();
        }        
    }
    
    private void PathMockHelper() {
        Mockito.doReturn(testQueryResList).when(qQuery).getResultList();
        Mockito.doReturn(yetAnotherQuery).when(session)
            .createNativeQuery(Mockito.anyString(), Mockito.eq(Visn.class));
        Mockito.doReturn(getVisnByIdQuery).when(yetAnotherQuery)
            .setParameter(Mockito.eq("id"), Mockito.anyLong());
        Mockito.doReturn(testVisn).when(getVisnByIdQuery).getSingleResult();
        Mockito.doReturn(getVisnQuery).when(yetAnotherQuery)
            .setParameter(Mockito.eq("name"), Mockito.anyString());
        Mockito.doReturn(testVisn).when(getVisnQuery).getSingleResult();
        Mockito.doReturn(facAnotherQuery).when(session)
            .createNativeQuery(testSqlBySN, Facility.class);
        Mockito.doReturn(ascFacQuery).when(session)
        .createNativeQuery(testSqlByAscId, Facility.class);
        Mockito.doReturn(getFacQuery).when(facAnotherQuery)
            .setParameter(Mockito.eq("stationNumber"), Mockito.anyString());
        Mockito.doReturn(venQuery).when(session)
            .createNativeQuery(Mockito.anyString(), Mockito.eq(Vendor.class));
        Mockito.doReturn(testVenList).when(venQuery).getResultList();
    }
    
    @Test
    public void getTopLevelReportNationalAdminPathGivenValidFormShouldReturnNonNullResponse() throws DAOException {
        form.setButtonSelection(TEST_BUTTON_SELECTION_NAT);
        Mockito.doReturn(yetAnotherQuery).when(session)
            .createNativeQuery(Mockito.anyString(), Mockito.eq(Vendor.class));
        assertNotNull(responseRateReportDao.getTopLevelReport(form));
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(testSqlbyDate);
    }
    
    @Test
    public void getTopLevelReportVisnPathNotAllVisnGivenValidFormShouldReturnNonNullResponse() throws DAOException {
        form.setButtonSelection(TEST_BUTTON_SELECTION_VISN);
        PathMockHelper();
        ResponseRateResults test = responseRateReportDao.getTopLevelReport(form);
        assertNotNull(test);
        assertTrue(!test.getResponseRateRecords().isEmpty());
        assertTrue(test.getResponseRateRecords().get(0).getVendorName().contains(TEST_FN_KNOWN));
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(testSqlbyDate + VISN_PATH_VAR);
    }
    
    @Test
    public void getTopLevelReportVisnPathforAllVisnGivenValidFormShouldReturnNonNullResponse() throws DAOException {
        form.setVisnId(ALL_VISN);
        form.setButtonSelection(TEST_BUTTON_SELECTION_VISN);
        PathMockHelper();
        ResponseRateResults test = responseRateReportDao.getTopLevelReport(form);
        assertNotNull(test);
        assertTrue(!test.getResponseRateRecords().isEmpty());
        assertTrue(test.getResponseRateRecords().get(0).getVendorName().contains(TEST_FN_KNOWN));
        Mockito.verify(qQuery, Mockito.atLeastOnce())
            .setParameter(VIS_ID, ZERO);
    }
    
    @Test
    public void getTopLevelReportVisnPathNotAllVisnGivenEmptyQueryResultsShouldReturnNullResponseRateRecords() throws DAOException {
        form.setButtonSelection(TEST_BUTTON_SELECTION_VISN);
        testQueryResList = setUpQueryResults(TEST_ID);
        PathMockHelper();
        ResponseRateResults test = responseRateReportDao.getTopLevelReport(form);
        assertNull(test.getResponseRateRecords());
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(testSqlbyDate + VISN_PATH_VAR);
    }
    
    @Test
    public void getTopLevelReportFacilityPathNotAllFacilityGivenValidFormShouldReturnNonNullResponse() throws DAOException {
         form.setButtonSelection(TEST_BUTTON_SELECTION_FAC); 
         Object[] testFacilities = setUpFacilitySet().toArray();
         PathMockHelper();
         Mockito.doReturn(testFacilities[ZERO]).when(getFacQuery).getSingleResult();
         ResponseRateResults test = responseRateReportDao.getTopLevelReport(form);
         assertNotNull(test);
         assertTrue(!test.getResponseRateRecords().isEmpty());
         assertTrue(test.getResponseRateRecords().get(0).getVendorName().contains(TEST_ID_STRING));
         Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(testSqlbyDate + FAC_PATH_VAR);
         Mockito.verify(session, Mockito.atLeastOnce())
             .createNativeQuery(testSqlBySN, Facility.class);
    }
    
    @Test
    public void getTopLevelReportFacilityPathForAllFacilitiesGivenValidFormShouldReturnNonNullResponse() throws DAOException {
        form.setButtonSelection(TEST_BUTTON_SELECTION_FAC);
        form.setFacilityId(ALL_FACS);
        List<Facility> testList = new ArrayList<Facility>();
        for (Facility facility : testFacilities) {
            testList.add(facility);
        }
        Object[] testFacilitiesArr = setUpFacilitySet().toArray();
        for (Object[] test : testQueryResList) {
            test[1] = (new String("1"));
        }
        PathMockHelper();
        Mockito.doReturn(testFacilitiesArr[ZERO]).when(getFacQuery).getSingleResult();
        Mockito.doReturn(testList).when(ascFacQuery).getResultList();
        ResponseRateResults test = responseRateReportDao.getTopLevelReport(form);
        assertNotNull(test);
        assertTrue(!test.getResponseRateRecords().isEmpty());
        assertTrue(test.getResponseRateRecords().get(0).getVendorName().contains(TEST_ID_STRING));
         
        Mockito.verify(session, Mockito.atLeastOnce()).createNativeQuery(testSqlbyDate + FAC_PATH_VAR);
        Mockito.verify(qQuery, Mockito.atLeastOnce())
            .setParameter(FAC_ID, null);
    }   
}

