package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSearchResult;
import gov.va.med.ccht.persistent.QIRDAO;
import gov.va.med.fw.ui.model.TermType;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class QIRDAOImplTest {
	
	private final static String UNIT_TEST_NAME = "unit_test";
	private final static int dateElement = 10;
	private final static int attachmentCountElement = 14;
	
	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	
	private QIRSearchParameters searchParams;
	private TermType testTerm;
	
	private QIRDAO dao;
	
	@Before
	public void setup() throws Exception{
		MockitoAnnotations.initMocks(this);
		Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
		Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
		dao = new QIRDAOImpl(sessionFactory);
}

	@Test 
	public void searchQIRWithMostQIRSearchParametersTrueShouldAddAndStatementsToWhereClause() {
		String sql ="SELECT "
				+" top 3 "
				+ "q.QIR_ID id, qt.NAME qirType, qst.NAME qirStatusType, "
				+ "cast(v.visn_name as varchar) visn, f.facility_id facilityCode, f.facility_name facility, "
				+ "mdmt.NAME deviceType, ven.VENDOR_NAME vendor, "
				+ "q.QIR_COMPLAINT complaint, q.QIR_SUMMARY headline, "
				+ "q.DATE_SUBMITTED submittedDate, q.SUBMITTED_BY submittedBy, q.RECORD_MODIFIED_COUNT, "
				+ "isnull(u.last_name,'') + ', ' + isnull(u.first_name,'')+' ' + isnull(u.middle_name,'') submittedByName, "
				+ "count(qda.QIR_DOC_ATTCHMNT_ID) attchmnt_cnt " + "FROM qir.QIR q "
				+ "join qir.QIR_STATUS_TYPES qst on q.QIR_STATUS_TYPE_ID = qst.ID "
				+ "join ht.app_user u on q.SUBMITTED_BY = u.user_name "
				+ "left join qir.QIR_TYPES qt on q.QIR_TYPE_ID = qt.id "
				+ "left join dbo.MEDICAL_DEVICE_MODEL_TYPE mdmt on q.MED_DVC_MODEL_TYPE_ID = mdmt.id "
				+ "left join dbo.VENDOR ven on q.VENDOR_ID = ven.Vendor_ID "
				+ "left join dbo.facilities f on q.FACILITIES_ID = f.ID "
				+ "left join dbo.visn v on q.visn_id = v.visn_id "
				+ "left outer join qir.QIR_DOC_ATTACHMENTS qda on q.QIR_ID = qda.QIR_ID " + "where 1=1 ";
		sql += "and qst.code in (:qirStatuses) ";
		sql += "and mdmt.code = :deviceType ";
		sql += "and ven.vendor_id = :vendor_id ";
		sql += "and qt.code = :qirType ";
		sql += "and f.facility_id = :station ";
		sql += "and v.visn_name = :visnName ";
		sql += "and q.submitted_by = :submittedBy ";
		sql += "and DATE_SUBMITTED >= :submittedFromDate ";
		sql += "and DATE_SUBMITTED < :submittedToDate ";
		sql += "and QIR_STATUS_TYPE_ID != :NewStatus ";
		sql += "and QIR_STATUS_TYPE_ID != :WithdrawnStatus ";
		sql += "group by q.QIR_ID, qt.NAME, qst.NAME, v.visn_name, f.facility_id, f.facility_name, mdmt.NAME, ven.VENDOR_NAME, q.QIR_COMPLAINT, q.QIR_SUMMARY, q.DATE_SUBMITTED, "
				+ "q.SUBMITTED_BY, q.RECORD_MODIFIED_COUNT, u.last_name, u.first_name, u.middle_name order by q.QIR_ID DESC";
		
		searchParams = new QIRSearchParameters();
		testTerm = new TermType(UNIT_TEST_NAME, UNIT_TEST_NAME);
		List<String> qirStatus = new ArrayList<>();
		qirStatus.add(UNIT_TEST_NAME);
		Date date = new Date();
		searchParams.setSubmittedBy(UNIT_TEST_NAME);
		searchParams.setVisn(testTerm);
		searchParams.setIsVendor(true);
		searchParams.setMaxRecords(3);
		searchParams.setId("");
		searchParams.setQirStatuses(qirStatus);
		searchParams.setDeviceType(testTerm);
		searchParams.setVendor(testTerm);
		searchParams.setQirType(testTerm);
		searchParams.setFacility(testTerm);
		searchParams.setSubmittedFromDate(date);
		searchParams.setSubmittedToDate(date);
		
		assertNotNull(dao.searchQIR(searchParams));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnName", testTerm.getValue());
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("qirStatuses", qirStatus);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("submittedFromDate", date);
	}
	
	@Test 
	public void searchQIRWithQIRSearchParameterIDSetShouldAddSingleAndStatementsToWhereClause() {
		String sql ="SELECT "
				+" top 3 "
				+ "q.QIR_ID id, qt.NAME qirType, qst.NAME qirStatusType, "
				+ "cast(v.visn_name as varchar) visn, f.facility_id facilityCode, f.facility_name facility, "
				+ "mdmt.NAME deviceType, ven.VENDOR_NAME vendor, "
				+ "q.QIR_COMPLAINT complaint, q.QIR_SUMMARY headline, "
				+ "q.DATE_SUBMITTED submittedDate, q.SUBMITTED_BY submittedBy, q.RECORD_MODIFIED_COUNT, "
				+ "isnull(u.last_name,'') + ', ' + isnull(u.first_name,'')+' ' + isnull(u.middle_name,'') submittedByName, "
				+ "count(qda.QIR_DOC_ATTCHMNT_ID) attchmnt_cnt " + "FROM qir.QIR q "
				+ "join qir.QIR_STATUS_TYPES qst on q.QIR_STATUS_TYPE_ID = qst.ID "
				+ "join ht.app_user u on q.SUBMITTED_BY = u.user_name "
				+ "left join qir.QIR_TYPES qt on q.QIR_TYPE_ID = qt.id "
				+ "left join dbo.MEDICAL_DEVICE_MODEL_TYPE mdmt on q.MED_DVC_MODEL_TYPE_ID = mdmt.id "
				+ "left join dbo.VENDOR ven on q.VENDOR_ID = ven.Vendor_ID "
				+ "left join dbo.facilities f on q.FACILITIES_ID = f.ID "
				+ "left join dbo.visn v on q.visn_id = v.visn_id "
				+ "left outer join qir.QIR_DOC_ATTACHMENTS qda on q.QIR_ID = qda.QIR_ID " + "where 1=1 ";
		sql += "and q.qir_id = :id ";
		sql += "group by q.QIR_ID, qt.NAME, qst.NAME, v.visn_name, f.facility_id, f.facility_name, mdmt.NAME, ven.VENDOR_NAME, q.QIR_COMPLAINT, q.QIR_SUMMARY, q.DATE_SUBMITTED, "
				+ "q.SUBMITTED_BY, q.RECORD_MODIFIED_COUNT, u.last_name, u.first_name, u.middle_name order by q.QIR_ID DESC";
		
		searchParams = new QIRSearchParameters();
		testTerm = new TermType(UNIT_TEST_NAME, UNIT_TEST_NAME);
		searchParams.setSubmittedBy(UNIT_TEST_NAME);
		searchParams.setVisn(testTerm);
		searchParams.setIsVendor(true);
		searchParams.setMaxRecords(3);
		searchParams.setId("1");
		
		assertNotNull(dao.searchQIR(searchParams));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("id", "1");
	}
	
	@Test 
	public void searchQIRWith2ResultRowsReturnedShouldCreate2RecordsSuccessfully() {
		searchParams = new QIRSearchParameters();
		searchParams.setId("1");
		List<Object[]> 	qirs = new ArrayList<>();
		Object[] obj = new Object[15];
		for(int i = 0; i < 2; i++){
			for(int j = 0; j < obj.length; j++){
				obj[j] = "1" + i;
				if(j == dateElement){
					obj[j] = Calendar.getInstance().getTime();
				}else if(j == attachmentCountElement){
					obj[j] = 1; 
				}
			}
			qirs.add(obj); 
		}
		Mockito.doReturn(qirs).when(nQuery).getResultList();
		List<QIRSearchResult> results = dao.searchQIR(searchParams);
		assertNotNull(results);
		assert(results.size() == 2);	
	}
	
	@Test 
	public void searchQIRWithNoResultRowsReturnedShouldCreateNoRecords() {
		searchParams = new QIRSearchParameters();
		searchParams.setId("1");
		List<Object[]> qirs = new ArrayList<>();
		Mockito.doReturn(qirs).when(nQuery).getResultList();
		List<QIRSearchResult> results = dao.searchQIR(searchParams);
		assertNotNull(results);
		assert(results.size() == 0);
	}
	
	@Test 
	public void searchQIRWithInvalidIdResultStringShouldCreateRuntimeException() {
		searchParams = new QIRSearchParameters();
		searchParams.setId("1");
		List<Object[]> qirs = new ArrayList<>();
		Object[] obj = new Object[15];
		for(int j = 0; j < obj.length; j++){
			obj[j] = "1 " + j;
			if(j == dateElement){
				obj[j] = Calendar.getInstance().getTime();
			}else if(j == attachmentCountElement){
				obj[j] = 1; 
			}
		}
		qirs.add(obj);
		Mockito.doReturn(qirs).when(nQuery).getResultList();
		List<QIRSearchResult> results = null;
		try{
			results = dao.searchQIR(searchParams);
		}
		catch(RuntimeException e){
			assert(results.size() == 0);
		}
	}
}