package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.*;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.med.ccht.model.pssreport.DistributionOfSurveysReport;
import gov.va.med.ccht.ui.model.PSSReportForm;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class PssDistributionReportDAOImplTest {
	
	private static final int ALL_SURVEYS = 1;
	private static final int PSS_V1 = 2;
	private static final int PSS_V2 = 3;
	
	public static final int NATIONAL_SELECTION = 1;
	public static final int VISN_SELECTION = 2;
	public static final int FACILITY_SELECTION = 3;
	
	private static final String ALL_VISNS = "-1";
	private static final String ALL_FACILITIES = "-1";
	
	private static final String DO_NOT_INCLUDE = "-2";
	private static final String ALL_MODALITIES = "-1";
	private static final String BROWSER = "11503";
	private static final String IVR = "11502";
	private static final String HOME_DEVICE = "11501";
	
	private static final int TEST_VALUE = 100;
	private static final int TEST_VALUE2 = 200;
	private static final int TEST_VALUE3 = 50;
	
	private static final String TEST_VENDOR_NAME = "TestVendor";
	private static final String TEST_VENDOR_NAME2 = "TestVendor2";
	private static final String TEST_VENDOR_NAME3 = "TestVendor3";
	
	// real modalities to be able to test business logic
	private static final String BROWSER_DATA = "Browser";
	private static final String IVR_DATA = "IVR";
	private static final String HOME_DEVICE_DATA = "Home Device";
	
	private static final SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
	
	private static final Date TEST_DATE = Calendar.getInstance().getTime();
	
	private PssDistributionReportDAOImpl DistReportDao;

	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	
	private DistributionOfSurveysReport records;

	@Before
	public void setUp() throws Exception {
	   MockitoAnnotations.initMocks(this);
	   
	   DistReportDao = new PssDistributionReportDAOImpl(sessionFactory);
	   
	   Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
	   Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
	   
	}
	
	@Test
	public void getNationalReportAllModalitiesSurvV2NoDataShouldReturnEmptyDistributionOfSurveysReport() {
		
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(ALL_MODALITIES);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(PSS_V2); // default selection
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getNationalReport(form);
		assertTrue(report.getReport().isEmpty());
		
	}
	
	@Test
	public void getNationalReportNoModalitiesSurvV1ShouldReturnDistributionOfSurveysReport() {
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(DO_NOT_INCLUDE);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(PSS_V1);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		// element zero is resurved for visn/facility names
		Object[] testRow = {"", TEST_VENDOR_NAME, TEST_VALUE};
		Object[] testRow2 = {"", TEST_VENDOR_NAME2, TEST_VALUE2};
		test.add(testRow);
		test.add(testRow2);
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getNationalReport(form);
		// should be 3, 2 data rows for and totals row.
		assertTrue(report.getReport().size() == test.size() + 1);
		// get totals row, get totals value
		assertTrue(Integer.parseInt(report.getReport().get(2)[2]) == 300);
	}
	
	@Test
	public void getNationalReportAllModalitiesAllSurvsShouldReturnDistributionOfSurveysReport() {
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(ALL_MODALITIES);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(ALL_SURVEYS);
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		// element zero is resurved for visn/facility names
		Object[] testRow = {"", TEST_VENDOR_NAME, TEST_VALUE, IVR_DATA};
		Object[] testRow2 = {"", TEST_VENDOR_NAME2, TEST_VALUE2, HOME_DEVICE_DATA};
		Object[] testRow3 = {"", TEST_VENDOR_NAME3, TEST_VALUE3, BROWSER_DATA};
		test.add(testRow);
		test.add(testRow2);
		test.add(testRow3);
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getNationalReport(form);
		// should be 4, 3 data rows for and totals row.
		assertTrue(report.getReport().size() == test.size() + 1);
		// get totals row, get totals value
		assertTrue(Integer.parseInt(report.getReport().get(test.size())[2]) == 350);
	}
	
	@Test
	public void getVisnReportAllModalitiesSurvV2ShouldReturnEmptyDistributionOfSurveysReport() {
		
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(ALL_MODALITIES);
		form.setVisnId(ALL_VISNS);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(PSS_V2); // default selection
		form.setTypeButtonSelection(VISN_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		// element zero is resurved for visn/facility names
		Object[] testRow = {"", TEST_VENDOR_NAME, TEST_VALUE, IVR_DATA};
		Object[] testRow2 = {"", TEST_VENDOR_NAME2, TEST_VALUE2, HOME_DEVICE_DATA};
		Object[] testRow3 = {"", TEST_VENDOR_NAME3, TEST_VALUE3, BROWSER_DATA};
		test.add(testRow);
		test.add(testRow2);
		test.add(testRow3);
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getVisnReport(form);
		assertTrue(report.getReport().size() == test.size() + 1);
		
	}
	
	@Test
	public void getFacilityReportAllModalitiesSurvV2ShouldReturnEmptyDistributionOfSurveysReport() {
		
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(ALL_MODALITIES);
		form.setFacilityId(ALL_FACILITIES);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(PSS_V2); // default selection
		form.setTypeButtonSelection(FACILITY_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		// element zero is resurved for visn/facility names
		Object[] testRow = {"", TEST_VENDOR_NAME, TEST_VALUE, IVR_DATA};
		Object[] testRow2 = {"", TEST_VENDOR_NAME2, TEST_VALUE2, HOME_DEVICE_DATA};
		Object[] testRow3 = {"", TEST_VENDOR_NAME3, TEST_VALUE3, BROWSER_DATA};
		test.add(testRow);
		test.add(testRow2);
		test.add(testRow3);
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getFacilityReport(form);
		assertTrue(report.getReport().size() == test.size() + 1);
		
	}
	
	@Test
	public void getNationalReportAllModalitiesSurvV2ShouldReturnAppend() {
		
		records = new DistributionOfSurveysReport();
		PSSReportForm form = new PSSReportForm();
		form.setModalityName(ALL_MODALITIES);
		form.setReportToDate(TEST_DATE);
		form.setReportFromDate(TEST_DATE);
		form.setReportVersionButtonSelection(PSS_V2); // default selection
		form.setTypeButtonSelection(NATIONAL_SELECTION);
		
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(nQuery).getResultList();
		DistributionOfSurveysReport report = DistReportDao.getNationalReport(form);
		
	}
	
//	@Test
//	public void getVisnReportAllModalitiesSurvV1ShouldReturnVisnSqlSurvyV1AppendAllModalities() {
//		
//		String expectedSql = "SELECT sa.VISN_ID AS [Breakout], " +
//		"v.vendor_name, COUNT(*) AS [NumSurveys] " +
//		",CASE Device.device_name WHEN 'IVR' THEN Device.device_name WHEN 'BROWSER' THEN Device.device_name ELSE 'Home Device' END " +
//		"FROM Surveyed_Activity AS sa " +
//		"INNER JOIN Surveyed_Activity_Response AS sar " +
//		"ON sa.Surveyed_Activity_ID = sar.Surveyed_Activity_ID " +
//		"INNER JOIN Vendor AS v ON sa.Vendor_ID = v.Vendor_ID " +
//		"INNER JOIN Device ON sa.Device_ID = Device.Device_ID " +
//		// formatting date must be done to match expected sql
//		"WHERE sar.completed_date >= '" +  df.format(TEST_DATE) + " 12:00:00 AM' " +
//		"AND sar.completed_date <= '" +  df.format(TEST_DATE) + " 11:59:59 PM' " +
//		
//		"AND sar.survey_accepted_status = 1 " +
//		"AND sar.survey_id = 2 " +
//		"AND device_name IS NOT NULL " +
//		"GROUP BY sa.VISN_ID, v.vendor_name " +
//		",CASE Device.device_name WHEN 'IVR' THEN Device.device_name WHEN 'BROWSER' THEN Device.device_name ELSE 'Home Device' END " +
//		"ORDER BY sa.VISN_ID, v.vendor_name ASC";
//		
//		records = new DistributionOfSurveysReport();
//		PSSReportForm form = new PSSReportForm();
//		form.setModalityName(ALL_MODALITIES);
//		form.setVisnId(ALL_VISNS);
//		form.setReportToDate(TEST_DATE);
//		form.setReportFromDate(TEST_DATE);
//		form.setReportVersionButtonSelection(PSS_V1);
//		form.setTypeButtonSelection(VISN_SELECTION);
//		
//		List<Object[]> test = new ArrayList<>();
//		Mockito.doReturn(test).when(nQuery).getResultList();
//		DistributionOfSurveysReport report = DistReportDao.getVisnReport(form);
//		
//		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(expectedSql);
//		
//	}
//	
//	@Test
//	public void getFacilityReportSingleModalitySurvAllShouldReturnVisnSqlSurvyV1AppendSingleModality() {
//		
//		String expectedSql = "SELECT f.facility_id + ' ' + f.facility_name AS [Breakout], " +
//				"v.vendor_name, COUNT(*) AS [NumSurveys] " +
//				"FROM Surveyed_Activity AS sa " +
//				"CASE Device.device_name WHEN 'IVR' THEN Device.device_name WHEN 'BROWSER' THEN Device.device_name ELSE 'Home Device' END " +
//				"INNER JOIN Surveyed_Activity_Response AS sar " +
//				"ON sa.Surveyed_Activity_ID = sar.Surveyed_Activity_ID " +
//				"INNER JOIN Vendor AS v " +
//				"ON sa.Vendor_ID = v.Vendor_ID " +
//				"INNER JOIN Device ON sa.Device_ID = Device.Device_ID " +
//				"INNER JOIN Facilities as f " +
//				"ON f.ID = sa.facility_ID " +
//				// formatting date must be done to match expected sql
//				"WHERE sar.completed_date >= '" +  df.format(TEST_DATE) + " 12:00:00 AM' " +
//				"AND sar.completed_date <= '" +  df.format(TEST_DATE) + " 11:59:59 PM' " +
//				
//				"AND sar.survey_accepted_status = 1 " +
//				"AND sar.survey_id = 2 AND device_name IS NOT NULL " +
//				"GROUP BY f.facility_id, f.facility_name, v.vendor_name ,CASE Device.device_name " +
//				"WHEN 'IVR' THEN Device.device_name " +
//				"WHEN 'BROWSER' THEN Device.device_name " +
//				"ELSE 'Home Device' END" +
//				"FROM Surveyed_Activity AS sa " +
//				"INNER JOIN Surveyed_Activity_Response AS sar " +
//				"ON sa.Surveyed_Activity_ID = sar.Surveyed_Activity_ID " +
//				"INNER JOIN Vendor AS v ON sa.Vendor_ID = v.Vendor_ID " +
//				"INNER JOIN Device ON sa.Device_ID = Device.Device_ID " +
//				"INNER JOIN Facilities as f ON f.ID = sa.facility_ID " +
//				"ORDER BY f.facility_id, f.facility_name, v.vendor_name ASC";
//		
//		records = new DistributionOfSurveysReport();
//		PSSReportForm form = new PSSReportForm();
//		form.setModalityName(IVR);
//		form.setFacilityId(ALL_FACILITIES);
//		form.setReportToDate(TEST_DATE);
//		form.setReportFromDate(TEST_DATE);
//		form.setReportVersionButtonSelection(PSS_V1);
//		form.setTypeButtonSelection(FACILITY_SELECTION);
//		
//		List<Object[]> test = new ArrayList<>();
//		Mockito.doReturn(test).when(nQuery).getResultList();
//		DistReportDao.getFacilityReport(form);
//		
//		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(expectedSql);
//		
//	}
//	
//	@Test
//	public void getNationalReportDoNotIncludeModalitySurvV2ShouldReturnNationalSqlV2Survey() {
//		
//		String expectedSql = "SELECT 'na' AS [Breakout], v.vendor_name, COUNT(*) AS [NumSurveys] " + 
//				"FROM Surveyed_Activity AS sa " + 
//				"INNER JOIN Surveyed_Activity_Response AS sar ON sa.Surveyed_Activity_ID = sar.Surveyed_Activity_ID " + 
//				"INNER JOIN Vendor AS v ON sa.Vendor_ID = v.Vendor_ID " +
//				"INNER JOIN Device ON sa.Device_ID = Device.Device_ID " +
//				// formatting date must be done to match expected sql
//				"WHERE sar.completed_date >= '" +  df.format(TEST_DATE) + " 12:00:00 AM' " +
//				"AND sar.completed_date <= '" +  df.format(TEST_DATE) + " 11:59:59 PM' " +
//				
//				"AND sar.survey_accepted_status = 1 " +
//				"AND sar.survey_id = 3 " +
//				"GROUP BY v.vendor_name " +
//				"ORDER BY v.vendor_name ASC";
//		
//		records = new DistributionOfSurveysReport();
//		PSSReportForm form = new PSSReportForm();
//		form.setModalityName(DO_NOT_INCLUDE);
//		form.setReportToDate(TEST_DATE);
//		form.setReportFromDate(TEST_DATE);
//		form.setReportVersionButtonSelection(PSS_V2);
//		form.setTypeButtonSelection(NATIONAL_SELECTION);
//		
//		List<Object[]> test = new ArrayList<>();
//		
//		Mockito.doReturn(test).when(nQuery).getResultList();
//		DistReportDao.getNationalReport(form);
//		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(expectedSql);
//		
//	}
}
