package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.*;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.med.ccht.model.car.CensusActivityRecord;
import gov.va.med.ccht.model.car.CensusActivitySubTotalRecord;
import gov.va.med.ccht.model.car.CensusActivityVisnRecord;
import gov.va.med.ccht.model.car.VendorCensusReportStatus;
import gov.va.med.ccht.model.report.AggregateModality;
import gov.va.med.ccht.persistent.CensusActivityReportDAO;
import gov.va.med.ccht.persistent.ReportsDAO;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.util.DateUtils;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class CensusActivityReportDAOImplTest {
	
	private static final String VENDOR_NAME = "vendorName";
	static final private String TESTNAME = "Name";
	static final private String WEEK_END = "weekEndDate";
	static final private String TESTID = "1";
	private final static String MESSAGE_BY_DEVICE_CODE = "6";
	static final private AggregateModality testModality = new AggregateModality();
	static final private AggregateModality test2Modality = new AggregateModality();
	static final private Long MODID = (long) 1;
    static final private int TESTTOTAL = 3;
	static final private String ALL = "-1"; 
	static final private Date TESTDATE = new Date();
	static final private Date repDate = null;
	
	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	@Mock
	private NativeQuery<Session> modalityQuery;
	@Mock
	private NativeQuery<Session> paramQuery;
	@Mock
	private NativeQuery<Session> param2Query;
	@Mock
	ReportsDAO reportsDao;
	
	private CensusActivityReportDAO censusReportDao;
	private CensusActivityReportDAOImpl censusReportDaoImpl;
	
	private CensusActivityReportForm form;
	private CensusActivityRecord recordResult;
	private List<CensusActivityRecord> result;
	private List<CensusActivitySubTotalRecord> subtotaledResult;
	private List<CensusActivityVisnRecord> visnResult;
	private Map<String, VendorCensusReportStatus> resultMap;
	private List<AggregateModality> modalities;
	
	@Before
	public void setup() {
		MockitoAnnotations.initMocks(this);
		
		Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
		Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
		
		Mockito.doReturn(modalityQuery).when(session).createNativeQuery("SELECT * FROM Modality_Aggregate_Type order by sort_order asc", AggregateModality.class);
		testModality.setId(MODID);
		testModality.setCode(TESTNAME);
		testModality.setDescription(TESTNAME);
		testModality.setName(TESTNAME);
		test2Modality.setId(MODID);
		test2Modality.setCode(MESSAGE_BY_DEVICE_CODE);
		test2Modality.setDescription(TESTNAME);
		test2Modality.setName(TESTNAME);
		modalities = new ArrayList<>();
		modalities.add(testModality);
		modalities.add(test2Modality);
		Mockito.doReturn(modalities).when(modalityQuery).getResultList();
		reportsDao = new ReportsDAOImpl(sessionFactory);
		censusReportDao = new CensusActivityReportDAOImpl(sessionFactory, reportsDao);
		censusReportDaoImpl = new CensusActivityReportDAOImpl(sessionFactory, reportsDao);
	}
	
	@Test
	public void getNationalReportWith2RowsShouldReturn6SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[2*modalities.size()];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		result = censusReportDao.getNationalReport(form);
		assertTrue(result.size() == test.size() + test.size()*modalities.size());
	}
	
	@Test
	public void getNationalReportWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		result = censusReportDao.getNationalReport(form);
		assertTrue(result.size() == test.size());
	}
	
	@Test
	public void getNationalReportWithModalityMessageByDeviceCodeAndOtherModalityShouldAppend2SQLStrings() throws DAOException{
		String sql = "select 'National Totals' AS [National Totals], "
				+ "sum(Facility_Activity.Patient_Totals) AS [Total Patients]";
		sql += ", sum(Facility_Activity.Name) AS [Name]";
		sql += ", sum(Facility_Activity.Name)-sum(IsNull(Facility_Activity.Browser,0)) AS [Name]";
		sql += " from Facility_Activity "
				+ "where Facility_Activity.Rpt_Wk= :weekEndingDate "
				+ "order by 1 ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getNationalReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
	}
	
	@Test
	public void getNationalReportWithNoModalitiesShouldNotAppendingAnyAdditionalSQL() throws DAOException{
		String sql = "select 'National Totals' AS [National Totals], "
				+ "sum(Facility_Activity.Patient_Totals) AS [Total Patients]"
				+ " from Facility_Activity "
				+ "where Facility_Activity.Rpt_Wk= :weekEndingDate "
				+ "order by 1 ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
        modalities = new ArrayList<>();
        Mockito.doReturn(modalities).when(modalityQuery).getResultList();
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getNationalReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
	}
	
	@Test
	public void getSubTotaledNationalReportWith2RowsShouldReturn3SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[2*modalities.size()];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		subtotaledResult = censusReportDao.getSubTotaledNationalReport(form);
		assertTrue(subtotaledResult.size() == modalities.size() + 1);
	}
	
	@Test
	public void getSubTotaledNationalReportWith0RowsShouldReturnNull() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		subtotaledResult = censusReportDao.getSubTotaledNationalReport(form);
		assertTrue(subtotaledResult == null);
	}
	
	@Test
	public void getNationalReportByVendorWith2RowsShouldReturn6SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[2*modalities.size()];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(param2Query).getResultList();
		result = censusReportDao.getNationalReportByVendor(form);
		assertTrue(result.size() == test.size() + test.size()*modalities.size());
	}
	
	@Test
	public void getNationalReportByVendorWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		result = censusReportDao.getNationalReportByVendor(form);
		assertTrue(result.size() == test.size());
	}
	
	@Test
	public void getNationalReportByModalityWith2RowsShouldReturn4SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[2*modalities.size()];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		result = censusReportDao.getNationalReportByModality(form);
		assertTrue(result.size() == test.size() * 2);
	}
	
	@Test
	public void getNationalReportByModalityWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		result = censusReportDao.getNationalReportByModality(form);
		assertTrue(result.size() == test.size());
	}
	
	@Test
	public void getSubtotaledNationalReportByModalityWith2RowsShouldReturn2SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[3];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		subtotaledResult = censusReportDao.getSubtotaledNationalReportByModality(form);
		assertTrue(subtotaledResult.size() == test.size());
	}
	
	@Test
	public void getSubtotaledNationalReportByModalityWith0RowsShouldReturnNullResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		subtotaledResult = censusReportDao.getSubtotaledNationalReportByModality(form);
		assertTrue(subtotaledResult == null);
	}
	
	@Test
	public void getNationalReportByModalityAndVendoryWith2RowsShouldReturn4SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[2*modalities.size()];
			obj[1] = TESTTOTAL + i;
			obj[2] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(param2Query).getResultList();
		result = censusReportDao.getNationalReportByModalityAndVendor(form);
		assertTrue(result.size() == test.size() * 2);
	}
	
	@Test
	public void getNationalReportByModalityAndVendorWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(param2Query).getResultList();
		result = censusReportDao.getNationalReportByModalityAndVendor(form);
		assertTrue(result.size() == test.size());
	}
	
	@Test
	public void getVisnReportWith2RowsShouldReturn3SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setVisnId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[15];
			obj[1] = TESTNAME + i;
			if(i == 0){
				obj[2] = null;
			}else{
				obj[2] = TESTTOTAL + i;
			}
			obj[4] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getVisnReport(form, modalities);
		assertTrue(visnResult.size() == 3);
	}
	
	@Test
	public void getVisnReportWith0RowsShouldReturn1ResultRow() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setVisnId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getVisnReport(form, modalities);
		assertTrue(visnResult.size() == 1);
	}
	
	@Test
	public void getVisnReportWithSingleVisnAndSingleVendorShouldAppend2SQLStrings() throws DAOException{
		String sql = "select VISN.VISN_ID, cast(VISN.VISN_Name As VARCHAR) As visn_name, "
				+ "sum(Facility_Activity.Patient_Totals) AS [Patient_Totals] "
				+ ", sum(Facility_Activity.Name) AS [Name]"
				+ ", sum(Facility_Activity.Name)-sum(IsNull(Facility_Activity.Browser,0)) AS [Name]"
				+ " from Facilities, Facility_Activity, VISN "
				+ "where Facility_Activity.Facility_ID=Facilities.ID and "
				+ "Facility_Activity.Rpt_Wk= :weekEndingDate and "
				+ "Facilities.VISN_ID=VISN.VISN_ID ";
			sql += "and VISN.VISN_ID= :visnId ";
			sql += "and Facility_Activity.VENDOR_ID = :vendorId ";
			sql += "group by VISN.VISN_ID, visn_name  order by VISN.VISN_ID ASC";

		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setReportEndDate(TESTDATE);
		form.setVisnId(TESTID);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getVisnReport(form, modalities));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("visnId", form.getVisnId());
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("vendorId", form.getVendorId());
	}
	
	@Test
	public void getSubtotaledVisnReportWith2RowsShouldReturn3SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setVisnId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[15];
			obj[1] = TESTNAME + i;
			obj[2] = TESTNAME + i;
			obj[3] = TESTTOTAL + i;
			obj[5] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getSubtotaledVisnReport(form, modalities);
		assertTrue(visnResult.size() == 3);
	}
	
	@Test
	public void getSubtotaledVisnReportWith0RowsShouldReturn1ResultRow() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(TESTID);
		form.setVisnId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getSubtotaledVisnReport(form, modalities);
		assertTrue(visnResult.size() == 1);
	}
	
	@Test
	public void getSubtotaledVisnReportWithSingleVisnShouldAppendASQLString() throws DAOException{
		String sql = "select VISN.VISN_ID, "
				+ "cast(VISN.VISN_Name As VARCHAR) As visn_name, "
				+ "Vendor.VENDOR_NAME as [Vendor_Name], "
				+ "sum(Facility_Activity.Patient_Totals) AS [Patient_Totals]"
				+ ", sum(Facility_Activity.Name) AS [Name]"
				+ ", sum(Facility_Activity.Name)-sum(IsNull(Facility_Activity.Browser,0)) AS [Name]"
				+ " from Facilities, Facility_Activity, VISN, Vendor "
				+ "where Facility_Activity.Facility_ID=Facilities.ID and "
				+ "Facility_Activity.Rpt_Wk= :weekEndingDate and "
				+ "Facilities.VISN_ID=VISN.VISN_ID "
				+ "and Vendor_Name <> 'None' "
			    + "and Facility_activity.VENDOR_ID = Vendor.VENDOR_ID ";
			sql += "and VISN.VISN_ID= :visnId ";
			sql += "group by VISN.VISN_ID, visn_name, Vendor_Name  order by VISN.VISN_ID, Vendor_Name ASC";
	
		form = new CensusActivityReportForm();
		form.setReportEndDate(TESTDATE);
		form.setVisnId(TESTID);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getSubtotaledVisnReport(form, modalities));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("visnId", form.getVisnId());
	}
	
	@Test
	public void getVisnDrilldownDataWith2RowsShouldReturn3SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[15];
			obj[1] = TESTNAME + i;
			obj[2] = TESTTOTAL + i;
			obj[3] = TESTNAME + i;
			obj[5] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getVisnDrilldownData(form, modalities);
		assertTrue(visnResult.size() == 3);
	}
	
	@Test
	public void getVisnDrilldownDataWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getVisnDrilldownData(form, modalities);
		assertTrue(visnResult.size() == 0);
	}
	
	@Test
	public void getVisnDrilldownDataWithSingleVisnFacilityAndVendorIdShouldAppend3SQLStrings() throws DAOException{
		String sql = "select CONVERT(VARCHAR(10), Facilities.Facility_ID), "
				+ "Facilities.Facility_Name AS [Facility],VISN.VISN_ID AS [VISN], "
				+ "cast(VISN.VISN_Name As VARCHAR) As visn_name, sum(Facility_Activity.Patient_Totals) AS [Total Patients]"
				+ ", sum(Facility_Activity.Name) AS [Name]"
				+ ", sum(Facility_Activity.Name)-sum(IsNull(Facility_Activity.Browser,0)) AS [Name]"
				+ " from Facilities, Facility_Activity, VISN "
				+ "where Facility_Activity.Facility_ID=Facilities.ID "
				+ "and Facility_Activity.Rpt_Wk= :weekEndingDate ";
			sql	+= "and VISN.VISN_ID = :visnId ";
			sql	+= "and Facilities.VISN_ID=VISN.VISN_ID ";
			sql	+= "and Facility_Activity.VENDOR_ID = :vendorId ";
			sql	+= "and Facilities.FACILITY_ID = :facilityId ";
			sql	+= "group by Facilities.facility_Name,VISN.VISN_ID, VISN.VISN_NAME, Facilities.facility_id "
				+ "order by Facilities.Facility_Name DESC";
		form = new CensusActivityReportForm();
		form.setReportEndDate(TESTDATE);
		form.setVisnId(TESTID);
		form.setVendorId(TESTID);
		form.setFacilityId(TESTID);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getVisnDrilldownData(form, modalities));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("visnId", form.getVisnId());
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("vendorId", form.getVendorId());
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("facilityId", form.getFacilityId());
	}
	
	@Test
	public void getSubtotaledFacilityReportWith2RowsShouldReturn3SuccessfulResults() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[15];
			obj[1] = TESTNAME + i;
			obj[2] = TESTTOTAL + i;
			obj[3] = TESTNAME + i;
			obj[5] = TESTTOTAL + i;
			obj[7] = TESTTOTAL + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getSubtotaledFacilityReport(form, modalities);
		assertTrue(visnResult.size() == 3);
	}
	
	@Test
	public void getSubtotaledFacilityReportWith0RowsShouldReturn1ResultRow() throws DAOException, ParseException {
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(TESTDATE);
        form.setVendorColumnName(TESTNAME);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(paramQuery).getResultList();
		visnResult = censusReportDao.getSubtotaledFacilityReport(form, modalities);
		assertTrue(visnResult.size() == 1);
	}
	
	@Test
	public void getSubtotaledFacilityReportWithSingleFacilityIdShouldAppendASQLStrings() throws DAOException{
		String sql = "select Facilities.Facility_Id, "
				+ "Facilities.Facility_Name, "
				+ "VISN.VISN_ID, "
				+ "cast(VISN.VISN_Name As VARCHAR) As visn_name, "
				+ "Vendor.VENDOR_NAME as [Vendor_Name], "
				+ "sum(Facility_Activity.Patient_Totals) AS [Patient_Totals]"
				+ ", sum(Facility_Activity.Name) AS [Name]"
				+ ", sum(Facility_Activity.Name)-sum(IsNull(Facility_Activity.Browser,0)) AS [Name]"
				+ " from Facilities, Facility_Activity, VISN, Vendor "
				+ "where Facility_Activity.Facility_ID=Facilities.ID and "
				+ "Facility_Activity.Rpt_Wk= :weekEndingDate "
				+ "and Facilities.VISN_ID=VISN.VISN_ID "
				+ "and Vendor_Name <> 'None' "
				+ "and Facility_activity.VENDOR_ID = Vendor.VENDOR_ID ";
			sql	+= "and Facilities.Facility_Id= :facilityId ";
			sql	+= "group by Facilities.Facility_Id, Facilities.Facility_Name, VISN.VISN_ID, visn_name, Vendor_Name order by Facilities.Facility_Name, Vendor_Name ASC";
			
		form = new CensusActivityReportForm();
		form.setReportEndDate(TESTDATE);
		form.setFacilityId(TESTID);
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		assertNotNull(censusReportDao.getSubtotaledFacilityReport(form, modalities));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("weekEndingDate", DateUtils.format(form.getReportEndDate(), DateUtils.MMDDYYYY));
		Mockito.verify(paramQuery, Mockito.atLeast(1)).setParameter("facilityId", form.getFacilityId());
	}
	
    @Test
    public void getAllVendorsReportStatusWithNonNullLastReportDateShouldReturnsssfulResults() throws DAOException, ParseException {
        List<Object[]> test = new ArrayList<>();
        
        
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
        cal.add(Calendar.DAY_OF_WEEK, -7);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        cal.add(Calendar.DAY_OF_WEEK, -6);
        Date repDate = cal.getTime();
        for(int i=0; i<2; i++){
            Object[] obj = new Object[15];
            obj[0] = TESTNAME + i;
            obj[1] = TESTTOTAL + i;
            obj[2] = TESTNAME + i;
            obj[3] = TESTTOTAL + i;
            test.add(obj);
        }
        Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.eq(WEEK_END), Mockito.any());
        Mockito.doReturn(test).when(paramQuery).getResultList();
        
        Mockito.doReturn(param2Query).when(nQuery).setParameter(Mockito.eq(VENDOR_NAME), Mockito.any());
        Mockito.doReturn(repDate).when(param2Query).getSingleResult();
        recordResult = censusReportDao.getAllVendorsReportStatus();
        assertTrue(recordResult.getVendorCensusReportStatus().size() == test.size());
    }
    
    @Test
    public void getAllVendorsReportStatusWith0RowsShouldReturn0Results() throws DAOException, ParseException {
        Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
        List<Object[]> test = new ArrayList<>();
        Mockito.doReturn(test).when(paramQuery).getResultList();
        recordResult = censusReportDao.getAllVendorsReportStatus();
        assertTrue(recordResult.getVendorCensusReportStatus().size() == test.size());
    }
    
    @Test
    public void getAllVendorsReportStatusWith2RowsShouldReturns2SuccessfulResults() throws DAOException, ParseException {
        List<Object[]> test = new ArrayList<>();
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_WEEK, Calendar.SATURDAY);
        cal.add(Calendar.DAY_OF_WEEK, -7);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        Date repDate = cal.getTime();
        for(int i=0; i<2; i++){
            Object[] obj = new Object[15];
            obj[0] = TESTNAME + i;
            obj[1] = TESTTOTAL + i;
            obj[2] = TESTNAME + i;
            obj[3] = TESTTOTAL + i;
            test.add(obj);
        }
        Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.eq(WEEK_END), Mockito.any());
        Mockito.doReturn(test).when(paramQuery).getResultList();
        
        Mockito.doReturn(param2Query).when(nQuery).setParameter(Mockito.eq(VENDOR_NAME), Mockito.any());
        Mockito.doReturn(repDate).when(param2Query).getSingleResult();
        recordResult = censusReportDao.getAllVendorsReportStatus();
        assertTrue(recordResult.getVendorCensusReportStatus().size() == 0);
    }
    
    @Test
    public void getActiveVendorsStatusForReportWeekWith2RowsShouldReturn2SuccessfulResults() throws DAOException, ParseException {
        String start="01/01/2005";
        String end="01/01/2018";
        Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
        Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
        List<Object[]> test = new ArrayList<>();
        for(int i=0; i<2; i++){
            Object[] obj = new Object[6];
            obj[0] = TESTNAME + i;
            obj[3] = TESTTOTAL + i;
            obj[4] = TESTNAME + i;
            obj[5] = TESTNAME + i;
            test.add(obj);
        }
        Mockito.doReturn(test).when(param2Query).getResultList();
        resultMap = censusReportDaoImpl.getActiveVendorsStatusForReportWeek(end, start);
        assertTrue(resultMap.size() == test.size());
    }
	
	@Test
	public void getActiveVendorsStatusForReportWeekWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		String start="01/01/2005";
		String end="01/01/2018";
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(param2Query).getResultList();
		resultMap = censusReportDaoImpl.getActiveVendorsStatusForReportWeek(end, start);
		assertTrue(resultMap.size() == test.size());
	}
	
	@Test
	public void getActiveVendorsForReportWeekWith2RowsShouldReturn2SuccessfulResults() throws DAOException, ParseException {
		String start="01/01/2005";
		String end="01/01/2018";
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		for(int i=0; i<2; i++){
			Object[] obj = new Object[6];
			obj[0] = TESTNAME + i;
			obj[3] = TESTTOTAL + i;
			obj[4] = TESTNAME + i;
			obj[5] = TESTNAME + i;
			test.add(obj);
		}
		Mockito.doReturn(test).when(param2Query).getResultList();
		resultMap = censusReportDaoImpl.getActiveVendorsForReportWeek(end, start);
		assertTrue(resultMap.size() == test.size());
	}
	
	@Test
	public void getActiveVendorsForReportWeekWith0RowsShouldReturn0Results() throws DAOException, ParseException {
		String start="01/01/2005";
		String end="01/01/2018";
		Mockito.doReturn(paramQuery).when(nQuery).setParameter(Mockito.anyString(), Mockito.any());
		Mockito.doReturn(param2Query).when(paramQuery).setParameter(Mockito.anyString(), Mockito.any());
		List<Object[]> test = new ArrayList<>();
		Mockito.doReturn(test).when(param2Query).getResultList();
		resultMap = censusReportDaoImpl.getActiveVendorsForReportWeek(end, start);
		assertTrue(resultMap.size() == test.size());
	}
	
	@Test
	public void getMinStartDateShouldReturnFixedMinStartDate() throws DAOException, ParseException {
		Date test = censusReportDaoImpl.getMinStartDate();
		SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
		String testDate = df.format(test);
		assertTrue(testDate.equals("09/10/2005"));
	}
}
