package gov.va.med.ccht.persistent.hibernate;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.NativeQuery;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import gov.va.med.ccht.model.coc.CategoryOfCareRecord;
import gov.va.med.ccht.persistent.CategoryOfCareReportDAO;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;

@ContextConfiguration(locations = "classpath:application-context-test.xml")
@RunWith(SpringJUnit4ClassRunner.class)
public class CategoryOfCareReportDAOImplTest {
	
	static final private String TEST_ID_NAME = "1";
	static final private int NAMEANDID = 0;
	static final private int ID = 1;
	static final private int RESULTS_NUMBER = 3;
	static final private int FACILITY_NAME = 4;
	
	static final private int NUMBER_OF_COC_TYPES = 6;
     
	static final private String ALL = "-1";
	
	@Mock
	private SessionFactory sessionFactory;
	@Mock
	private Session session;
	@Mock
	private NativeQuery<Session> nQuery;
	@Mock
	private NativeQuery<Session> visnOrFacilityQuery;
	@Mock
	private NativeQuery<Session> vendorQuery;
	
	private CategoryOfCareReportDAO reportDao;
	
	private CensusActivityReportForm form;
	private List<CategoryOfCareRecord> result;
	private List<Object[]> records;
	private List<Integer> visnsOrFacilities;
	private List<String> vendors;
	
	@Before
	public void setup() {
		MockitoAnnotations.initMocks(this);
		
		Mockito.doReturn(session).when(sessionFactory).getCurrentSession();
		Mockito.doReturn(nQuery).when(session).createNativeQuery(Mockito.anyString());
		
		visnsOrFacilities = new ArrayList<Integer>();
		visnsOrFacilities.add(1);
		visnsOrFacilities.add(2);
		Mockito.doReturn(visnOrFacilityQuery).when(session).createNativeQuery("SELECT DISTINCT VISN_ID FROM VISN ");	
		Mockito.doReturn(visnOrFacilityQuery).when(session).createNativeQuery("SELECT DISTINCT FACILITY_ID FROM FACILITIES ");
		Mockito.doReturn(visnsOrFacilities).when(visnOrFacilityQuery).getResultList();
		
		vendors = new ArrayList<String>();
		vendors.add(TEST_ID_NAME);
		Mockito.doReturn(vendorQuery).when(session).createNativeQuery("SELECT DISTINCT vendor_name FROM vendor");
		Mockito.doReturn(vendors).when(vendorQuery).getResultList();
		reportDao = new CategoryOfCareReportDAOImpl(sessionFactory);
	}
		
	@Test
	public void getNationalReportWithSingleVendorShouldAppendVendorIdToSql() {		
		String sql =  "SELECT DISTINCT c.vendor_name, d.id, d.category_name, COUNT(distinct patient_id) AS [Num] "
				+ "FROM census a, censuspatient b, vendor c, levelofcare d "
				+ "WHERE a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND a.vendor_id = c.vendor_id "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date IS NULL OR b.disenrollment_date >= :reportStartDate) ";
		sql += "AND c.vendor_id = :vendorId ";
		sql += "GROUP BY c.vendor_name, d.id, d.category_name "
				+ "ORDER BY c.vendor_name, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getNationalReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
	}
	
	@Test
	public void getNationalReportWithAllVendorsShouldNotAddAnyAdditionalParameters(){	
		String sql =  "SELECT DISTINCT c.vendor_name, d.id, d.category_name, COUNT(distinct patient_id) AS [Num] "
				+ "FROM census a, censuspatient b, vendor c, levelofcare d "
				+ "WHERE a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND a.vendor_id = c.vendor_id "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date IS NULL OR b.disenrollment_date >= :reportStartDate) "
				+ "GROUP BY c.vendor_name, d.id, d.category_name "
				+ "ORDER BY c.vendor_name, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getNationalReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
	}
	
	@Test
	public void getNationalReportWithNoResultRowsReturnedShouldCreate2EmptyRecords(){	
		records = new ArrayList<>();
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getNationalReport(form);
		assertTrue(result.size() == 2);
		assertTrue(result.get(0).getTotal() == 0);
	}
	
	@Test
	public void getNationalReportWith3ResultRowsReturnedShouldCreate3RecordsSuccessfully(){	
		records = new ArrayList<>();
		for (int i = 1; i <= NUMBER_OF_COC_TYPES; i++){
			Object[] obj = new Object[5];
			obj[NAMEANDID] = TEST_ID_NAME;
			obj[ID] = i;
			obj[RESULTS_NUMBER] = RESULTS_NUMBER;
			obj[FACILITY_NAME] = TEST_ID_NAME;
			records.add(obj);
		}		
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getNationalReport(form);
		assertTrue(result.size() == 3);
		assertTrue(result.get(0).getTotal() == RESULTS_NUMBER * NUMBER_OF_COC_TYPES);
	}
	
	@Test
	public void getVisnReportWithSingleVendorAndSingleVisnShouldAppendVendorIdAndVisnIdToSql() {		
		String sql =  "SELECT DISTINCT e.visn_id, d.id, d.category_name, COUNT(distinct patient_id) "
				+ "FROM census a, censuspatient b, facilities c, levelofcare d, visn e, vendor f "
				+ "WHERE b.facility_id = c.id AND c.visn_id = e.visn_id "
				+ "AND a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date is null OR b.disenrollment_date >= :reportStartDate) ";
		sql += "AND e.visn_id = :visnId ";
		sql += "AND a.vendor_id = f.vendor_id ";
		sql += "AND f.vendor_id = :vendorId  ";
		sql += "GROUP BY e.visn_id, d.id, d.category_name ORDER BY e.visn_id, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setVisnId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getVisnReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", TEST_ID_NAME);	
	}
	
	@Test
	public void getVisnReportWithAllVendorsAndSingleVisnShouldAppendVisnIdToSql(){	
		String sql =  "SELECT DISTINCT e.visn_id, d.id, d.category_name, COUNT(distinct patient_id) "
				+ "FROM census a, censuspatient b, facilities c, levelofcare d, visn e, vendor f "
				+ "WHERE b.facility_id = c.id AND c.visn_id = e.visn_id "
				+ "AND a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date is null OR b.disenrollment_date >= :reportStartDate) ";
		sql += "AND e.visn_id = :visnId ";
		sql += "GROUP BY e.visn_id, d.id, d.category_name ORDER BY e.visn_id, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getVisnReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("visnId", TEST_ID_NAME);
	}
	
	@Test
	public void getVisnReportWithSingleVendorAndAllVisnShouldAppendVendorId() {		
		String sql =  "SELECT DISTINCT e.visn_id, d.id, d.category_name, COUNT(distinct patient_id) "
				+ "FROM census a, censuspatient b, facilities c, levelofcare d, visn e, vendor f "
				+ "WHERE b.facility_id = c.id AND c.visn_id = e.visn_id "
				+ "AND a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date is null OR b.disenrollment_date >= :reportStartDate) ";
		sql += "AND a.vendor_id = f.vendor_id ";
		sql += "and a.vendor_id = :vendorId ";
		sql += "GROUP BY e.visn_id, d.id, d.category_name ORDER BY e.visn_id, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getVisnReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
	}
	
	@Test
	public void getVisnReportWithAllVendorsAndAllVisnShouldNotAddAnyAdditionalParameters(){	
		String sql =  "SELECT DISTINCT e.visn_id, d.id, d.category_name, COUNT(distinct patient_id) "
				+ "FROM census a, censuspatient b, facilities c, levelofcare d, visn e, vendor f "
				+ "WHERE b.facility_id = c.id AND c.visn_id = e.visn_id "
				+ "AND a.census_id = b.census_id "
				+ "AND a.report_start_date = :reportStartDate "
				+ "AND b.level_of_care = d.id "
				+ "AND (b.disenrollment_date is null OR b.disenrollment_date >= :reportStartDate) ";
		sql += "GROUP BY e.visn_id, d.id, d.category_name ORDER BY e.visn_id, d.id ASC";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getVisnReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
	}
	
	@Test
	public void getVisnReportWithNoResultRowsReturnedShouldCreate2EmptyRecords(){	
		records = new ArrayList<>();
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getVisnReport(form);
		assertTrue(result.size() == 2);
		assertTrue(result.get(0).getTotal() == 0);
	}
	
	@Test
	public void getVisnReportWith3ResultRowsReturnedShouldCreate3RecordsSuccessfully(){	
		records = new ArrayList<>();
		for (int i = 1; i <= NUMBER_OF_COC_TYPES; i++){
			Object[] obj = new Object[5];
			obj[NAMEANDID] = TEST_ID_NAME;
			obj[ID] = i;
			obj[RESULTS_NUMBER] = RESULTS_NUMBER;
			obj[FACILITY_NAME] = TEST_ID_NAME;
			records.add(obj);
		}		
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getVisnReport(form);
		assertTrue(result.size() == 3);
		assertTrue(result.get(0).getTotal() == RESULTS_NUMBER * NUMBER_OF_COC_TYPES);
	}
	
	@Test
	public void getFacilityReportByVisnWithSingleVendorShouldAppendVendorIdToSql() {		
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, visn f, vendor g  "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and c.visn_id = f.visn_id "
				+ "and f.visn_id = :visnId "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "and a.vendor_id = :vendorId ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name ";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setVisnId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReportByVisn(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
	}
	
	@Test
	public void getFacilityReportByVisnWithAllVendorsShouldNotAddAnyAdditionalParameters(){	
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, visn f, vendor g  "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and c.visn_id = f.visn_id "
				+ "and f.visn_id = :visnId "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name ";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReportByVisn(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
	}
	
	@Test
	public void getFacilityReportByVisnWithNoResultRowsReturnedShouldCreate2EmptyRecords(){	
		records = new ArrayList<>();
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getFacilityReportByVisn(form);
		assertTrue(result.size() == 2);
		assertTrue(result.get(0).getTotal() == 0);
	}
	
	@Test
	public void getFacilityReportByVisnWith3ResultRowsReturnedShouldCreate3RecordsSuccessfully(){	
		records = new ArrayList<>();
		for (int i = 1; i <= NUMBER_OF_COC_TYPES; i++){
			Object[] obj = new Object[5];
			obj[NAMEANDID] = TEST_ID_NAME;
			obj[ID] = i;
			obj[RESULTS_NUMBER] = RESULTS_NUMBER;
			obj[FACILITY_NAME] = TEST_ID_NAME;
			records.add(obj);
		}		
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setVisnId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getFacilityReportByVisn(form);
		assertTrue(result.size() == 3);
		assertTrue(result.get(0).getTotal() == RESULTS_NUMBER * NUMBER_OF_COC_TYPES);
	}
	
	/**************************************************************************/
	@Test
	public void getFacilityReportWithSingleVendorAndSingleFacilityShouldAppendVendorIdAndFacilityIdToSql() {		
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, vendor f "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "and c.facility_id = :facilityId ";
		sql += "and a.vendor_id = :vendorId ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name "
				+ "order by c.facility_id ";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setFacilityId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID_NAME);	
	}
	
	@Test
	public void getFacilityReportWithAllVendorsAndSingleFacilityShouldAppendFacilityIdToSql(){	
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, vendor f "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "and c.facility_id = :facilityId ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name "
				+ "order by c.facility_id ";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID_NAME);
	}
	
	@Test
	public void getFacilityReportWithSingleVendorAndAllFacilityShouldAppendVendorId() {		
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, vendor f "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "and a.vendor_id = :vendorId ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name "
				+ "order by c.facility_id ";
		form = new CensusActivityReportForm();
		form.setVendorId(TEST_ID_NAME);
		form.setFacilityId(ALL);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("vendorId", TEST_ID_NAME);
	}
	
	@Test
	public void getFacilityReportWithAllVendorsAndAllFacilityShouldNotAddAnyAdditionalParameters(){	
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, vendor f "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name "
				+ "order by c.facility_id ";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
	}
	
	@Test
	public void getFacilityReportWithAllVendorsAndFacilitySelectedFromDropdownShouldAppendFacilityIdToSql(){	
		String sql =  "select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name "
				+ "from census a, censuspatient b, facilities c, "
				+ "levelofcare d, patient e, vendor f "
				+ "where b.patient_id = e.patient_id  "
				+ "and b.facility_id = c.id  "
				+ "and a.census_id = b.census_id "
				+ "and a.report_start_date = :reportStartDate "
				+ "and b.level_of_care = d.id  "
				+ "and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) "
				+ "and left(e.patient_ssn,5)<>'00000' ";
		sql += "and c.facility_id = :facilityId ";
		sql += "group by c.facility_id, c.facility_name, d.id, d.category_name "
				+ "order by c.facility_id ";
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(","+TEST_ID_NAME);
		form.setReportEndDate(new Date());
		assertNotNull(reportDao.getFacilityReport(form));
		Mockito.verify(session, Mockito.atLeast(1)).createNativeQuery(sql);
		Mockito.verify(nQuery, Mockito.atLeast(1)).setParameter("facilityId", TEST_ID_NAME);
	}
	@Test
	public void getFacilityReportWithNoResultRowsReturnedShouldCreate2EmptyRecords(){	
		records = new ArrayList<>();
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getFacilityReport(form);
		assertTrue(result.size() == 2);
		assertTrue(result.get(0).getTotal() == 0);
	}
	
	@Test
	public void getFacilityReportWith3ResultRowsReturnedShouldCreate3RecordsSuccessfully(){	
		records = new ArrayList<>();
		for (int i = 1; i <= NUMBER_OF_COC_TYPES; i++){
			Object[] obj = new Object[5];
			obj[NAMEANDID] = TEST_ID_NAME;
			obj[ID] = i;
			obj[RESULTS_NUMBER] = RESULTS_NUMBER;
			obj[FACILITY_NAME] = TEST_ID_NAME;
			records.add(obj);
		}		
		Mockito.doReturn(records).when(nQuery).getResultList();		
		
		form = new CensusActivityReportForm();
		form.setVendorId(ALL);
		form.setFacilityId(ALL);
		form.setReportEndDate(new Date());
		result = reportDao.getFacilityReport(form);
		assertTrue(result.size() == 3);
		assertTrue(result.get(0).getTotal() == RESULTS_NUMBER * NUMBER_OF_COC_TYPES);
	}
}
