/********************************************************************
 * Copyriight 2005 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.service.transaction;

import java.sql.Timestamp;
import java.util.Date;

import org.apache.commons.lang.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.ccht.persistent.TimestampDAO;
import gov.va.med.fw.cache.TransactionTimestampCacheManager;
import gov.va.med.fw.service.AbstractComponent;

/**
 * Manages the creation and storage of the "global Transaction timestamp". The
 * "global Transaction timestamp" is meant to synchronize all database flushes
 * in the same Transaction such that they all use the same event date. Typically
 * this timestamp will be created and cached when the Transaction is started by
 * the TransactionSynchronizationInitializer class.
 * 
 * Timestamp retrieval is either from the database (if they
 * CurrentTimestampAction is injected) or from the application server (no
 * CurrentTimestampAction injection).
 * 
 * Created Nov 3, 2005 2:59:33 PM
 * 
 * DNS
 * 
 * @see gov.va.med.fw.service.transaction.TransactionSynchronizationInitializer
 */
@Component
public class TransactionTimestampManager extends AbstractComponent { 
	
	@Autowired
	private TransactionTimestampCacheManager timestampCacheManager;
	@Autowired
	private TimestampDAO timestampDAO;
	
	public Timestamp intializeTransactionTimestamp(boolean forceOverride) {
		if (timestampCacheManager.hasTimestamp() && forceOverride)
			timestampCacheManager.removeTimestamp();

		Timestamp ts = createNewTransactionTimestamp();
		timestampCacheManager.storeTransactionTimestamp(ts);
		return ts;
	}

	/**
	 * If called in a scenario that does not have a Timestamp already created,
	 * one will be created but not stored. Therefore, subsequent calls in that
	 * non-Transactional context will return a different timestamp every time.
	 * 
	 * @return
	 */
	public Timestamp getTransactionTimestamp() {
		Timestamp ts = timestampCacheManager.getTransactionTimestamp();
		if (ts == null) {
			logger
					.debug("There is no Timestamp tied to the current (if any) Transaction. Creating new one");
			// create an unbound ts
			ts = createNewTransactionTimestamp();
		}
		return ts;
	}

	private Timestamp createNewTransactionTimestamp() {
		Timestamp ts = null;
		if (timestampDAO != null) {
			return timestampDAO.getCurrentTimestampFromDatabase();
		} else {
			ts = new Timestamp(System.currentTimeMillis());
			if (logger.isDebugEnabled())
				logger.debug("###Initialized ts (from JVM) at JVM time [" + new Date()
						+ "]: using jvm ts [" + ts + "]");
		}
		return ts;
	}

	/**
	 * @return Returns the timestampCacheManager.
	 */
	public TransactionTimestampCacheManager getTimestampCacheManager() {
		return timestampCacheManager;
	}

	/**
	 * @param timestampCacheManager
	 *            The timestampCacheManager to set.
	 */
	public void setTimestampCacheManager(TransactionTimestampCacheManager timestampCacheManager) {
		this.timestampCacheManager = timestampCacheManager;
	}

	public void afterPropertiesSet() {
		Validate.notNull(timestampCacheManager);
		Validate.notNull(timestampDAO);
	}
}
