/********************************************************************
 * Copyriight 2007 VHA. All rights reserved
 ********************************************************************/

package gov.va.med.fw.service.jms;

import java.io.Serializable;
import java.util.Enumeration;
import java.util.Map;

import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.Session;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jms.support.converter.MessageConversionException;
import org.springframework.jms.support.converter.SimpleMessageConverter;

/**
 * Since outbound HL7 messages requires a JMS TextMessage from a Serializable
 * HL7Message object, custom conversion is required here (as opposed to just
 * using Spring's SimpleMessageConverter).
 * 
 * <p>
 * Also allows for "attempted" String'ification of JMS payloads.
 * 
 * Created Jun 20, 2007 2:00:43 PM
 * 
 * DNS
 */
public class DefaultMessageConverter extends SimpleMessageConverter {
	protected final Log logger = LogFactory.getLog(getClass());
	private StringifyPayloadConverter stringifyPayloadConverter;

	@Override
	public Message toMessage(Object object, Session session) throws JMSException,
			MessageConversionException {
		// Create a JMS message based on an input object's class type
		// The rule here is as follow:
		// a. byte[] = BytesMessage
		// b. String = TextMessage
		// c. gov.va.med.fw.hl7.Message = TextMessage
		// d. Map = MapMessage
		// e. Serializable = ObjectMessage
		// f. Anything else = Not supported throw exception

		if (object instanceof Message) {
			return (Message) object;
		}
		// use a fully-qualified path to avoid collision with a
		// javax.jms.Message
		else if (object instanceof gov.va.med.fw.hl7.Message) {
			return createMessageForString(((gov.va.med.fw.hl7.Message) object).getMessageData(),
					session);
		} else if (object instanceof String) {
			return createMessageForString((String) object, session);
		} else if (object instanceof byte[]) {
			return createMessageForByteArray((byte[]) object, session);
		} else if (object instanceof Map) {
			MapMessage mapMessage = createMessageForMap((Map) object, session);
			if (stringifyPayloadConverter != null) {
				String key = null;
				Object val = null;
				for (Enumeration e = mapMessage.getMapNames(); e.hasMoreElements();) {
					key = (String) e.nextElement();
					val = mapMessage.getObject(key);
					if (!(val instanceof Serializable))
						throw new MessageConversionException("JMS Payloads must be Serializable");
					val = toFinalPayload((Serializable) val);
					mapMessage.setString(key, val != null ? val.toString() : (String) val);
				}
			}
			return mapMessage;
		}
		// do this test last to avoid processing above special cases
		else if (object instanceof Serializable) {
			return createMessageForSerializable(toFinalPayload((Serializable) object), session);
		} else {
			throw new MessageConversionException("Cannot convert object [" + object
					+ "] to JMS message");
		}
	}

	@Override
	public Object fromMessage(Message message) throws JMSException, MessageConversionException {
		Object obj = super.fromMessage(message);
		if (stringifyPayloadConverter != null && obj instanceof String) {
			obj = stringifyPayloadConverter.fromString((String) obj);
		}
		return obj;
	}

	protected final Serializable toFinalPayload(Serializable payload) {
		return stringifyPayloadConverter == null ? payload : stringifyPayloadConverter
				.toString(payload);
	}

	public StringifyPayloadConverter getStringifyPayloadConverter() {
		return stringifyPayloadConverter;
	}

	public void setStringifyPayloadConverter(StringifyPayloadConverter stringifyPayloadConverter) {
		this.stringifyPayloadConverter = stringifyPayloadConverter;
	}

}
