/********************************************************************
 * Copyriight 2004 VHA. All rights reserved
 ********************************************************************/
// Package
package gov.va.med.fw.service.jms;

// Java classes
import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.Message;

import org.apache.commons.lang.Validate;
import org.springframework.jms.JmsException;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.core.MessagePostProcessor;

import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.util.StopWatchLogger;

/**
 * Generic JMS Message Producer that can be wired for a Queue or Topic
 * destination. Handles wrapping message payload into the appropriate type of
 * JMS Mesage.
 * 
 * <p>
 * Supports configurable fallback targets.
 * </p>
 * 
 * <p>
 * Relies on JEE container to pool JMS ConnectionFactory's.
 * </p>
 * 
 * Created Jun 18, 2007 3:50:11 PM
 * 
 * DNS
 */
public class AsynchronousMessageProducerServiceImpl extends AbstractComponent
		implements MessageProducerService {
	/**
	 * An instance of serialVersionUID
	 */
	private static final long serialVersionUID = 2808696406833332275L;

	/**
	 * this is not real-time as it indicates the longest period of time in
	 * between two requests (lazily checked)
	 */
	private int checkIntervalSecsAfterFallback = 120;

	private boolean supportsFailover = true;
	private JmsTemplate jmsTemplatePrimary;
	private List<JmsTemplate> jmsTemplatesFallback;

	// state tracking
	private JmsTemplate jmsTemplateActive;
	private Date switchedToFallbackAtThisTime;

	private boolean supportsDistributedTransactions;

	public boolean isSupportsDistributedTransactions() {
		return supportsDistributedTransactions;
	}

	public void setSupportsDistributedTransactions(
			boolean supportsDistributedTransactions) {
		this.supportsDistributedTransactions = supportsDistributedTransactions;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		super.afterPropertiesSet();
		Validate.notNull(jmsTemplatePrimary, "jmsTemplatePrimary is required");
		if (this.supportsDistributedTransactions) {
			jmsTemplatePrimary.setSessionTransacted(false);

			// set also on fallbacks
			if (jmsTemplatesFallback != null) {
				for (JmsTemplate target : jmsTemplatesFallback) {
					target.setSessionTransacted(false);
				}
			}
		}
	}

	/**
	 * Sends to injected Destination (priority) or injected default Destination.
	 */
	public void send(final Serializable payload)
			throws MessageProducerException {
		send(payload, (Properties) null);
	}

	public void sendToAll(final Serializable payload)
			throws MessageProducerException {
		sendToAll(payload, (Properties) null);
	}

	public void send(MessagePayload payload) throws MessageProducerException {
		Map<String, Serializable> props = new HashMap<String, Serializable>();
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_TYPE, payload
				.getTargetServiceDescriptor());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_INITIATER,
				SecurityContextHelper.getUserName());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_ORIGINATING_TIMEZONE,
				SecurityContextHelper.getUserTimeZone());

		// TODO: make this conditional based on deployed environment so not to
		// expose potential PII
		if (logger.isDebugEnabled()) {
			props.put("payloadSummary", payload.getPayload().toString());
		}

		send(payload.getPayload(), props);
	}

	public void sendToAll(MessagePayload payload)
			throws MessageProducerException {
		Map<String, Serializable> props = new HashMap<String, Serializable>();
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_TYPE, payload
				.getTargetServiceDescriptor());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_INITIATER,
				SecurityContextHelper.getUserName());
		props.put(ConfigurationConstants.DEFAULT_MESSAGE_ORIGINATING_TIMEZONE,
				SecurityContextHelper.getUserTimeZone());

		// TODO: make this conditional based on deployed environment so not to
		// expose potential PII
		if (logger.isDebugEnabled()) {
			props.put("payloadSummary", payload.getPayload().toString());
		}

		sendToAll(payload.getPayload(), props);
	}

	public void send(Serializable payload,
			final Map<String, Serializable> headerProperties)
			throws MessageProducerException {
		StopWatchLogger watch = new StopWatchLogger("send JMS");
		watch.start();
		boolean attemptedFallback = false;

		JmsTemplate targetJmsTemplate = getActiveJmsTemplate();
		try {
			targetJmsTemplate.convertAndSend(payload,
					new MessagePostProcessor() {
						public Message postProcessMessage(Message message)
								throws JMSException {
							return copyJMSProperties(message, headerProperties);
						}
					});
		} catch (Exception e) {
			attemptedFallback = true;
			attemptFallbackConnections(e, targetJmsTemplate, payload,
					headerProperties);
		} finally {
			watch.stopAndLog("attemptedFallback=" + attemptedFallback);
		}
	}

	public void sendToAll(Serializable payload,
			final Map<String, Serializable> headerProperties)
			throws MessageProducerException {
		StopWatchLogger watch = new StopWatchLogger("send JMS");
		watch.start();
		boolean attemptedFallback = false;

		JmsTemplate targetJmsTemplate = getActiveJmsTemplate();
		try {
			targetJmsTemplate.convertAndSend(payload,
					new MessagePostProcessor() {
						public Message postProcessMessage(Message message)
								throws JMSException {
							return copyJMSProperties(message, headerProperties);
						}
					});
		} catch (Exception e) {
			getLogger().error(e);
		}
		for (JmsTemplate jmsTemplate : jmsTemplatesFallback) {
			try {
				jmsTemplate.convertAndSend(payload, new MessagePostProcessor() {
					public Message postProcessMessage(Message message)
							throws JMSException {
						return copyJMSProperties(message, headerProperties);
					}
				});
			} catch (Exception e) {
				getLogger().error(e);
			}
		}
		watch.stopAndLog("");
	}

	public void send(Serializable payload, final Properties headerProperties)
			throws MessageProducerException {
		StopWatchLogger watch = new StopWatchLogger("send JMS");
		watch.start();
		boolean attemptedFallback = false;

		JmsTemplate targetJmsTemplate = getActiveJmsTemplate();
		try {
			targetJmsTemplate.convertAndSend(payload,
					new MessagePostProcessor() {
						public Message postProcessMessage(Message message)
								throws JMSException {
							return copyJMSProperties(message, headerProperties);
						}
					});
		} catch (Exception e) {
			attemptFallbackConnections(e, targetJmsTemplate, payload,
					headerProperties);
		} finally {
			watch.stopAndLog("attemptedFallback=" + attemptedFallback);
		}
	}

	public void sendToAll(Serializable payload,
			final Properties headerProperties) throws MessageProducerException {
		StopWatchLogger watch = new StopWatchLogger("send JMS");
		watch.start();
		boolean attemptedFallback = false;

		JmsTemplate targetJmsTemplate = getActiveJmsTemplate();
		try {
			targetJmsTemplate.convertAndSend(payload,
					new MessagePostProcessor() {
						public Message postProcessMessage(Message message)
								throws JMSException {
							return copyJMSProperties(message, headerProperties);
						}
					});
		} catch (Exception e) {
			getLogger().error(e);
		}
		for (JmsTemplate jmsTemplate : jmsTemplatesFallback) {
			try {
				jmsTemplate.convertAndSend(payload, new MessagePostProcessor() {
					public Message postProcessMessage(Message message)
							throws JMSException {
						return copyJMSProperties(message, headerProperties);
					}
				});
			} catch (Exception e) {
				getLogger().error(e);
			}
		}
		watch.stopAndLog("");
	}

	private JmsTemplate getActiveJmsTemplate() {
		if (jmsTemplateActive == null) {
			jmsTemplateActive = jmsTemplatePrimary;
			switchedToFallbackAtThisTime = null;
		} else if (switchedToFallbackAtThisTime == null) {
			jmsTemplateActive = jmsTemplatePrimary;
		} else if (switchedToFallbackAtThisTime != null) {
			if (((System.currentTimeMillis() - switchedToFallbackAtThisTime
					.getTime()) / 1000) >= checkIntervalSecsAfterFallback) {
				if (logger.isInfoEnabled()) {
					logger
							.info(getBeanName()
									+ " had previously falled back to a fallback JmsTemplate....checkInterval of "
									+ checkIntervalSecsAfterFallback
									+ " has expired.....trying primary again");
				}
				jmsTemplateActive = jmsTemplatePrimary;
				switchedToFallbackAtThisTime = null;
			}
		}

		return jmsTemplateActive;
	}

	private boolean isConfiguredForFallback() {
		return supportsFailover && jmsTemplatesFallback != null
				&& !jmsTemplatesFallback.isEmpty();
	}

	private void attemptFallbackConnections(Exception e,
			JmsTemplate jmsTemplateFailure, Serializable payload,
			final Map headerProperties) throws JMSServiceException {
		if (!isConfiguredForFallback())
			throw new JMSServiceException(
					"Unable to send JMS Message and not configured for failover to fallback JmsTemplates - exception:",
					e);

		if (logger.isWarnEnabled()) {
			logger
					.warn(
							"Unable to send JMS Message, attempting failover to fallback JmsTemplates due to exception:",
							e);
		}

		// fallback
		switchedToFallbackAtThisTime = new Date();
		int i = -1;
		boolean fallbackResult = false;
		for (JmsTemplate target : jmsTemplatesFallback) {
			i++;
			if (target == jmsTemplateFailure) // don't retry
				continue;

			jmsTemplateActive = target;

			fallbackResult = attemptFallbackConnection(jmsTemplateActive,
					payload, headerProperties, "index: " + i);
			break;
		}

		// one last check....if tried all fallbacks, also try primary if not
		// original failure (in case a fallback failed)
		if (!fallbackResult && jmsTemplateFailure != jmsTemplatePrimary) {
			if (attemptFallbackConnection(jmsTemplatePrimary, payload,
					headerProperties, "looped back to primary")) {
				// reset
				jmsTemplateActive = jmsTemplatePrimary;
				switchedToFallbackAtThisTime = null;
			}
		}
	}

	private boolean attemptFallbackConnection(JmsTemplate target,
			Serializable payload, final Map headerProperties, String logMessage) {
		try {
			target.convertAndSend(payload, new MessagePostProcessor() {
				public Message postProcessMessage(Message message)
						throws JMSException {
					return copyJMSProperties(message, headerProperties);
				}
			});

			if (logger.isInfoEnabled()) {
				logger.info("[SUCCESS JMS FALLBACK]: " + logMessage);
			}

			return true;
		} catch (JmsException e) {
			if (logger.isErrorEnabled()) {
				logger.error("[FAILURE JMS FALLBACK]: " + logMessage, e);
			}
		}
		return false;
	}

	protected Message copyJMSProperties(Message message, Map headerProperties)
			throws JMSException {
		if (headerProperties != null) {
			// put all properties in a message's properties
			for (Object key : headerProperties.keySet()) {
				message.setStringProperty(key.toString(), headerProperties.get(
						key).toString());
			}
		}
		return message;
	}

	/**
	 * @return the jmsTemplatePrimary
	 */
	public JmsTemplate getJmsTemplatePrimary() {
		return jmsTemplatePrimary;
	}

	/**
	 * @param jmsTemplatePrimary
	 *            the jmsTemplatePrimary to set
	 */
	public void setJmsTemplatePrimary(JmsTemplate jmsTemplatePrimary) {
		this.jmsTemplatePrimary = jmsTemplatePrimary;
	}

	/**
	 * @return the jmsTemplatesFallback
	 */
	public List<JmsTemplate> getJmsTemplatesFallback() {
		return jmsTemplatesFallback;
	}

	/**
	 * @param jmsTemplatesFallback
	 *            the jmsTemplatesFallback to set
	 */
	public void setJmsTemplatesFallback(List<JmsTemplate> jmsTemplatesFallback) {
		this.jmsTemplatesFallback = jmsTemplatesFallback;
	}

	/**
	 * @return the supportsFailover
	 */
	public boolean isSupportsFailover() {
		return supportsFailover;
	}

	/**
	 * @param supportsFailover
	 *            the supportsFailover to set
	 */
	public void setSupportsFailover(boolean supportsFailover) {
		this.supportsFailover = supportsFailover;
	}

	/**
	 * @return the checkIntervalSecsAfterFallback
	 */
	public int getCheckIntervalSecsAfterFallback() {
		return checkIntervalSecsAfterFallback;
	}

	/**
	 * @param checkIntervalSecsAfterFallback
	 *            the checkIntervalSecsAfterFallback to set
	 */
	public void setCheckIntervalSecsAfterFallback(
			int checkIntervalSecsAfterFallback) {
		this.checkIntervalSecsAfterFallback = checkIntervalSecsAfterFallback;
	}

}