package gov.va.med.fw.security;

//Java imports
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;

import org.apache.commons.lang.Validate;

import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.util.EncoderDecoderUtil;
import gov.va.med.fw.util.StringUtils;

/**
 * This is the API clients uses for getting secret key, encrypting and
 * decrypting. Thsi class uses SunJCE provider to generate secret key,
 * encrypting and decrypting the information.
 * 
 * @author Muddaiah Ranga
 * @version 3.0
 */
public class EncryptionServiceImpl extends AbstractComponent implements EncryptionService {
	private static final String SECURE_RANDOM_ALGORITHM = "SHA1PRNG";
	private static final String PROVIDER = "SUN";

	protected String algorithm;
	protected int keySize;
	protected boolean encryptionFlag;
	protected String defaultCharacterEncoding;
	protected KeyManager keyManager;
	private EncoderDecoderUtil encoderDecoder;

	// Load SunJCE provider
	static {
		Provider sunJce = new com.sun.crypto.provider.SunJCE();
		Security.addProvider(sunJce);
	}

	public EncryptionServiceImpl() {
	}

	/**
	 * Gets the encryption algorithm.
	 */
	public String getAlgorithm() {
		return algorithm;
	}

	/**
	 * Sets the encryption algorithm.
	 */
	public void setAlgorithm(String algorithm) {
		this.algorithm = algorithm;
	}

	/**
	 * Gets the key size from the configuration file.
	 */
	public int getKeySize() {
		return keySize;
	}

	/**
	 * Gets the key size from the configuration file.
	 */
	public void setKeySize(int keySize) {
		this.keySize = keySize;
	}

	/**
	 * Checks to see the encryption is enabled or not.
	 */
	public boolean isEncryptionEnabled() {
		return encryptionFlag;
	}

	/**
	 * Sets the encryption flag.
	 */
	public void setEncryptionFlag(boolean encryptionFlag) {
		this.encryptionFlag = encryptionFlag;
	}

	/**
	 * Gets the default Character Encoding.
	 */
	public String getDefaultCharacterEncoding() {
		return defaultCharacterEncoding;
	}

	/**
	 * Sets the default Character Encoding.
	 */
	public void setDefaultCharacterEncoding(String defaultCharacterEncoding) {
		this.defaultCharacterEncoding = defaultCharacterEncoding;
	}

	/**
	 * Gets the KeyManager object.
	 */
	public KeyManager getKeyManager() {
		return keyManager;
	}

	/**
	 * Sets the KeyManager object.
	 */
	public void setKeyManager(KeyManager keyManager) {
		this.keyManager = keyManager;
	}

	/**
	 * Checks to see if all the properties are set.
	 */
	public void afterPropertiesSet() throws Exception {
		if (StringUtils.isEmpty(algorithm)) {
			throw new EncryptionServiceException("encryption algorithm is not set.");
		}
		if (keyManager == null) {
			throw new EncryptionServiceException("key manager is not set.");
		}
		if (StringUtils.isEmpty(defaultCharacterEncoding)) {
			throw new EncryptionServiceException("default character encoding is not set.");
		}

		Validate.notNull(encoderDecoder, "encoderDecoder is null");
	}

	/**
	 * Encrypt the given text using the given secret key. uses default crypto
	 * suffic for padding
	 * 
	 * @param key
	 *            the secret key
	 * @param clearText
	 *            the test to be encrypted
	 * @param characterEncoding
	 *            the character encoding
	 * @return the encrypted text
	 * @throws EncryptionServiceException
	 *             thrown when there was an encryption error
	 */
	public String encrypt(SecretKey key, String characterEncoding, String clearText)
			throws EncryptionServiceException {
		return encrypt(key, characterEncoding, clearText, EncryptionService.CRYPTO_SUFFIX);
	}

	/**
	 * 
	 * @param key
	 * @param characterEncoding
	 * @param clearText
	 * @param cryptoSuffix
	 * @return
	 * @throws EncryptionServiceException
	 */
	public String encrypt(SecretKey key, String characterEncoding, String clearText,
			String cryptoSuffix) throws EncryptionServiceException {
		String encrypted = clearText;
		if (key != null && (clearText != null && clearText.length() > 0)) {
			try {
				Cipher cipher = Cipher.getInstance(key.getAlgorithm());

				cipher.init(Cipher.ENCRYPT_MODE, key);

				if (!StringUtils.isEmpty(cryptoSuffix)) {
					clearText = clearText + cryptoSuffix;
				}

				if (StringUtils.isEmpty(characterEncoding)) {
					characterEncoding = defaultCharacterEncoding;
				}
				// Encode the string into bytes
				byte[] utf8 = clearText.getBytes(characterEncoding);

				// Encrypt
				byte[] enc = cipher.doFinal(utf8);

				// Encode bytes to base64 to get a string
				encrypted = encoderDecoder.encode(enc);
			} catch (Exception ex) {
				throw new EncryptionServiceException("error while encrypting" + ex);
			}
		}
		return encrypted;
	}

	/**
	 * default decrypt applies the default crypto suffix string
	 */
	public String decrypt(SecretKey key, String characterEncoding, String encryptedText)
			throws EncryptionServiceException {
		return decrypt(key, characterEncoding, encryptedText, EncryptionService.CRYPTO_SUFFIX);
	}

	/**
	 * Decrypt the given text using the given secret key.
	 * 
	 * @param key
	 *            the secret key
	 * @param encryptedText
	 *            the encrypted text
	 * @param characterEncoding
	 *            the character encoding
	 * @return the decrypted text
	 * @throws DecryptionException
	 *             thrown when there was a decryption error
	 */
	public String decrypt(SecretKey key, String characterEncoding, String encryptedText,
			String cryptoSuffix) throws EncryptionServiceException {
		String decryptedText = encryptedText;
		if (key != null && (encryptedText != null && encryptedText.length() > 0)) {
			try {
				Cipher cipher = Cipher.getInstance(key.getAlgorithm());
				cipher.init(Cipher.DECRYPT_MODE, key);

				// Decode base64 to get bytes
				byte[] dec = encoderDecoder.decode(encryptedText);
				// Decrypt
				byte[] utf8 = cipher.doFinal(dec);

				if (StringUtils.isEmpty(characterEncoding)) {
					characterEncoding = defaultCharacterEncoding;
				}
				// Decode bytes to the string
				String clearText = new String(utf8, characterEncoding);

				if (!StringUtils.isEmpty(cryptoSuffix)) {
					if (!clearText.endsWith(cryptoSuffix)) {
						throw new EncryptionServiceException("decryption error.");
					}

					// Remove the suffix
					decryptedText = clearText.substring(0, clearText.lastIndexOf(cryptoSuffix));
				} else {
					decryptedText = clearText;
				}
			} catch (Exception ex) {
				throw new EncryptionServiceException("error while decrypting the text", ex);
			}
		}
		return decryptedText;
	}

	/********************************************************************************/
	/************************ Private/protected Methods *****************************/
	/********************************************************************************/

	/**
	 * Gets/creates the secret key. If the new key created, it will be added to
	 * the key cache.
	 * 
	 * @param keyCache
	 *            the key cache.
	 * @param createSecretKey
	 *            a flag to indicate the creation of the new key. If the this
	 *            flag is <code>true</code> and key is not present in the key
	 *            cache a new key will be created. If this flag is
	 *            <code>false</code> and key is not present in the key cache
	 *            null will be returned.
	 * 
	 * @return the key from the key cache or a newl key.
	 * @throws EncryptionServiceException
	 *             when there is an error while creating a new key or the key
	 *             manager is null
	 */
	protected SecretKey getSecretKey(KeyCache keyCache, boolean createSecretKey)
			throws EncryptionServiceException {
		if (keyManager == null) {
			throw new EncryptionServiceException("keyManager is null");
		}
		SecretKey secretKey = keyManager.getKey(keyCache);
		if (secretKey == null && createSecretKey == true) {
			secretKey = createSecretKey();
			keyCache.setKey(secretKey);
		}
		return secretKey;
	}

	/**
	 * Generates a secret key using the algorithm and key size from the
	 * configuration. If the keysize is not found, the secret key is generated
	 * using the default key size.
	 * 
	 * @return the secret key
	 * @throws EncryptionServiceException
	 *             thrown when the algorithm is not supported
	 */
	private SecretKey createSecretKey() throws EncryptionServiceException {
		try {
			if (algorithm == null) {
				throw new EncryptionServiceException("algorithm is not initialised");
			}
			KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithm);
			SecureRandom sRandom = SecureRandom.getInstance(SECURE_RANDOM_ALGORITHM, PROVIDER);
			if (keySize > 0) {
				keyGenerator.init(keySize, sRandom);
			} else {
				keyGenerator.init(sRandom);
				logger.trace("default keysize is used.");
			}
			return keyGenerator.generateKey();
		} catch (NoSuchAlgorithmException ex) {
			logger.trace("error generating secret key.", ex);
			throw new EncryptionServiceException("error generating secret key.", ex);
		} catch (NoSuchProviderException ex) {
			logger.trace("error generating secret key.", ex);
			throw new EncryptionServiceException("error generating secret key.", ex);
		}
	}

	public EncoderDecoderUtil getEncoderDecoder() {
		return encoderDecoder;
	}

	public void setEncoderDecoder(EncoderDecoderUtil encoderDecoder) {
		this.encoderDecoder = encoderDecoder;
	}
}
