package gov.va.med.config;

import static org.owasp.esapi.ESAPI.validator;

import java.util.Properties;

import javax.naming.NamingException;
import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AvailableSettings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.PropertySource;
import org.springframework.core.env.Environment;
import org.springframework.jndi.JndiTemplate;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;

/**
 * 
 * @author James Garnhart
 * 
 */
@Configuration
@EnableTransactionManagement
@PropertySource(value = { "classpath:env/server.properties" })
public class HibernateConfig {

	@Autowired
	private Environment env;

	/**
	 * Initialize dataSource
	 * 
	 * @return DataSource
	 * @throws NamingException 
	 */
	
	private Logger logger = Logger.getLogger(getClass());
	
	public DataSource getDataSource() throws NamingException {
		JndiTemplate jndiTemplate = new JndiTemplate();
		String jdbcUrl = env.getProperty("jdbc.url");
	
		 //Fix Fortify Issues
		ESAPIValidator.validateStringInput(jdbcUrl, ESAPIValidationType.JdbcUrl_Whitelist);
		
		return (DataSource) jndiTemplate.lookup(jdbcUrl);
	}

	/**
	 * Initialize hibernate properties
	 * 
	 * @return Properties
	 */
	private Properties getHibernateProperties() { 
		Properties properties = new Properties();
		properties.put(AvailableSettings.DIALECT, env.getRequiredProperty("hibernate.dialect"));
		properties.put(AvailableSettings.SHOW_SQL, env.getRequiredProperty("hibernate.show_sql"));
		properties.put(AvailableSettings.DEFAULT_SCHEMA, env.getRequiredProperty("hibernate.default_schema"));
		properties.put(AvailableSettings.STATEMENT_BATCH_SIZE, env.getRequiredProperty("hibernate.batch.size"));
		properties.put(AvailableSettings.HBM2DDL_AUTO, env.getRequiredProperty("hibernate.hbm2ddl.auto"));
		properties.put(AvailableSettings.CURRENT_SESSION_CONTEXT_CLASS, env.getRequiredProperty("hibernate.current.session.context.class"));
		return properties;
	}
	
	@Bean
	public LocalSessionFactoryBean getSessionFactory() throws NamingException {
		LocalSessionFactoryBean sessionFactory = new LocalSessionFactoryBean();
		sessionFactory.setDataSource(getDataSource());
		sessionFactory.setPackagesToScan(
			new String[] { 
					"gov.va.med.ccht.controller",
					"gov.va.med.ccht.model",
					"gov.va.med.ccht.persistent.hibernate" ,
					"gov.va.med.ccht.service.common.impl",
					"gov.va.med.ccht.service.dmpreport.impl",
					"gov.va.med.ccht.service.qir.impl",
					"gov.va.med.ccht.service.report.impl",
					"gov.va.med.fw.mail",
					"gov.va.med.fw.scheduling",
					"gov.va.med.fw.security",
					"gov.va.med.fw.model"
			});
		sessionFactory.setHibernateProperties(getHibernateProperties());
		return sessionFactory;
	}


	/**
	 * Initialize Transaction Manager
	 * 
	 * @param sessionFactory
	 * @return HibernateTransactionManager
	 */
	@Bean
	public HibernateTransactionManager transactionManager(SessionFactory sessionFactory) {
		HibernateTransactionManager txManager = new HibernateTransactionManager();
		txManager.setSessionFactory(sessionFactory);
		return txManager;
	}
}
