package gov.va.med.ccht.ui.interceptor;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.httpclient.util.DateUtil;
import org.hibernate.Hibernate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.view.RedirectView;

import gov.va.med.ccht.controller.SecurityController;
import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.ui.ajax.AjaxRequestHandler;
import gov.va.med.ccht.ui.common.UserNotifier;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.ccht.ui.util.WebUtil;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.UserPrincipal;

@Component
public class CommonReferenceDataInterceptor implements HandlerInterceptor, InitializingBean {
	private static final Logger logger = LoggerFactory.getLogger(CommonReferenceDataInterceptor.class);

	public static final String ONE_TIME_USER_NOTIFICATION = "oneTimeUserNotification";

	@Value("${useMinifiedDependencies}")
	private String useMinifiedDependenciesStr;
	@Value("${session.pollingIntervalSeconds}")
	private String sessionPollingIntervalSecondsStr;
	@Value("${session.failedRequestsCount}")
	private String sessionFailedRequestsCountStr;
	@Value("${session.heartBeatTimeoutMillis}")
	private String sessionHeartBeatTimeoutMillisStr;
	@Value("${session.idleAfterSeconds}")
	private String sessionIdleAfterSecondsStr;
	@Value("${logout.manual.url}")
	private String logoutManualUrl;
	@Value("${logout.timeout.url}")
	private String logoutTimeoutUrl;
	
	@Autowired
	private SecurityService securityService;

	private boolean useMinifiedDependencies;
	private int sessionPollingIntervalSeconds;
	private int sessionFailedRequestsCount;
	private int sessionHeartBeatTimeoutMillis;
	private int sessionIdleAfterSeconds;

	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
			throws Exception {
		return true;
	}

	@Override
	public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
			throws Exception {
	}

	protected boolean isAjax(HttpServletRequest request) {
		return AjaxRequestHandler.isAjax(request);
	}

	@Override
	public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
			ModelAndView modelAndView) throws Exception {
		if (isAjax(request)) {
			logger.debug("Request was an AJAX request, no need to populate reference data");
			return;
		}

		if (modelAndView == null) {
			logger.debug("No modelAndView returned, the controller must have dealt with the response manually.");
			return;
		}

		Map<String, Object> model = modelAndView.getModel();
		if (model == null) {
			logger.error(
					"No model in the ModelAndView, that is unexpected. Returning without adding common reference data.");
			return;
		}

		if (modelAndView.getView() instanceof RedirectView
				|| (modelAndView.isReference() && modelAndView.getViewName().startsWith("redirect:"))) {
			logger.debug("View is null or a redirect, no need to populate reference data");
			return;
		}

		request.setAttribute("logoutManualUrl", logoutManualUrl);

		HttpSession session = request.getSession();

		session.setAttribute("sessionPollingIntervalSeconds", new Integer(sessionPollingIntervalSeconds));
		session.setAttribute("sessionFailedRequestsCount", new Integer(sessionFailedRequestsCount));
		session.setAttribute("sessionHeartBeatTimeoutMillis", new Integer(sessionHeartBeatTimeoutMillis));
		session.setAttribute("sessionIdleAfterSeconds", new Integer(sessionIdleAfterSeconds));
		session.setAttribute("logoutTimeoutUrl", logoutTimeoutUrl);

		String val = UserNotifier.getAndRemoveAttributeFromSession(session);
		if (val != null)
			model.put(ONE_TIME_USER_NOTIFICATION, val);

		/*
		 * Add all roles and permissions to model for standard use in
		 * <security:authorize> JSP tags - CPB
		 */
		WebUtil.addClassConstantsToModel(Permission.class, model, true);

		/*
		 * Add all date util constants as they are defined (without adding
		 * "DATE_UTIL_" before their names) - CPB
		 */
		WebUtil.addClassConstantsToModel(DateUtil.class, model, false);

		/* Add constants from WebUtils for global error display */
		WebUtil.addClassConstantsToModel(WebUtil.class, model);

		// waiting on Spring 4 - CPB
		// model.put("AJAX_CONTEXT_PATH_PREFIX",
		// CommonWebConfig.AJAX_CONTEXT_PATH_PREFIX);

		model.put("useMinifiedDependencies", useMinifiedDependencies);
		
		if(getCurrentUserAsOrNull(User.class) != null) {
			UserPrincipal currentUser = getCurrentUser();
			Integer visnId = null;
			Integer facilityId = null;
			if (currentUser.isPermissionGranted(SecurityController.NATIONAL_ADMIN)) {
				
				if(!securityService.getNewRegistrations(visnId, facilityId).isEmpty()) {
					model.put("notifyPendingRegistrations", true);
				}
				else {
					model.put("notifyPendingRegistrations", false);
				}
			} else if (currentUser.isPermissionGranted(SecurityController.VISN_ADMIN)) {
				visnId = ((User) currentUser).getVisn().getId();
				if(!securityService.getNewRegistrations(visnId, facilityId).isEmpty()) {
					model.put("notifyPendingRegistrations", true);
				}
				else {
					model.put("notifyPendingRegistrations", false);
				}
			} else if (currentUser.isPermissionGranted(SecurityController.FACILITY_ADMIN)) {
				facilityId = ((User) currentUser).getFacility().getId();
				if(!securityService.getNewRegistrations(visnId, facilityId).isEmpty()) {
					model.put("notifyPendingRegistrations", true);
				}
				else {
					model.put("notifyPendingRegistrations", false);
				}
			}
		}
		
		if (!model.containsKey("user"))
			model.put("user", getCurrentUserAsOrNull(User.class));
		if (!model.containsKey("userName"))
			model.put("userName", getCurrentUsername());
		if (!model.containsKey("userDisplayName"))
			model.put("userDisplayName", getCurrentUserDisplayName());

	}

	public static String getCurrentUserDisplayName() {
		User currentUser = getCurrentUserAsOrNull(User.class);
		if (currentUser == null)
			return "";

		StringBuilder userName = new StringBuilder();
		userName.append(currentUser.getLastName());
		userName.append(", ");
		userName.append(currentUser.getFirstName());

		String middleName = currentUser.getMiddleName();
		if (middleName != null && middleName.length() > 0) {
			userName.append(" ");
			userName.append(middleName);
		}

		return userName.toString();
	}

	@SuppressWarnings("unchecked")
	public static <T extends UserPrincipal<T>> T getCurrentUserAs(Class<T> type) {
		return (T) getCurrentUser();
	}

	@SuppressWarnings("unchecked")
	public static <T extends UserPrincipal<T>> T getCurrentUserAsOrNull(Class<T> type) {
		UserPrincipal<?> userPrincipal = SecurityContextHelper.getSecurityContext().getUserPrincipal();
		if (userPrincipal == null)
			return null;
		if (type.isAssignableFrom(Hibernate.getClass(userPrincipal)))
			return (T) userPrincipal;
		return null;
	}

	public static UserPrincipalImpl getCurrentUser() {
		@SuppressWarnings("rawtypes")
		UserPrincipal p = SecurityContextHelper.getSecurityContext().getUserPrincipal();
		if (p == null)
			return null;
		if (p instanceof User)
			return (User) p;
		throw new IllegalStateException("Unsupported type of user in security context: " + p.getClass());
	}

	public static String getCurrentUsername() {
		User currentUser = getCurrentUserAsOrNull(User.class);
		if (currentUser == null || currentUser.getUserCredentials() == null)
			return null;
		return currentUser.getUserCredentials().getUserID();
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		useMinifiedDependencies = Boolean.parseBoolean(useMinifiedDependenciesStr);
		sessionPollingIntervalSeconds = Integer.parseInt(sessionPollingIntervalSecondsStr);
		sessionFailedRequestsCount = Integer.parseInt(sessionFailedRequestsCountStr);
		sessionHeartBeatTimeoutMillis = Integer.parseInt(sessionHeartBeatTimeoutMillisStr);
		sessionIdleAfterSeconds = Integer.parseInt(sessionIdleAfterSecondsStr);
	}

}
