/**
 * 
 */
package gov.va.med.ccht.service.qir.impl;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.CCHTPermissions;
import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRAttachment;
import gov.va.med.ccht.model.qir.QIRSearchParameters;
import gov.va.med.ccht.model.qir.QIRSearchResult;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.qir.QIRStatusTypeEnum;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.persistent.QIRDAO;
import gov.va.med.ccht.service.common.MergeService;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.impl.AbstractBusinessService;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.StringUtils;

/**
 * @author DNS
 *
 */
@Service
public class QIRServiceImpl extends AbstractBusinessService implements QIRService {

	public final int SATURDAY = 7;
	public final int SUNDAY = 1;

	private String qirNotificationJMSPayLoadName;
	
	@Autowired
	@Qualifier("qirNotificationMailMessage")
	private SimpleMailMessage qirNotificationMessage;
	@Autowired
	@Qualifier("qirVendorResponseDueMailMessage")
	private SimpleMailMessage qirVendorResponseDueMailMessage;
	@Autowired
	private SecurityService securityService;
	@Autowired
	private MergeService mergeService;
	@Autowired
	private QIRDAO qirDAO;
	
	public String getQirNotificationJMSPayLoadName() {
		return qirNotificationJMSPayLoadName;
	}

	public void setQirNotificationJMSPayLoadName(String qirNotificationJMSPayLoadName) {
		this.qirNotificationJMSPayLoadName = qirNotificationJMSPayLoadName;
	}

	@Override
	public QIR getQIR(final long id) throws ServiceException {
		return qirDAO.findById(id);
	}

	public Date getDueDateFromHistory(String qirId) throws ServiceException {
		Date vendorResponseDueDate = null;

		if (qirId != null) {
			vendorResponseDueDate = qirDAO.getVendorResponseDueDate(Long.parseLong(qirId));
		}
		return vendorResponseDueDate;
	}

	@Override
	public QIRAttachment getQIRAttachment(final long attachmentId) throws ServiceException {
		return qirDAO.findAttachmentById(attachmentId);
	}

	@Override
	public List<QIRSearchResult> searchQIR(QIRSearchParameters qirSearchParameters) throws ServiceException {
		User user = getCurrentUser();
		if (user.isPermissionGranted(CCHTPermissions.VENDOR)) {
			Vendor vendor = user.getVendor();
			if (vendor != null) {
				qirSearchParameters.setVendor(new TermType(vendor.getName(), vendor.getCode()));
				qirSearchParameters.setIsVendor(true);
			} else {
				qirSearchParameters.setVendor(new TermType("_INVALID_MISSING_VENDOR", "-1"));
			}
		}
		else
		{
			qirSearchParameters.setIsVendor(false);
		}
		return qirDAO.searchQIR(qirSearchParameters);
	}

	public void updateQIR(QIR qir) throws ServiceException {
		try {
			User user = getCurrentUser();

			QIRStatusType existingStatus = new QIRStatusType();
			existingStatus.setCode(QIRStatusType.NEW);
			if (qir.getId() != null) {
				QIR onFile = qirDAO.findById(qir.getId());
				existingStatus = onFile.getQirStatusType();
			}

			QIRStatusType newStatus = qir.getQirStatusType();

			boolean updatedDueDate = false;
			if (existingStatus.isNew() && qir.isApproved()) {
				updatedDueDate = true;
			}
			if (existingStatus.isAgreed() && newStatus.isAgreed()
					&& user.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN)) {
				updatedDueDate = true;
			}
			if (existingStatus.isReplied() && newStatus.isReplied() // case NA updates Replied QIR
					&& user.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN)) {
				updatedDueDate = true;
			}
			if (existingStatus.isApproved() && newStatus.isApproved() // case NA updates approved QIR
					&& user.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN)) {
				updatedDueDate = true;
			}

			if (qir.isSameVendor(user)
					&& (existingStatus.isApproved() || existingStatus.isAgreed() || existingStatus.isReplied())
					&& newStatus.isReplied()) {
				qir.setVendorResponseDueDate(null);
				qir.setEmailReminderSentDate(null);
			}
			
			if (updatedDueDate) {
				/**
				 * Starts a vendor response due date clock on the next day email
				 * remind sent date to null
				 */
				Date start = DateUtils.getWeekDate();
				
				qir.setVendorResponseDueDate(DateUtils.forwardBusinessDays(start, 10));
				qir.setEmailReminderSentDate(null);
				Date end = qir.getVendorResponseDueDate();		

				Calendar cld = Calendar.getInstance();
				
				int thisYear = cld.get(Calendar.YEAR);
				List<FederalHoliday> holidays = this.securityService.getHolidays(thisYear);
				int currentMonth = cld.get(Calendar.MONTH);
				
				// if QIR is created end of current year and is due in preceding year because holiday has 
				// current years year and will not see it.
				if(currentMonth == Calendar.DECEMBER) { // works
					int nextYear = cld.get(Calendar.YEAR) + 1;
					List<FederalHoliday> nextYearsHolidays = this.securityService.getHolidays(nextYear);
					holidays.addAll(nextYearsHolidays);
				}
				
				/**
				 * If a federal holiday falls between a date when a vendor
				 * response due date is set and a day when a vendor response is
				 * due, forward a vendor response due date by one day. If a
				 * vendor response due date falls on either Saturday or Sunday,
				 * forward a vendor response due date by one day.
				 */
				
				for (FederalHoliday holiday : holidays) {
					Date fdate = DateUtils.truncate(holiday.getDate(), Calendar.DAY_OF_MONTH);
					
					if (DateUtils.isDateBetweenIgnoreTime(fdate, start, end)) {
						
						cld.setTime(qir.getVendorResponseDueDate());
						DateUtils.add(cld);
						qir.setVendorResponseDueDate(cld.getTime());
					}
				}

			}
			if (qir.getId() != null) {
				QIR onFile = (QIR) qirDAO.findById(qir.getId());
				mergeService.mergeQIR(qir, onFile);
				QIR persisted = this.saveQIR(onFile);
				if (persisted.getId() != null && onFile.getId() == null) {
					onFile.setId(persisted.getId());
				}
				sendQIRNotificationMail(onFile);
			} else {
				// Fill the audit fields
				qir.setRecordModifiedCount((short) 1);
				qir.setRecordCreatedBy(user.getUsername());
				qir.setRecordModifiedBy(user.getUsername());
				final Date currentDate = DateUtils.getCurrentDateTime();
				qir.setRecordCreatedDate(currentDate);
				qir.setRecordModifiedDate(currentDate);
				QIR persisted = this.saveQIR(qir);
				if (persisted.getId() != null && qir.getId() == null) {
					qir.setId(persisted.getId());
				}
				sendQIRNotificationMail(qir);
			}
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public QIR saveQIR(QIR qir) throws ServiceException {
		return qirDAO.saveQIR(qir);
	}

	public void updateQIRAttachment(QIRAttachment qirAttachment) throws ServiceException {
		qirDAO.updateAttachment(qirAttachment);
	}

	public void deleteQIRAttachment(long qirAttachmentId) throws ServiceException {
		
		QIRAttachment onFile = qirDAO.findAttachmentById(qirAttachmentId);
		
		final String user = getCurrentUser().getName();
		final String userCreatedAttachemnt = onFile.getRecordCreatedBy();
		
		if (user.equalsIgnoreCase(userCreatedAttachemnt)) {

			qirDAO.deleteAttachment(qirAttachmentId);

		} else {

			throw new ServiceException("The user " + user
					+ " is not authorized to delete a record created by " + userCreatedAttachemnt);

		}

	}

	public Integer generateVendorResponseDueNotifications() throws ServiceException {
		try {
			List<String> qirIds = qirDAO.getVendorResponseDueQIRIds();
			
			for (String id : qirIds) {
				sendVendorResponseDueEMail(id);
			}
			
			return qirIds.size();
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}

	public void sendVendorResponseDueEMail(String qirId) throws ServiceException {
		
		try {
			QIR qir = qirDAO.findById(Long.parseLong(qirId));
			sendVendorResponseDueNotification(qir);
			qir.setEmailReminderSentDate(new Date());
			qirDAO.saveQIR(qir);
		} catch (Exception e) {
			throw new ServiceException(e.getMessage(), e);
		}
	}
	
	@Override
	public QIRStatusType findQIRStatusTypeByName(String name) {
		return qirDAO.findQIRStatusTypeByName(name);
	}

	private void sendQIRNotificationMail(QIR qir) {
		
		
		try {
			List<User> userList = new ArrayList<User>();
			String qirStatusCode = qir.getQirStatusType().getCode();

			List<User> nationalAdmins = getNationalAdmins();
			List<User> visnAdmins = getVisnAdmins(qir.getVisn());
			List<User> facilityAdmins = getFacilityAdmins(qir.getFacility());
			List<User> submitters = getSubmitter(qir.getSubmittedBy());

			if (nationalAdmins != null)
				userList.addAll(nationalAdmins);
			if (visnAdmins != null)
				userList.addAll(visnAdmins);
			if (facilityAdmins != null)
				userList.addAll(facilityAdmins);
			if (submitters != null)
				userList.addAll(submitters);

			if (QIRStatusTypeEnum.New.toString().equalsIgnoreCase(qirStatusCode) || 
					QIRStatusTypeEnum.Withdrawn.toString().equalsIgnoreCase(qirStatusCode)) {
				// default list
			} else {
				List<User> vendors = getVendors(qir.getVendor());
				List<User> nac = getNAC();
				if (vendors != null)
					userList.addAll(vendors);
				if (nac != null)
					userList.addAll(nac);
			}

			// Only active users will receive e-mails
			List<String> emailList = getEmailList(userList);
			if (emailList.size() > 0) {
				Map<String, Object> data = new HashMap<String, Object>();
				data.put("qir", qir);
				data.put("webAppUrl", getWebAppUrl());
				if (submitters != null && submitters.size() > 0) {
					User submitter = submitters.get(0);
					qir.setSubmittedByName(submitter.getFullName());
					qir.setEmailSubmittedByName(submitter.getFirstName() + " " + submitter.getLastName());
				}
				UserPrincipal currentUser = getCurrentUser();
				if (currentUser != null) {
					qir.setModifiedByName(getCurrentUser().getFullName());
				}
				SimpleMailMessage template = processMailTemplate(data, qirNotificationMessage);
				String[] toAddress = new String[emailList.size()];
				for (int i = 0; i < emailList.size(); i++) {
					toAddress[i] = emailList.get(i);
				}
				template.setTo(toAddress);
				send(template);
			}
		} catch (ServiceException e) {
			logger.error("QIR Service : sendQIRNotificationMail Failed", e);
		}
	}

	private void sendVendorResponseDueNotification(QIR qir) {
		
		try {
			List<User> userList = new ArrayList<User>();
			List<User> nationalAdmins = getNationalAdmins();
			List<User> vendors = getVendors(qir.getVendor());
			List<User> nac = getNAC();
			List<User> visnAdmins = getVisnAdmins(qir.getVisn());
			List<User> facilityAdmins = this.getFacilityAdmins(qir.getFacility());
			List<User> submitter = this.getSubmitter(qir.getSubmittedBy());

			if (nationalAdmins != null)
				userList.addAll(nationalAdmins);
			if (vendors != null)
				userList.addAll(vendors);
			if (nac != null)
				userList.addAll(nac);
			if (visnAdmins != null)
				userList.addAll(visnAdmins);
			if (facilityAdmins != null)
				userList.addAll(facilityAdmins);
			if (submitter != null)
				userList.addAll(submitter);

			// Only active users will receive e-mails
			List<String> emailList = getEmailList(userList);

			if (logger.isDebugEnabled()) {
				logger.debug("Sending e-mail to the following: " + emailList + " for QIR " + qir.getId());
			}
			if (emailList.size() > 0) {
				Map<String, Object> data = new HashMap<String, Object>();
				data.put("qir", qir);
				data.put("webAppUrl", getWebAppUrl());
				SimpleMailMessage template = processMailTemplate(data, qirVendorResponseDueMailMessage);
				String[] toAddress = new String[emailList.size()];
				for (int i = 0; i < emailList.size(); i++) {
					toAddress[i] = emailList.get(i);
				}
				template.setTo(toAddress);

				// Send a message with no attachment
				send(template, null, Priority.HIGH);
				if (logger.isDebugEnabled()) {
					logger.debug(
							"E-mail was sent successfully to the following: " + emailList + " for QIR " + qir.getId());
				}
			}
		} catch (ServiceException e) {
			logger.error("QIR Service : sendVendorResponseDueNotification Failed", e);
		}
	}

	private List<User> getNAC() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.NAC);
		searchCriteria.setRoles(roles);
		return securityService.findAppUsers(searchCriteria);
	}

	private List<User> getNationalAdmins() throws ServiceException {
		SearchCriteria searchCriteria = new SearchCriteria();
		List<String> roles = new ArrayList<String>();
		roles.add(CCHTRoles.NATIONAL_ADMIN);
		searchCriteria.setRoles(roles);
		return securityService.findAppUsers(searchCriteria);
	}

	private List<User> getVisnAdmins(SimpleVisn visn) throws ServiceException {
		if (visn != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String>();
			roles.add(CCHTRoles.VISN_ADMIN);
			searchCriteria.setRoles(roles);
			List<String> visns = new ArrayList<String>();
			visns.add(visn.getCode());
			searchCriteria.setVisns(visns);
			List<User> admins = securityService.findAppUsers(searchCriteria);
			return admins;
		} else {
			return new ArrayList<User>();
		}
	}

	private List<User> getFacilityAdmins(SimpleFacility facility) throws ServiceException {
		if (facility != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String>();
			roles.add(CCHTRoles.FACILITY_ADMIN);
			searchCriteria.setRoles(roles);
			List<String> facilities = new ArrayList<String>();
			facilities.add(facility.getCode());
			searchCriteria.setStations(facilities);
			List<User> admins = securityService.findAppUsers(searchCriteria);
			return admins;
		} else {
			return new ArrayList<User>();
		}
	}

	private List<User> getVendors(Vendor vendor) throws ServiceException {
		if (vendor != null) {
			SearchCriteria searchCriteria = new SearchCriteria();
			List<String> roles = new ArrayList<String>();
			roles.add(CCHTRoles.VENDOR);
			searchCriteria.setRoles(roles);
			List<String> vendors = new ArrayList<String>();
			vendors.add(vendor.getName());
			searchCriteria.setVendors(vendors);
			List<User> vendorpocs = securityService.findAppUsers(searchCriteria);
			return vendorpocs;
		} else {
			return new ArrayList<User>();
		}
	}

	private List<User> getSubmitter(String username) throws ServiceException {
		if (StringUtils.isNotEmpty(username)) {
			SearchCriteria searchCriteria = new SearchCriteria();
			searchCriteria.setSAMAccountName(username);
			return securityService.findAppUsers(searchCriteria);
		} else {
			return new ArrayList<User>();
		}
	}

	@Override
	public List<QIRStatusType> getAllQIRStatusTypes() {
		return qirDAO.findAllQIRStatusTypes();
	}

}
