/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.service.common.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.ldap.CommunicationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.SimpleUser;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.ccht.service.common.MergeService;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.RolePermission;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.ldap.LdapPersonService;

/**
 * Implements the security services to manage system users and user profiles
 * 
 * @author DNS
 */
@Service
public class SecurityServiceImpl extends AbstractComponent implements SecurityService {

	@Autowired
	private LdapPersonService ldapPersonService;
	@Autowired
	private SecurityDAO securityDAO;
	@Autowired
	private MergeService mergeService;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getRole(gov.va.med.fw
	 * .model.EntityKey)
	 */
	public Role getRole(EntityKey<?> entityKey) throws ServiceException {
		Validate.notNull(entityKey);
		Role role = (Role) securityDAO.getByKey(entityKey);
		return role;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getRole(java.lang.String
	 * )
	 */
	public Role getRole(String roleName) throws ServiceException {
		Validate.notNull(roleName);
		try {
			return (Role) securityDAO.getRoleByName(roleName);
		} catch (Exception e) {
			throw new ServiceException("SecurityServiceImpl.getRole: Error while retrieving Role", e);
		}
	}
	
	@Override
	public Role getRoleById(long id) {
		return securityDAO.getRoleById(id);
	}	
	
	/* * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#getUser(java.lang.String
	 * )
	 */
	public User getUser(String userName) throws ServiceException {
		Validate.notNull(userName);
		User user = null;
		// Get the user from database
		try {
			 user = (User) securityDAO.getUserByName(userName);
		} catch (UsernameNotFoundException e) {
		}
		catch(Exception e) {
		
		}

		// if user doesn't exist in DB create atem user for creation and
		// return
		if (user == null) {
			 return null;
		}

		return user;
	}

	public LdapPerson getLdapUser(String samAcountName) throws ServiceException {
		return ldapPersonService.findBySamAccountName(samAcountName);
	}

	public Boolean authenticateLdapUser(String userDn, String password) throws ServiceException {
		try {
			if (ldapPersonService.authenticate(userDn, password)) {
				return true;
			} else {
				return false;
			}
		} catch (CommunicationException e) {
			throw new ServiceException("Authentication Failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#createUser(gov.va.med
	 * .ccht.service.security.User)
	 */
	public void createUser(User user) throws ServiceException {
		// save user in the ciss database
		try {
			 // make sure user is a valid ldap user
			 LdapPerson ldapPerson = ldapPersonService.findBySamAccountName(user.getName());
			 if (ldapPerson == null) {
					throw new ServiceException("Invalid User credentials");
			 }
			// TODO: Do we need this? Is this an OHRS requirement?
			 //user.setDisabled(ldapPerson.isDisabled());
			 
			 // update the username to ldap returned user id
			 user.setName(ldapPerson.getSamAccountName());
			 user.setTelephoneNumber(ldapPerson.getTelephoneNumber());
			 user.setEmail(ldapPerson.getEmail());
			 // create the new user in the ciss database
			// TODO: Do we need this? Is this an OHRS requirement?
			 //updateRoles(user);
			 securityDAO.createUser(user);
		} catch (DAOException e) {
			 throw new ServiceException("SecurityServiceImpl.createUser: Error while creating the User", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateUser(gov.va.med
	 * .ccht.service.security.User)
	 */
	public void updateUser(User user) throws ServiceException {
		try {
			// update user in ccht database
			securityDAO.updateUser(user);
		} catch (Exception e) {
			logger.error("Error while saving User " + user.getUsername() + " ID: " + user.getId(), e);
			throw new ServiceException("Error while saving User " + user.getUsername() + " ID: " + user.getId(), e);
		}
	}
	
	public void createRole(Role role) throws ServiceException {
		try {
			 securityDAO.persist(role);
		} catch (DataAccessException dae) {
			 throw new ServiceException("SecurityServiceImpl.createRole: failed", dae);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateRole(gov.va.med
	 * .ccht.service.security.cchtRole)
	 */
	public void updateRole(Role role) throws ServiceException {

		try {
			List<SimpleRole> allRoles = securityDAO.findAllSimpleRoles();
			for (SimpleRole r : allRoles) {
				if (r.getName().trim().equalsIgnoreCase(role.getName())
						&& (role.getId() == null || !role.getId().equals(r.getId())))
					throw new ServiceException("Duplicate role name found");
			}
			
			if(role.getInternalPermissions() == null){
				Set<RolePermission> emptySet = new HashSet<RolePermission>();
				role.setInternalPermissions(emptySet);
				
			}
			
			if (role.getId() == null) {
				
				securityDAO.updateRole(role);
			} else {

				Role oldRole = securityDAO.getRoleById(role.getId());
				securityDAO.deleteRolePermissionsFromRole(oldRole); // must delete old role Permissions first
				securityDAO.updateRole(role);
			}
				
		} catch (DAOException e) {
			throw new ServiceException("SecurityServiceImpl.updateRole: failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#deleteRole(gov.va.med
	 * .ccht.model.security.cchtRole)
	 */
	public void deleteRole(Role role) throws ServiceException {
		// TODO
		try {
			 securityDAO.remove(role);
		} catch (Exception e) {
			 throw new ServiceException("SecurityServiceImpl.deletRole failed", e);
		}
	}
	
	@Override
	public List<Role> getAllRoles() throws ServiceException {
		try {
			return securityDAO.findAllRoles();

		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoleNames failed", e);
		}
	}

	public List<FederalHoliday> getHolidays(int year) throws ServiceException {
		try {
			return securityDAO.findHolidays(year);
		} catch (DAOException e) {
			throw new ServiceException("getHolidays failed", e);
		}
	}

	public List<SimpleRole> getAllSimpleRoles() throws ServiceException {
		try {
			return securityDAO.findAllSimpleRoles();
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoles failed", e);
		}
	}

	public List<SimpleRole> getAllDMPRoles() throws ServiceException {
		try {
			return securityDAO.findAllDMPRoles();
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoles failed", e);
		}
	}

	/**
	 * Get Permission names associated with the roles
	 */
	public Set<String> getRolePermissionNames(Set<Role> roles) throws ServiceException {
		Set<String> permissionNames = new HashSet<String>();
		if (roles == null)
			return permissionNames;

		try {
			for (Role role : roles) {
				List<String> permissions = securityDAO.getRolePermissionNames(role.getName());
				permissionNames.addAll(permissions);
			}
			return permissionNames;
		} catch (DAOException e) {
			throw new ServiceException("getAvailableRoleNames failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateFailedLogin(java
	 * .lang.String)
	 */
	public boolean updateFailedLogin(String userName) throws ServiceException {
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userName);
			if (user != null && user.getId() != null) {
				// set failed login date
				// TimeZone currentTimeZone =
				// TimeZoneUtils.getThreadCurrentTimezone();
				// DateWithTimeZone dt = new
				// DateWithTimeZone(getTransactinTime(), currentTimeZone);
				user.setLoginFailedDate(new Date());
				// increment failed login count
				Short failedCount = (short) (user.getLoginFailedCount() == null ? 1 : user.getLoginFailedCount().shortValue() + 1);
				user.setLoginFailedCount(failedCount);

				// lock the account if the failed count >=3
				if (user.getLoginFailedCount().intValue() >= 3) {
					if (!user.isAccountLocked()) {
						user.setAccountLockDate(new Date());
						// user.setAccountLocked(true);
					}
				}
				securityDAO.updateUser(user);
				return user.isAccountLocked();
			}
			return false;

		} catch (Exception e) {
			throw new ServiceException("Get All Permissions Failed", e);
		}
	}

	public void updateSuccessFulLogin(String userName) throws ServiceException {
		updateSuccessFulLogin(userName, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.ccht.service.security.SecurityService#updateSuccessFulLogin
	 * (java.lang.String)
	 */
	public void updateSuccessFulLogin(String userName, String credential) throws ServiceException {
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userName);
			if (user == null) {
				user = new User(userName);
			}
			if (credential != null) {
				user.setPassword(credential);
				user.getUserCredentials().hashUserCredentials();
			}
			updateLoginDate(user);

			securityDAO.updateUser(user);
		} catch (Exception e) {
			throw new ServiceException("updateSuccessFulLogin Failed", e);
		}
	}

	private void updateLoginDate(User user) {
		// set success login date
		user.setLastLoginDate(new Date());
		// set failed login count to 0
		user.setLoginFailedCount((short) 0);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.fw.security.SecurityService#updateSuccessFulLogin(gov.va.med
	 * .fw.security.UserPrincipal)
	 */
	public void updateSuccessFulLogin(UserPrincipal userPrincipal, boolean updateRoles) throws ServiceException {
		Validate.notNull(userPrincipal);
		try {
			// get the user and update
			User user = securityDAO.getUserByName(userPrincipal.getUsername());
			if (user == null && userPrincipal instanceof User) {
				user = (User) userPrincipal;
			} else {
				user.getUserCredentials().setPassword(userPrincipal.getPassword());				
				user.setFirstName(userPrincipal.getFirstName());
				user.setMiddleName(userPrincipal.getMiddleName());
				user.setLastName(userPrincipal.getLastName());
			}

			user.getUserCredentials().hashUserCredentials();
			updateLoginDate(user);

			securityDAO.updateUser(user);
		} catch (Exception e) {
			throw new ServiceException("updateSuccessFulLogin Failed", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.ccht.service.security.SecurityService#getAllPermissions()
	 */
	public List<Permission> getAllPermissions() throws ServiceException {
		try {
			return securityDAO.findAllPermissions();
		} catch (DAOException e) {
			throw new ServiceException("Get All Permissions Failed", e);
		}
	}

	public List<User> findAppUsers(SearchCriteria searchCriteria) throws ServiceException {
		try {
			List<User> Users = new ArrayList<User>();
			List<SimpleUser> users = securityDAO.findAppUsers(searchCriteria);
			if (users != null && users.size() > 0) {
				for (SimpleUser person : users) {
					Users.add(new User(person));
				}
			}
			return Users;
		} catch (DAOException e) {
			throw new ServiceException("Securityservice:findAppUsers failed", e);
		}
	}

	public UserPrincipal getAuditUser(String userName) throws ServiceException {
		try {
			return securityDAO.getAuditUser(userName);
		} catch (DAOException e) {
			throw new ServiceException("Securityservice:getAuditUser failed", e);
		}
	}

	public List<String> getUserIds() throws ServiceException {
		try {
			return securityDAO.getUserIds();
		} catch (Exception e) {
			logger.error("SecurityService.getUserIds() failed", e);
			throw new ServiceException("SecurityService.getUserIds() failed", e);
		}
	}

	public int updateUsers(List<String> userIds) throws ServiceException {
		int updateCount = 0;
		try {
			if (userIds != null && userIds.size() > 0) {
				for (String userId : userIds) {
					User user = getUser(userId);
					if (user != null) {
						updateUser(user);
						updateCount = updateCount + 1;
					} else {
						logger.info("SecurityService.updateUsers(): User not found for UserId = " + userId);
					}
				}
			}
		} catch (Exception e) {
			logger.error("SecurityService.updateUsers(): failed", e);
			throw new ServiceException("SecurityService.updateUsers(): failed", e);
		}
		return updateCount;
	}

	public List<User> getSubmittedRegistrations(Integer visnId, Integer facilityId) throws ServiceException {
		try {
			return securityDAO.getSubmittedRegistrations(visnId, facilityId);
		} catch (Exception e) {
			logger.error("SecurityService.getSubmittedRegistrations(): failed", e);
			throw new ServiceException("SecurityService.getSubmittedRegistrations(): failed", e);
		}
	}
	
	public List<User> getNewRegistrations(Integer visnId, Integer facilityId) throws ServiceException {
		try {
			return securityDAO.getNewRegistrations(visnId, facilityId);
		} catch (Exception e) {
			logger.error("SecurityService.getNewRegistrations(): failed", e);
			throw new ServiceException("SecurityService.getNewRegistrations(): failed", e);
		}
	}

	public List<User> getSubmittedRegistrationsForDmp(Integer visnId, Integer facilityId) throws ServiceException {
		try {
			return securityDAO.getSubmittedRegistrationsForDmp(visnId, facilityId);
		} catch (Exception e) {
			logger.error("SecurityService.getSubmittedRegistrationsForDmp(): failed", e);
			throw new ServiceException("SecurityService.getSubmittedRegistrationsForDmp(): failed", e);
		}
	}

	public SecurityDAO getSecurityDAO() {
		return securityDAO;
	}

	public void setSecurityDAO(SecurityDAO securityDAO) {
		this.securityDAO = securityDAO;
	}

	public LdapPersonService getLdapPersonService() {
		return ldapPersonService;
	}

	public void setLdapPersonService(LdapPersonService ldapPersonService) {
		this.ldapPersonService = ldapPersonService;
	}

	public MergeService getMergeService() {
		return mergeService;
	}

	public void setMergeService(MergeService mergeService) {
		this.mergeService = mergeService;
	}
}
