package gov.va.med.ccht.persistent.hibernate;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.pssreport.VendorSubmissionReportData;
import gov.va.med.ccht.persistent.VendorSubmissionReportDAO;
import gov.va.med.fw.persistent.DAOException;

@Repository
@Transactional
public class VendorSubmissionReportDAOImpl implements VendorSubmissionReportDAO {

	static final private int VENDOR_NAME = 0;
	static final private int DATE = 1;
	static final private int NUM_DAYS_SINCE_SUB = 2;
	static final private int SURVEY_TOTAL = 3;
	static final private int SURVEY_ID = 4;
	
	static final private int SURVEY_ID_PS_VERION_ONE = 2;
	static final private int SURVEY_ID_PS_VERION_TWO = 3;

	
	@Autowired
	private SessionFactory sessionFactory;

	public VendorSubmissionReportDAOImpl() {
	}
	
	public VendorSubmissionReportDAOImpl(SessionFactory factory) {
		sessionFactory = factory;
	}
	
	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@Override
	public List<VendorSubmissionReportData> getVendorSubmissionReport() {
		
		StringBuilder sql = new StringBuilder();
		
		sql.append("SELECT DISTINCT Vendor.Vendor_Name as [Vendor], ");
		sql.append("MAX(Surveyed_Activity_Response.Received_Date) AS [Last_Date_Received], ");
		sql.append("DATEDIFF(dd, CONVERT(varchar, MAX(Surveyed_Activity_Response.Received_Date), 101), GETDATE()) AS [NumDays], ");
		sql.append("COUNT(*) AS SurveyCount, Survey.Survey_ID,CASE Survey.Survey_ID ");
		sql.append("WHEN 2 THEN 'PS Ver. 1.0' ");
		sql.append("WHEN 3 THEN 'PS Ver. 2.0' ");
		sql.append("ELSE '' END as [SurveyName] ");
		sql.append("FROM Vendor ");
		sql.append("INNER JOIN Surveyed_Activity ");
		sql.append("INNER JOIN Surveyed_Activity_Response ");
		sql.append("ON Surveyed_Activity.Surveyed_Activity_ID = Surveyed_Activity_Response.Surveyed_Activity_ID ");
		sql.append("ON Vendor.Vendor_ID = Surveyed_Activity.Vendor_ID ");
		sql.append("INNER JOIN Survey ON Surveyed_Activity_Response.Survey_ID = Survey.Survey_ID ");
		sql.append("WHERE survey_accepted_status = 1 and Surveyed_Activity_Response.Survey_ID != 1");
		sql.append("GROUP BY Vendor.Vendor_Name, Survey.Survey_ID ");
		sql.append("ORDER BY Vendor.Vendor_Name, 2 ASC, Survey.Survey_ID ");
		
		final Query query = getSession().createNativeQuery(sql.toString());
		
		List<Object[]> report = query.getResultList();
		
		List<VendorSubmissionReportData> records = new ArrayList<VendorSubmissionReportData>();
		int totalNumberOfSurveysByVendor = 0;
		int totalNumberOfSurveys = 0;
		int totalNumberOfSurveysVersionOne = 0;
		int totalNumberOfSurveysVersionTwo = 0;
		
		List<String> vendors = new ArrayList<String>();
		
		Map<Integer, Date> datesSubmitted = new HashMap<Integer, Date>();
		
		for(Object[] record : report ) {
			
			if(vendors.indexOf(record[VENDOR_NAME].toString()) == -1) { // vendor not in list so add it
				
				vendors.add(record[VENDOR_NAME].toString());
			}
		}
		
		for(String vendor : vendors) {
		
			VendorSubmissionReportData newRecord = new VendorSubmissionReportData();
			
			for(Object[] record : report) {
				
				
				if(record[VENDOR_NAME].toString().equalsIgnoreCase(vendor)) {
					
					if(newRecord.getVendorName() == null) {
						newRecord.setVendorName(record[VENDOR_NAME].toString());
					}
					
					datesSubmitted.put((int) record[NUM_DAYS_SINCE_SUB], (Date) record[DATE]);
					
					int surveyId = (int) record[SURVEY_ID];

				    switch (surveyId) {
					    case SURVEY_ID_PS_VERION_ONE:
					    	newRecord.setSurveyVersionOneTotal(record[SURVEY_TOTAL].toString());
					    	totalNumberOfSurveysByVendor += (int) record[SURVEY_TOTAL];
					    	totalNumberOfSurveysVersionOne += (int) record[SURVEY_TOTAL];
						break;
					    case SURVEY_ID_PS_VERION_TWO:
					    	newRecord.setSurveyVersionTwoTotal(record[SURVEY_TOTAL].toString());
					    	totalNumberOfSurveysByVendor += (int) record[SURVEY_TOTAL];
					    	totalNumberOfSurveysVersionTwo += (int) record[SURVEY_TOTAL];
						break;
				    }
				}
			} // records loop
			
			if(newRecord.getVendorName() != null) { // we only want to add non-null records to list
				newRecord.setDateLastSurveyWithDays(buildDate(datesSubmitted));
				newRecord.setTotalNumberOfSurveys(String.valueOf(totalNumberOfSurveysByVendor));
				records.add(newRecord);
			}
			datesSubmitted = new HashMap<Integer, Date>();
			totalNumberOfSurveysByVendor = 0;
		} // vendor loop
		
		VendorSubmissionReportData totalsRow = new VendorSubmissionReportData();
		totalNumberOfSurveys = totalNumberOfSurveysVersionOne + totalNumberOfSurveysVersionTwo;
		
		totalsRow.setVendorName("Total for All Vendors");
		totalsRow.setSurveyVersionOneTotal(String.valueOf(totalNumberOfSurveysVersionOne));
		totalsRow.setSurveyVersionTwoTotal(String.valueOf(totalNumberOfSurveysVersionTwo));
		totalsRow.setTotalNumberOfSurveys(String.valueOf(totalNumberOfSurveys));
		records.add(totalsRow);
		
		return records;
	}

	private String buildDate(Map<Integer, Date> datesSubmitted) {
		
		List<Integer> daysList = new ArrayList<Integer>(datesSubmitted.keySet());
		
		int leastDays = daysList.get(0); // get initial element to compare
		
		for(int days: daysList) {
			
			if(leastDays > days) {
				leastDays = days;
			}
		}
		
		SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
				
		String formattedDate = df.format(datesSubmitted.get(leastDays)) + " - " + leastDays + " Day(s)";
		
		return formattedDate;
	}
	
	@Override
	public Date getMinReportDate() throws DAOException {
		String sql = "SELECT MIN(rpt_wk) FROM Facility_Activity";
		return (Date) getSession()
				.createNativeQuery(sql)
				.getSingleResult();
	}
}
