package gov.va.med.ccht.persistent.hibernate;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.pssreport.SurveyTrendChartResult;
import gov.va.med.ccht.persistent.SurveyTrendChartsDAO;
import gov.va.med.ccht.service.common.facility.FacilityService;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.ui.model.PSSReportForm;

/**
 * 
 * @author DNS
 *
 */

@Repository
@Transactional
public class SurveyTrendChartsDAOImpl implements SurveyTrendChartsDAO, 
	CensusActivityReportsConstants {

	private static final int MONTH = 0;
	private static final int VISN = 0;
	private static final int NUM_OF_SURVEYS_ALL_VISNS = 1;
	private static final int NUM_OF_SURVEYS_FACILITIES = 1;
	private static final int NUM_OF_SURVEYS = 2;
	
	public static final int NATIONAL_SELECTION = 1;
	public static final int VISN_SELECTION = 2;
	public static final int FACILITY_SELECTION = 3;
	
	@Autowired
	private SessionFactory sessionFactory;

	public SurveyTrendChartsDAOImpl() {
	}
	
	public SurveyTrendChartsDAOImpl(SessionFactory factory) {
		sessionFactory = factory;
	}

	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@SuppressWarnings("unused")
	@Override
	public LinkedList<SurveyTrendChartResult> getSurveyTrendChartsReportData(PSSReportForm form) {
		
		Query query = null;
		
		switch (form.getTypeButtonSelection()) {
		
		case  NATIONAL_SELECTION:
			query = getNationalReportQuery(form);
			break;
		case VISN_SELECTION:
			if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) {
				query = getAllVisnReportQuery(form);
			}
			else {
				query = getVisnReportQuery(form);
				// if query is for a single visn sets query parameter for that visn
				query.setParameter("visnId", form.getVisnId());
			}
			break;
		case FACILITY_SELECTION:
			if(form.getFacilityId().equalsIgnoreCase(ALL_FACILITIES)) {
				query = getAllFacilityReport(form);
			}
			else {
				query = getFacilityReportQuery(form);
				// if query is for a single facility sets query parameter for that facility
				query.setParameter("facilityId", form.getFacilityId());
			}
			break;
		default:
			return new LinkedList<SurveyTrendChartResult>();
		}
		
		//fortify Fix, Null Dereference
		if(query == null) {
			return new LinkedList<SurveyTrendChartResult>();
		}
		
		//fortify Fix, Null Dereference
		// Should never be null, has default value
		if(form.getShortFormatReportFromDate() == null || form.getShortFormatReportToDate() == null) {
			// return empty list to avoid errors, will result in empty report.
			return new LinkedList<SurveyTrendChartResult>();
		} 
		else {
			query.setParameter("fromDate", form.getShortFormatReportFromDate());
			
			//Calendar defaults to the 1st of the month so adding an additional month
			//to include the entire month in the report
			Calendar toDate = Calendar.getInstance();
			toDate.setTime(form.getShortFormatReportToDate());
			toDate.add(Calendar.MONTH, 1);
			
			query.setParameter("toDate", toDate.getTime());
		}
		
		List<Object[]> reportData = query.getResultList();
		
		LinkedList <SurveyTrendChartResult> results = new LinkedList<SurveyTrendChartResult>();
		
		//If report is being run for All visns the data returned is each visn 
		// & the number of surveys over the selected time period
		if(form.getTypeButtonSelection() == VISN_SELECTION && form.getVisnId().equals(ALL_VISNS)) {
			for(Object[] entry : reportData) {
				SurveyTrendChartResult row = new SurveyTrendChartResult();
				row.setVisn(entry[VISN].toString());
				row.setNumOfSurveys(Integer.parseInt(entry[NUM_OF_SURVEYS_ALL_VISNS].toString()));
				results.add(row);
			}
		}
		//For all other reports the data is the month and the total number of surveys for that month
		else if(form.getTypeButtonSelection() == FACILITY_SELECTION){
			for(Object[] entry : reportData) {
				SurveyTrendChartResult row = new SurveyTrendChartResult();
				row.setMonth(entry[MONTH].toString());
				row.setNumOfSurveys(Integer.parseInt(entry[NUM_OF_SURVEYS_FACILITIES].toString()));
				results.add(row);
			}
		}else{
			for(Object[] entry : reportData) {
				SurveyTrendChartResult row = new SurveyTrendChartResult();
				row.setMonth(entry[MONTH].toString());
				row.setNumOfSurveys(Integer.parseInt(entry[NUM_OF_SURVEYS].toString()));
				results.add(row);
			}
		}
		return results;
	}
	
	/**
	* Creates query for National totals
	 * @param form
	 * @return query for national totals
	 */
	private Query getNationalReportQuery(PSSReportForm form) {

		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("datepart(month, received_date) as [MonthNum], ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include VR-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by convert(char(3), received_date),datepart(month, received_date) ");
		sql.append("order by 2");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for All VISNS
     * which returns results by visn instead of by month
	 * @param form
	 * @return query for all visns by visn instead of month
	 */
	private Query getAllVisnReportQuery(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		
		sql.append("SELECT 'Visn' + convert(varchar(2), b.visn_id) ");
		sql.append("as [VISN], ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by b.visn_id  ");
		sql.append("order by b.visn_id");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single visn   
	 * @param form
	 * @return query for single visn by month
	 */
	private Query getVisnReportQuery(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();	
		
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("datepart(month, received_date) as [MonthNum], ");
		sql.append("count(*) as [Count] ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and b.visn_id= :visnId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("group by convert(char(3), received_date),datepart(month, received_date), b.visn_id ");
		sql.append("order by 2, b.visn_id ");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for All Facilities
	 * @param form
	 * @return query for all facilities by month
	 */
	private Query getAllFacilityReport(PSSReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date) ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b ");
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("GROUP BY CONVERT(char(3), a.Received_Date),DATEPART(month, a.Received_Date) ");
		sql.append("order by 3");
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	/**
	 * Creates query for a single facility   
	 * @param form
	 * @return query for single facility by month
	 */
	private Query getFacilityReportQuery(PSSReportForm form) {

		StringBuilder sql = new StringBuilder();
		sql.append("SELECT convert(char(3), received_date) as [Month], ");
		sql.append("count(*) as [Count], ");
		sql.append("datepart(month, received_date) ");
		sql.append("from Surveyed_Activity_Response a, surveyed_activity b, facilities c ");
		sql.append("where a.surveyed_activity_id= b.surveyed_activity_id ");
		sql.append("and c.id = b.Facility_id and b.Facility_id= :facilityId ");
		sql.append("and (a.survey_id != 1) "); //do not include vr-12 data
		sql.append("and a.survey_accepted_status = 1 ");
		sql.append("and received_date ");
		sql.append("BETWEEN :fromDate AND :toDate ");
		sql.append("GROUP BY CONVERT(char(3), a.Received_Date),");
		sql.append("DATEPART(month, a.Received_Date), c.Facility_Name ");
		sql.append("order by 3");	
		
		return getSession().createNativeQuery(sql.toString());
	}
	
	@SuppressWarnings("deprecation")
	@Override
	public List<Date> getDatesByMonth() {
		// TODO: Connect to the database for the dates if necessary.
		// For now, the business logic always starts in January of 2006.

		Calendar lastDate = Calendar.getInstance();
		lastDate.set(2006, Calendar.JANUARY, 1);
		List<Date> datesList = new ArrayList<Date>();
		Calendar current = Calendar.getInstance();
		Date today = Calendar.getInstance().getTime();
		current.set(today.getYear() + 1900, today.getMonth(), 1);
		while(current.getTime().after(lastDate.getTime())) {
			datesList.add(current.getTime());
			current.add(Calendar.MONTH, -1);
		}
		datesList.add(current.getTime()); // add last month
		return datesList;
	}
}
