/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.persistent.hibernate;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.med.ccht.model.SimpleUser;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserRole;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.terminology.FederalHoliday;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.RolePermission;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;

/**
 * 
 * 
 * @author DNS
 */
@Repository
@Transactional
public class SecurityDAOImpl implements SecurityDAO {
	
	//constructors for Impl file so we can mock session
    public SecurityDAOImpl() {
    }

    public SecurityDAOImpl(SessionFactory factory) {
        sessionFactory = factory;
    }
	
	private Logger logger = Logger.getLogger(SecurityDAOImpl.class);

    @Autowired
	private SessionFactory sessionFactory;
    
	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}
    
	/** 
	 * Get user by name
	 * 
	 * @param userId
	 * @return
	 * @throws DAOException
	 */
	public User getUserByName(String userId) {

		final String sql = "select * from ht.app_user where user_name='" + userId +"'";
		return getSession().createNativeQuery(sql, User.class).getSingleResult();
	}
	
	/**
	 * Delete a role
	 * 
	 * @param identifier
	 * @return
	 */
	@Override
	public void deleteUserRole(UserRole userRole) {
		
		Validate.notNull(userRole, "Role identifier can not be null.");
		Validate.notNull(userRole.getId(), "Role identifier key can not be null.");

		getSession().remove(userRole);
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<User> getSubmittedRegistrations(Integer visnId, Integer facilityId) {
		String sql = "select l from " + User.class.getName() + " l where l.registrationStatus.code != :registrationStatus";
		Query query;
		
		if (visnId != null && visnId > 0) {
			sql += " and l.visn.id = :visnId";
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.APPROVED);
			query.setParameter("visnId", visnId);
			return query.getResultList();
		}else if (facilityId != null && facilityId > 0) {
			sql += " and l.facility.id = :facilityId";
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.APPROVED);
			query.setParameter("facilityId", facilityId);
			return query.getResultList();
		}
		else {
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.APPROVED);
			return query.getResultList();
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<User> getNewRegistrations(Integer visnId, Integer facilityId) {
		String sql = "select l from " + User.class.getName() + " l where l.registrationStatus.code = :registrationStatus";
		Query query;
		
		if (visnId != null && visnId > 0) {
			sql += " and l.visn.id = :visnId";
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.SUBMITTED);
			query.setParameter("visnId", visnId);
			return query.getResultList();
		}else if (facilityId != null && facilityId > 0) {
			sql += " and l.facility.id = :facilityId";
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.SUBMITTED);
			query.setParameter("facilityId", facilityId);
			return query.getResultList();
		}
		else {
			query = getSession().createQuery(sql);
			query.setParameter("registrationStatus", RegistrationStatus.SUBMITTED);
			return query.getResultList();
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<User> getSubmittedRegistrationsForDmp(Integer visnId, Integer facilityId) throws DAOException {
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("registrationStatus", RegistrationStatus.APPROVED);	
			
			String sql = "select l from " + User.class.getName() + " l where l.registrationStatus.code != :registrationStatus and (l.registrationReason.name like '%DMP%' or l.registrationReason.name = 'Training Center Staff')";
			if (visnId != null) {
				sql += " and l.visn.id = :visnId";
				params.put("visnId", visnId);
			}else if (facilityId != null) {
				sql += " and l.facility.id = :facilityId";
				params.put("facilityId", facilityId);
			}
			
			final Query query = getSession().createQuery(sql);
			for(String field: params.keySet()) {
				query.setParameter(field, params.get(field));
			}
			return query.getResultList();
		} catch (Exception e) {
			throw new DAOException("getSubmittedRegistrationsForDmp failed: ", e);
		}
	}
	
	/**
	 * Create user in the IHTA database
	 * 
	 * @param user
	 * @throws DAOException
	 */
	public void createUser(User user) {
		getSession().persist(user);
	}

	/**
	 * Update user in the IHTA database
	 * 
	 * @param user
	 * @throws DAOException
	 */
	public void updateUser(User user) {

		for (UserRole ur : user.getRoles()) {
			ur.setUser(user);
		}
		
		getSession().saveOrUpdate(user);
	}

	/**
	 * Update role in the IHTA database
	 * 
	 * @param role
	 * @throws DAOException
	 */
	public void updateRole(Role role) throws DAOException {
		
		getSession().saveOrUpdate(role);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.IHTA.persistent.SecurityDAO#findAllRoles()
	 */
	@SuppressWarnings("unchecked")
	public List<Role> findAllRoles() throws DAOException {
		
		try {
			String sql = "select l from " + Role.class.getName() + " l";
			final Query query = getSession().createQuery(sql);
			return query.getResultList();
		} catch (Exception e) {
			throw new DAOException("findAllRoles failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleRole> findAllSimpleRoles() throws DAOException {
		try {
			String sql = "select l from " + SimpleRole.class.getName() + " l where l.id != 2020 order by l.name";
			return getSession().createQuery(sql).getResultList();
		} catch (Exception e) {
			throw new DAOException("findAllRoles failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleRole> findAllDMPRoles() throws DAOException {
		try {
			String sql = "select l from " + SimpleRole.class.getName() + " l where l.name like '%DMP%' order by l.name";
			return getSession().createQuery(sql).getResultList();
		} catch (Exception e) {
			throw new DAOException("findAllDMPRoles failed: ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#findAllPermissions(gov.va.med.
	 * fw.security.Application)
	 */
	@SuppressWarnings("unchecked")
	public List<Permission> findAllPermissions() throws DAOException {

		try {
			final String sql = "select l from " + Permission.class.getName() + " l";
			return getSession().createQuery(sql).getResultList();
		} catch (Exception e) {
			throw new DAOException("findAllPermissions failed: ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#getRoleByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public Role getRoleByName(String roleName) {
		
		String sql = "select l from " + Role.class.getName() + " l where UPPER(l.name) = :roleName";
		final Query query = getSession().createQuery(sql);
		query.setParameter("roleName", roleName);
		
		List<Role> list = query.getResultList();
		if (list != null && list.size() > 0)
			return list.get(0);
		return null;
	}
	
	@Override
	public Role getRoleById(long id) {
		return getSession().get(Role.class, id);
	}	
	
	

	public SimpleRole getSimpleRoleByName(String roleName) throws DAOException {
		try {
			String sql = "select l from " + SimpleRole.class.getName()
					+ " l where l.name = :roleName";
			final Query query = getSession().createQuery(sql);
			query.setParameter("roleName", roleName);
			return (SimpleRole) query.getSingleResult();
		} catch (Exception e) {
			throw new DAOException("getSimpleRoleByName Failed ", e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.IHTA.persistent.SecurityDAO#getAvailableRoleNames()
	 */
//	@SuppressWarnings("unchecked")
//	public List<String> getAvailableRoleNames(Application application) throws DAOException {
//		
//		logger.error("$$$$$$ fix me");
//		return null;
////		try {
////			String query = "select l.name from " + Role.class.getName() + " l ";
////			Map<String, Object> params = new HashMap<String, Object>();
////			if (application != null) {
////				query = query + " where l.application.applicationName = :applicationName";
////				params.put("applicationName", application.getApplicationName());
////			}
////			return getJpaTemplate().findByNamedParams(query, params);
////		} catch (Exception e) {
////			throw new DAOException("getAvailableRoleNames failed", e);
////		}
//	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.IHTA.persistent.SecurityDAO#getRolePermissions(java.lang.String
	 * )
	 */
	@SuppressWarnings("unchecked")
	public List<String> getRolePermissionNames(String roleName) throws DAOException {
		try {
			String sql = "select r.internalPermissions.permission.name from "
					+ Role.class.getName() + " r where r.name = :roleName";
			final Query query = getSession().createQuery(sql).setParameter("roleName", roleName);
			return query.getResultList();
		} catch (Exception e) {
			throw new DAOException("getRolePermissionNames failed", e);
		}
	}

	@SuppressWarnings("unchecked")
	public UserPrincipal getAuditUser(String userName) throws DAOException {
		Validate.notNull(userName);
		try {
			final String sql = "select USER_NAME, FIRST_NAME, MIDDLE_NAME, LAST_NAME"
					+ " from ht.app_user"
					+ " where upper(USER_NAME) = :userName";
			final Query query = getSession().createNativeQuery(sql).setParameter("userName", userName);

			List<Object[]> results = query.getResultList();

			if (results != null) {
				for (Object[] result : results) {
					UserPrincipalImpl user = new UserPrincipalImpl((String) result[0]);
					user.setFirstName((String) result[1]);
					user.setMiddleName((String) result[2]);
					user.setLastName((String) result[3]);
					return user;
				}
			}
			return new UserPrincipalImpl(userName);

		} catch (Exception e) {
			throw new DAOException("getAuditUser failed", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<FederalHoliday> findHolidays(int year) throws DAOException {
		
		try {
			String sql = "select h from " + FederalHoliday.class.getName() + " h where year(h.activeDate) = :year";
			final Query query = getSession().createQuery(sql).setParameter("year", year);
			
			return query.getResultList();
		} catch (Exception e) {
			throw new DAOException("findHolidays failed: ", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> getUserIds() throws DAOException {
		try {
			final String sql = "select USER_NAME from ht.app_user";
			final Query query = getSession().createNativeQuery(sql);

			return query.getResultList();
			
		} catch (Exception e) {
			throw new DAOException("getAuditUser failed", e);
		}
	}

	@SuppressWarnings("unchecked")
	public List<SimpleUser> findAppUsers(SearchCriteria searchCriteria) throws DAOException {
		try {
			//user name search
			if (StringUtils.isNotEmpty(searchCriteria.getSAMAccountName())){
				final String sql = "select l from " + SimpleUser.class.getName()
						+ " l where upper(l.userCredentials.userID) = :userId";

				return getSession().createQuery(sql)
						.setParameter("userId", searchCriteria.getSAMAccountName())
						.getResultList();							
			}//Last name and first name search
			else if (StringUtils.isNotEmpty(searchCriteria.getLastName())){
				String sql = "select l from " + SimpleUser.class.getName()
					+ " l where l.lastName = :lastName";
				Query query = getSession().createQuery(sql);
				if (StringUtils.isNotEmpty(searchCriteria.getFirstName())) {
					sql = sql + " and l.firstName like :firstName";
					query = getSession().createQuery(sql);
					// Find first names that start with the value of the first name search criteria.
					query.setParameter("firstName", searchCriteria.getFirstName() + "%");
				}
				query.setParameter("lastName", searchCriteria.getLastName());
				//return the result set				
				return query.getResultList();	
			}//Role and VISN or facility search 
			else if (searchCriteria.getRoles().size() > 0) {	
				final Map<String,Object> map = new HashMap<String,Object>();
				map.put("roles", searchCriteria.getRoles());
				String sql = "select l from " + SimpleUser.class.getName()
				+ " l where l.id in (select distinct ur.user.id from " + UserRole.class.getName()
				+ " ur where ur.role.name in (:roles))";
				if (searchCriteria.getVisns() != null && searchCriteria.getVisns().size() > 0) {
					sql += " and l.visn.id in (select v.id from " + SimpleVisn.class.getName() +
					" v where v.name in (:visns))";
					map.put("visns", searchCriteria.getVisns());
				}
				if (searchCriteria.getStations() != null && searchCriteria.getStations().size() > 0) {
					sql += " and l.facility.id in (select f.id from " + SimpleFacility.class.getName() +
					" f where f.stationNumber in (:stations))";
					map.put("stations", searchCriteria.getStations());
				}					
				if (searchCriteria.getVendors() != null && searchCriteria.getVendors().size() > 0) {
					sql += " and l.vendor.id in (select ven.id from " + Vendor.class.getName() +
					" ven where ven.name in (:vendors))";
					map.put("vendors", searchCriteria.getVendors());
				}
				
				final Query query = getSession().createQuery(sql);
				for(String field: map.keySet()) {
					query.setParameter(field, map.get(field));
				}
				
				return query.getResultList();
			} else {
				throw new Exception("findAppUsers failed: Invalid Search criteria");
			}
		}catch (Exception e){
			throw new DAOException("findAppUsers failed", e);
		}
	}
	
	@Override
   public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
		try {
			final User user = getUserByName(username);
			if (user == null)
				throw new UsernameNotFoundException("Sorry, no user with the name '" + username + "' was found.");
			Collection<SimpleRole> simpleRoles = getRolesByUserName(username);

			List<String> masterRoles = new ArrayList<>();
			// changed for SSO
			for (Object obj : simpleRoles) {
				if(obj instanceof SimpleRole ) {
					masterRoles.add(((SimpleRole)obj).getName());
				}
				else if(obj instanceof Role) {
					masterRoles.add(((Role)obj).getName());
				}
			}

			// Add default role for all valid users
			if (!masterRoles.contains("CCHT_USER")) {
				masterRoles.add("CCHT_USER");
			}
			user.setLdapRoles(new HashSet<>(masterRoles));

			final Set<String> permissions = new HashSet<String>();
			// now loop through and get permissions for each role and add to
			// the authorities
			for (Object element : masterRoles) {
				String roleName = (String) element;
				Role role = getRoleByName(roleName);
				if (role != null && role.getPermissions() != null) {
					for (Object element2 : role.getPermissions()) {
						Permission permission = (Permission) element2;
						permissions.add(permission.getName());
					}
				}
			}
			permissions.addAll(masterRoles);

			// TODO: OHRS specific code?
			// Set<String> appSpecificPermissions = new HashSet<String>();
			// for (String permName : permissions) {
			// if (permName.startsWith(subsystemPrefixToCheck)) {
			// appSpecificPermissions.add(permName.substring(subsystemPrefixToCheck.length(),
			// permName.length())
			// + subsystemSuffixToAdd);
			// }
			// }
			// permissions.addAll(appSpecificPermissions);

			// add user name
			permissions.add(username);

			List<GrantedAuthority> roles = new ArrayList<>(permissions.size());

			for (String element : permissions) {
				GrantedAuthority authority = new SimpleGrantedAuthority(element);
				roles.add(authority);
			}

			user.setAuthorities(roles);
			return user;
		} catch(NoResultException e) {
			throw new UsernameNotFoundException("No result found for username = " + username, e);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public List<SimpleRole> getRolesByUserName(String userName) {
		final String sql = "select distinct ur.role from " + UserRole.class.getName()
				+ " ur where ur.user.userCredentials.userID = :userName";
		final Query query = getSession().createQuery(sql).setParameter("userName", userName);
		
		return query.getResultList();
	}

	@Override
	public Map<String, UserPrincipal<?>> findUsersWithAnyRole(List<String> roleNames) {
		
		if(roleNames == null || roleNames.size() == 0) {
			return new HashMap<String, UserPrincipal<?>>();
		}
		
		String sql = "select distinct ur.user.userCredentials.userID, ur.user from "
				+ UserRole.class.getName() + " ur where ur.role.name in (:roleNames)";
		
		@SuppressWarnings("unchecked")
		List<Object[]> rows = getSession().createQuery(sql).setParameterList("roleNames", roleNames).getResultList();

		Map<String, UserPrincipal<?>> results = new HashMap<>();
		for (Object[] row : rows)
			results.put((String) row[0], (UserPrincipal<?>) row[1]);
		return results;
	}

	@Override
	public void deleteEntity(Object obj) throws DAOException {
		// TODO Auto-generated method stub
		getSession().delete(obj);
		
	}

	@Override
	public void flush() {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void persist(Role role) {
		getSession().persist(role);
	}
	
	@Override
	public void persist(UserRole userRole) {
		getSession().persist(userRole);
	}

	@Override
	public void remove(Role role) {
		getSession().remove(role);
	}
	
	@Override
	public void remove(UserRole userRole) {
		getSession().remove(userRole);
	}

	@Override
	public Object getByKey(EntityKey<?> entityKey) {
		Object entity = getSession().get(entityKey.getEntityClass(), entityKey.getKeyValue());
		return entity;
	}

	@Override
	public void deleteRole(Role role) {
		getSession().remove(role);
	}
	
	public void deleteRolePermissionsFromRole(Role role) {
		
		if (role.getInternalPermissions() != null ) {
			for (final RolePermission rolePermission : role.getInternalPermissions()) {
				getSession().delete(rolePermission);
			}
		}
	}
}
