package gov.va.med.ccht.persistent.hibernate;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.pssreport.CategoryOfCare;
import gov.va.med.ccht.model.report.DeviceModality;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.persistent.DeviceModalityDAO;
import gov.va.med.ccht.persistent.SatisfactionSurveyReportDAO;
import gov.va.med.ccht.persistent.VendorDAO;
import gov.va.med.ccht.persistent.utils.PSSReportUtils;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;

@Repository
@Transactional
public class SatisfactionSurveyReportDAOImpl  implements SatisfactionSurveyReportDAO, 
	CensusActivityReportsConstants {
	
	private Logger logger = Logger.getLogger(SatisfactionSurveyReportDAOImpl.class);

	private String selectedModality = null;
	
	@Autowired
	private VendorDAO vendorDao;
	
	@Autowired
	private DeviceModalityDAO deviceModalityDao;
	
	@Autowired
	private SessionFactory sessionFactory;

	public SatisfactionSurveyReportDAOImpl() {
	}
	
	public SatisfactionSurveyReportDAOImpl(SessionFactory factory, VendorDAO vd, DeviceModalityDAO dm) {
		sessionFactory = factory;
		vendorDao = vd;
		deviceModalityDao = dm;
	}

	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@Override
	public List<CategoryOfCare> getCategoriesOfCare() {
		String sql = "SELECT * FROM LevelOfCare WHERE Category_Name != 'Unknown'";
		return getSession()
				.createNativeQuery(sql, CategoryOfCare.class)
				.getResultList();
	}
	
	@SuppressWarnings("deprecation")
	@Override
	public List<Date> getDatesByMonth() {
		// TODO: Connect to the database for the dates if necessary.
		// For now, the business logic always starts in January of 2006.

		Calendar lastDate = Calendar.getInstance();
		lastDate.set(2006, Calendar.JANUARY, 1);
		List<Date> datesList = new ArrayList<Date>();
		Calendar current = Calendar.getInstance();
		Date today = Calendar.getInstance().getTime();
		current.set(today.getYear() + 1900, today.getMonth(), 1);
		while(current.getTime().after(lastDate.getTime())) {
			datesList.add(current.getTime());
			current.add(Calendar.MONTH, -1);
		}
		datesList.add(current.getTime()); // add last month
		return datesList;
	}

	@Override
	public List<SatisfactionSurveyQuestionResult> getSatisfactionSurveyReport(
			PSSReportForm form) throws DAOException, NumberFormatException, ServiceException {

		List<SatisfactionSurveyQuestionResult> satisfactionSurveyQuestionResults = null;
		Query query = null;
		selectedModality = form.getModalityName();
		
		switch (form.getTypeButtonSelection())
		{
			case 1:
				query = getNationalSatisfactionSurveyReport(form);
				break;
			case 2:
				if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) {
					// Same query as national
					query = getNationalSatisfactionSurveyReport(form);
				}
				else {
					query = getSatisfactionSurveyReportByVisn(form);
				}
				break;
			case 3:
				if(form.getFacilityId().equalsIgnoreCase(ALL_FACILITIES)) {
					// Same query as national
					query = getNationalSatisfactionSurveyReport(form);
				}
				else {
					query = getSatisfactionSurveyReportByFacilities(form);
				}
				break;
		}

		try
		{
			if(query == null) { // if null, expected value was not selected, return empty list.
				return new ArrayList<SatisfactionSurveyQuestionResult>();
			}
			
			List<?> report = query.getResultList();

			int numIterations = 0;
			if(report != null) {
				numIterations = report.size();
			}

			satisfactionSurveyQuestionResults = new ArrayList<SatisfactionSurveyQuestionResult>();

			Map<Integer, SatisfactionSurveyQuestionResult> resultsByQuestion = new HashMap<Integer, SatisfactionSurveyQuestionResult>();
			
			// for loop will not execute if numIterations is 0;
			for (int j = 0; j < numIterations; j++) 
			{
				Object[] record = (Object[]) report.get(j);
				if (record != null) 
				{
					Integer questionNumber = (Integer) record[0];
					String questionText = (String) record[1];
					int month = (Integer) record[2];
					int year = (Integer) record[3];
					Integer answerCount = (Integer) record[4];
					
					double avgAnswer = 0;
					BigDecimal avgAnswerAsBD = (BigDecimal) record[5];
					avgAnswer = avgAnswerAsBD.doubleValue();
					 
					String mod = (String) record[6];
					SatisfactionSurveyQuestion ssq = new SatisfactionSurveyQuestion();
					ssq.setQuestionNumber(questionNumber);
					ssq.setQuestionText(questionText);
					ssq.setModality(mod);
					ssq.setAvgAnswer(avgAnswer);
					ssq.setMonthAsInt(month);
					ssq.setYear(year);
					ssq.setAnswerCount(answerCount);
					ssq.setMonthAsString();
					ssq.setHeaderString(false);
					ssq.setTheDate();
					if (resultsByQuestion.containsKey(questionNumber)) 
					{
						SatisfactionSurveyQuestionResult tempResult = resultsByQuestion
								.remove(questionNumber);
						tempResult.addToResultsKeyedByMonthMap(ssq);

						resultsByQuestion.put(questionNumber, tempResult);
					} 
					else 
					{
						SatisfactionSurveyQuestionResult newResult = new SatisfactionSurveyQuestionResult();
						newResult.setQuestionNumber(questionNumber);
						newResult.setQuestionText(questionText);
						newResult.setModality(selectedModality);

						newResult.addToResultsKeyedByMonthMap(ssq);
						resultsByQuestion.put(questionNumber, newResult);
					}
				}

			}
			
			satisfactionSurveyQuestionResults.addAll(resultsByQuestion.values());
		}
		catch(Exception e)
		{
			throw new DAOException(e.getMessage(), e);
		}

		return satisfactionSurveyQuestionResults;
	}

	private Query getNationalSatisfactionSurveyReport(
			PSSReportForm form) throws DAOException, NumberFormatException, ServiceException {
		
		StringBuilder sql = new StringBuilder();
		//For all modalities most get total column union with modality columns
		if(selectedModality.equals(ALL_MODALITIES)){
			// BUILD SELECT SECTION OF SQL
			sql.append(getCommonSelectSQL());
			sql.append(getAverageClause(form.getReportVersionButtonSelection()));
			sql.append(" 'Total' AS [Modality] ");
			
			// BUILD FROM SECTION OF SQL
			sql.append(getCommonFromSQL());
			
			Vendor vendorMod = null;
			if(!form.getVendorId().equals(ALL_VENDORS)) {
				vendorMod = vendorDao.getVendorById(new Long(form.getVendorId()));
			}

			// BUILD WHERE SECTION OF SQL
			sql.append(getCommonWhereSQL(form, vendorMod));
			
			sql.append(" GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
			sql.append(" c.Question_Number, c.Question_Text_Short ");
			sql.append(" Union ");
		}
		// BUILD SELECT SECTION OF SQL
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));
		
		// BUILD FROM SECTION OF SQL
		sql.append(getCommonFromSQL());
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}

		// BUILD WHERE SECTION OF SQL
		sql.append(getCommonWhereSQL(form, vendor));
		
		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);
		sql.append(" GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append(" ORDER BY Yr, Mon, c.Question_Number, Modality");
		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));
		
		addDateParams(query, form);
		
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;
	}

	private Query getSatisfactionSurveyReportByVisn(final PSSReportForm form) throws DAOException, NumberFormatException, ServiceException {
		
		StringBuilder sql = new StringBuilder();
		//For all modalities most get total column union with modality columns
		if(selectedModality.equals(ALL_MODALITIES)){
			// BUILD SELECT SECTION OF SQL
			sql.append(getCommonSelectSQL());
			sql.append(getAverageClause(form.getReportVersionButtonSelection()));
			sql.append(" 'Total' AS [Modality] ");
			
			// BUILD FROM SECTION OF SQL
			sql.append(getCommonFromSQL());
			
			Vendor vendorMod = null;
			if(!form.getVendorId().equals(ALL_VENDORS)) {
				vendorMod = vendorDao.getVendorById(new Long(form.getVendorId()));
			}
			// BUILD WHERE SECTION OF SQL
			sql.append(getCommonWhereSQL(form, vendorMod));
			sql.append("AND d.visn_id = :visnId ");
			
			sql.append(" GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
			sql.append(" c.Question_Number, c.Question_Text_Short ");
			sql.append(" Union ");
		}
		
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));

		sql.append(getCommonFromSQL());
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}
		
		sql.append(getCommonWhereSQL(form, vendor));
		sql.append("AND d.visn_id = :visnId ");

		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);

		sql.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));

		addDateParams(query, form);

		query.setParameter("visnId", form.getVisnId());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;
	}

	private Query getSatisfactionSurveyReportByFacilities(final PSSReportForm form) throws DAOException, NumberFormatException, ServiceException {
		
		StringBuilder sql = new StringBuilder();
		//For all modalities most get total column union with modality columns
		if(selectedModality.equals(ALL_MODALITIES)){
			// BUILD SELECT SECTION OF SQL
			sql.append(getCommonSelectSQL());
			sql.append(getAverageClause(form.getReportVersionButtonSelection()));
			sql.append(" 'Total' AS [Modality] ");
			
			// BUILD FROM SECTION OF SQL
			sql.append(getCommonFromSQL());
			sql.append("INNER JOIN facilities as g ON d.Facility_ID = g.ID ");
			
			Vendor vendorMod = null;
			if(!form.getVendorId().equals(ALL_VENDORS)) {
				vendorMod = vendorDao.getVendorById(new Long(form.getVendorId()));
			}
			// BUILD WHERE SECTION OF SQL
			sql.append(getCommonWhereSQL(form, vendorMod));
			sql.append("and d.facility_id = :facilityId ");
			
			sql.append(" GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
			sql.append(" c.Question_Number, c.Question_Text_Short ");
			sql.append(" Union ");
		}
		
		sql.append(getCommonSelectSQL());
		sql.append(getAverageClause(form.getReportVersionButtonSelection()));
		sql.append(PSSReportUtils.getModalityString(selectedModality));

		sql.append(getCommonFromSQL());
		sql.append("INNER JOIN facilities as g ON d.Facility_ID = g.ID ");
		sql.append(PSSReportUtils.getModalityInnerJoinString(selectedModality));
		
		Vendor vendor = null;
		if(!form.getVendorId().equals(ALL_VENDORS)) {
			vendor = vendorDao.getVendorById(new Long(form.getVendorId()));
		}
		
		sql.append(getCommonWhereSQL(form, vendor));
		sql.append("and d.facility_id = :facilityId ");

		String modalityInString = PSSReportUtils.getModalityInString(selectedModality);
		sql.append(modalityInString);

		sql.append("GROUP BY DATEPART(yy, b.Completed_Date), DATEPART(mm, b.Completed_Date), ");
		sql.append(" c.Question_Number, c.Question_Text_Short ");
		sql.append(PSSReportUtils.getModalityStringForGroupByClause(selectedModality));

		sql.append("ORDER BY Yr, Mon, c.Question_Number, Modality");

		String temp = sql.toString();
		// This is done with a replace rather than setting a parameter because something is wrong with
		// how Hibernate is producing the SQL for the String array parameter.
		Query query = getSession().createNativeQuery(temp.replace(":selectedQuestionList", StringUtils.join(form.getQuestionNumbers(), ", ")));

		addDateParams(query, form);

		query.setParameter("facilityId", form.getFacilityId());

		// TODO: IMPLEMENT METHOD FOR GETTING THE NEEDED MODALITY_TYPE ID'S
		if (selectedModality != null && !selectedModality.equalsIgnoreCase(DO_NOT_INCLUDE) && !selectedModality.equalsIgnoreCase(ALL_MODALITIES)) {
			query.setParameter("modalityIdList", getModalityIds(selectedModality));
		}

		return query;

	}

	private String getCommonSelectSQL() {
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT c.Question_Number, c.Question_Text_Short, ");
		sb.append("DATEPART(mm, b.Completed_Date) AS Mon, DATEPART(yy, b.Completed_Date) AS Yr, COUNT(*) AS AnsCount, ");
		
		return sb.toString();
	}

	private String getCommonFromSQL() {
		StringBuilder sb = new StringBuilder();
		sb.append("FROM Surveyed_Activity AS d ");
		sb.append("INNER JOIN Surveyed_Activity_Response AS b ON d.Surveyed_Activity_ID = b.Surveyed_Activity_ID ");
		sb.append("INNER JOIN Survey_Response AS a ON b.Surveyed_Activity_Response_ID = a.Surveyed_Activity_Response_ID ");
		sb.append("INNER JOIN Survey_Question AS c ON a.Survey_Question_ID = c.Survey_Question_ID ");
		sb.append("INNER JOIN Device ON d.Device_ID = Device.Device_ID ");
		sb.append("INNER JOIN (select cp.patient_id, cp.level_of_care from censuspatient cp ");
		sb.append("inner join (select distinct MAX(census_id) as censusid, patient_id, disenrollment_date ");
		sb.append("from censuspatient where disenrollment_date is null group by patient_id, disenrollment_date) as lcpe ");
		sb.append("on cp.census_id = lcpe.censusid and cp.patient_id = lcpe.patient_id) f ");
		sb.append("on d.patient_id = f.patient_id ");

		return sb.toString();
	}

	private String getCommonWhereSQL(final PSSReportForm form, final Vendor vendor) {
		StringBuilder sb = new StringBuilder();

		sb.append(" WHERE ");

		// If both dates are not null append dates to the query
		if (form.getShortFormatReportFromDate() != null && form.getShortFormatReportToDate() != null) {
			sb.append("(b.Completed_Date between :fromDate AND :toDate ) ");
		}

		sb.append(PSSReportUtils.getSurveyIdClause(form.getReportVersionButtonSelection()));

		if (vendor != null && vendor.getName() != null) {
			if (!vendor.getName().equalsIgnoreCase("All Vendors")) {
				sb.append("AND d.Vendor_ID = " + vendor.getId() + " ");
			}
		}
		
		if(form.getCategoriesOfCare() != null) {
			if(form.getCategoriesOfCare().length == 0 || 
					(form.getCategoriesOfCare()[0].equalsIgnoreCase("-1"))) {
				// Default to all COC
				String[] temp = {"1", "2", "3", "4", "5"};
				form.setCategoriesOfCare(temp);
			}
			else if (form.getCategoriesOfCare()[0].equalsIgnoreCase("-2")) {
				// All except L2
				String[] temp = {"1", "2", "3", "4"};
				form.setCategoriesOfCare(temp);
			}

			sb.append("AND f.level_of_care in (");
			sb.append(StringUtils.join(form.getCategoriesOfCare(), ", ") +") ");
		}

		sb.append("AND c.question_number in ");
		sb.append("( :selectedQuestionList ) ");

		return sb.toString();
	}
	
	private String getAverageClause(final int patientSatisfactionVersion)
	{
		StringBuilder qString = new StringBuilder();
		// Ignore answer #6 for PS Ver. 1.0. This is "No experience" 
	    // since there are only 4 choices for PS Ver. 2.0 the same condition can be used.
		// More "versions" can be used if necessary.
		switch(patientSatisfactionVersion)
		{
		// Pat. Sat. Version 1
		case 1:
			qString
			.append(" AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/5.0) AS AvgAnswer, ");
			break;
		// Pat. Sat. Version 2
		case 2:
			qString
			.append(" AVG((CASE a.answer WHEN 6 THEN NULL ELSE a.answer END)/4.0) AS AvgAnswer, ");
			break;
		}
		
		return qString.toString();
	}

	private List<Long> getModalityIds(String modalityString)
	{
		List<DeviceModality> modalities = deviceModalityDao.getAllDeviceModalities();
		List<Long> modalityIds = new ArrayList<Long>(modalities.size()); 
		/*
		 * To match CNS IHTA no longer needs to generate a list of all the modality Ids when ALL modalities is selected
		 * because CNS uses "Home Device as a catch all" & not a strict definition. Final determinations have not been
		 * made as to which solution should be used 11/05/18
		 */
//		if(modalityString.equalsIgnoreCase(ALL_MODALITIES))
//		{
//			for(DeviceModality modality: modalities) {
//				modalityIds.add(modality.getId());
//			}
//		}
//		else {
			modalityIds.add(new Long(modalityString));
//		}
		
		return modalityIds;
	}
	
	private Query addDateParams(Query query, PSSReportForm form){
		query.setParameter("fromDate", form.getShortFormatReportFromDate());
		
		//Calendar defaults to the 1st of the month so adding an additional month minus 1 day
		//to include the entire month in the report and setting the time to the last millisecond
		//of the last day in the month
		Calendar toDate = Calendar.getInstance();
		toDate.setTime(form.getShortFormatReportToDate());

		toDate.set(Calendar.MILLISECOND, 999);
		toDate.set(Calendar.SECOND, 59);
		toDate.set(Calendar.MINUTE, 59);
		toDate.set(Calendar.HOUR_OF_DAY, 23);
		toDate.add(Calendar.MONTH, 1);
		toDate.add(Calendar.DATE, -1);
		
		query.setParameter("toDate", toDate.getTime());
		
		return query;
	}
}
