package gov.va.med.ccht.persistent.hibernate;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import javax.transaction.Transactional;

import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.query.Query;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.responserate.ResponseRateRecord;
import gov.va.med.ccht.model.responserate.ResponseRateResults;
import gov.va.med.ccht.persistent.FacilityDAO;
import gov.va.med.ccht.persistent.ResponseRateReportDAO;
import gov.va.med.ccht.persistent.VendorDAO;
import gov.va.med.ccht.persistent.VisnDAO;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.fw.persistent.DAOException;

@Repository
@Transactional
public class ResponseRateReportDAOImpl implements ResponseRateReportDAO, CensusActivityReportsConstants {
    
    public ResponseRateReportDAOImpl() {
    }
    
    public ResponseRateReportDAOImpl(SessionFactory factory) {
        sessionFactory = factory;
    }
    
    public ResponseRateReportDAOImpl(SessionFactory factory, 
            VendorDAO ven, VisnDAO vis, FacilityDAO fac) {
        sessionFactory = factory;
        vendorDao = ven;
        visnDao = vis;
        facilityDao = fac;
    }
    
    private Logger logger = Logger.getLogger(getClass());

	@Autowired
	private SessionFactory sessionFactory;
	@Autowired
	private VendorDAO vendorDao;
	@Autowired
	private VisnDAO visnDao;
	@Autowired
	private FacilityDAO facilityDao;

	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}
	
	@Override
	public ResponseRateResults getTopLevelReport(CensusActivityReportForm form) throws DAOException {
		String queryString = getResponseRateSQL(form);
		Query query = getSession()
				.createNativeQuery(queryString)
				.setParameter("mode", form.getButtonSelection())
				.setParameter("reportEndDate", form.getReportEndDate());
		if(form.getButtonSelection() == VISN_BUTTON_SELECTION) {
			if(!form.getVisnId().equals(ALL_VISNS)) {
				query.setParameter("visnId", form.getVisnId());
			}
			else {
				query.setParameter("visnId", 0);
			}
			return getVisnResults(query.getResultList());
		}
		else if(form.getButtonSelection() == FACILITY_BUTTON_SELECTION) {
			query.setParameter("visnId", 0);
			if(!form.getFacilityId().equals(ALL_FACILITIES)) {
				query.setParameter("facility_id", form.getFacilityId());
			}
			else {
				query.setParameter("facility_id", null);
			}
			return getFacilityResults(query.getResultList(), form);
		}
		else {
			return getNationalResults(query.getResultList());
		}
	}
	
	@Override
	public ResponseRateResults getVisnDrillDownReport(CensusActivityReportForm form) throws NumberFormatException, DAOException {
		List<?> queryResults = getSession()
				.createNativeQuery(getVisnDrillDownSQL(form))
				.setParameter("reportEndDate", form.getReportEndDate())
				.setParameter("visnId", form.getVisnId())
				.getResultList();
		return getFacilityResults(queryResults, form);
	}
	
	private ResponseRateResults getNationalResults(List<?> queryResults) {
		List<ResponseRateRecord> records = new ArrayList<ResponseRateRecord>();
		for (int j = 0; j < queryResults.size(); j++) {
			ResponseRateRecord nextRecord = new ResponseRateRecord();
			Object[] record = (Object[]) queryResults.get(j);
			Integer id = (Integer) record[0];
			Integer dummy = (Integer) record[1];
			String vendorName = (String) record[2];
			Integer responseCode = (Integer) record[3];
			Integer patientCount = (Integer) record[4];
			BigDecimal maxDataAgeAsBd = (BigDecimal) record[5];
	
			Integer maxDataAge = 0;
	
			if (maxDataAgeAsBd != null) {
				maxDataAge = maxDataAgeAsBd.intValueExact();
			}
	
			nextRecord.setId(id);
			// Will read "0" for national results
			nextRecord.setStationNumber(dummy + "");
			nextRecord.setVendorName(vendorName);
			nextRecord.setResponseCode(responseCode);
			nextRecord.setPatientCount(patientCount);
			nextRecord.setMaxDataAge(maxDataAge);
	
			records.add(nextRecord);
		}
	
		ResponseRateResults retVal = new ResponseRateResults();
		if (queryResults.size() > 0) {
			retVal.setResponseRateRecords(records);
			retVal.setRecordsKeyedByVendor();
			retVal.buildResponseRateRowsByVendor(vendorDao.getAllVendors());
			retVal.setTotalsRow();
		}
		return retVal;
	}
	
	private ResponseRateResults getVisnResults(List<?> queryResults) {
		List<ResponseRateRecord> records = new ArrayList<ResponseRateRecord>();
		for (int j = 0; j < queryResults.size(); j++) {
			ResponseRateRecord nextRecord = new ResponseRateRecord();
			Object[] record = (Object[]) queryResults.get(j);
			Integer visnId = (Integer) record[0];
			Integer dummy = (Integer) record[1];
			Integer responseCode = (Integer) record[3];
			Integer patientCount = (Integer) record[4];
			BigDecimal maxDataAgeAsBd = (BigDecimal) record[5];
	
			Integer maxDataAge = 0;
	
			if (maxDataAgeAsBd != null) {
				maxDataAge = maxDataAgeAsBd.intValueExact();
			}
	
			nextRecord.setId(visnId);
			// Will read "0" for visn results
			nextRecord.setStationNumber(dummy + "");
			// Set the name so that we can use it later in the table
			nextRecord.setVendorName(visnDao.getVisnById(new Long(visnId)).getName());
			nextRecord.setResponseCode(responseCode);
			nextRecord.setPatientCount(patientCount);
			nextRecord.setMaxDataAge(maxDataAge);
	
			records.add(nextRecord);
		}
	
		ResponseRateResults retVal = new ResponseRateResults();
		if (queryResults.size() > 0) {
			retVal.setResponseRateRecords(records);
			retVal.setRecordsKeyedById();
			retVal.buildResponseRateRowsByVisn(visnDao.getVisns());
			retVal.setTotalsRow();
		}
		return retVal;
	}
	
	private ResponseRateResults getFacilityResults(List<?> queryResults, CensusActivityReportForm form) throws NumberFormatException, DAOException {
		List<ResponseRateRecord> records = new ArrayList<ResponseRateRecord>();
		for (int j = 0; j < queryResults.size(); j++) {
			ResponseRateRecord nextRecord = new ResponseRateRecord();
			Object[] record = (Object[]) queryResults.get(j);
			if(form.getFacilityId() == null || form.getFacilityId().equals(ALL_FACILITIES)) {
				nextRecord.setId((Integer) record[0]);
				nextRecord.setStationNumber((String) record[1]);
				// Set the facility name so that we can use it later in the table
				nextRecord.setVendorName((String) record[2]);
			}
			else {
				nextRecord.setId(facilityDao.getFacility(form.getFacilityId()).getId());
				nextRecord.setStationNumber(form.getFacilityId());
				// Set the facility name so that we can use it later in the table
				nextRecord.setVendorName((String) record[2]);
			}
			Integer responseCode = (Integer) record[3];
			Integer patientCount = (Integer) record[4];
			BigDecimal maxDataAgeAsBd = (BigDecimal) record[5];
	
			Integer maxDataAge = 0;
	
			if (maxDataAgeAsBd != null) {
				maxDataAge = maxDataAgeAsBd.intValueExact();
			}
	
			nextRecord.setResponseCode(responseCode);
			nextRecord.setPatientCount(patientCount);
			nextRecord.setMaxDataAge(maxDataAge);
	
			records.add(nextRecord);
		}
	
		ResponseRateResults retVal = new ResponseRateResults();
		if (queryResults.size() > 0) {
			retVal.setResponseRateRecords(records);
			if(form.getFacilityId() == null || form.getFacilityId().equals(ALL_FACILITIES)) {
				retVal.setRecordsKeyedByVendor();
				retVal.buildResponseRateRowsByFacility(facilityDao.getFacilitiesAscById());
			}
			else {
				retVal.setRecordsKeyedByVendor();
				retVal.buildResponseRateRowsByVendor(vendorDao.getAllVendors());
			}
			retVal.setTotalsRow();
		}
		return retVal;
	}
	
	private String getResponseRateSQL(CensusActivityReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("EXEC GetResponseReportData :mode, :reportEndDate");
		if(form.getButtonSelection() == VISN_BUTTON_SELECTION) {
			sql.append(", :visnId");
		}
		else if(form.getButtonSelection() == FACILITY_BUTTON_SELECTION) {
			sql.append(", :visnId");
			sql.append(", :facility_id");
		}
		return sql.toString();
	}
	
	private String getVisnDrillDownSQL(CensusActivityReportForm form) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT c.id, ");
		sql.append("c.facility_id, ");
		sql.append("c.facility_name, ");
		sql.append("a.responsecode, ");
		sql.append("count (DISTINCT a.PatientID) PatientCnt, ");
		sql.append("MAX (a.data_age) MaxDataAge ");
		sql.append("FROM (SELECT a.facility_id AS FacilityID, ");
		sql.append("dbo.GetWeeklyCompliance3 (a.rr_res, a.data_age) ");
		sql.append("AS responsecode, ");
		sql.append("CONVERT (NUMERIC (5), (REPLACE (REPLACE (a.data_age, 'NA', '0'), '-', '0'))) AS data_age, ");
		sql.append("a.patient_id AS PatientID ");
		sql.append("FROM censuspatient a, census b ");
		sql.append("WHERE a.census_id = b.census_id ");
		sql.append("AND (a.disenrollment_date IS NULL OR a.disenrollment_date >= dateadd (day, -6, :reportEndDate)) ");
		sql.append("AND a.enrollment_date <= :reportEndDate ");
		sql.append("AND b.report_start_date = dateadd (day, -6, :reportEndDate) ");
		sql.append("AND left (a.patient_ssn, 5) <> '00000' ");
		sql.append("UNION ");
		sql.append("SELECT a.facility_id AS FacilityID, ");
		sql.append("5 AS responsecode, ");
		sql.append("CONVERT (NUMERIC (5), (REPLACE (REPLACE (a.data_age, 'NA', '0'), '-', '0'))) ");
		sql.append("AS data_age, ");
		sql.append("a.patient_id AS PatientID ");
		sql.append("FROM censuspatient a, census b ");
		sql.append("WHERE a.census_id = b.census_id ");
		sql.append("AND (a.disenrollment_date IS NULL OR a.disenrollment_date >= dateadd (day, -6, :reportEndDate)) ");
		sql.append("AND a.enrollment_date <= :reportEndDate ");
		sql.append("AND b.report_start_date = dateadd (day, -6, :reportEndDate) ");
		sql.append("AND left (a.patient_ssn, 5) <> '00000' ");
		sql.append("AND CONVERT (NUMERIC (5), (REPLACE (REPLACE (a.data_age, 'NA', '0'), '-', '0'))) > 30) a ");
		sql.append("JOIN facilities c ");
		sql.append("ON a.FacilityID = c.ID ");
		sql.append("AND c.visn_id = :visnId ");
		sql.append("GROUP BY c.facility_id, c.facility_name, c.id, a.responsecode ");
		sql.append("ORDER BY c.facility_id, c.facility_name, c.id, a.responsecode ASC");
		return sql.toString();
	}
}
