package gov.va.med.ccht.persistent.hibernate;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;

import javax.persistence.Query;
import javax.transaction.Transactional;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import gov.va.med.ccht.model.coc.CategoryOfCareRecord;
import gov.va.med.ccht.persistent.CategoryOfCareReportDAO;
import gov.va.med.ccht.ui.model.CensusActivityReportForm;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.util.DateUtils;

@Repository
@Transactional
public class CategoryOfCareReportDAOImpl implements CategoryOfCareReportDAO
{
	
	 static final private int VENDOR_NAME = 0;
	 static final private int VISN_ID = 0;
	 static final private int FACILITY_ID = 0;
	 static final private int ID = 1;
	 static final private int FACILITY_NAME = 4;
	 static final private int RESULTS_NUMBER = 3;
	 
	 private static final int NIC = 1;
	 private static final int NICLR = 6;
     private static final int CHRONIC_CARE = 2;
     private static final int ACUTE_CARE = 3;
     private static final int HEALTH_PROMO_DISEASE_PREVENTION = 4;
     private static final int LOW_ACTUITY_LOW_INTENSITY = 5; 
     private static final int TOTALS_TOTAL = 100;
     
 	static final private String ALL_VENDORS = "-1"; 
 	static final private String ALL_FACILITIES = "-1"; 
 	static final private String ALL_VISNS = "-1";
	@Autowired
	private SessionFactory sessionFactory;

	public CategoryOfCareReportDAOImpl() {
	}

	public CategoryOfCareReportDAOImpl(SessionFactory factory) {
		sessionFactory = factory;
	}
	
	protected Session getSession() {
		return this.sessionFactory.getCurrentSession();
	}

    @Override
    public List<CategoryOfCareRecord> getNationalReport(CensusActivityReportForm form)
    {
    	
    	boolean appendVendorId = false;
    	
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT DISTINCT c.vendor_name, d.id, d.category_name, COUNT(distinct patient_id) AS [Num] ");
		sql.append("FROM census a, censuspatient b, vendor c, levelofcare d ");
		sql.append("WHERE a.census_id = b.census_id ");
		sql.append("AND a.report_start_date = :reportStartDate ");
		sql.append("AND a.vendor_id = c.vendor_id ");
		sql.append("AND b.level_of_care = d.id ");
		sql.append("AND (b.disenrollment_date IS NULL OR b.disenrollment_date >= :reportStartDate) ");
		
		if(!form.getVendorId().equalsIgnoreCase(ALL_VENDORS)) {
			sql.append("AND c.vendor_id = :vendorId ");	
			appendVendorId = true;
		}
		
		sql.append("GROUP BY c.vendor_name, d.id, d.category_name ");
		sql.append("ORDER BY c.vendor_name, d.id ASC");

		final Query query = getSession().createNativeQuery(sql.toString());
		
		String reportStartDate = getReportStartDate(form.getReportEndDate());
		query.setParameter("reportStartDate", reportStartDate);
		
		if(appendVendorId) {
			query.setParameter("vendorId", form.getVendorId());
		}

		return generateNationalReport(query);
    }
    
    public List<CategoryOfCareRecord> getVisnReport(CensusActivityReportForm form) {
    	
    	boolean appendVisn = false;
    	boolean appendVendorId = false;
    	
    	StringBuilder sql = new StringBuilder();		
		sql.append("SELECT DISTINCT e.visn_id, d.id, d.category_name, COUNT(distinct patient_id) ");
		sql.append("FROM census a, censuspatient b, facilities c, levelofcare d, visn e, vendor f ");
		sql.append("WHERE b.facility_id = c.id AND c.visn_id = e.visn_id ");
		sql.append("AND a.census_id = b.census_id ");
		sql.append("AND a.report_start_date = :reportStartDate ");
		sql.append("AND b.level_of_care = d.id ");
		sql.append("AND (b.disenrollment_date is null OR b.disenrollment_date >= :reportStartDate) ");
		
		if(form.getVendorId().equalsIgnoreCase(ALL_VENDORS)) {
			// all vendor selections
			if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) { // all vendors and all visns
				// do not add any criteria
			}
			else { // all vendors and single visn
				sql.append("AND e.visn_id = :visnId ");
				appendVisn = true;
			}
		} 
		else { 
			// single vendor selections
			if(form.getVisnId().equalsIgnoreCase(ALL_VISNS)) { // single vendor all Visns
				sql.append("AND a.vendor_id = f.vendor_id ");
				sql.append("and a.vendor_id = :vendorId ");
				appendVendorId = true;
			} 
			else {// single vendor and single facility
				sql.append("AND e.visn_id = :visnId ");
				sql.append("AND a.vendor_id = f.vendor_id ");
				sql.append("AND f.vendor_id = :vendorId  ");
				appendVisn = true;
				appendVendorId = true;
			}						
		}
		
		sql.append("GROUP BY e.visn_id, d.id, d.category_name ORDER BY e.visn_id, d.id ASC");
		
		final Query query = getSession().createNativeQuery(sql.toString());
		
		String reportStartDate = getReportStartDate(form.getReportEndDate());
		query.setParameter("reportStartDate", reportStartDate);
		
		if(appendVendorId) {
			query.setParameter("vendorId", form.getVendorId());
		}
		if(appendVisn) {
			query.setParameter("visnId", form.getVisnId());
		}

		return generateVisnReport(query);
    }
    
    public List<CategoryOfCareRecord> getFacilityReportByVisn(CensusActivityReportForm form) {
    	
    	boolean appendVendorId = false;
    	
    	StringBuilder sql = new StringBuilder();		
		sql.append("select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name ");
		sql.append("from census a, censuspatient b, facilities c, ");
		sql.append("levelofcare d, patient e, visn f, vendor g  ");
		sql.append("where b.patient_id = e.patient_id  ");
		sql.append("and b.facility_id = c.id  ");
		sql.append("and c.visn_id = f.visn_id ");
		sql.append("and f.visn_id = :visnId ");
		sql.append("and a.census_id = b.census_id "); 
		sql.append("and a.report_start_date = :reportStartDate "); 
		sql.append("and b.level_of_care = d.id  ");
		sql.append("and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) ");
		sql.append("and left(e.patient_ssn,5)<>'00000' ");
		
		if(!form.getVendorId().equalsIgnoreCase(ALL_VENDORS)) {
			
			sql.append("and a.vendor_id = :vendorId ");
			appendVendorId = true;
		}						
		
		sql.append("group by c.facility_id, c.facility_name, d.id, d.category_name ");
		
		String reportStartDate = getReportStartDate(form.getReportEndDate());
		String visnId = form.getVisnId();
		
		final Query query = getSession().createNativeQuery(sql.toString());
		query.setParameter("reportStartDate", reportStartDate);
		query.setParameter("visnId", visnId);
		
		if(appendVendorId) {
			query.setParameter("vendorId", form.getVendorId());
		}
		
		return generateFacilitiesReport(query);
    }
    
    public List<CategoryOfCareRecord> getFacilityReport(CensusActivityReportForm form) {
    	
    	boolean appendFacId = false;
    	boolean appendVendorId = false;
    	
    	StringBuilder sql = new StringBuilder();		
		sql.append("select distinct c.facility_id, d.id, d.category_name, count(distinct b.patient_id), c.facility_name ");
		sql.append("from census a, censuspatient b, facilities c, ");
		sql.append("levelofcare d, patient e, vendor f ");
		sql.append("where b.patient_id = e.patient_id  ");
		sql.append("and b.facility_id = c.id  ");
		sql.append("and a.census_id = b.census_id "); 
		sql.append("and a.report_start_date = :reportStartDate "); 
		sql.append("and b.level_of_care = d.id  ");
		sql.append("and (b.disenrollment_date is null or b.disenrollment_date >= :reportStartDate) ");
		sql.append("and left(e.patient_ssn,5)<>'00000' ");
		
		String[] facilityId = form.getFacilityId().split(",");
		String facility = "";
		if(!facilityId[0].equals("")){
			facility = facilityId[0];
		}
		else {
			facility = facilityId[1];
		}
		
		if(form.getVendorId().equalsIgnoreCase(ALL_VENDORS)) {
			// all vendor selections
			if(!facility.equalsIgnoreCase(ALL_FACILITIES)) {
				sql.append("and c.facility_id = :facilityId "); // correct (oldway)
				appendFacId = true;
			}
		} 
		else { 
			// single vendor selections
			if(facility.equals(ALL_FACILITIES)) { // single vendor all facilities
				sql.append("and a.vendor_id = :vendorId ");
				appendVendorId = true;
			} 
			else {// single vendor and single facility
				sql.append("and c.facility_id = :facilityId ");
				sql.append("and a.vendor_id = :vendorId ");
				appendFacId = true;
				appendVendorId = true;
			}						
		}
		
		sql.append("group by c.facility_id, c.facility_name, d.id, d.category_name ");
		sql.append("order by c.facility_id ");
		
		final Query query = getSession().createNativeQuery(sql.toString());
		
		String reportStartDate = getReportStartDate(form.getReportEndDate());
		query.setParameter("reportStartDate", reportStartDate);
		
		if(appendVendorId) {
			query.setParameter("vendorId", form.getVendorId());
		}
		if(appendFacId) {
			query.setParameter("facilityId", facility);
		}
		
		return generateFacilitiesReport(query);
    }

    private List<CategoryOfCareRecord> generateNationalReport(Query query)
    {
		List<CategoryOfCareRecord> records = new ArrayList<CategoryOfCareRecord>();
		@SuppressWarnings("unchecked")
		List<Object[]> report = query.getResultList();
		List<String> allVendors = getAllVendors();
		
		// running total of call categories, will turn into "totals" column.
		int nicTotal = 0;
		int niclrTotal = 0;
		int chronicCareTotal = 0;
		int acuteCareTotal = 0;
		int hpdpTotal = 0;
		int LowActLowIntTotal = 0;
		int totalsTotal = 0; // a count of all of the vendor's totals.
		String vendorName = "";
		
		for(String vendor : allVendors) {
		
			CategoryOfCareRecord newRecord = new CategoryOfCareRecord();
			
			for(Object[] record : report) {
				
				if(record[VENDOR_NAME].toString().equalsIgnoreCase(vendor)) { 
					
					if(newRecord.getName() == null) {
						// Validate Vendor Names only have upper and lowercase letters, Fortify Fix
						vendorName = ESAPIValidator.validateStringInput(record[VENDOR_NAME].toString(),ESAPIValidationType.UpperAndLowerCaseWithNumbers_WhiteList);
						newRecord.setName(vendorName);
					}
			    	
				    int recordId = (int) record[ID];

				    switch (recordId) {
					    case NIC:
					    	newRecord.setNic((int)record[RESULTS_NUMBER]);
					    	nicTotal += (int)record[RESULTS_NUMBER];
						break;
					    case NICLR:
					    	newRecord.setNiclr((int)record[RESULTS_NUMBER]);
					    	niclrTotal += (int)record[RESULTS_NUMBER];
						break;
					    case CHRONIC_CARE:
					    	newRecord.setChronicCareMgnt((int)record[RESULTS_NUMBER]);
					    	chronicCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case ACUTE_CARE:
					    	newRecord.setAcuteCareMgnt((int)record[RESULTS_NUMBER]);
					    	acuteCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case HEALTH_PROMO_DISEASE_PREVENTION:
					    	newRecord.setHpdp((int)record[RESULTS_NUMBER]);
					    	hpdpTotal += (int)record[RESULTS_NUMBER];
						break;
					    case LOW_ACTUITY_LOW_INTENSITY: 
					    	newRecord.setLowActLowInt((int)record[RESULTS_NUMBER]);
					    	LowActLowIntTotal += (int)record[RESULTS_NUMBER];
						break;
				    }
				
				newRecord.setTotal();
				}
			} // records, report loop
			if(newRecord.getName() != null) { // we only want to add non-null records to list
				records.add(newRecord);
				totalsTotal += newRecord.getTotal();
			}
		}
		
		HashMap<Integer, Integer> categoryTotals = new HashMap<Integer, Integer>();
		
		categoryTotals.put(NIC, nicTotal);
		categoryTotals.put(NICLR,niclrTotal);
		categoryTotals.put(CHRONIC_CARE, chronicCareTotal);
		categoryTotals.put(ACUTE_CARE, acuteCareTotal);
		categoryTotals.put(HEALTH_PROMO_DISEASE_PREVENTION, hpdpTotal);
		categoryTotals.put(LOW_ACTUITY_LOW_INTENSITY, LowActLowIntTotal);
		categoryTotals.put(TOTALS_TOTAL, totalsTotal);
		
		records.add(generateTotalsRow(categoryTotals));
		records.add(generatePercentRow(categoryTotals));
		
		return records;
    }
    
    private List<CategoryOfCareRecord> generateVisnReport(Query query)
    {
		List<CategoryOfCareRecord> records = new ArrayList<CategoryOfCareRecord>();
		@SuppressWarnings("unchecked")
		// Any fields output to user are validated against a whitelist in the appropriate setter of CategoryOfCareRecord.java 
		List<Object[]> report = query.getResultList();
		List<Integer> allVisns = getAllVisn();
		
		// running total of call categories, will turn into "totals" column.
		int nicTotal = 0;
		int niclrTotal = 0;
		int chronicCareTotal = 0;
		int acuteCareTotal = 0;
		int hpdpTotal = 0;
		int LowActLowIntTotal = 0;
		int vendorsTotalsTotal = 0;
		String visnName = "";
	
		for(int visn : allVisns) {
			
			CategoryOfCareRecord newRecord = new CategoryOfCareRecord();
			
			for(Object[] record : report) {
				
				if(Integer.parseInt(record[VISN_ID].toString()) == visn) { 
					
					if(newRecord.getName() == null) {
						visnName = ESAPIValidator.validateStringInput(record[VISN_ID].toString(), ESAPIValidationType.Numbers_WhiteList);
						newRecord.setName("VISN " + visnName);
						newRecord.setVisnId(visnName); // Id is just Name without " VISN " appended.
					}
			    	
				    int recordId = (int) record[ID];

				    switch (recordId) {
					    case NIC:
					    	newRecord.setNic((int)record[RESULTS_NUMBER]);
					    	nicTotal += (int)record[RESULTS_NUMBER];
						break;
					    case NICLR:
					    	newRecord.setNiclr((int)record[RESULTS_NUMBER]);
					    	niclrTotal += (int)record[RESULTS_NUMBER];
						break;
					    case CHRONIC_CARE:
					    	newRecord.setChronicCareMgnt((int)record[RESULTS_NUMBER]);
					    	chronicCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case ACUTE_CARE:
					    	newRecord.setAcuteCareMgnt((int)record[RESULTS_NUMBER]);
					    	acuteCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case HEALTH_PROMO_DISEASE_PREVENTION:
					    	newRecord.setHpdp((int)record[RESULTS_NUMBER]);
					    	hpdpTotal += (int)record[RESULTS_NUMBER];
						break;
					    case LOW_ACTUITY_LOW_INTENSITY: 
					    	newRecord.setLowActLowInt((int)record[RESULTS_NUMBER]);
					    	LowActLowIntTotal += (int)record[RESULTS_NUMBER];
						break;
				    }
				
				newRecord.setTotal();
				}
			} // records, report loop
			if(newRecord.getName() != null) { // we only want to add non-null records to list
				// will only contain sql if single visn is selected.
				if(!query.toString().contains("AND e.visn_id = :visnId")) { 
					newRecord.calulateInvidualPercentColumns();
				}
				records.add(newRecord);
				vendorsTotalsTotal += newRecord.getTotal();
			}
		}
		
		HashMap<Integer, Integer> categoryTotals = new HashMap<Integer, Integer>();
		
		categoryTotals.put(NIC, nicTotal);
		categoryTotals.put(NICLR,niclrTotal);
		categoryTotals.put(CHRONIC_CARE, chronicCareTotal);
		categoryTotals.put(ACUTE_CARE, acuteCareTotal);
		categoryTotals.put(HEALTH_PROMO_DISEASE_PREVENTION, hpdpTotal);
		categoryTotals.put(LOW_ACTUITY_LOW_INTENSITY, LowActLowIntTotal);
		categoryTotals.put(TOTALS_TOTAL, vendorsTotalsTotal);
		
		records.add(generateTotalsRow(categoryTotals));
		records.add(generatePercentRow(categoryTotals));
		
		return records;
    }
    
    private List<CategoryOfCareRecord> generateFacilitiesReport(Query query)
    {
		List<CategoryOfCareRecord> records = new ArrayList<CategoryOfCareRecord>();
		@SuppressWarnings("unchecked")
		// Any fields output to user are validated against a whitelist in the appropriate setter of CategoryOfCareRecord.java 
		List<Object[]> report = query.getResultList();
		List<String> allFacilities = getFacilities("-1"); // only used to iterate through a list, and compare, value not assigned.
		
		
		// running total of call categories, will turn into "totals" column.
		int nicTotal = 0;
		int niclrTotal = 0;
		int chronicCareTotal = 0;
		int acuteCareTotal = 0;
		int hpdpTotal = 0;
		int LowActLowIntTotal = 0;
		int totalsTotal = 0;
		String FacilityName = "";
		
	
		for(Object facility : allFacilities) {
			
			CategoryOfCareRecord newRecord = new CategoryOfCareRecord();
			
			for(Object[] record : report) {
				
				if(record[FACILITY_ID].toString().equalsIgnoreCase(facility.toString())) { 
					
					if(newRecord.getName() == null) {
						FacilityName = ESAPIValidator.validateStringInput(record[FACILITY_ID].toString(), ESAPIValidationType.FacilitiesName_WhiteList)
								+ " " 
								+ ESAPIValidator.validateStringInput(record[FACILITY_NAME].toString(), ESAPIValidationType.FacilitiesName_WhiteList);
						newRecord.setName("Facility " + FacilityName);
						// Not directly output to screen.
						newRecord.setVisnId(record[VISN_ID].toString());
					}
			    	
				    int recordId = (int) record[ID]; // category of care id 

				    switch (recordId) {
					    case NIC:
					    	newRecord.setNic((int)record[RESULTS_NUMBER]);
					    	nicTotal += (int)record[RESULTS_NUMBER];
						break;
					    case NICLR:
					    	newRecord.setNiclr((int)record[RESULTS_NUMBER]);
					    	niclrTotal += (int)record[RESULTS_NUMBER];
						break;
					    case CHRONIC_CARE:
					    	newRecord.setChronicCareMgnt((int)record[RESULTS_NUMBER]);
					    	chronicCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case ACUTE_CARE:
					    	newRecord.setAcuteCareMgnt((int)record[RESULTS_NUMBER]);
					    	acuteCareTotal += (int)record[RESULTS_NUMBER];
						break;
					    case HEALTH_PROMO_DISEASE_PREVENTION:
					    	newRecord.setHpdp((int)record[RESULTS_NUMBER]);
					    	hpdpTotal += (int)record[RESULTS_NUMBER];
						break;
					    case LOW_ACTUITY_LOW_INTENSITY: 
					    	newRecord.setLowActLowInt((int)record[RESULTS_NUMBER]);
					    	LowActLowIntTotal += (int)record[RESULTS_NUMBER];
						break;
				    }
				
				newRecord.setTotal();
				}
			}
			if(newRecord.getName() != null) { // we only want to add non-null records to list
				// will only contain sql if single facility is selected.
				if(!query.toString().contains("and c.facility_id = :facilityId")) { 
					newRecord.calulateInvidualPercentColumns();
				}
				records.add(newRecord);
				totalsTotal += newRecord.getTotal();
			}
		}
		
		HashMap<Integer, Integer> categoryTotals = new HashMap<Integer, Integer>();
		
		categoryTotals.put(NIC, nicTotal);
		categoryTotals.put(NICLR,niclrTotal);
		categoryTotals.put(CHRONIC_CARE, chronicCareTotal);
		categoryTotals.put(ACUTE_CARE, acuteCareTotal);
		categoryTotals.put(HEALTH_PROMO_DISEASE_PREVENTION, hpdpTotal);
		categoryTotals.put(LOW_ACTUITY_LOW_INTENSITY, LowActLowIntTotal);
		categoryTotals.put(TOTALS_TOTAL, totalsTotal);
		
		records.add(generateTotalsRow(categoryTotals));
		records.add(generatePercentRow(categoryTotals));
		
		return records;
    }
    
    private CategoryOfCareRecord generateTotalsRow(HashMap<Integer, Integer> totals) {
    	
    	CategoryOfCareRecord totalsRow = new CategoryOfCareRecord();
		totalsRow.setRowName("TOTALS by CATEGORY");
		totalsRow.setNic(totals.get(NIC));
		totalsRow.setNiclr(totals.get(NICLR));
		totalsRow.setChronicCareMgnt(totals.get(CHRONIC_CARE));
		totalsRow.setAcuteCareMgnt(totals.get(ACUTE_CARE));
		totalsRow.setHpdp(totals.get(HEALTH_PROMO_DISEASE_PREVENTION));
		totalsRow.setLowActLowInt(totals.get(LOW_ACTUITY_LOW_INTENSITY));
		totalsRow.setColumnName("TOTAL PNTS CATEGORIZED");
		
		return totalsRow;
    }
    
    private CategoryOfCareRecord generatePercentRow(HashMap<Integer, Integer> totals) {
    	
    	CategoryOfCareRecord percentageRow = new CategoryOfCareRecord();
		percentageRow.setRowName("PERCENT OF TOTAL");
		percentageRow.setNicPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(NIC)));
		percentageRow.setNiclrPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(NICLR)));
		percentageRow.setChronicCareMgntPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(CHRONIC_CARE)));
		percentageRow.setAcuteCareMgntPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(ACUTE_CARE)));
		percentageRow.setHpdpPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(HEALTH_PROMO_DISEASE_PREVENTION)));
		percentageRow.setLowActLowIntPercentTotal(calculatePercent(totals.get(TOTALS_TOTAL), totals.get(LOW_ACTUITY_LOW_INTENSITY)));
		percentageRow.setTotalsTotal(totals.get(TOTALS_TOTAL));
		
		return percentageRow;
    }
    
    private String calculatePercent(int total, int part) {
    	
    	float percent = (float)part/total;
    	
    	return String.valueOf(percent);
    }
    /****** Script for SelectTopNRows command from SSMS  ******/
    private List<String> getAllVendors() {
    	
    	StringBuilder sql = new StringBuilder();
		sql.append("SELECT DISTINCT vendor_name FROM vendor");
		
		final Query query = getSession().createNativeQuery(sql.toString());
    	
    	List<String> vendors = query.getResultList();
    	return vendors;
    }
    
    private List<Integer> getAllVisn() {
    	
    	StringBuilder sql = new StringBuilder();
		sql.append("SELECT DISTINCT VISN_ID FROM VISN ");
		
		final Query query = getSession().createNativeQuery(sql.toString());
    	
    	List<Integer> visn = query.getResultList();
    	return visn;
    }
    
    private List<String> getFacilities(String visn) {
    	
    	StringBuilder sql = new StringBuilder();
		sql.append("SELECT DISTINCT FACILITY_ID FROM FACILITIES ");
		
		if(!visn.equals("-1")) { // get facilities from all visns
			sql.append("WHERE VISN_ID = " + visn);
		}
		
		final Query query = getSession().createNativeQuery(sql.toString());
    	
    	List<String> facilities = query.getResultList();
    	return facilities;
    }
    
    private String getReportStartDate(Date reportEndDate)
    {
	String startDate = null;
	
	GregorianCalendar gc = new GregorianCalendar();
	gc.setTime(reportEndDate);
	
	gc.set(Calendar.DATE, (gc.get(Calendar.DATE) - 6 ));
	SimpleDateFormat format = new SimpleDateFormat(DateUtils.MMDDYYYY);			
	startDate = format.format(gc.getTime());
	return startDate;
    }
}
