package gov.va.med.ccht.model.responserate;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.common.Visn;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;

public class ResponseRateResults implements Serializable {
	private static final long serialVersionUID = 5329731773474249792L;

	private static final int RESPONDERS = 1;
	private static final int PARTIAL_RESPONDERS = 2;
	private static final int NON_RESPONDERS = 3;
	private static final int UNKOWN = 4;
	private static final int DATE_COLS = 5;

	private List<ResponseRateRow> responseRateRows = null;
	private List<ResponseRateRecord> responseRateRecords = null;
	private Map<String, List<ResponseRateRecord>> recordsKeyedByVendor = null;
	private Map<Integer, List<ResponseRateRecord>> recordsKeyedById = null;

	public List<ResponseRateRecord> getResponseRateRecords() {
		return responseRateRecords;
	}

	public void setResponseRateRecords(List<ResponseRateRecord> responseRateRecords) {
		this.responseRateRecords = responseRateRecords;
	}

	public List<ResponseRateRow> getResponseRateRows() {
		return responseRateRows;
	}

	public void setResponseRateRows(List<ResponseRateRow> responseRateRows) {
		this.responseRateRows = responseRateRows;
	}

	public void setTotalsRow() {
		if (responseRateRows == null || responseRateRows.size() == 0) {
			return;
		}

		ResponseRateRow row = new ResponseRateRow();
		row.setVendorName("TOTALS BY RESPONSE RATE");
		int numResponders = 0;
		int partialResponders = 0;
		int nonResponders = 0;
		int unknown = 0;
		int total = 0;

		for (ResponseRateRow currentRow : responseRateRows) {
			numResponders += currentRow.getResponders();
			partialResponders += currentRow.getPartialResponders();
			nonResponders += currentRow.getNonResponders();
			unknown += currentRow.getUnknown();
			total += currentRow.getTotal();
		}

		row.setResponders(numResponders);
		row.setPartialResponders(partialResponders);
		row.setNonResponders(nonResponders);
		row.setUnknown(unknown);
		row.setTotal(total);
		row.setColumnFive("TOTAL PTS REPORTED");
		row.setColumnSix("");
		row.setColumnSeven("");

		int overAllVendorTotal = row.getTotal();
		ResponseRateRow percentageRow = new ResponseRateRow();
		percentageRow.setVendorName("PERCENT OF TOTAL");
		int numRespondersTotal = row.getResponders();
		int partialRespondersTotal = row.getPartialResponders();
		int nonRespondersTotal = row.getNonResponders();
		int unknownTotal = row.getUnknown();

		percentageRow.setColumnOne(getPercentOfTotal(numRespondersTotal, overAllVendorTotal));
		percentageRow.setColumnTwo(getPercentOfTotal(partialRespondersTotal, overAllVendorTotal));
		percentageRow.setColumnThree(getPercentOfTotal(nonRespondersTotal, overAllVendorTotal));
		percentageRow.setColumnFour(getPercentOfTotal(unknownTotal, overAllVendorTotal));

		percentageRow.setTotal(overAllVendorTotal);
		percentageRow.setColumnSix("");
		percentageRow.setColumnSeven("");

		responseRateRows.add(row);
		responseRateRows.add(percentageRow);
	}

	private String getPercentOfTotal(int total, int overAllTotal) {
		Double percent = (((double) total) / overAllTotal);
		return percent.toString();
	}

	public void setRecordsKeyedByVendor() {
		if (responseRateRecords != null) {
			recordsKeyedByVendor = new TreeMap<String, List<ResponseRateRecord>>();

			for (ResponseRateRecord currentRecord : responseRateRecords) {
				String vendor = currentRecord.getVendorName();
				List<ResponseRateRecord> temp = recordsKeyedByVendor.get(vendor);

				if (temp == null) {
					temp = new ArrayList<ResponseRateRecord>();
				}

				temp.add(currentRecord);
				recordsKeyedByVendor.put(vendor, temp);
			}
		}
	}
	
	public void setRecordsKeyedById() {
		if(responseRateRecords != null) {
			recordsKeyedById = new TreeMap<Integer, List<ResponseRateRecord>>();
			
			for(ResponseRateRecord currentRecord: responseRateRecords) {
				List<ResponseRateRecord> temp = recordsKeyedById.get(currentRecord.getId());
				if(temp == null) {
					temp = new ArrayList<ResponseRateRecord>();
				}
				temp.add(currentRecord);
				recordsKeyedById.put(currentRecord.getId(), temp);
			}
		}
	}

	public void buildResponseRateRowsByVendor(List<Vendor> vendors) {
		responseRateRows = new ArrayList<ResponseRateRow>();
		// Need a list of vendors
		// Iterate through the list and add things to the rows as needed

		for(Vendor vendor: vendors) {
			// use Facilities WhiteList because of the Vendor name, Visn, and Facilities that all use this method, it is the least restrictive
			// while still providing Ample Protection. The other lists would not allow Facility Names to Pass.
			ResponseRateRow row = getResponseRateRowByVendor(ESAPIValidator.validateStringInput
					(vendor.getName(), ESAPIValidationType.UpperAndLowerCaseWithNumbers_WhiteList));
			if (row != null) {
				responseRateRows.add(row);
			}
		}
	}
	
	public void buildResponseRateRowsByVisn(List<Visn> visns) {
		responseRateRows = new ArrayList<ResponseRateRow>();
		// Need a list of vendors
		// Iterate through the list and add things to the rows as needed

		for(Visn visn: visns) {
			ResponseRateRow row = getResponseRateRowById(visn.getId());
			if (row != null) {
				responseRateRows.add(row);
			}
		}
	}
	
	public void buildResponseRateRowsByFacility(List<Facility> facilities) {
		responseRateRows = new ArrayList<ResponseRateRow>();
		// Need a list of vendors
		// Iterate through the list and add things to the rows as needed

		for(Facility facility: facilities) {
			// use Facilities WhiteList because of the Vendor name, Visn, and Facilities that all use this method, it is the least restrictive
			// while still providing Ample Protection. The other lists would not allow Facility Names to Pass.
			ResponseRateRow row = getResponseRateRowByVendor(ESAPIValidator.validateStringInput(
					facility.getName(), ESAPIValidationType.FacilitiesName_WhiteList)); // use Facilities WhiteList because of the Vendor name
			if (row != null) {
				responseRateRows.add(row);
			}
		}
	}

	public ResponseRateRow getResponseRateRowByVendor(String vendorName) {
		List<ResponseRateRecord> currentRow = recordsKeyedByVendor.get(vendorName);
		ResponseRateRow row = null;

		if (currentRow != null) {
			row = new ResponseRateRow();
			// use Facilities WhiteList because of the Vendor name, Visn, and Facilities that all use this method, it is the least restrictive
			// while still providing Ample Protection. The other lists would not allow Facility Names to Pass.
			row.setVendorName(ESAPIValidator.validateStringInput(
					vendorName, ESAPIValidationType.FacilitiesName_WhiteList));
			row.setStationNumber(ESAPIValidator.validateStringInput(
					currentRow.get(0).getStationNumber(), ESAPIValidationType.FacilitiesName_WhiteList));
			setTotals(currentRow, row);
		}

		return row;
	}
	
	public ResponseRateRow getResponseRateRowById(Integer visnId) {
		List<ResponseRateRecord> currentRow = recordsKeyedById.get(visnId);
		ResponseRateRow row = null;

		if (currentRow != null) {
			row = new ResponseRateRow();
			// Set the vendor name to the visn/Facilities name
			// use Facilities WhiteList because of the Vendor name, Visn, and Facilities that all use this method, it is the least restrictive
						// while still providing Ample Protection. The other lists would not allow Facility Names to Pass.
			row.setVendorName(ESAPIValidator.validateStringInput(
					currentRow.get(0).getVendorName(), ESAPIValidationType.FacilitiesName_WhiteList));
			row.setVisnId(visnId);
			row.setStationNumber(ESAPIValidator.validateStringInput(
					currentRow.get(0).getStationNumber(), ESAPIValidationType.FacilitiesName_WhiteList));
			setTotals(currentRow, row);
		}

		return row;
	}
	
	private void setTotals(List<ResponseRateRecord> currentRow, ResponseRateRow row) {
		for (ResponseRateRecord record : currentRow) {
			int responseCode = record.getResponseCode();

			switch (responseCode) {
			case RESPONDERS:
				row.setResponders(record.getPatientCount());
				break;
			case PARTIAL_RESPONDERS:
				row.setPartialResponders(record.getPatientCount());
				break;
			case NON_RESPONDERS:
				row.setNonResponders(record.getPatientCount());
				break;
			case UNKOWN:
				row.setUnknown(record.getPatientCount());
				break;
			case DATE_COLS:
				row.setThirtyDaysNonResponse(record.getPatientCount());
				row.setGreatestDaysNonResponse(record.getMaxDataAge());
				break;
			}
		}
		row.setTotal();
	}

}
