package gov.va.med.ccht.model;

import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.validation.Valid;

import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.terminology.InactiveReason;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.security.SecurityContextHelper;

@Entity
@Table(schema="ht", name="APP_USER")
public class User extends UserPrincipalImpl {

	private static final long serialVersionUID = 8175242732620450379L;
	
	private Long id;

	@Valid
	private InactiveReason inactiveReason;
	@Valid
	private RegistrationStatus registrationStatus;
	@Valid
	private RegistrationDeniedReason registrationDeniedReason;
	@Valid
	private SimpleVisn visn;
	@Valid
	private SimpleFacility facility;
	@Valid
	private SimpleFacility secondaryFacility;
	@Valid
	private Vendor vendor;

	private Date accountExpireDate;
	private Date accountLockDate;
	private Date inactiveDate;
	private Date loginFailedDate;
	private Date lastLoginDate;
	private Short loginFailedCount;
	@Valid
	private RegistrationReason registrationReason;
	@Valid
	private Set<UserRole> internalRoles;
	@Valid
	private Set<AppUserGroup> internalUserGroups;
	@Valid
	private Set<UserPreference> internalPreferences;

	// -------------------------------------------------------- Constructors

	public User() {
		super();
	}

	public User(String name) {
		super(name);
	}

	public User(SimpleUser simpleUser) {
		super(simpleUser.getUsername());
		setFirstName(simpleUser.getFirstName());
		setMiddleName(simpleUser.getMiddleName());
		setLastName(simpleUser.getLastName());
		setEmail(simpleUser.getEmail());
		setTelephoneNumber(simpleUser.getTelephoneNumber());
		setInactiveDate(simpleUser.getInactiveDate());
	}

	public User(LdapPerson person) {
		setLdapPerson(person);
	}

	// -------------------------------------------------------- Business Methods

	public void addRole(UserRole userRole) {
		if (userRole != null) {
			userRole.setUser(this);
			getInternalRoles().add(userRole);
		}
	}

	public void removeRole(UserRole userRole) {
		if (userRole != null) {
			userRole.setUser(null);
			getInternalRoles().remove(userRole);
		}
	}

	public void addUserGroup(AppUserGroup userGroup) {
		if (userGroup != null) {
			userGroup.setUser(this);
			getInternalUserGroups().add(userGroup);
		}
	}

	public void removeUserGroup(AppUserGroup userGroup) {
		if (userGroup != null) {
			userGroup.setUser(null);
			getInternalUserGroups().remove(userGroup);
		}
	}

	public String getPreferenceValue(String name) {
		if (name != null) {
			for (UserPreference userPreference : getInternalPreferences()) {
				if (userPreference.getName().equalsIgnoreCase(name)) {
					return userPreference.getValue();
				}
			}
		}
		return null;
	}

	public void addPreference(UserPreference userPreference) {
		if (userPreference != null) {
			userPreference.setUser(this);
			getInternalPreferences().add(userPreference);
		}
	}

	public void addOrUpdatePreference(String name, String value) {
		if (name != null) {
			
			String username = SecurityContextHelper.getCurrentUser().getUsername();
			Date currentDate = Calendar.getInstance().getTime();
			
			for (UserPreference userPreference : getInternalPreferences()) {
				if (userPreference.getName().equalsIgnoreCase(name)) {
					
					userPreference.setValue(value);
					userPreference.setRecordModifiedDate(currentDate);
					userPreference.setRecordModifiedBy(username);
					userPreference.setRecordModifiedCount((short) (userPreference.getRecordModifiedCount() + 1));
					return;
				}
			}
			// not found, new value
			UserPreference userPreference = new UserPreference();
			userPreference.setName(name);
			userPreference.setValue(value);
			userPreference.setRecordCreatedBy(username);
			userPreference.setRecordCreatedDate(currentDate);
			userPreference.setRecordModifiedDate(currentDate);
			userPreference.setRecordModifiedBy(username);
		}
	}

	public void removePreference(UserPreference userPreference) {
		if (userPreference != null) {
			userPreference.setUser(null);
			getInternalPreferences().remove(userPreference);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.fw.model.UserPrincipalImpl#getCachedRoles()
	 */
	@Override
	@Transient
	public Set<String> getCachedRoles() {
		Set<String> cachedRoles = new HashSet<String>();
		for (UserRole role : getInternalRoles()) {
			cachedRoles.add(role.getRole().getName());
		}
		return cachedRoles;
	}

	@Override
	@Transient
	public boolean isAccountLocked() {
		return accountLockDate != null;
	}

	@Override
	@Transient
	public boolean isInactive() {
		return inactiveDate != null && inactiveDate.before(new Date());
	}
	
	

	// -------------------------------------------------------- Common Methods

	public String toString() {
		return getUsername();
	}

	// -------------------------------------------------------- Accessor Methods

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	@Column(name = "APP_USER_ID", unique = true, nullable = false)
	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}
	
	@ManyToOne
	@JoinColumn(name="ACCOUNT_INACTIVE_REASON_ID")
	public InactiveReason getInactiveReason() {
		return inactiveReason;
	}

	public void setInactiveReason(InactiveReason inactiveReason) {
		this.inactiveReason = inactiveReason;
	}

	@ManyToOne
	@JoinColumn(name="REGISTRATION_STATUS_ID")
	public RegistrationStatus getRegistrationStatus() {
		return registrationStatus;
	}

	public void setRegistrationStatus(RegistrationStatus registrationStatus) {
		this.registrationStatus = registrationStatus;
	}

	@ManyToOne
	@JoinColumn(name="REG_DENIED_REASON_ID")
	public RegistrationDeniedReason getRegistrationDeniedReason() {
		return registrationDeniedReason;
	}

	public void setRegistrationDeniedReason(RegistrationDeniedReason registrationDeniedReason) {
		this.registrationDeniedReason = registrationDeniedReason;
	}

	@ManyToOne
	@JoinColumn(name="VISN_ID")
	public SimpleVisn getVisn() {
		return visn;
	}

	public void setVisn(SimpleVisn visn) {
		this.visn = visn;
	}

	@Column(name = "ACCOUNT_EXPIRE_DATE")
	public Date getAccountExpireDate() {
		return accountExpireDate;
	}

	public void setAccountExpireDate(Date accountExpireDate) {
		this.accountExpireDate = accountExpireDate;
	}

	@Column(name = "ACCOUNT_LOCK_DATE")
	public Date getAccountLockDate() {
		return accountLockDate;
	}

	public void setAccountLockDate(Date accountLockDate) {
		this.accountLockDate = accountLockDate;
	}

	@Column(name = "INACTIVE_DATE")
	public Date getInactiveDate() {
		return inactiveDate;
	}

	public void setInactiveDate(Date inactiveDate) {
		this.inactiveDate = inactiveDate;
	}

	@Column(name = "LOGIN_FAILED_DATE")
	public Date getLoginFailedDate() {
		return loginFailedDate;
	}

	public void setLoginFailedDate(Date loginFailedDate) {
		this.loginFailedDate = loginFailedDate;
	}

	@Column(name = "LAST_LOGIN_DATE")
	public Date getLastLoginDate() {
		return lastLoginDate;
	}

	public void setLastLoginDate(Date lastLoginDate) {
		this.lastLoginDate = lastLoginDate;
	}

	@Column(name = "LOGIN_FAILED_COUNT")
	public Short getLoginFailedCount() {
		return loginFailedCount;
	}

	public void setLoginFailedCount(Short loginFailedCount) {
		this.loginFailedCount = loginFailedCount;
	}

	@ManyToOne
	@JoinColumn(name="HT_JOB_DESCRIPTION_ID")
	public RegistrationReason getRegistrationReason() {
		return registrationReason;
	}

	public void setRegistrationReason(RegistrationReason registrationReason) {
		this.registrationReason = registrationReason;
	}

	@ManyToOne
	@JoinColumn(name="PRI_FACILITIES_ID")
	public SimpleFacility getFacility() {
		return facility;
	}

	public void setFacility(SimpleFacility facility) {
		this.facility = facility;
	}

	@ManyToOne
	@JoinColumn(name="SEC_FACILITIES_ID")
	public SimpleFacility getSecondaryFacility() {
		return secondaryFacility;
	}

	public void setSecondaryFacility(SimpleFacility secondaryFacility) {
		this.secondaryFacility = secondaryFacility;
	}

	@ManyToOne
	@JoinColumn(name="VENDOR_ID")
	public Vendor getVendor() {
		return vendor;
	}

	public void setVendor(Vendor vendor) {
		this.vendor = vendor;
	}

	@OneToMany(cascade=CascadeType.ALL, mappedBy="user", fetch=FetchType.EAGER, orphanRemoval=true)
	private Set<UserRole> getInternalRoles() {
		if (internalRoles == null)
			internalRoles = new HashSet<UserRole>();
		return internalRoles;
	}

	@SuppressWarnings("unused")
	private void setInternalRoles(Set<UserRole> internalRoles) {
		this.internalRoles = internalRoles;
	}

	@OneToMany(cascade=CascadeType.ALL, mappedBy="user", fetch=FetchType.EAGER)
	private Set<AppUserGroup> getInternalUserGroups() {
		if (internalUserGroups == null)
			internalUserGroups = new HashSet<AppUserGroup>();
		return internalUserGroups;
	}

	@SuppressWarnings("unused")
	private void setInternalUserGroups(Set<AppUserGroup> internalUserGroups) {
		this.internalUserGroups = internalUserGroups;
	}

	@OneToMany(cascade=CascadeType.ALL, mappedBy="user", fetch=FetchType.EAGER)
	private Set<UserPreference> getInternalPreferences() {
		if (internalPreferences == null)
			internalPreferences = new HashSet<UserPreference>();
		return internalPreferences;
	}

	@SuppressWarnings("unused")
	private void setInternalPreferences(Set<UserPreference> internalPreferences) {
		this.internalPreferences = internalPreferences;
	}

	@Transient
	public Set<UserRole> getRoles() {
		return Collections.unmodifiableSet(getInternalRoles());
	}

	@Transient
	public Set<AppUserGroup> getUserGroups() {
		return Collections.unmodifiableSet(getInternalUserGroups());
	}

	@Transient
	public Set<UserPreference> getPreferencees() {
		return Collections.unmodifiableSet(getInternalPreferences());
	}

}