package gov.va.med.ccht.jobs;

import java.text.ParseException;
import java.util.Date;

import org.apache.commons.lang.Validate;
import org.quartz.CronExpression;
import org.quartz.CronScheduleBuilder;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.impl.StdSchedulerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.ccht.persistent.JobDAO;
import gov.va.med.ccht.persistent.QuartzDAO;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.ccht.service.report.StandardReportService;
import gov.va.med.fw.service.AbstractComponent;

@Component
public class JobScheduler extends AbstractComponent {

	public static final String QIR_VENDOR_RESPONSE_DUE_TRIGGER = "QIRVendorResponseDueCronTrigger";
	public static final String PURGE_COMPLETED_JOBS_TRIGGER = "purgeCompletedReportsCronTrigger";

	public static final String QIR_VENDOR_RESPONSE_DUE_JOB_NAME = "QIRVendorResponseDue";
	public static final String PURGE_COMPLETED_JOBS_JOB_NAME = "PurgeCompletedReports";

	public static final String JOBS_GROUP = "ccht-batch";
	public static final String TRIGGERS_GROUP = "common.static.triggers";

	Scheduler scheduler;

	@Autowired
	private JobDAO jobDao;

	@Autowired
	private QuartzDAO quartzDAO;

	@Autowired
	private QIRService qirService;
	
	@Autowired
	private StandardReportService standardReportService;

	@Override
	public void afterPropertiesSet() throws Exception {

		Validate.notNull(jobDao, "jobDao must not be null");

		JobDataMap jdm = new JobDataMap();
		jdm.put("jobDao", jobDao);
		jdm.put("qirService", qirService);
		jdm.put("reportService", standardReportService);

		JobDetail purgeJob = JobBuilder.newJob(PurgeCompletedJobsJob.class).usingJobData(jdm)
				.storeDurably()
				.withDescription(quartzDAO.findJobDescription(PURGE_COMPLETED_JOBS_JOB_NAME, JOBS_GROUP))
				.withIdentity(PURGE_COMPLETED_JOBS_JOB_NAME, JOBS_GROUP).build();

		String purgeCronExpression = quartzDAO.findCronExpression(PURGE_COMPLETED_JOBS_TRIGGER, TRIGGERS_GROUP);

		Trigger purgeTrigger = TriggerBuilder.newTrigger().withIdentity(PURGE_COMPLETED_JOBS_TRIGGER, TRIGGERS_GROUP)
				.withDescription(quartzDAO.findTriggerDescription(PURGE_COMPLETED_JOBS_TRIGGER, TRIGGERS_GROUP))
				.withSchedule(CronScheduleBuilder.cronSchedule(purgeCronExpression)).build();

		JobDetail qirVendorJob = JobBuilder.newJob(QIRVendorResponseDueJob.class).usingJobData(jdm)
				.storeDurably()
				.withDescription(quartzDAO.findJobDescription(QIR_VENDOR_RESPONSE_DUE_JOB_NAME, JOBS_GROUP))
				.withIdentity(QIR_VENDOR_RESPONSE_DUE_JOB_NAME, JOBS_GROUP).build();

		String qirCronExpression = quartzDAO.findCronExpression(QIR_VENDOR_RESPONSE_DUE_TRIGGER, TRIGGERS_GROUP);

		Trigger qirVendorTrigger = TriggerBuilder.newTrigger()
				.withIdentity(QIR_VENDOR_RESPONSE_DUE_TRIGGER, TRIGGERS_GROUP)
				.withDescription(quartzDAO.findTriggerDescription(QIR_VENDOR_RESPONSE_DUE_TRIGGER, TRIGGERS_GROUP))
				.withSchedule(CronScheduleBuilder.cronSchedule(qirCronExpression)).build();


		try {
			scheduler = new StdSchedulerFactory().getScheduler();
			
			// Schedule purge job only if it is valid
			try {
				CronExpression purgeExpression = new CronExpression(purgeCronExpression);
				if(purgeExpression.getNextValidTimeAfter(new Date()) != null) {
					scheduler.scheduleJob(purgeJob, purgeTrigger);
				}
				else {
					scheduler.addJob(purgeJob, true);
				}
				if (quartzDAO.isPaused(PURGE_COMPLETED_JOBS_TRIGGER, TRIGGERS_GROUP)) {
					scheduler.pauseJob(new JobKey(PURGE_COMPLETED_JOBS_JOB_NAME, JOBS_GROUP));
				}
			}
			catch(ParseException e) {
				e.printStackTrace();
			}
			
			// Schedule qir job only if it is valid. Otherwise add the job and leave it dormant.
			try {
				CronExpression qirExpression = new CronExpression(qirCronExpression);
				if(qirExpression.getNextValidTimeAfter(new Date()) != null) {
					scheduler.scheduleJob(qirVendorJob, qirVendorTrigger);
				}
				else {
					scheduler.addJob(qirVendorJob, true);
				}
				if (quartzDAO.isPaused(PURGE_COMPLETED_JOBS_TRIGGER, TRIGGERS_GROUP)) {
					scheduler.pauseJob(new JobKey(QIR_VENDOR_RESPONSE_DUE_JOB_NAME, JOBS_GROUP));
				}
			}
			catch(ParseException e) {
				e.printStackTrace();
			}
			
			scheduler.start();

		} catch (final SchedulerException e) {
			e.printStackTrace();
		}
	}

	public static String getTriggerName(final String jobName) {
		if (jobName.equalsIgnoreCase(PURGE_COMPLETED_JOBS_JOB_NAME)) {
			return PURGE_COMPLETED_JOBS_TRIGGER;
		} else {
			return QIR_VENDOR_RESPONSE_DUE_TRIGGER;
		}
	}

	public Scheduler getScheduler() {
		return scheduler;
	}

}
