package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.UserPreference;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.service.common.RegistrationService;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.ui.common.ControllerResult;
import gov.va.med.ccht.ui.model.RoleForm;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.fw.model.EntityKeyFactory;
import gov.va.med.fw.model.ldap.SearchCriteria;
import gov.va.med.fw.security.Permission;
import gov.va.med.fw.security.Role;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.StringUtils;

@Controller
public class SecurityController implements CCHTRoles {

	public static final TermType NATIONAL_ADMIN_ROLE = new TermType(NATIONAL_ADMIN, NATIONAL_ADMIN);
	public static final TermType VISN_ADMIN_ROLE = new TermType(VISN_ADMIN, VISN_ADMIN);
	public static final TermType FACILITY_ADMIN_ROLE = new TermType(FACILITY_ADMIN, FACILITY_ADMIN);
	public static final TermType APPLICATION_ADMIN_ROLE = new TermType(APPLICATION_ADMIN, APPLICATION_ADMIN);
	public static final TermType MANAGEMENT_ROLE = new TermType(MANAGEMENT, MANAGEMENT);
	public static final TermType VENDOR_ROLE = new TermType(VENDOR, VENDOR);
	public static final TermType NAC_POC_ROLE = new TermType(NAC, NAC);
	public static final TermType SYSTEM_ADMIN_ROLE = new TermType(SYSTEM_ADMIN, SYSTEM_ADMIN);
	public static final TermType CARE_COORDINATOR_ROLE = new TermType(CARE_COORDINATOR, CARE_COORDINATOR);
	public static final TermType PROGRAM_ASSISTANT_ROLE = new TermType(PROGRAM_ASSISTANT, PROGRAM_ASSISTANT);
	public static final TermType QIR_ORIGINATOR_ROLE = new TermType(QIR_ORIGINATOR, QIR_ORIGINATOR);

	@Autowired
	private SecurityConversionService securityConversionService;
	
	@Autowired
	protected SecurityService securityService;
	
	@Autowired
	protected RegistrationService registrationService;
	
	@Autowired
	protected TerminologyCache terminologyCache;

	public Boolean isServerActive() {
		return true;
	}

	// Manage roles and Permissions
	public RoleForm getRole(String roleName) throws Exception {
		Role role = securityService.getRole(roleName);
		RoleForm rf = new RoleForm();
		rf.setId(String.valueOf(role.getId()));
		rf.setName(role.getName());
		rf.setDescription(role.getDescription());
		List<TermType> permissions = new ArrayList<TermType>();
		for (Permission permission : role.getPermissions()) {
			permissions.add(new TermType(permission.getName(), permission.getName()));
		}
		// rf.setPermissions(permissions);
		List<TermType> allPermissions = getAllPermissions();
		allPermissions.removeAll(permissions);
		// rf.setAvailablePermissions(allPermissions);
		return rf;
	}

	public Map<Long, SimpleRole> getAllRolesLookUpMap() throws Exception {
		Map<Long, SimpleRole> allRoles = new LinkedHashMap<Long, SimpleRole>();
		List<SimpleRole> roles = securityService.getAllSimpleRoles();
		for (SimpleRole role : roles) {
			allRoles.put(role.getId(), role);
		}
		return allRoles;
	}

	public Map<Long, String> getAllRolesMap(Map<Long, SimpleRole> roles) throws Exception {
		Map<Long, String> allRoles = new LinkedHashMap<Long, String>();
		for (Map.Entry<Long, SimpleRole> role : roles.entrySet()) {
			SimpleRole entry = role.getValue();
			allRoles.put(entry.getId(), entry.getName());
		}
		return allRoles;
	}

	public List<TermType> getAllRoles() throws Exception {
		List<TermType> allRoles = new ArrayList<TermType>();
		List<SimpleRole> roles = securityService.getAllSimpleRoles();
		for (SimpleRole role : roles) {
			allRoles.add(new TermType(role.getDescription(), role.getName()));
		}
		return allRoles;

	}

	public List<TermType> getAllPermissions() throws Exception {
		List<TermType> allPermissions = new ArrayList<TermType>();
		List<Permission> permissions = securityService.getAllPermissions();
		for (Permission permission : permissions) {
			allPermissions.add(new TermType(permission.getName(), permission.getName()));
		}
		return allPermissions;

	}

	public UserForm getUser(String userName) throws Exception {
		UserPrincipal currentUser = SecurityContextHelper.getSecurityContext().getUserPrincipal();
		User user = securityService.getUser(userName);
		UserForm userForm = new UserForm();
		securityConversionService.convert(user, userForm);
		List<TermType> assignedRoles = userForm.getRoles();
		List<TermType> availableRoles = new ArrayList<TermType>(); // getAllRoles();
		List<TermType> allRoles = new ArrayList<TermType>(); // getAllRoles();
		List<SimpleRole> roles = securityService.getAllSimpleRoles();

		for (SimpleRole role : roles) {
			allRoles.add(new TermType(role.getName(), role.getName()));
		}
		// filter the available roles based on the user role
		if (currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN)) {
			// no filtering
			availableRoles = allRoles;
		} else {
			// availableRoles.remove(NATIONAL_ADMIN_ROLE);
			// availableRoles.remove(APPLICATION_ADMIN_ROLE);
			// availableRoles.remove(MANAGEMENT_ROLE);
			User loginUser = (User) currentUser;
			boolean canAddRole = false;
			if (currentUser.isPermissionGranted(CCHTRoles.FACILITY_ADMIN)) {

				if (loginUser.getFacility() != null) {
					if (user.getFacility() != null && (loginUser.getFacility().getId() == user.getFacility()
							.getId())) {
						canAddRole = true;
					} else if (user.getSecondaryFacility() != null && (loginUser.getFacility().getId() == user.getSecondaryFacility().getId())) {
						canAddRole = true;
					}
				} else if (loginUser.getSecondaryFacility() != null) {
					if (user.getFacility() != null && (loginUser.getSecondaryFacility().getId() == user
							.getFacility().getId())) {
						canAddRole = true;
					} else if (user.getSecondaryFacility() != null && (loginUser.getSecondaryFacility().getId() == user.getSecondaryFacility().getId())) {
						canAddRole = true;
					}
				}
				if (canAddRole) {
					availableRoles.add(FACILITY_ADMIN_ROLE);
					availableRoles.add(CARE_COORDINATOR_ROLE);
					availableRoles.add(PROGRAM_ASSISTANT_ROLE);
					availableRoles.add(APPLICATION_ADMIN_ROLE);
				}
			}
			canAddRole = false;
			if (currentUser.isPermissionGranted(CCHTRoles.VISN_ADMIN)) {
				// keep Facility Administrator, Care Coordinator, QIR
				// Originator, Program Support Assistant, Application
				// Administrator
				// if user and current user are in the same VISN, otherwise,
				// remove all
				if (loginUser.getVisn() != null && user.getVisn() != null
						&& (loginUser.getVisn().getId() == user.getVisn().getId())) {
					canAddRole = true;
				}
				if (canAddRole) {
					if (!availableRoles.contains(VISN_ADMIN_ROLE))
						availableRoles.add(VISN_ADMIN_ROLE);
					if (!availableRoles.contains(FACILITY_ADMIN_ROLE))
						availableRoles.add(FACILITY_ADMIN_ROLE);
					if (!availableRoles.contains(CARE_COORDINATOR_ROLE))
						availableRoles.add(CARE_COORDINATOR_ROLE);
					if (!availableRoles.contains(QIR_ORIGINATOR_ROLE))
						availableRoles.add(QIR_ORIGINATOR_ROLE);
					if (!availableRoles.contains(PROGRAM_ASSISTANT_ROLE))
						availableRoles.add(PROGRAM_ASSISTANT_ROLE);
				}
			}
		}

		availableRoles.removeAll(assignedRoles);
		userForm.setAvailableRoles(availableRoles);

		// Preferences
		// Set a default value of MDI to false for HTIE_CodeCR436
		userForm.setMultipleWindows(getBooleanValue(user.getPreferenceValue(UserPreference.MULTIPLE_WINDOWS), false));

		return userForm;
	}

	public UserForm getUser(UserForm userForm) throws Exception {
		return getUser(userForm.getUserName());
	}

	public List<UserForm> findUsers(SearchCriteria searchCriteria) throws Exception {
		List<UserForm> users = new ArrayList<UserForm>();
		List<User> cchtUsers = securityService.findAppUsers(searchCriteria);
		// if only single user is found return the object
		if (cchtUsers != null && cchtUsers.size() > 0) {
			// found users
			if (cchtUsers.size() > 1) {
				for (User user : cchtUsers) {
					UserForm userForm = new UserForm();
					securityConversionService.convert(user, userForm);
					users.add(userForm);
				}
			} else {
				User user = cchtUsers.get(0);
				UserForm userForm = getUser(user.getName());
				users.add(userForm);
			}
		}
		return users;
	}

	public ControllerResult updateUser(UserForm userForm) throws Exception {
		User user = securityService.getUser(userForm.getUserName());
		String oldRegStatusCode = user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();
		securityConversionService.convert(userForm, user);
		securityService.updateUser(user);
		String newRegStatusCode = user.getRegistrationStatus() == null ? "" : user.getRegistrationStatus().getCode();
		if (!oldRegStatusCode.equals(newRegStatusCode)) {
			if (RegistrationStatus.APPROVED.equals(newRegStatusCode)) {
				registrationService.sendWelcomeMail(user);
			} else if (RegistrationStatus.REJECTED.equals(newRegStatusCode)) {
				registrationService.sendDeniedMail(user);
			} else {
				// submitted nothing to do moved to pending status
			}
		}
		return new ControllerResult(ControllerResult.SUCCESS);
	}

	public ControllerResult updateUserProfile(UserForm userForm) throws Exception {
		User user = securityService.getUser(userForm.getUserName());
		user.addOrUpdatePreference(UserPreference.MULTIPLE_WINDOWS, String.valueOf(userForm.getMultipleWindows()));
		securityService.updateUser(user);
		return new ControllerResult(ControllerResult.SUCCESS);
	}

	public List<UserForm> getSubmittedRegistrations() throws Exception {
		List<UserForm> users = new ArrayList<UserForm>();
		Integer visnId = -1;
		Integer facilityId = -1;
		UserPrincipal currentUser = SecurityContextHelper.getSecurityContext().getUserPrincipal();
		List<User> registrations;
		if (currentUser.isPermissionGranted(NATIONAL_ADMIN)) {
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
			// all visns and sites
		} else if (currentUser.isPermissionGranted(VISN_ADMIN)) {
			visnId = ((User) currentUser).getVisn().getId();
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
		} else if (currentUser.isPermissionGranted(FACILITY_ADMIN)) {
			facilityId = ((User) currentUser).getFacility().getId();
			registrations = securityService.getSubmittedRegistrations(visnId, facilityId);
		} else {
			return users;
		}
		// convert
		for (User registration : registrations) {
			UserForm userForm = new UserForm();
			userForm.setUserName(registration.getName());
			userForm.setGivenName(registration.getFirstName());
			userForm.setFamilyName(registration.getLastName());
			userForm.setMiddleName(registration.getMiddleName());
			userForm.setTelephoneNumber(registration.getTelephoneNumber());
			RegistrationStatus registrationStatus = registration.getRegistrationStatus();
			if (registrationStatus != null) {
				userForm.setRegistrationStatus(
						new TermType(registrationStatus.getName(), registrationStatus.getCode()));
			}
			SimpleVisn visn = registration.getVisn();
			if (visn != null) {
				userForm.setVisn(new TermType(visn.getName(), String.valueOf(visn.getId())));
			}
			SimpleFacility facility = registration.getFacility();
			if (facility != null) {
				userForm.setFacility(new TermType(facility.getName(), facility.getStationNumber()));
			}
			RegistrationReason registrationReason = registration.getRegistrationReason();
			if (registrationReason != null) {
				userForm.setRegistrationReason(
						new TermType(registrationReason.getName(), registrationReason.getCode()));
			}
			RegistrationDeniedReason registrationDeniedReason = registration.getRegistrationDeniedReason();
			if (registrationDeniedReason != null) {
				userForm.setRegistrationDeniedReason(
						new TermType(registrationDeniedReason.getName(), registrationDeniedReason.getCode()));
			}
			users.add(userForm);
		}
		return users;
	}

	public List<UserForm> approveRegistration(UserForm userForm) throws Exception {
		User user = securityService.getUser(userForm.getUserName());
		user.setRegistrationStatus(
				terminologyCache.getTermByCode(RegistrationStatus.class, RegistrationStatus.APPROVED));
		// update
		registrationService.approveUser(user);
		return getSubmittedRegistrations();
	}

	public List<UserForm> rejectRegistration(UserForm userForm) throws Exception {
		User user = securityService.getUser(userForm.getUserName());
		user.setRegistrationStatus(
				terminologyCache.getTermByCode(RegistrationStatus.class, RegistrationStatus.REJECTED));
		registrationService.rejectUser(user);
		return getSubmittedRegistrations();
	}

	private Boolean getBooleanValue(String value, Boolean defaultValue) {
		if (value != null) {
			return Boolean.parseBoolean(value);
		}
		return defaultValue;
	}
}
