/********************************************************************
 * Copyright � 2010 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.ccht.controller;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.Schedule;
import gov.va.med.ccht.model.common.DIReportParameters;
import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.ReportWeek;
import gov.va.med.ccht.model.common.SimpleVisn;
import gov.va.med.ccht.model.report.ReportSchedule;
import gov.va.med.ccht.model.report.ReportSetup;
import gov.va.med.ccht.model.report.ScheduledReport;
import gov.va.med.ccht.model.report.SimpleCompletedReport;
import gov.va.med.ccht.model.report.scheduled.DateRangeBoundary;
import gov.va.med.ccht.model.report.scheduled.DateRangeBoundaryType;
import gov.va.med.ccht.model.report.scheduled.FixedDateType;
import gov.va.med.ccht.model.report.scheduled.MultiDayTimeUnitType;
import gov.va.med.ccht.model.report.scheduled.SimpleTimeUnitType;
import gov.va.med.ccht.model.terminology.DayOfMonth;
import gov.va.med.ccht.model.terminology.DayOfQuarter;
import gov.va.med.ccht.model.terminology.DayOfWeek;
import gov.va.med.ccht.model.terminology.Month;
import gov.va.med.ccht.model.terminology.ReportFormat;
import gov.va.med.ccht.model.terminology.ReportType;
import gov.va.med.ccht.model.terminology.RunFrequency;
import gov.va.med.ccht.model.terminology.ScheduleType;
import gov.va.med.ccht.model.terminology.StandardReport;
import gov.va.med.ccht.model.terminology.WeekOfMonth;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.service.common.TerminologyService;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.ui.common.AbstractConversionService;
import gov.va.med.ccht.ui.model.CompletedReportForm;
import gov.va.med.ccht.ui.model.FormattedDateForm;
import gov.va.med.ccht.ui.model.ReportParametersForm;
import gov.va.med.ccht.ui.model.ReportScheduleForm;
import gov.va.med.ccht.ui.model.ReportSetupForm;
import gov.va.med.ccht.ui.model.ScheduledReportForm;
import gov.va.med.ccht.ui.model.StandardReportForm;
import gov.va.med.ccht.util.TransferUtils;
import gov.va.med.fw.model.AbstractEntity;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.report.ReportExportedType;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.ui.conversion.ConversionService;
import gov.va.med.fw.ui.conversion.ConversionServiceException;
import gov.va.med.fw.ui.model.Form;
import gov.va.med.fw.ui.model.TermType;
//import gov.va.med.fw.util.DateUtils;
import gov.va.med.fw.util.date.DateWithTimeZone;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * 
 * @author DNS
 */
@Service
public class ReportConversionService extends AbstractConversionService implements ConversionService { 
	
	private Logger logger = Logger.getLogger(ReportConversionService.class);
	
	@Override
	public void convert(Schedule bom, ReportScheduleForm form) { 
		logger.error("$$$$$$$ fix me.");
	}
	
	@Override
	public void convert(AbstractEntity bom, Form form) throws ConversionServiceException {
		if (bom instanceof ScheduledReport && form instanceof ScheduledReportForm) {
			toScheduledReportForm((ScheduledReport) bom, (ScheduledReportForm) form);
		} else if (bom instanceof SimpleCompletedReport) {
			toCompletedReportForm((SimpleCompletedReport) bom, (CompletedReportForm) form);
		} else if (bom instanceof ReportSetup && form instanceof ReportSetupForm) {
			toReportSetupForm((ReportSetup) bom, (ReportSetupForm) form);
		} else if (bom instanceof ScheduledReport && form instanceof ReportSetupForm) {
			toReportSetupForm((ScheduledReport) bom, (ReportSetupForm) form);			
		} else {
			throw new ConversionServiceException("Unsupported conversion exception");
		}
	}

	@Override
	public void convert(Form form, AbstractEntity bom) throws ConversionServiceException {
		try {
			if (form instanceof ReportSetupForm && bom instanceof ScheduledReport) {
				toScheduledReport((ReportSetupForm) form, (ScheduledReport) bom);
			}
			else if (form instanceof ReportSetupForm && bom instanceof ReportSetup) {
				toReportSetup((ReportSetupForm) form, (ReportSetup) bom);
			}else {
				throw new Exception ("Unknown Parameter types: " + 
						form.getClass().getName() + ", " + bom.getClass().getName());
			}
		} catch (Exception te) {
			throw new ConversionServiceException(te.getMessage(), te);
		}
	}

	private void toScheduledReport(ReportSetupForm reportSetupForm, ScheduledReport scheduledReport)
	throws Exception {
		scheduledReport.setId(reportSetupForm.getId() != null ? new Long(reportSetupForm.getId())
				: null);
		
		// convert standard report (terminology lookup)
		StandardReport standardReport = (StandardReport) getByCode(StandardReport.class,
				reportSetupForm.getStandardReport().getValue());
		scheduledReport.setStandardReport(standardReport);
		// Output file type
		ReportExportedType fileType = (ReportExportedType) TransferUtils.transferTermTypeToLookup(
				getTerminologyCache().getTerminologyService(), ReportExportedType.class,
				reportSetupForm.getFileType());
		scheduledReport.setFileType(fileType);
		
		// convert e-mail address
		scheduledReport.setEmail(reportSetupForm.getEmailAddress());
		// Parameters
		DIReportParameters parameters = new DIReportParameters();
		ReportParametersForm rpForm = reportSetupForm.getReportParameters();
		toReportParameters(rpForm, parameters);
		scheduledReport.setReportParameters(parameters);
		
		// Schedule
		ReportSchedule reportSchedule = new ReportSchedule();
		ReportScheduleForm reportScheduleForm = reportSetupForm.getReportSchedule();
		toSchedule(reportScheduleForm, reportSchedule);
		scheduledReport.setReportSchedule(reportSchedule);
	}

	
	private void toReportSetup(ReportSetupForm reportSetupForm, ReportSetup reportSetup)
			throws Exception {
		reportSetup.setId(reportSetupForm.getId() != null ? new Long(reportSetupForm.getId())
				: null);

		// convert standard report (terminology lookup)
		StandardReport standardReport = (StandardReport) getByCode(StandardReport.class,
				reportSetupForm.getStandardReport().getValue());
		reportSetup.setStandardReport(standardReport);
		// Output file type
		ReportExportedType fileType = (ReportExportedType) TransferUtils.transferTermTypeToLookup(
				getTerminologyCache().getTerminologyService(), ReportExportedType.class,
				reportSetupForm.getFileType());
		reportSetup.setFileType(fileType);

		// convert e-mail address
		reportSetup.setEmailAddress(reportSetupForm.getEmailAddress());
		// Parameters
		DIReportParameters parameters = new DIReportParameters();
		ReportParametersForm rpForm = reportSetupForm.getReportParameters();
		toReportParameters(rpForm, parameters);
		reportSetup.setReportParameters(parameters);
		
		// Schedule
		ReportSchedule reportSchedule = new ReportSchedule();
		ReportScheduleForm reportScheduleForm = reportSetupForm.getReportSchedule();
		toSchedule(reportScheduleForm, reportSchedule);
		reportSetup.setReportSchedule(reportSchedule);
	}

	private void toReportParameters(ReportParametersForm rpForm, DIReportParameters parameters)
			throws Exception {
		// convert report parameters		
		parameters.setUserTimeZone(getCurrentUser().getCurrentTimeZone());
		parameters.setToDate(rpForm.getToDate());
		parameters.setFromDate(rpForm.getFromDate());
		parameters.setAsOfDate(rpForm.getAsOfDate());
		parameters.setReportName(rpForm.getReportName());
		if (rpForm.getReportSummaryType() != null) {
			parameters.setReportSummaryType(rpForm.getReportSummaryType().getValue());
		}
		if (rpForm.getSubTotalType() != null) {
			parameters.setSubTotalType(rpForm.getSubTotalType().getValue());
		}
		parameters.setReportWeek(rpForm.getReportWeek());
		if (ReportConstants.REPORT_SUMMARY_TYPE_VISN.equals(parameters.getReportSummaryType()) && rpForm.getVisn() != null) {				
			if (SimpleVisn.ALL_VISNS.getName().equals(rpForm.getVisn().getValue())) {
				parameters.setAllVisns(true);
			}else {		
				parameters.setAllVisns(false);
				parameters.setVisn(getVisnService().getSimpleVisn(rpForm.getVisn().getValue()));
			}
		}
		if (ReportConstants.REPORT_SUMMARY_TYPE_FACILITY.equals(parameters.getReportSummaryType()) && rpForm.getFacility() != null) {
			if (Facility.ALL_FACILITIES.getName().equals(rpForm.getFacility().getValue())) {
				parameters.setAllFacilities(true);
			}else {
				parameters.setAllFacilities(false);
				parameters.setFacility(
					getFacilityService().getFacility(rpForm.getFacility().getValue()));
			}
		}	
		//device statuses
		parameters.setAllDeviceStatuses(rpForm.getAllDeviceStatuses());

		
		//from and to dates for a scheduled report
		TermType scheduledReportFromType = rpForm.getScheduledReportFromType();
		if (scheduledReportFromType != null) {
			DateRangeBoundary fromBoundary = new DateRangeBoundary();
			fromBoundary.setFixedDateType(toLookup(FixedDateType.class, rpForm
					.getScheduledReportFromFixedDateType()));
			fromBoundary.setMultiDayTimeUnitType(toLookup(MultiDayTimeUnitType.class, rpForm
					.getScheduledReportFromMultiDayTimeUnitType()));
			fromBoundary.setNumUnitsPriorToCurrentDay(rpForm
					.getScheduledReportFromNumUnitsPriorToCurrentDay());
			fromBoundary.setSpecificFixedDate(rpForm.getScheduledReportFromSpecificFixedDate());
			fromBoundary.setType(toLookup(DateRangeBoundaryType.class, rpForm
					.getScheduledReportFromType()));
			fromBoundary.setUnitPriorToCurrentDay(toLookup(SimpleTimeUnitType.class, rpForm
					.getScheduledReportFromUnitPriorToCurrentDay()));
			fromBoundary.setUsePreviousDatePeriod(rpForm
					.getScheduledReportFromUsePreviousDatePeriod());
			parameters.setScheduledReportFromDate(fromBoundary);
		}

		TermType scheduledReportToType = rpForm.getScheduledReportToType();
		if (scheduledReportToType != null) {
			DateRangeBoundary toBoundary = new DateRangeBoundary();
			toBoundary.setFixedDateType(toLookup(FixedDateType.class, rpForm
					.getScheduledReportToFixedDateType()));
			toBoundary.setMultiDayTimeUnitType(toLookup(MultiDayTimeUnitType.class, rpForm
					.getScheduledReportToMultiDayTimeUnitType()));
			toBoundary.setNumUnitsPriorToCurrentDay(rpForm
					.getScheduledReportToNumUnitsPriorToCurrentDay());
			toBoundary.setSpecificFixedDate(rpForm.getScheduledReportToSpecificFixedDate());
			toBoundary.setType(toLookup(DateRangeBoundaryType.class, rpForm
					.getScheduledReportToType()));
			toBoundary.setUnitPriorToCurrentDay(toLookup(SimpleTimeUnitType.class, rpForm
					.getScheduledReportToUnitPriorToCurrentDay()));
			toBoundary.setUsePreviousDatePeriod(rpForm.getScheduledReportToUsePreviousDatePeriod());
			parameters.setScheduledReportToDate(toBoundary);
		}

		Integer daysToKeepCompletedReports = rpForm.getScheduledReportDaysToKeepCompletedReports();
		if (daysToKeepCompletedReports != null)
			parameters.setDaysToKeepCompletedReports(daysToKeepCompletedReports);

		// Format
		if (rpForm.getReportFormat() != null) {
			ReportFormat reportFormat = (ReportFormat) getByCode(ReportFormat.class, rpForm
					.getReportFormat().getValue());
			parameters.setReportFormat(reportFormat);
		}
	}

	private void toReportScheduleForm(Schedule rs, ReportScheduleForm form) {
		form.setDateToGenerate(new FormattedDateForm(rs.getDateToGenerate()));
		form.setDayOfMonth(toTermType(rs.getDayOfMonth()));
		form.setDayOfQuarter(toTermType(rs.getDayOfQuarter()));
		form.setDayOfWeek(toTermType(rs.getDayOfWeek()));
		form.setDaysOfWeek(toTermTypeList(rs.getDaysOfWeek()));
		form.setHour(rs.getHour());
		form.setMinute(rs.getMinute());
		form.setMonth(toTermType(rs.getMonth()));
		form.setNumericDayOfMonth(rs.getNumericDayOfMonth());
		form.setRunFrequency(toTermType(rs.getRunFrequency()));
		form.setScheduleType(toTermType(rs.getScheduleType()));
		form.setWeekOfMonth(toTermType(rs.getWeekOfMonth()));
		form.setCronExpression(rs.getCronExpression());
		form.setFriendlyDescription(rs.getFriendlyDescription());
	}
	
	private void toSchedule(ReportScheduleForm reportScheduleForm, Schedule reportSchedule)
			throws TerminologyException{
		// convert report schedule
		if (reportScheduleForm != null) {
			TerminologyService terminologyService = getTerminologyCache().getTerminologyService();
			RunFrequency reportRunFrequency = TransferUtils.transferTermTypeToLookup(
					terminologyService, RunFrequency.class, reportScheduleForm
							.getRunFrequency());
			reportSchedule.setRunFrequency(reportRunFrequency);
			ScheduleType scheduleType = TransferUtils.transferTermTypeToLookup(
					terminologyService, ScheduleType.class, reportScheduleForm
							.getScheduleType());
			reportSchedule.setScheduleType(scheduleType);
			DayOfMonth dayOfMonth = TransferUtils.transferTermTypeToLookup(terminologyService,
					DayOfMonth.class, reportScheduleForm.getDayOfMonth());
			reportSchedule.setDayOfMonth(dayOfMonth);
			WeekOfMonth weekOfMonth = TransferUtils.transferTermTypeToLookup(terminologyService,
					WeekOfMonth.class, reportScheduleForm.getWeekOfMonth());
			reportSchedule.setWeekOfMonth(weekOfMonth);
			// private DayOfWeek dayOfWeek; stored as set daysOfWeek
			DayOfQuarter dayOfQuarter = TransferUtils.transferTermTypeToLookup(terminologyService,
					DayOfQuarter.class, reportScheduleForm.getDayOfQuarter());
			reportSchedule.setDayOfQuarter(dayOfQuarter);
			Month month = TransferUtils.transferTermTypeToLookup(terminologyService, Month.class,
					reportScheduleForm.getMonth());
			reportSchedule.setMonth(month);
			reportSchedule.setNumericDayOfMonth(reportScheduleForm.getNumericDayOfMonth());
			reportSchedule.setHour(reportScheduleForm.getHour());
			reportSchedule.setMinute(reportScheduleForm.getMinute());
			// daysOfWeek
			Set<DayOfWeek> reportDaysOfWeek = new HashSet<DayOfWeek>();
			List<TermType> daysOfWeek = reportScheduleForm.getDaysOfWeek();
			if (daysOfWeek != null && daysOfWeek.size() > 0) {
				for (TermType termType : daysOfWeek) {
					DayOfWeek dayOfWeek = (DayOfWeek) getByCode(DayOfWeek.class, termType
							.getValue());
					reportDaysOfWeek.add(dayOfWeek);
				}
				reportSchedule.setDaysOfWeek(reportDaysOfWeek);
			}
			DayOfWeek dayOfWeek = (DayOfWeek) TransferUtils.transferTermTypeToLookup(
					terminologyService, DayOfWeek.class, reportScheduleForm.getDayOfWeek());
			reportSchedule.setDayOfWeek(dayOfWeek);
			DateWithTimeZone dt = new DateWithTimeZone(reportScheduleForm.getDateToGenerate()
					.getDate(), SecurityContextHelper.getCurrentUser().getCurrentTimeZone());
			reportSchedule.setDateToGenerate(dt);
		}
	}

	private void toCompletedReportForm(SimpleCompletedReport completedReport,
			CompletedReportForm completedRepForm) throws ConversionServiceException {
		try {
			StandardReport standardReport = completedReport.getStandardReport();
			completedRepForm.setId(String.valueOf(completedReport.getId()));
			StandardReportForm stdRepForm = new StandardReportForm();
			ReportType reportType = standardReport.getReportType();
			stdRepForm.setStandardReport(new TermType(standardReport.getName(), standardReport
					.getCode()));
			stdRepForm.setReportType(new TermType(reportType.getName(), reportType.getCode()));
			completedRepForm.setStandardReport(stdRepForm);
			FormattedDateForm completedDate = new FormattedDateForm();
			completedDate.setDate(completedReport.getCompletedDate());
			completedDate.setFormattedDate(TimeZoneUtils.convertDateToTimeZone(completedReport
					.getCompletedDate(), TimeZone.getDefault()));
			completedRepForm.setCompletedDate(completedDate);
			ReportExportedType fileType = completedReport.getFileType();
			completedRepForm.setFileType(new TermType(fileType.getName(), fileType.getCode()));
			completedRepForm
					.setReportParameterText(tohtml(completedReport.getReportParameterText()));
			completedRepForm.setUserName(completedReport.getUserName());
			completedRepForm.setUserFullName(completedReport.getUserFullName());
			if (completedReport.getExpirationDate() != null) {
				FormattedDateForm expirationdDate = new FormattedDateForm();
				expirationdDate.setDate(completedReport.getExpirationDate());
				expirationdDate.setFormattedDate(TimeZoneUtils.convertDateToTimeZone(
						completedReport.getExpirationDate(), TimeZone.getDefault()));
				completedRepForm.setExpirationDate(expirationdDate);
			}
			completedRepForm.setReportName(completedReport.getReportName());
		} catch (Exception e) {
			throw new ConversionServiceException(
					"Conversion from SimpleCompletedReport to  CompletedReportForm failed ", e);
		}
	}

	private void toReportSetupForm(ReportSetup reportSetup, ReportSetupForm reportSetupForm)
			throws ConversionServiceException {
		// convert report setup
		StandardReport standardReport = reportSetup.getStandardReport();
		reportSetupForm.setId(reportSetup.getId() == null ? null : String.valueOf(reportSetup
				.getId()));
		reportSetupForm.setStandardReport(new TermType(standardReport.getName(), standardReport
				.getCode()));
		reportSetupForm.setFileType(new TermType(reportSetup.getFileType().getName(), reportSetup
				.getFileType().getCode()));
		reportSetupForm.setEmailAddress(reportSetup.getEmailAddress());
		// convert report parameters
		DIReportParameters parameters = (DIReportParameters) reportSetup.getReportParameters();
		ReportParametersForm rpForm = new ReportParametersForm();
		reportSetupForm.setReportParameters(rpForm);
		toReportParametersForm(parameters, rpForm);
	}
	
	/**
	 * Convert ScheduledReport to ReportSetupForm
	 * @param scheduledReport
	 * @param reportSetupForm
	 * @throws ConversionServiceException
	 */
	private void toReportSetupForm(ScheduledReport scheduledReport, ReportSetupForm reportSetupForm)
	throws ConversionServiceException {
		
		StandardReport standardReport = scheduledReport.getStandardReport();
		reportSetupForm.setId(scheduledReport.getId() == null ? null : String.valueOf(scheduledReport
				.getId()));
		reportSetupForm.setStandardReport(new TermType(standardReport.getName(), standardReport
				.getCode()));
		reportSetupForm.setFileType(new TermType(scheduledReport.getFileType().getName(), scheduledReport
				.getFileType().getCode()));
		reportSetupForm.setEmailAddress(scheduledReport.getEmail());
		
		// convert schedule
		ReportSchedule reportSchedule = scheduledReport.getReportSchedule();
		ReportScheduleForm reportScheduleForm = new ReportScheduleForm();
		reportSetupForm.setReportSchedule(reportScheduleForm);
		toReportScheduleForm(reportSchedule, reportScheduleForm);
		
		// convert report parameters
		DIReportParameters parameters = (DIReportParameters) scheduledReport.getReportParameters();
		ReportParametersForm rpForm = new ReportParametersForm();
		reportSetupForm.setReportParameters(rpForm);
		toReportParametersForm(parameters, rpForm);
	}
	
	private void toScheduledReportForm(ScheduledReport scheduledReport,
			ScheduledReportForm scheduledReportForm) throws ConversionServiceException {
		scheduledReportForm.setId(String.valueOf(scheduledReport.getId()));
		StandardReport standardReport = scheduledReport.getStandardReport();
		StandardReportForm stdRepForm = new StandardReportForm();
		ReportType reportType = standardReport.getReportType();
		stdRepForm.setStandardReport(new TermType(standardReport.getName(), standardReport
				.getCode()));
		stdRepForm.setReportType(new TermType(reportType.getName(), reportType.getCode()));
		scheduledReportForm.setStandardReport(stdRepForm);
		ReportExportedType fileType = scheduledReport.getFileType();
		scheduledReportForm.setFileType(new TermType(fileType.getName(), fileType.getCode()));
		scheduledReportForm.setSetupUser(scheduledReport.getUserName());
		scheduledReportForm.setEmailAddress(scheduledReport.getEmail());
		//scheduledReportForm.setReportParametersText(scheduledReport.getReportParameterText());
		//scheduledReportForm.setReportScheduleText(scheduledReport.getReportScheduleText());
		
		//Schedule
		scheduledReportForm.setFrequency(scheduledReport.getReportSchedule() != null ? scheduledReport
				.getReportSchedule().getFriendlyDescription() : null);
		
		//TODO Date Range is not used yet for the scheduled reports
		DIReportParameters rp = (DIReportParameters) scheduledReport.getReportParameters();
		if (rp != null) {
			if (rp.getFromDate() != null)
				//scheduledReportForm.setFromDate(DateUtils.format(rp.getFromDate(),null));
			if (rp.getToDate() != null)
				//scheduledReportForm.setToDate(DateUtils.format(rp.getToDate(),null));
			
			if (rp.getReportWeek() != null) {
				ReportWeek rw = rp.getReportWeek();
				//scheduledReportForm.setFromDate(DateUtils.format(rw.getStartDate(),null));
				//scheduledReportForm.setToDate(DateUtils.format(rw.getEndDate(),null));
			}
		}
	}

	private void toReportParametersForm(DIReportParameters parameters, ReportParametersForm rpForm)
			throws ConversionServiceException {

		// parameters.setUserTimeZone(getCurrentUser().getCurrentTimeZone());
		rpForm.setToDate(parameters.getToDate());
		rpForm.setFromDate(parameters.getFromDate());
		rpForm.setAsOfDate(parameters.getAsOfDate());
		rpForm.setReportWeek(parameters.getReportWeek());
		rpForm.setReportFormat(toTermType(parameters.getReportFormat()));
		if (StringUtils.isNotEmpty(parameters.getReportSummaryType())) {
			rpForm.setReportSummaryType(new TermType(parameters.getReportSummaryType(),parameters.getReportSummaryType()));
		}
		if (parameters.getAllVisns()) {
			SimpleVisn allVisns = SimpleVisn.ALL_VISNS;
			rpForm.setVisn(new TermType(allVisns.getName(), allVisns.getName()));			
		}else if (parameters.getVisn() != null){
			rpForm.setVisn(toTermType(parameters.getVisn()));
		}
		if (parameters.getAllFacilities()) {
			Facility allFacilities = Facility.ALL_FACILITIES;
			rpForm.setFacility(new TermType(allFacilities.getName(), allFacilities.getName()));
		}else if (parameters.getFacility() != null){
			rpForm.setFacility(toTermType(parameters.getFacility()));
		}		
		if (StringUtils.isNotEmpty(parameters.getSubTotalType())) {
			rpForm.setSubTotalType(new TermType(parameters.getSubTotalType(),parameters.getSubTotalType()));
		}
		
		//device statuses
		rpForm.setAllDeviceStatuses(parameters.getAllDeviceStatuses());
		
		//Scheduling parameters
		DateRangeBoundary srFromDate = parameters.getScheduledReportFromDate();
		if (srFromDate != null) {
			rpForm.setScheduledReportFromFixedDateType(toTermType(srFromDate.getFixedDateType()));
			rpForm.setScheduledReportFromMultiDayTimeUnitType(toTermType(srFromDate
					.getMultiDayTimeUnitType()));
			rpForm.setScheduledReportFromNumUnitsPriorToCurrentDay(srFromDate
					.getNumUnitsPriorToCurrentDay());
			rpForm.setScheduledReportFromSpecificFixedDate(srFromDate.getSpecificFixedDate());
			rpForm.setScheduledReportFromType(toTermType(srFromDate.getType()));
			rpForm.setScheduledReportFromUnitPriorToCurrentDay(toTermType(srFromDate
					.getUnitPriorToCurrentDay()));
			rpForm.setScheduledReportFromUsePreviousDatePeriod(srFromDate
					.getUsePreviousDatePeriod());
		}

		DateRangeBoundary srToDate = parameters.getScheduledReportToDate();
		if (srToDate != null) {
			rpForm.setScheduledReportToFixedDateType(toTermType(srToDate.getFixedDateType()));
			rpForm.setScheduledReportToMultiDayTimeUnitType(toTermType(srToDate
					.getMultiDayTimeUnitType()));
			rpForm.setScheduledReportToNumUnitsPriorToCurrentDay(srToDate
					.getNumUnitsPriorToCurrentDay());
			rpForm.setScheduledReportToSpecificFixedDate(srToDate.getSpecificFixedDate());
			rpForm.setScheduledReportToType(toTermType(srToDate.getType()));
			rpForm.setScheduledReportToUnitPriorToCurrentDay(toTermType(srToDate
					.getUnitPriorToCurrentDay()));
			rpForm.setScheduledReportToUsePreviousDatePeriod(srToDate.getUsePreviousDatePeriod());
		}

		rpForm.setScheduledReportDaysToKeepCompletedReports(parameters
				.getDaysToKeepCompletedReports());
		
		
	}

	protected <T extends Lookup> Set<T> toLookupSet(Class<T> clazz, Collection<TermType> termTypes)
			throws Exception {
		Set<T> lookups = new HashSet<T>();
		if (termTypes != null && termTypes.size() > 0) {
			for (TermType termType : termTypes) {
				T lookup = getByCode(clazz, termType.getValue());
				lookups.add(lookup);
			}
		}
		return lookups;
	}

	protected Set<Boolean> toBooleanSet(Collection<? extends TermType> termTypes) throws Exception {
		Set<Boolean> set = new HashSet<Boolean>();
		if (termTypes != null && termTypes.size() > 0) {
			for (TermType termType : termTypes) {
				Boolean selectedValue = Boolean.parseBoolean(termType.getValue());
				set.add(selectedValue);
			}
		}
		return set;
	}

	protected Set<String> toStringSet(Collection<TermType> termTypes) throws Exception {
		Set<String> set = new HashSet<String>();
		if (termTypes != null && termTypes.size() > 0) {
			for (TermType termType : termTypes) {
				set.add(termType.getValue());
			}
		}
		return set;
	}

	protected <T extends Lookup> T toLookup(Class<T> clazz, TermType termType) throws TerminologyException{
		return TransferUtils.transferTermTypeToLookup(getTerminologyCache().getTerminologyService(),
				clazz, termType);
	}

	private String tohtml(String parametersText) {
		StringBuilder formattedText = new StringBuilder();
		if (StringUtils.isNotEmpty(parametersText)) {
			String[] pars = parametersText.split(",");
			for (int i = 0; i < pars.length; i++) {
				String parValue = pars[i];
				parValue = parValue.replace(":", ",");
				formattedText.append(parValue);
				formattedText.append("\n");// formattedText.append("<br>");
			}
		}

		return formattedText.toString();
	}

}
