package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.SortedSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import edu.emory.mathcs.backport.java.util.Collections;
import gov.va.med.ccht.model.AppUserGroup;
import gov.va.med.ccht.model.RegistrationForm;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.FacilityRegComparator;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.common.Visn;
import gov.va.med.ccht.model.terminology.RegistrationReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.model.terminology.UserGroup;
import gov.va.med.ccht.service.common.JobDescription;
import gov.va.med.ccht.service.common.RegistrationService;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.TerminologyService;
import gov.va.med.ccht.service.common.facility.FacilityService;
import gov.va.med.ccht.service.common.vendor.VendorService;
import gov.va.med.ccht.service.common.visn.VisnService;
import gov.va.med.fw.model.ldap.LdapPerson;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.StringUtils;

@Controller
public class RegistrationController {
	
	private static final Logger logger = LoggerFactory.getLogger(RegistrationController.class);

	@Autowired
	private SecurityService securityService;
	@Autowired
	private VendorService vendorService;
	@Autowired
	private TerminologyService terminologyService;
	@Autowired
	private FacilityService facilityService;
	@Autowired
	private VisnService visnService;
	@Autowired
	private RegistrationService registrationService;

	private static final String JOB_DESCRIPTION_REQUIRED = "jobDescription.required";
	private static final String USER_NAME_REQUIRED = "username.required";
	private static final String MESSAGE_REG_ACCOUNT_NOTFOUND = "user.notfound";
	private static final String MESSAGE_REG_ACCOUNT_EXISTS = "user.found";
	private static final String MESSAGE_REGISTRATION_APPROVED = "registration.approved";
	private static final String MESSAGE_REGISTRATION_DENIED = "registration.denied";
	private static final String MESSAGE_REGISTRATION_PENDING = "registration.pending";
	private static final String MESSAGE_FACILITY_REQUIRED = "select.facility";
	private static final String MESSAGE_VISN_REQUIRED = "select.visn";
	private static final String MESSAGE_VENDOR_REQUIRED = "select.vendor";
	private static final String MESSAGE_FACILITY_VISN_MISMATCH = "registrationForm.getFacility()";
	public static final String MESSAGE_SYSTEM_ERROR = "System Error: Please contact your System Administrator.";
	
	public static final String VISN_REQUIRED = "visnRequired";
	public static final String FACILITY_REQUIRED = "facilityRequired";
	public static final String VENDOR_REQUIRED = "vendorRequired";
	public static final String REGISTER_1 = "register_1";
	public static final String REGISTER_2 = "register_2";
	public static final String REGISTRATION_CONFIRM = "registrationConfirm";
	public static final String REGISTRATION_BUTTON = "REGISTRATIONBUTTON";
	
	//Add @InitBinder to fix Fortify issue
	@InitBinder
    public void initBinder(WebDataBinder binder)
    {   
    	String[] allowedFields = {"id", "username", "visn", "facility", "vendor", "description", "registerSubmit"};
    	binder.setAllowedFields(allowedFields);
    }
	
	@RequestMapping(value = "/register_1.html", method = RequestMethod.GET)
	public String showRegistrationForm1(Model model) throws ServiceException {
		
		RegistrationForm registrationForm = new RegistrationForm();
		addModelAttributes(model);
		model.addAttribute("command", registrationForm);
		
		return REGISTER_1;
	}
	
	
	@RequestMapping(value = "/register_1.html", method = RequestMethod.POST)
	public String register(@ModelAttribute("command") RegistrationForm registrationForm,
			BindingResult result, Model model) throws ServiceException {
		
		// Validate and process a registration wizard 1 of 2
		String username = registrationForm.getUsername();
		String description = registrationForm.getDescription();
		if (registrationForm.isRegisterSubmit()) {
			// Validate required fields
			if (StringUtils.isEmpty(registrationForm.getUsername()) || StringUtils.isEmpty(registrationForm.getDescription())) {
				if(StringUtils.isEmpty(registrationForm.getUsername())) {
					result.rejectValue("username", USER_NAME_REQUIRED, "User name is required.");
				}
				if(StringUtils.isEmpty(registrationForm.getDescription())) {
					result.rejectValue("description", JOB_DESCRIPTION_REQUIRED, "Description is required.");
				}
				// Check description is in the list of descriptions that require VISN and Facility
				if(StringUtils.equals(JobDescription.NATIONAL_ADMIN.getValue(), description)) {
					model.addAttribute(REGISTRATION_BUTTON, true);
				}
				
				addModelAttributes(model);
				return REGISTER_1;
			}
			
			// Check to see if the provided user name is valid
			// Authenticate the user and get the user details
			try {
				User oldUser = securityService.getUser(username);
				if (oldUser != null) {
					// verify the registration status
					RegistrationStatus regStatus = oldUser.getRegistrationStatus();
					if (regStatus == null) {
						result.rejectValue("description", MESSAGE_REG_ACCOUNT_EXISTS);
					} else if (RegistrationStatus.REJECTED.equalsIgnoreCase(regStatus.getCode())) {
						result.rejectValue("description", MESSAGE_REGISTRATION_DENIED);
					} else if (RegistrationStatus.SUBMITTED.equalsIgnoreCase(regStatus.getCode())) {
						result.rejectValue("description", MESSAGE_REGISTRATION_PENDING);
					} else if (RegistrationStatus.APPROVED.equalsIgnoreCase(regStatus.getCode())) {
						result.rejectValue("description", MESSAGE_REGISTRATION_APPROVED);
					}
					
					if(StringUtils.equals(JobDescription.NATIONAL_ADMIN.getValue(), description)) {
						model.addAttribute(REGISTRATION_BUTTON, true);
					}
					
					addModelAttributes(model);
					return REGISTER_1;
				} else {
					LdapPerson ldapPerson = securityService.getLdapUser(username);
				
				// VISN and Facility are required for Care Coordinator, Program Support Assistant, and Facility admin
				// VISN only required for VISN Admin
				// Vendor only required for Vendor
				if(ldapPerson != null) {
					model.addAttribute("username", username);
					model.addAttribute("description", description);
					
					// default values
					model.addAttribute(FACILITY_REQUIRED, true);
					model.addAttribute(VISN_REQUIRED, true);
					model.addAttribute(VENDOR_REQUIRED, false);
					if(StringUtils.equals(JobDescription.VISN_ADMIN.getValue(), description)) {
						model.addAttribute(FACILITY_REQUIRED, false);
					}
					else if (StringUtils.equals(JobDescription.VENDOR.getValue(), description)) {
						model.addAttribute(FACILITY_REQUIRED, false);
						model.addAttribute(VISN_REQUIRED, false);
						model.addAttribute(VENDOR_REQUIRED, true);
					} 
					else if (StringUtils.equals(JobDescription.NATIONAL_ADMIN.getValue(), description)) {
						return processVARegistration(registrationForm, result, model);
					}
					addModelAttributes(model);
					return REGISTER_2;
				}

				else {
						result.rejectValue("username", MESSAGE_REG_ACCOUNT_NOTFOUND);
						
						if(StringUtils.equals(JobDescription.NATIONAL_ADMIN.getValue(), description) ||
							StringUtils.equals(JobDescription.VENDOR.getValue(), description)) {
							model.addAttribute(REGISTRATION_BUTTON, true);
						}
							
						addModelAttributes(model);
						return REGISTER_1;
					}
				}
			} catch (ServiceException e) {
				result.rejectValue("username", MESSAGE_REG_ACCOUNT_NOTFOUND);
				if(StringUtils.equals(JobDescription.NATIONAL_ADMIN.getValue(), description) ||
					StringUtils.equals(JobDescription.VENDOR.getValue(), description)) {
					model.addAttribute(REGISTRATION_BUTTON, true);
				}

				addModelAttributes(model);
				return REGISTER_1;
			}
		}
		// Display a registration wizard 1 of 2
		// to capture user name and reason
		addModelAttributes(model);
		return REGISTER_1;
	}
	
	@RequestMapping(value = "/register_2.html", method = RequestMethod.GET)
	public String showRegistrationForm2(Model model,
			@ModelAttribute("command") RegistrationForm registrationForm) throws ServiceException {
		addModelAttributes(model);
		model.addAttribute("command", registrationForm);
		
		return REGISTER_2;
	}

	@RequestMapping(value = "/register_2.html", method = RequestMethod.POST)
	public String registerVAUser(@ModelAttribute("command") RegistrationForm registrationForm,
			@RequestParam(name = "facilityRequired", required = false) boolean facilityRequired,
			@RequestParam(name = "vendorRequired", required = false) boolean vendorRequired,
			@RequestParam(name = "visnRequired", required = false) boolean visnRequired,
			BindingResult result, Model model) throws Exception {

		// For CC and PSA, validate facility
		// For Lead CC, validate VISN, and facility
		// For vendor, validate vendor
		
		// Set request attributes for later so that they are consistent according to the first registration page.
		if (facilityRequired) {
			model.addAttribute(FACILITY_REQUIRED, true);
		}
		if (visnRequired) {
			model.addAttribute(VISN_REQUIRED, true);
		}
		if (vendorRequired) {
			model.addAttribute(VENDOR_REQUIRED, true);
		}
		model.addAttribute("username", registrationForm.getUsername());
		model.addAttribute("description", registrationForm.getDescription());

		if (facilityRequired) {
			if (StringUtils.isEmpty(registrationForm.getFacility())) {
				result.rejectValue("facility", MESSAGE_FACILITY_REQUIRED, "Facility are required.");
			}
		}

		if (visnRequired) {
			if (StringUtils.isEmpty(registrationForm.getVisn())) {
				result.rejectValue("visn", MESSAGE_VISN_REQUIRED, "VISN is required.");
			}
		}
		
		if (vendorRequired) {
			if (StringUtils.isEmpty(registrationForm.getVendor())) {
				result.rejectValue("vendor", MESSAGE_VENDOR_REQUIRED, "Vendor is required.");
			}
		}
		
		// Check Selected Facility is in Selected VISN if applicable
		// if visn and/or facility not able to be selected, will be null.
		if(registrationForm.getFacility() != null && registrationForm.getVisn() != null) {
			if(!registrationForm.getFacility().isEmpty() && !registrationForm.getVisn().isEmpty()) {
				Facility facility = facilityService.getFacility(registrationForm.getFacility());
				Visn facilityVisn = facility.getVisn();
				String facilityVisnName = facilityVisn.getName();
				if(!facilityVisnName.equals(registrationForm.getVisn())) {
					result.rejectValue("facility", 
							MESSAGE_FACILITY_VISN_MISMATCH, 
							"The selected Facility is not in the VISN selected. Please select a matching Facility and VISN.");
				}
			}
		}

		// Return to input if there are errors
		if (result.hasErrors()) {
			addModelAttributes(model);
			return REGISTER_2;
		} else {
			// Just in case there are errors, put back request attributes
			return processVARegistration(registrationForm, result, model);
		}
	}
	
	@RequestMapping(value = "/registrationConfirm.html", method = RequestMethod.GET)
	public String confirmRegistration() {
		return "registrationConfirm";
	}
	
	private String processVARegistration(RegistrationForm registrationForm,
			BindingResult result, Model model) {

		String username = registrationForm.getUsername();
		String description = registrationForm.getDescription(); 
		try {
			User oldUser = securityService.getUser(username);
			if (oldUser != null) {
				// verify the registration status
				RegistrationStatus regStatus = oldUser.getRegistrationStatus();
				if (regStatus == null) {
					result.rejectValue("username", MESSAGE_REG_ACCOUNT_EXISTS, "User account already exists.");
				} else if (RegistrationStatus.REJECTED.equalsIgnoreCase(regStatus.getCode())) {
					result.rejectValue("username", MESSAGE_REGISTRATION_DENIED, "Registration has been denied"
							+ " for this user.");
				} else if (RegistrationStatus.SUBMITTED.equalsIgnoreCase(regStatus.getCode())) {
					result.rejectValue("username", MESSAGE_REGISTRATION_PENDING, "Registration for this user is"
							+ " currently pending.");
				} else if (RegistrationStatus.APPROVED.equalsIgnoreCase(regStatus.getCode())) {
					result.rejectValue("username", MESSAGE_REGISTRATION_APPROVED, "Registration for this user has"
							+ " already been approved");
				}
			} else {
				// Authenticate the user and get the user details
				LdapPerson ldapPerson = securityService.getLdapUser(username);

				if (ldapPerson != null) {
					// create user and send e-mails to admin and users
					User user = new User(ldapPerson.getSamAccountName());
					user.setLdapPerson(ldapPerson);
					
					user.setRecordCreatedBy(user.getUsername());
					user.setRecordModifiedBy(user.getUsername());
					final Date date = new Date();
					user.setRecordCreatedDate(date);
					user.setRecordModifiedDate(date);
					user.setRecordModifiedCount((short) 1);

					// Group
					UserGroup ug = terminologyService.getByCode(UserGroup.class, UserGroup.VA_USER);
					AppUserGroup aug = new AppUserGroup();
					aug.setRecordCreatedBy(user.getUsername());
					aug.setRecordModifiedBy(user.getUsername());
					aug.setRecordCreatedDate(date);
					aug.setRecordModifiedDate(date);
					aug.setRecordModifiedCount((short) 1);
					aug.setUserGroup(ug);
					user.addUserGroup(aug);

					user.setLoginFailedCount((short) 0);
					RegistrationStatus regStatus = terminologyService.getByCode(RegistrationStatus.class,
							RegistrationStatus.SUBMITTED);
					user.setRegistrationStatus(regStatus);
					if (StringUtils.isNotEmpty(registrationForm.getDescription())) {
						user.setRegistrationReason(terminologyService.getByCode(RegistrationReason.class, description));
					}
					if (StringUtils.isNotEmpty(registrationForm.getVisn())) {
						user.setVisn(visnService.getSimpleVisn(registrationForm.getVisn()));
					}
					if (StringUtils.isNotEmpty(registrationForm.getFacility())) {
						user.setFacility(facilityService.getSimpleFacility(registrationForm.getFacility()));
						user.setVisn(user.getFacility().getVisn());
					}
					if (StringUtils.isNotEmpty(registrationForm.getVendor())) {
						user.setVendor(vendorService.getVendor(registrationForm.getVendor()));
					}

					registrationService.addVAUser(user);
					
					Authentication authentication =  new UsernamePasswordAuthenticationToken(user, null, user.getAuthorities());
					SecurityContextHolder.getContext().setAuthentication(authentication);
					model.asMap().clear();
					
					return "redirect:/registrationConfirm.html";
				} else {
					result.rejectValue("username", MESSAGE_REG_ACCOUNT_NOTFOUND, "The user cannot be found.");
					return REGISTER_1;
				}
			}
		} catch (Exception e) {
			logger.error(e.toString());
			result.rejectValue("username", "system.error", MESSAGE_SYSTEM_ERROR);
		}
		return "TODO";
	}
	
	

	
	private void addModelAttributes(Model model) throws ServiceException {
		if(!model.containsAttribute("registrationReasons")) {
			List<RegistrationReason> registrationReasons = terminologyService.findAllActive(RegistrationReason.class);
			
			if(registrationReasons != null) {
				model.addAttribute("registrationReasons", registrationReasons);
			}
		}
		if(!model.containsAttribute("facilities")) {
			// Sort facilities based on VISN first.
			SortedSet<SimpleFacility> facilities = facilityService.getSimpleFacilities();
			List<SimpleFacility> facilitiesList = new ArrayList<SimpleFacility>(facilities);
			Collections.sort(facilitiesList, new FacilityRegComparator());
			model.addAttribute("facilities", facilitiesList);
		}
		if(!model.containsAttribute("visns")) {
			model.addAttribute("visns", visnService.getSimpleVisns());
		}
		if(!model.containsAttribute("vendors")) {
			model.addAttribute("vendors", vendorService.getActiveVendors());
		}
	}
}