/**
 * 
 */
package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.DeviceType;
import gov.va.med.ccht.model.common.SimpleFacility;
import gov.va.med.ccht.model.qir.DmpQirType;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRRemarks;
import gov.va.med.ccht.model.qir.QIRSimpleAttachment;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.qir.QIRType;
import gov.va.med.ccht.model.qir.QIRVendorAction;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.ui.common.AbstractConversionService;
import gov.va.med.ccht.ui.model.AttachmentForm;
import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.date.TimeZoneUtils;

/**
 * @author
 *
 */
@Service
public class QIRConversionService extends AbstractConversionService {
	
	private static final char NON_BREAKING_SPACE = '\u00A0';
	
	@Autowired
	private TerminologyCache terminologyCache;
	
	public TerminologyCache getTerminologyCache() {
		return terminologyCache;
	}

	public void setTerminologyCache(TerminologyCache terminologyCache) {
		this.terminologyCache = terminologyCache;
	}

	public void convertToQIR(QIRForm form, QIR bom, User currentUser) {
		try {
			toQIR(form, bom, currentUser);
		} catch (TerminologyException e) {
			e.printStackTrace();
		}
	}
	
	public void convertToQIRForm(QIR bom, QIRForm form) {
		try {
			toQIRForm(bom, form);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public <T extends Lookup> T getLookup(Class<T> terminologyType, TermType termType)
			throws TerminologyException { 
			return (termType != null) ? 
					terminologyCache.getTermByCode(terminologyType, termType.getValue()) : null;
		}
	

	private void toQIRForm(QIR qir, QIRForm qirForm) {
		qirForm.setId(String.valueOf(qir.getId()));
		qirForm.setDmpQirType(toTermType(qir.getDmpQirType()));
		qirForm.setQirType(toTermType(qir.getQirType()));
		qirForm.setQirStatusType(qir.getQirStatusType());
		qirForm.setVisn(toTermType(qir.getVisn()));
		qirForm.setFacility(toTermType(qir.getFacility()));
		qirForm.setDeviceType(toTermType(qir.getDeviceType()));
		qirForm.setVendor(toTermType(qir.getVendor()));
		qirForm.setSerialNumber(qir.getSerialNumber());
		qirForm.setPurchaseOrderNumber(qir.getPurchaseOrderNumber());
		qirForm.setSubmittedDate(qir.getSubmittedDate());
		qirForm.setVendorResponseDueDate(qir.getVendorResponseDueDate());
		qirForm.setEmailReminderSentDate(qir.getEmailReminderSentDate());
		qirForm.setStatusChangeDate(qir.getStatusChangeDate());
		qirForm.setComplaint(qir.getComplaint());
		qirForm.setHeadline(qir.getSummary());
		qirForm.setSubmittedBy(qir.getSubmittedBy());
		qirForm.setSubmittedByName(qir.getSubmittedByName());
		qirForm.setIsDmpIssue(qir.getIsDmpIssue());

		qirForm.setSelectedDmpQirTypeId();
		String facilityNumAndName = "";
		if (qir.getFacility() != null) {
			if (StringUtils.isNotEmpty(qir.getFacility().getStationNumber())) {
				facilityNumAndName = qir.getFacility().getStationNumber();
			}
			if (StringUtils.isNotEmpty(qir.getFacility().getName())) {
				facilityNumAndName = facilityNumAndName + " " + qir.getFacility().getName();
			}
		}
		qirForm.setFacilityNumAndName(facilityNumAndName);

		// Remarks
		String remarksHistory = "";
		List<QIRRemarks> remarksList = new ArrayList<QIRRemarks>(qir.getRemarks());
		Collections.sort(remarksList, Collections.reverseOrder());
		for (QIRRemarks remarks : remarksList) {

			// qirForm.setRemarks(remarks.getRemarks());

			remarksHistory = remarksHistory
					+ TimeZoneUtils.convertDateToTimeZone(remarks.getDate(), TimeZone.getDefault()) + ", "
					+ remarks.getSubmittedByName() + ": " + remarks.getRemarks() + "\n";

		}
		qirForm.setRemarksHistory(remarksHistory);

		// /* Strip facility from front if present */
		// String remarks = qirForm.getRemarks();
		// if (StringUtils.isNotBlank(remarks)) {
		// int noBrSpaceIndex = remarks.indexOf(NON_BREAKING_SPACE);
		// if (noBrSpaceIndex != -1)
		// qirForm.setRemarks(remarks.substring(noBrSpaceIndex + 1));
		// }

		// Vendor Actions
		String vendorActionsHistory = "";
		List<QIRVendorAction> vendorActionsList = new ArrayList<QIRVendorAction>(qir.getVendorActions());
		Collections.sort(vendorActionsList, Collections.reverseOrder());
		for (QIRVendorAction vendorActions : vendorActionsList) {

			if (qirForm.getLastRepliedDate() == null) {
				qirForm.setLastRepliedDate(vendorActions.getDate());
			} else if (vendorActions.getDate().after(qirForm.getLastRepliedDate())) {
				qirForm.setLastRepliedDate(vendorActions.getDate());
			}

			vendorActionsHistory = vendorActionsHistory
					+ TimeZoneUtils.convertDateToTimeZone(vendorActions.getDate(), TimeZone.getDefault()) + ", "
					+ vendorActions.getSubmittedByName() + ": " + vendorActions.getAction() + "\n";
		}
		qirForm.setVendorActionsHistory(vendorActionsHistory);

		// Attachments
		List<AttachmentForm> attachments = new ArrayList<AttachmentForm>();
		for (QIRSimpleAttachment attachment : qir.getAttachments()) {
			AttachmentForm attachmentForm = new AttachmentForm();
			attachmentForm.setId(String.valueOf(attachment.getId()));
			attachmentForm.setDate(attachment.getDate());
			attachmentForm.setName(attachment.getDocumentName());
			attachmentForm.setType(toTermType(attachment.getDocumentType()));
			attachmentForm.setModifiedBy(attachment.getRecordCreatedBy());
			attachmentForm.setLength(attachment.getLength());
			attachments.add(attachmentForm);
		}
		qirForm.setAttachments(attachments);
	}

	private void toQIR(QIRForm qirForm, QIR qir, User currentUser) throws TerminologyException { 

		
		if(qir.getQirType() == null) {
			qir.setQirType(getLookup(QIRType.class, qirForm.getQirType()));
		}
		if(qirForm.getQirStatusType() != null) {
			qir.setQirStatusType(qirForm.getQirStatusType());
		}
		qir.setVisn(getVisn(qirForm.getVisn()));
		SimpleFacility facility = getFacility(qirForm.getFacility());
		if(facility != null) {
			qir.setFacility(facility);
		}
		qir.setDeviceType(getLookup(DeviceType.class, qirForm.getDeviceType()));
		if (qir.getDeviceType() != null) {
			qir.setVendor(qir.getDeviceType().getVendor());
		} else {
			if(qirForm.getVendor() != null) {
				qir.setVendor(getVendor(qirForm.getVendor()));
			}
		}
		if(qirForm.getSerialNumber() != null) {
			qir.setSerialNumber(qirForm.getSerialNumber());
		}
		qir.setPurchaseOrderNumber(qirForm.getPurchaseOrderNumber());
		qir.setSubmittedDate(qirForm.getSubmittedDate());
		qir.setVendorResponseDueDate(qirForm.getVendorResponseDueDate());
		qir.setEmailReminderSentDate(qirForm.getEmailReminderSentDate());
		qir.setStatusChangeDate(qirForm.getStatusChangeDate());
		qir.setComplaint(qirForm.getComplaint());
		qir.setSummary(qirForm.getHeadline());
		qir.setSubmittedBy(qirForm.getSubmittedBy());
		qir.setIsDmpIssue(qirForm.getIsDmpIssue());
		
		if(qir.getDmpQirType() != null) {
			qir.setDmpQirType(qir.getDmpQirType());
		}

		String selectedDmpQirTypeId = qirForm.getSelectedDmpQirTypeId();

		TermType previousDmpQirType = qirForm.getDmpQirType();

		if (previousDmpQirType != null) {

			String previousDmpQirTypeId = qirForm.getDmpQirType().getValue();
			if (previousDmpQirTypeId.equalsIgnoreCase(selectedDmpQirTypeId)) {
				qir.setDmpQirType(getLookup(DmpQirType.class, qirForm.getDmpQirType()));
			} else {
				TermType newDmpQirType = new TermType("", selectedDmpQirTypeId);
				qir.setDmpQirType(getLookup(DmpQirType.class, newDmpQirType));
			}
		} else {
			TermType newDmpQirType = new TermType("", selectedDmpQirTypeId);
			qir.setDmpQirType(getLookup(DmpQirType.class, newDmpQirType));
		}
		// copy editable properties
		if (StringUtils.isNotEmpty(qirForm.getRemarks())) {
			QIRRemarks remarks = new QIRRemarks();
			remarks.setRecordCreatedBy(currentUser.getName());
			remarks.setRecordModifiedBy(currentUser.getName());
			final Date currentDate = new Date();
			remarks.setDate(currentDate);
			remarks.setRecordCreatedDate(currentDate);
			remarks.setRecordModifiedDate(currentDate);
			remarks.setRecordModifiedCount((short) 1);
			String finalRemarks = "";
			if (qir.getRemarks().isEmpty() && facility != null) {
				finalRemarks = facility.toString() + " - " + NON_BREAKING_SPACE;
			}
			finalRemarks += qirForm.getRemarks();
			remarks.setRemarks(finalRemarks);

			qir.addRemarks(remarks);
		}
		if (StringUtils.isNotEmpty(qirForm.getVendorActions())) {
			QIRVendorAction actions = new QIRVendorAction();
			actions.setDate(new Date());
			actions.setAction(qirForm.getVendorActions());
			qir.addVendorActions(actions);
			final Date currentDate = new Date();
			actions.setDate(currentDate);
			actions.setRecordCreatedDate(currentDate);
			actions.setRecordCreatedBy(qirForm.getSubmittedBy());
			actions.setRecordModifiedDate(currentDate);
			actions.setRecordModifiedBy(qirForm.getSubmittedBy());
			actions.setRecordModifiedCount((short) 1);
		}
		// Attachments handled separately
	}
}
