package gov.va.med.ccht.controller;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import com.ibm.icu.util.Calendar;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.FontFactory;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.ColumnText;
import com.lowagie.text.pdf.PdfCopy;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;

import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.pssreport.CategoryOfCare;
import gov.va.med.ccht.model.pssreport.DistributionOfSurveysReport;
import gov.va.med.ccht.model.pssreport.PSSQuestionChoice;
import gov.va.med.ccht.model.pssreport.SurveyTrendChartResult;
import gov.va.med.ccht.model.pssreport.VendorSubmissionReportData;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestion;
import gov.va.med.ccht.model.satisfactionsurvey.SatisfactionSurveyQuestionResult;
import gov.va.med.ccht.service.common.facility.FacilityService;
import gov.va.med.ccht.service.common.vendor.VendorService;
import gov.va.med.ccht.service.common.visn.VisnService;
import gov.va.med.ccht.service.htreports.CensusActivityReportsConstants;
import gov.va.med.ccht.service.htreports.PssReportService;
import gov.va.med.ccht.service.report.ReportConstants;
import gov.va.med.ccht.ui.model.PSSReportForm;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.DateUtils;


@Controller
public class PssReportController implements ReportConstants, CensusActivityReportsConstants{
	
	/**
	 * Spring uses a comma to separate multiple values coming to a single path.
	 */
	private String INPUT_SEPARATOR = ",";
	
	public static final int NATIONAL_BUTTON_SELECTION = 1;
	public static final int VISN_BUTTON_SELECTION = 2;
	public static final int FACILITY_BUTTON_SELECTION = 3;
	
	public static final int ALL_SURVEYS_BUTTON_SELECTION = 1;
	public static final int PSS_V1_BUTTON_SELECTION = 2;
	public static final int PSS_V2_BUTTON_SELECTION = 3;
	
	public static final int LINE_GRAPH = 1;
	public static final int BAR_GRAPH = 2;
	public static final int TABLE = 3;
	
	private static final String DISPLAY_SURVEYTRENDS_REP_FORM = "pssSurveyTrendChartsMenu";
	private static final String DISPLAY_STATISTICS_REP_FORM = "pssStatisticsReportMenu";
	private static final String DISPLAY_DISTRIBUTION_REP_FORM = "pssDistributionReportMenu";
	private static final String DO_NOT_INCLUDE = "-2";
	private static final String ALL_MODALITIES = "-1";
	private static final String BROWSER = "11503";
	private static final String IVR = "11502";
	private static final String HOME_DEVICE = "11501";
	private static final String START_DATE_DISTRIBUTION_REPORT = "10/02/2005";
	private static final String START_DATE_SURVEYTRENDS_REPORT = "01/2009";
	private static final String ALL_EXCEPT_L2_VALUE = "-2";
	
	private static final String MESSAGE_QUESTION_NUMBER_REQUIRED = "questionNumbers.selection.required";
	private static final String MESSAGE_SHORT_DATE_SELECTION_ERROR = "date.selection.invalid";
	private static final String MESSAGE_BAD_STATION_SELECTION_ERROR = "bad.station.number";

	@Autowired
	private PssReportService pssReportService;
	@Autowired
	private FacilityService facilityService;
	@Autowired
	private VendorService vendorService;
	@Autowired
	private VisnService visnService;
	
	@InitBinder
    public void initBinder(WebDataBinder binder)
    {
		String[] allowedFields = {"reportFromDate", "reportToDate", "typeButtonSelection", "visnId", 
				"facilityId", "vendorId", "modalityName", "reportVersionButtonSelection", "questionNumbers", 
				"categoriesOfCare", "shortFormatReportFromDate", "shortFormatReportToDate", "chartType"};
		binder.setAllowedFields(allowedFields);
    }
	
	@RequestMapping(value = "/pssVendorSubReport.html", method = RequestMethod.GET)
	public String showVendorSubmissionReport(Model model) throws ServiceException{
		
		List<VendorSubmissionReportData> report = pssReportService.getVendorSubmissionReport();
		
		model.addAttribute("vendorSubReport", report);
		model.addAttribute("reportTitle", "National Distribution of All Surveys by Vendor");
		
		return "pssVendorSubReport";
	}
	
	@RequestMapping(value = "/pssPatientSatReport.html", method = RequestMethod.GET)
	public String showPatientSatisfactionSurveyResultsReport(Model model) throws ServiceException{
		
		addCommonAttributes(model);
		model.addAttribute("patientSatisfactionSurveyResultsReport", true);
		
		return "pssPatientSatReportMenu";
	}	
	
	@RequestMapping(value = "/pssSurveyTrendCharts.html", method = RequestMethod.GET)
	public String showSurveyTrendCharts(Model model) throws ServiceException{
		addCommonAttributes(model);
		PSSReportForm form = new PSSReportForm();
		//Set default type selection
		form.setTypeButtonSelection(NATIONAL_TOTAL);
		form.setChartType(BAR_GRAPH);
		model.addAttribute("form", form);
		model.addAttribute("dateByMonths", pssReportService.getDatesByMonth());
		SimpleDateFormat df = new SimpleDateFormat("MM/yyyy");
		Date today = Calendar.getInstance().getTime();
		model.addAttribute("fromDate", START_DATE_SURVEYTRENDS_REPORT);
		model.addAttribute("toDate", df.format(today));
		return DISPLAY_SURVEYTRENDS_REP_FORM;
	}

	@RequestMapping(value = "/pssSurveyTrendCharts.html", method = RequestMethod.POST)
	public String generateSurveyTrendCharts(@ModelAttribute("form") PSSReportForm form, 
			BindingResult result, Model model) throws ServiceException{
		// Validate the user defined facility.
		validateShortDate(form, result);
		Facility facility = null;
		if(form.getFacilityId() != null) {
			int lastComma = form.getFacilityId().lastIndexOf(INPUT_SEPARATOR);
			if(lastComma != 0) {
				String userStationNumber = form.getFacilityId().substring(0,lastComma);
				// Avoid adding the comma separator back into the facility id text field.
				form.setFacilityId(userStationNumber);
				try {
					facility = facilityService.getFacility(userStationNumber);
					form.setFacilityId(facility.getId() + "");
				}
				catch(ServiceException e) {
					result.rejectValue("facilityId",MESSAGE_BAD_STATION_SELECTION_ERROR);
					addErrorCommonAttributes(model);
					model.addAttribute("fromDate", form.getShortFormatReportFromDate());
					model.addAttribute("toDate", form.getShortFormatReportToDate());
					model.addAttribute("facilityId", form.getFacilityId());					
					model.addAttribute("form", form);
					return DISPLAY_SURVEYTRENDS_REP_FORM;
				}
			}
			else {
				form.setFacilityId(form.getFacilityId().substring(1));
				if(!form.getFacilityId().equals(ALL_FACILITIES)) {
					//Remove the input separator
					facility = facilityService.getFacility(form.getFacilityId());
					form.setFacilityId(facility.getId() + "");
				}
			}
		}
		// Return to input if there are errors
		if (result.hasErrors()) {			
			addErrorCommonAttributes(model);
			model.addAttribute("fromDate", form.getShortFormatReportFromDate());
			model.addAttribute("toDate", form.getShortFormatReportToDate());
			Facility userFacility;
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (!form.getFacilityId().equals(ALL_FACILITIES))) {
				userFacility = facilityService.getFacilityById(Long.parseLong(form.getFacilityId()));
				String stationNumber = userFacility.getStationNumber();
				form.setFacilityId(stationNumber);
			}
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (form.getFacilityId().equals(ALL_FACILITIES))) {
				form.setFacilityId("");
			}
			model.addAttribute("form", form);			
			return DISPLAY_SURVEYTRENDS_REP_FORM;
		}
		LinkedList<SurveyTrendChartResult> report =  pssReportService.getSurveyTrendChartsReportData(form);
		model.addAttribute("report", report);
		
		List<String> months = new ArrayList<String>();
		List<Integer> surveyCounts = new ArrayList<Integer>();
		List<String> visns = new ArrayList<String>();
		Boolean isAllVISNS = form.getTypeButtonSelection()==VISN_BUTTON_SELECTION && form.getVisnId().equals(ALL_VISNS);
		model.addAttribute("isAllVISNS", isAllVISNS);
			
		//If report is being run for All visns the data returned is each visn 
		// & the number of surveys over the selected time period
		if(isAllVISNS) {
			for(SurveyTrendChartResult entry : report) {
				visns.add(entry.getVisn());
				surveyCounts.add(entry.getNumOfSurveys());
			}
			model.addAttribute("xAxis", visns);
		}
		//For all other reports the data is the month and the total number of surveys for that month
		else {
			for(SurveyTrendChartResult entry : report) {
				months.add(entry.getMonth());
				surveyCounts.add(entry.getNumOfSurveys());
			}
			model.addAttribute("xAxis", months);
		}

		model.addAttribute("surveyReportTotals", surveyCounts);
		model.addAttribute("dateRange", buildReportDateRange(form));
		String reportPageTitle = "Survey Trend Chart " + buildReportDateRange(form);
		model.addAttribute("reportPageTitle", reportPageTitle);
		
		//Checks if the query successfully returned results. Checking in the jsp resulted in a length of negative infinity
		Boolean hasData = surveyCounts.size() > 0;
		model.addAttribute("hasData", hasData);

		model.addAttribute("reportMenuUrl", "pssSurveyTrendCharts.html");
		
		//The graph title varies based on selection of National/VISN or facility
		model.addAttribute("graphTitle", buildTrendsReportTitle(form, facility));


		switch(form.getChartType()) {
		case BAR_GRAPH:
			model.addAttribute("generateBarGraph", true);
			model.addAttribute("generateLineGraph", false);
			model.addAttribute("generateTable", false);
			break;
		case LINE_GRAPH:
			model.addAttribute("generateLineGraph", true);
			model.addAttribute("generateBarGraph", false);
			model.addAttribute("generateTable", false);
			break;
		case TABLE:
			model.addAttribute("generateLineGraph", false);
			model.addAttribute("generateBarGraph", false);
			model.addAttribute("generateTable", true);
			break;
		default:
			// if selection is not applicable, do not generate report. and return menu.
			return DISPLAY_SURVEYTRENDS_REP_FORM;
		}
		return "pssSurveyTrendCharts";
	}
	
	@RequestMapping(value = "/pssDistributionReport.html", method = RequestMethod.GET)
	public String showDistributionOfSurveyReport(Model model) throws ServiceException{
		PSSReportForm form = new PSSReportForm();
		addCommonAttributes(model);
		//Set default type selection
		form.setTypeButtonSelection(NATIONAL_TOTAL);
		model.addAttribute("form", form);
		SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
		String todaysDate = df.format(Calendar.getInstance().getTime());
		model.addAttribute("fromDate", START_DATE_DISTRIBUTION_REPORT);
		model.addAttribute("toDate", todaysDate);
		return DISPLAY_DISTRIBUTION_REP_FORM;
	}
	
	@RequestMapping(value = "/pssDisributionReportsMenu.html", method = RequestMethod.POST)
	public String generateDistributionOfSurveyReport(@ModelAttribute("form") PSSReportForm form, 
			BindingResult result, Model model) throws ServiceException {
		//set model attribute value in case report menu button is selected
		model.addAttribute("reportMenuUrl", "pssDistributionReport.html");
		// Validate the user defined dates.
		validateReportDate(form, result);		
		// Validate the user defined facility.
		Facility facility = null;
		if(form.getFacilityId() != null) {
			//determines if selection is typed in or from drop down or both
			int lastComma = form.getFacilityId().lastIndexOf(INPUT_SEPARATOR);
			
			if(lastComma != 0) {
				String userStationNumber = form.getFacilityId().substring(0,lastComma);
				// Avoid adding the comma separator back into the facility id text field.
				form.setFacilityId(userStationNumber);				
				try {
					// check for valid facility
					facilityService.getFacility(userStationNumber);
					form.setFacilityId(userStationNumber);
				}
				catch(ServiceException e) {
					result.rejectValue("facilityId",MESSAGE_BAD_STATION_SELECTION_ERROR);
					addErrorCommonAttributes(model);
					model.addAttribute("reportVersionButtonSelection", form.getReportVersionButtonSelection());					
					model.addAttribute("facilityId", form.getFacilityId());					
					model.addAttribute("PSSReportForm", form);
					return DISPLAY_DISTRIBUTION_REP_FORM;
				}
			}
			else {
				//if drop down selection is not all facilities, get id to set the value in the form
				form.setFacilityId(form.getFacilityId().substring(1));
				if(!form.getFacilityId().equals(ALL_FACILITIES)) {
					//Remove the input separator
					facility = facilityService.getFacility(form.getFacilityId());
					form.setFacilityId(facility.getStationNumber());
				}
			}
		}
		
		if (result.hasErrors()) {			
			addErrorCommonAttributes(model);
			model.addAttribute("fromDate", form.getReportFromDate());
			model.addAttribute("toDate", form.getReportToDate());
			model.addAttribute("reportVersionButtonSelection", form.getReportVersionButtonSelection());
			Facility userFacility;
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (!form.getFacilityId().equals(ALL_FACILITIES))) {
				userFacility = facilityService.getFacilityById(Long.parseLong(form.getFacilityId()));
				String stationNumber = userFacility.getStationNumber();
				form.setFacilityId(stationNumber);
			}
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (form.getFacilityId().equals(ALL_FACILITIES))) {
				form.setFacilityId("");
			}
			model.addAttribute("pssReportForm", form);			
			return DISPLAY_DISTRIBUTION_REP_FORM;
		}
		
		DistributionOfSurveysReport report = new DistributionOfSurveysReport();
		
		int surveyId = form.getTypeButtonSelection();
		switch(surveyId) {
		
		case NATIONAL_BUTTON_SELECTION:
			
			report = pssReportService.getDistributionNationalReport(form);
			model.addAttribute("isBreakoutRow", false);
			break;
		case VISN_BUTTON_SELECTION:
			
			report = pssReportService.getDistributionVisnReport(form);
			model.addAttribute("isBreakoutRow", true);
			model.addAttribute("breakoutRow", "VISN");
			break;
		case FACILITY_BUTTON_SELECTION:
			
			report = pssReportService.getDistributionFacilityReport(form);
			model.addAttribute("isBreakoutRow", true);
			model.addAttribute("breakoutRow", "Facility");
			break;
		default:
			return DISPLAY_DISTRIBUTION_REP_FORM;
		}
		
		if(form.getModalityName().equals(ALL_MODALITIES)) {
			model.addAttribute("isModalityBreakoutRows", true); 
			model.addAttribute("modalityBreakoutRowIvr", "Surveys via IVR");
			model.addAttribute("modalityBreakoutRowBrowser", "Surveys via Browser");
			model.addAttribute("modalityBreakoutRowHomeDevice", "Surveys via Home Device");
		}
		else model.addAttribute("isModalityBreakoutRows", false);
				
		model.addAttribute("reportTitle", generateDistOfSurveysReportTitle(form));
		model.addAttribute("report", report);		
		return "pssDistributionReport";
	}
	
	private String generateDistOfSurveysReportTitle(PSSReportForm form ){
		
		String title = "Distribution Of Surveys";
		
		int reportType = form.getTypeButtonSelection();
		
		switch(reportType) {
			
		case NATIONAL_BUTTON_SELECTION:
			
			title += " National Totals for";
			break;
		case VISN_BUTTON_SELECTION:
			
			title += " by VISN for";
			break;
		case FACILITY_BUTTON_SELECTION:
			
			title += " by Facility for";
			break;
		default:
			break;
		}
		
		int pssVersion = form.getReportVersionButtonSelection();
		
		switch(pssVersion) {
			
		case ALL_SURVEYS_BUTTON_SELECTION:
			
			title += "<br>All Survey Types for All Vendors";
			break;
		case PSS_V1_BUTTON_SELECTION:
			
			title += "<br>Patient Satisfaction Survey Ver. 1 for All Vendors";
			break;
		case PSS_V2_BUTTON_SELECTION:
			
			title += "<br>Patient Satisfaction Survey Ver. 2 for All Vendors";
			break;
		default:
			break;
		}
		
		String modalitySelection = form.getModalityName();
		
		switch(modalitySelection) {
		
		case BROWSER:
			title += " Modality Browser";
            break;
        case IVR:
        	title += " Modality IVR";
            break;
        case HOME_DEVICE:
        	title += " Modality Home Device";
            break;
        case DO_NOT_INCLUDE:
            break;
        case ALL_MODALITIES:
        	title += " for All Modalities";
            break;
		}
		
		SimpleDateFormat df = new SimpleDateFormat("MMM dd, yyyy");
		
		String fromDate = df.format(form.getReportFromDate());
		String toDate = df.format(form.getReportToDate());
		
		String titleDate = "<br>" + fromDate + " through " + toDate;
		
		title += titleDate;
		
		return title;
	}
	
	@RequestMapping(value="/pssStatisticsReport.html", method = RequestMethod.GET)
	public String showPssStatisticsMenu(Model model) throws ServiceException {
		addCommonAttributes(model);
		PSSReportForm form = new PSSReportForm();
		//Set default type selection
		form.setTypeButtonSelection(NATIONAL_TOTAL);
		//set default version of questions
		form.setReportVersionButtonSelection(PSS_V1_BUTTON_SELECTION);
		//set default COC value to all except L2 option
		String[] categoriesOfCare = new String[8];
		categoriesOfCare[1] = ALL_EXCEPT_L2_VALUE;
		form.setCategoriesOfCare(categoriesOfCare);
		model.addAttribute("pssReportForm", form);
		model.addAttribute("questions", getQuestionNumbers(form));
		model.addAttribute("categoriesOfCare", pssReportService.getCategoriesOfCare());
		model.addAttribute("dateByMonths", pssReportService.getDatesByMonth());		
		SimpleDateFormat df = new SimpleDateFormat("MM/yyyy");
		Calendar fromDate = Calendar.getInstance();
		Date today = Calendar.getInstance().getTime();
		fromDate.add(Calendar.MONTH, -2);
		
		model.addAttribute("fromDate", df.format(fromDate.getTime()));
		model.addAttribute("toDate", df.format(today));
		
		return DISPLAY_STATISTICS_REP_FORM;
	}
	
	@RequestMapping(value="/pssStatisticsReport.html", method = RequestMethod.POST)
	public String generatePssStatisticsReport(Model model, @ModelAttribute("pssReportForm") PSSReportForm form, 
			BindingResult result) throws ServiceException, NumberFormatException, DAOException {	
		
		validateShortDate(form, result);
		
		if((form.getQuestionNumbers() == null) || (form.getQuestionNumbers().length == 0)) {
			result.rejectValue("questionNumbers", MESSAGE_QUESTION_NUMBER_REQUIRED, "Question Numbers is a required field.");
		}
		// Validate the user defined facility.
		Facility facility = null;
		if(form.getFacilityId() != null) {
			int lastComma = form.getFacilityId().lastIndexOf(INPUT_SEPARATOR);
			if(lastComma != 0) {
				String userStationNumber = form.getFacilityId().substring(0,lastComma);
				// Avoid adding the comma separator back into the facility id text field.
				form.setFacilityId(userStationNumber);				
				try {
					facility = facilityService.getFacility(userStationNumber);
					form.setFacilityId(facility.getId() + "");
				}
				catch(ServiceException e) {
					result.rejectValue("facilityId",MESSAGE_BAD_STATION_SELECTION_ERROR);
					addErrorCommonAttributes(model);
					model.addAttribute("questions", getQuestionNumbers(form));
					model.addAttribute("reportVersionButtonSelection", form.getReportVersionButtonSelection());
					model.addAttribute("fromDate", form.getShortFormatReportFromDate());
					model.addAttribute("toDate", form.getShortFormatReportToDate());
					model.addAttribute("facilityId", form.getFacilityId());
					model.addAttribute("PSSReportForm", form);
					return DISPLAY_STATISTICS_REP_FORM;
				}
			}
			else {
				form.setFacilityId(form.getFacilityId().substring(1));
				if(!form.getFacilityId().equals(ALL_FACILITIES)) {
					//Remove the input separator
					facility = facilityService.getFacility(form.getFacilityId());
					form.setFacilityId(facility.getId() + "");
				}
			}
		}
		Date currentTime = Calendar.getInstance().getTime();
		model.addAttribute("reportTitle", buildReportTitle(form, facility));
		model.addAttribute("currentTime", DateUtils.format(currentTime, DateUtils.MMDDYYYYHHMMSS));
		// Return to input if there are errors
		if (result.hasErrors()) {			
			addErrorCommonAttributes(model);			
			model.addAttribute("fromDate", form.getShortFormatReportFromDate());
			model.addAttribute("toDate", form.getShortFormatReportToDate());
			model.addAttribute("questions", getQuestionNumbers(form));
			model.addAttribute("reportVersionButtonSelection", form.getReportVersionButtonSelection());
			Facility userFacility;
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (!form.getFacilityId().equals(ALL_FACILITIES))) {
				userFacility = facilityService.getFacilityById(Long.parseLong(form.getFacilityId()));
				String stationNumber = userFacility.getStationNumber();
				form.setFacilityId(stationNumber);
			}
			if((form.getTypeButtonSelection() == FACILITY_BUTTON_SELECTION) && (form.getFacilityId().equals(ALL_FACILITIES))) {
				form.setFacilityId("");
			}
			model.addAttribute("pssReportForm", form);			
			return DISPLAY_STATISTICS_REP_FORM;
		}
		List<SatisfactionSurveyQuestionResult> questionResults = pssReportService.getSatisfactionSurveyReport(form);
		// TODO: MAY NOT NEED TO CALL RESULTS AN EXTRA TIME TO SET THE INITIAL DATA.
		if(questionResults != null && questionResults.size() > 0) {
			questionResults.get(0).getResults();
			model.addAttribute("averageRow", getAverageRow(questionResults));
		}
		model.addAttribute("satSurveyResultsReport", questionResults);
		int totalSurveyCount = 0;
		if(questionResults != null && (!questionResults.isEmpty()))
			totalSurveyCount += questionResults.get(0).getAmountOfSurveysForQuestion(form);
		model.addAttribute("totalSurveys", totalSurveyCount);
		model.addAttribute("dateRange", buildReportDateRange(form));
		model.addAttribute("reportMenuUrl", "pssStatisticsReport.html");
		return "pssStatisticsReport";
	}
	
	private void addErrorCommonAttributes(Model model) throws ServiceException {
		// Add facilities, days of week, modalities, ect to the model
		model.addAttribute("visns", visnService.getVisns());
		model.addAttribute("facilities", facilityService.getFacilities());
		model.addAttribute("vendors", vendorService.getAllVendors());
		model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
		model.addAttribute("dateByMonths", pssReportService.getDatesByMonth());		
	}
	
	private void addCommonAttributes(Model model) throws ServiceException {
		// Add facilities, days of week, modalities, ect to the model
		model.addAttribute("visns", visnService.getVisns());
		model.addAttribute("facilities", facilityService.getFacilities());
		model.addAttribute("vendors", vendorService.getAllVendors());
		model.addAttribute("pssReportForm", new PSSReportForm());
		model.addAttribute("modalities", pssReportService.getAllDeviceModalities());
	}
	
	/*
	 * Method for validating the correct date order to ensure From dates are earlier than To dates
	 */
	private void validateShortDate(PSSReportForm form, BindingResult result) {
		
		// allows for short date fields to be checked.
		if(form.getShortFormatReportFromDate().after(form.getShortFormatReportToDate())) {
			result.rejectValue("shortFormatReportFromDate", MESSAGE_SHORT_DATE_SELECTION_ERROR);			
		}		
	}
	
		
	// method can applied to all reports with TO and FROM date fields.
	private void validateReportDate(PSSReportForm form, BindingResult result) {
	
		// allows for all fields to be check instead of returning after one failure.
				
		if(form.getReportFromDate() == null) {
			result.rejectValue("reportFromDate", "date.From.required", "Report From Date is required");			
		}
		
		if(form.getReportToDate() == null) {
			result.rejectValue("reportToDate", "date.To.required", "Report To Date is required");
		}
		
		if(form.getReportFromDate().after(form.getReportToDate())) {
			result.rejectValue("reportFromDate", MESSAGE_SHORT_DATE_SELECTION_ERROR);			
		}		
	}
	
	private String buildTrendsReportTitle(final PSSReportForm form, final Facility facility) throws NumberFormatException, ServiceException {
		StringBuilder sb = new StringBuilder("Survey Trends ");
		if(form.getTypeButtonSelection() == NATIONAL_TOTAL) {
			sb.append("by Month from ");
		}
		else if(form.getTypeButtonSelection() == VISN_REPORT) {
			if(form.getVisnId().equals(ALL_VISNS)) {
				sb.append("by VISN from ");
			}
			else {
				sb.append("for VISN " + form.getVisnId() + " from ");
			}
		}
		else {
			if(form.getFacilityId().equals(ALL_FACILITIES)) {
				sb.append("for All Facilities from ");
			}
			else {
				sb.append("for Facility " + facility.getStationNumber() + " " + facility.getName() + " from ");
			}
		}
		sb.append(buildReportDateRange(form));
		
		return ESAPIValidator.validateStringInput(sb.toString(), ESAPIValidationType.ReportTitle_WhiteList);
	}
	
	private List<PSSQuestionChoice> getQuestionNumbers(PSSReportForm form) {
		List<PSSQuestionChoice> questions = new ArrayList<PSSQuestionChoice>();
		String[] oldSelections = form.getQuestionNumbers();
		form.setQuestionNumbers(new String[MAX_QUESTION_NUMBER]);
		for(int i = 1; i <= MAX_QUESTION_NUMBER; i++) {
			if(oldSelections == null) {
				form.getQuestionNumbers()[i-1] = "" + i;
			}
			PSSQuestionChoice question = new PSSQuestionChoice();
			question.setChoiceNumber(i);
			question.setChoiceText(i + ")");
			questions.add(question);
		}
		
		if(oldSelections != null) {
			for(int i = 0; i < oldSelections.length; i++) {
				form.getQuestionNumbers()[Integer.parseInt(oldSelections[i])-1] = oldSelections[i];
			}
		}

		return questions;
	}
	
	private String buildReportDateRange(final PSSReportForm form) throws NumberFormatException, ServiceException {
		StringBuilder sb = new StringBuilder(" ");
		sb.append(DateUtils.format(form.getShortFormatReportFromDate(), DateUtils.MMMDYYYY));
		
		// Need to set "to" date to end of the last month to reflect how the report runs
		Calendar calToDate = Calendar.getInstance();
		Calendar today = Calendar.getInstance();
		calToDate.setTime(form.getShortFormatReportToDate());
		calToDate.set(Calendar.DAY_OF_MONTH, calToDate.getActualMaximum(Calendar.DAY_OF_MONTH));
		
		if(calToDate.after(today)) {
			sb.append(" through " + DateUtils.format(today.getTime(), DateUtils.MMMDYYYY));
		}
		else {
			sb.append(" through " + DateUtils.format(calToDate.getTime(), DateUtils.MMMDYYYY));
		}
		return ESAPIValidator.validateStringInput(sb.toString(), ESAPIValidationType.ReportTitle_WhiteList);
	}
	
	private String buildReportTitle(final PSSReportForm form, final Facility facility) throws NumberFormatException, ServiceException {
		StringBuilder sb = new StringBuilder("Statistics for Patient Satisfaction Surveys - ");
		if(form.getTypeButtonSelection() == NATIONAL_TOTAL) {
			sb.append(NATIONAL_TITLE);
		}
		else if(form.getTypeButtonSelection() == VISN_REPORT) {
			if(form.getVisnId().equals(ALL_VISNS)) {
				sb.append("All VISNs");
			}
			else {
				sb.append("VISN " + form.getVisnId());
			}
		}
		else {
			if(form.getFacilityId().equals(ALL_FACILITIES)) {
				sb.append("All Facilities");
			}
			else {
				sb.append("Facility " + facility.getStationNumber() + " " + facility.getName());
			}
		}
		if(form.getVendorId().equals(ALL_VENDORS)) {
			sb.append(", All Vendors");
		}
		else {
			Vendor vendor = vendorService.getVendorById(new Long(form.getVendorId()));
			sb.append(", " + vendor.getName());
		}
		if(form.getCategoriesOfCare()[0].equals(PSS_ALL_COC)) {
			sb.append(", All Categories of Care");
		}
		else if(form.getCategoriesOfCare()[0].equals(PSS_ALL_COC_EXCEPT_L2)){
			sb.append(", All Categories of Care Except L2");
		}
		else {
			for(String categoryString: form.getCategoriesOfCare()) {
				// No great way to do the lookup, so use the least DB calls because this report title
				// method is not called often
				for (CategoryOfCare category: pssReportService.getCategoriesOfCare()) {
					if(category.getId() == Integer.parseInt(categoryString)) {
						sb.append(", " + category.getName());
					}
				}
			}
		}
		if(form.getReportVersionButtonSelection() == 1) {
			sb.append(", Pat Sat Version 1");
		}
		else if (form.getReportVersionButtonSelection() == 2){
			sb.append(", Pat Sat Version 2");
		}
		sb.append(",");
		sb.append(buildReportDateRange(form));
		
		return ESAPIValidator.validateStringInput(sb.toString(), ESAPIValidationType.ReportTitle_WhiteList);
	}

	private SatisfactionSurveyQuestionResult getAverageRow(List<SatisfactionSurveyQuestionResult> questionResults) {
		SatisfactionSurveyQuestionResult averageRow = new SatisfactionSurveyQuestionResult();
		averageRow.setFormattedQuestionText("Patient Satisfaction Index");
		List<SatisfactionSurveyQuestion> psiAverages = new ArrayList<SatisfactionSurveyQuestion>();
		for(int i = 0; i < questionResults.get(0).getResults().size(); i++) {
			int numAnswers = 0;
			// Not using double to avoid losing precision for averaging
			BigDecimal averagedAnswer = new BigDecimal(0);
			for(SatisfactionSurveyQuestionResult result: questionResults) {
				numAnswers += result.getResults().get(i).getAnswerCount();
				averagedAnswer = averagedAnswer.add(
						new BigDecimal(result.getResults().get(i).getAnswerCount() * result.getResults().get(i).getAvgAnswer()));
			}
			averagedAnswer = averagedAnswer.divide(new BigDecimal(numAnswers), 4, RoundingMode.HALF_UP);
			SatisfactionSurveyQuestion averageCell = new SatisfactionSurveyQuestion();
			averageCell.setAnswerCount(numAnswers);
			averageCell.setAvgAnswer(averagedAnswer.doubleValue());
			psiAverages.add(averageCell);
		}
		averageRow.setResults(psiAverages);
		return averageRow;
	}
	
	@RequestMapping(value = "/PatSatSurveyV1Questions.html",  params = {"attachmentId"},
			method = RequestMethod.GET)
	public String viewAttachment(Model model, HttpSession session,
			@RequestParam int attachmentId,
			HttpServletRequest request, HttpServletResponse response) throws ServiceException, IOException, DocumentException {	

		String documentName = "";
		String fileName = "";
		if(attachmentId == 1) {
			documentName = "PatientSatisfactionSurveyQuestions_V1.pdf";
			fileName = "PatSatSurveyV1.pdf";
		} 
		else {
			documentName = "PatientSatisfactionSurveyQuestions_V2.pdf";
			fileName = "PatSatSurveyV2.pdf";
		}
		
		PdfReader reader = null;
		ClassPathResource resource = new ClassPathResource(fileName);
		
		try {
		reader = new PdfReader(resource.getInputStream());
		int pages = reader.getNumberOfPages();
		Document document = new Document();
		ByteArrayOutputStream result = new ByteArrayOutputStream();
		PdfCopy copy = new PdfCopy(document,result);
		document.open();
		
		PdfImportedPage page;
        PdfCopy.PageStamp stamp;
        for (int i = 0; i < pages; i++) {
            page = copy.getImportedPage(reader, (i+1));
            stamp = copy.createPageStamp(page);
            
            // add page numbers
            ColumnText.showTextAligned(
                    stamp.getUnderContent(), Element.ALIGN_RIGHT,
                    new Phrase(String.format("Page %d of %d", (i+1), pages),FontFactory.getFont(FontFactory.HELVETICA,8)),
                      580.5f, 28, 0);
            stamp.alterContents();
            copy.addPage(page);
        }
        
        copy.close();
        document.close();
        result.close();
			
		response.setHeader("X-Content-Type-Options", "nosniff");
		response.setHeader("X-XSS-Protection", "1");
		response.setHeader("Content-Disposition","attachment; filename=" + documentName);
		response.setContentType("application/pdf");
		
		OutputStream outStream = response.getOutputStream();
		outStream.write(result.toByteArray());
		outStream.close();
		} finally {
			if(reader != null) {
				safeCloseReader(reader);
			}
			
		}
		return null;		
	}
	
// close method required by fortify:
	public static void safeCloseReader(PdfReader reader) {
		  if (reader != null) {
		    try {
		    	reader.close();
		    } catch (Exception e) {
		    }
		  }
	}
}
