package gov.va.med.ccht.controller;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.naming.SizeLimitExceededException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.validation.Valid;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Errors;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.support.SessionStatus;

import gov.va.med.ccht.controller.validators.QIRFormValidator;
import gov.va.med.ccht.model.CCHTPermissions;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.qir.QIR;
import gov.va.med.ccht.model.qir.QIRStatusType;
import gov.va.med.ccht.model.qir.QIRStatusTypeEnum;
import gov.va.med.ccht.model.qir.QIRType;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.ccht.ui.common.UserNotifier;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.ccht.util.ESAPIValidationType;
import gov.va.med.ccht.util.ESAPIValidator;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.ui.conversion.ConversionServiceException;
import gov.va.med.fw.ui.model.TermType;
import gov.va.med.fw.util.DateUtils;


@Controller
//@SessionAttributes(value = { "command" })
public class EditQIRController implements InitializingBean {
	
	private Logger logger = Logger.getLogger(getClass());

	@Autowired
	private QIRConversionService qiRConversionService;
	@Autowired
	private QIRFormValidator validator;
	@Value("${qir.maxCombinedAttachmentBytes}")
	private int maxCombinedAttachmentLength;
	
	@Autowired
	private QIRService qirService;
	@Autowired
	protected TerminologyCache terminologyCache;
	@Autowired
	protected UserNotifier userNotifier;
	
	private QIRStatusType qirStatusTypeNew;
	private QIRStatusType qirStatusTypeApproved;
	private QIRStatusType qirStatusTypeClosed;
	private QIRStatusType qirStatusTypeWithdrawn;
	private QIRStatusType qirStatusTypeReplied;
	private QIRStatusType qirStatusTypeAgreed;
	
	public int getMaxCombinedAttachmentLength() {
		return maxCombinedAttachmentLength;
	}

    @InitBinder
    public void initBinder(WebDataBinder binder)
    {
    	String[] allowedFields = { "isDmpIssue", "selectedDmpQirTypeId", "dmpQirType", "complaint" , "remarks", 
    			"vendorActions", "attachments", "attachmentFiles", "id", "headline", "statusChangeDate", "closed", "newStatus",
    			"approved", "agreed", "withdrawn", "replied", "fromPage"};
    	binder.setAllowedFields(allowedFields);
    }
    
    
	@RequestMapping(value = "/manageQirs.html", method = RequestMethod.GET)
	public String showEditQirForm(@RequestParam String selectedQIR, @RequestParam String fromPage,
			@RequestParam String editMode, Model model) throws Exception {
		QIRForm qirForm = buildForm(selectedQIR, fromPage, "View".equals(editMode));
		model.addAttribute("command", qirForm);
		addReferenceData(model, qirForm);
		return "editQir";
	}

	@RequestMapping(value = "/manageQirs.html", params = { "PDF" }, method = RequestMethod.GET)
	public String doPDFRequest(@RequestParam String selectedQIR, Model model) throws Exception {
		return "CCHT";
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST }, params = { "Update" })
	public String submitUpdateQIR(@ModelAttribute("command") @Valid QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpSession session, HttpServletRequest request) throws Exception {
		validator.validateEditQir(command, result); // PLEASE USE FRONT END JS VALIDATION UNLESS WE MOVE ALL VALIDATION TO JAVA! 
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, null);
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST }, params = { "Approve" })
	public String submitApproveQIR(@ModelAttribute("command") QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpServletRequest request) throws Exception {
		validator.validateApproveQir(command, result); // PLEASE USE FRONT END JS VALIDATION UNLESS WE MOVE ALL VALIDATION TO JAVA!
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, qirStatusTypeApproved);
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST, RequestMethod.GET }, params = {
			"Withdraw" })
	public String submitWithdrawQIR(@ModelAttribute("command") QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpServletRequest request) throws Exception {
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, qirStatusTypeWithdrawn);
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST }, params = { "Reply" })
	public String submitReplyQIR(@ModelAttribute("command") QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpServletRequest request) throws Exception {
		validator.validateReplyQir(command, result);
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, qirStatusTypeReplied);
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST }, params = { "Agree" })
	public String submitAgreeQIR(@ModelAttribute("command") QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpServletRequest request) throws Exception {
		validator.validateAgreeQir(command, result); // PLEASE USE FRONT END JS VALIDATION UNLESS WE MOVE ALL VALIDATION TO JAVA! 
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, qirStatusTypeAgreed);
	}
	
	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST }, params = { "Close" })
	public String submitCloseQIR(@ModelAttribute("command") QIRForm command, BindingResult result, Model model,
			SessionStatus status, HttpServletRequest request) throws Exception {
		sanitizeInput(command, result);
		return processSubmission(command, result, model, status, request, qirStatusTypeClosed);
	}

	private String processSubmission(QIRForm command, BindingResult result, Model model, SessionStatus status,
			HttpServletRequest request, QIRStatusType newStatus)
			throws Exception, ServiceException, ConversionServiceException, SizeLimitExceededException {
		if (result.hasErrors()) {
			qiRConversionService.convertToQIRForm(qirService.getQIR(new Long(command.getId())), command);
			addReferenceData(model, command);
			return "editQir";
		}
		
		QIR qir = qirService.getQIR(new Long(command.getId()));
		if (qir != null) {
			if (newStatus != null) {
				command.setQirStatusType(newStatus);
				if(!newStatus.equals(qir.getQirStatusType())) {
					command.setStatusChangeDate(new Date());
				}
			}
			command.setStatusChangeDate(qir.getStatusChangeDate());
			command.setSubmittedBy(qir.getSubmittedBy());
			command.setSubmittedDate(qir.getSubmittedDate());
			qiRConversionService.convertToQIR(command, qir, getCurrentUser());
			qirService.updateQIR(qir); // this step assigned a VRDD

			try {
				NewQIRController.addAttachments(command, qir, terminologyCache, qirService, request,
						maxCombinedAttachmentLength);
			} catch (SizeLimitExceededException e) {
				result.reject("fileSizeExceeded");
				addReferenceData(model, command);
				return "editQir";
			}
		}

		userNotifier.notifyUserOnceWithMessage(request, "QIR updated successfully.");
		status.setComplete();
		return "redirect:" + command.getFromPage();
	}

	@RequestMapping(value = "/editQir.html", method = { RequestMethod.POST, RequestMethod.GET }, params = { "PDF" })
	public String showPDF(Model model) {
		return "CCHT";
	}

	private boolean enableUpdate(QIRForm qir) {
		User user = getCurrentUser();
		if (canApprove())
			// national admin
			return true;

		if (qir.isSameSubmittedBy(user) && (canCreate() || canWithdraw() || canUpdate()))
			return true;

		if (isVendor() && canUpdate() && qir.isSameVendor(user))
			return true;

		if (isVisnAdmin() && qir.isSameVisn(user))
			return true;

		if (isFacilityAdmin() && qir.isSameFacility(user))
			return true;

		return false;
	}

	private void addReferenceData(Model model, QIRForm qirForm) throws Exception {
		User user = getCurrentUser();

		boolean enableSubmit = false;
		boolean enableReply = false;
		boolean enableApprove = false;
		boolean enableWithdraw = false;
		boolean enableAgree = false;
		boolean enableClose = false;
		boolean enableVendorResponseDate = false;
		boolean enableRemarks = false;
		boolean enableActions = false;

		Date vendorResponseDueDate = qirForm.getVendorResponseDueDate();
		if (!qirForm.isClosed() && !qirForm.isWithdrawn()) {
			enableSubmit = enableUpdate(qirForm);
			enableVendorResponseDate = vendorResponseDueDate != null;

			if (qirForm.isNew()) {
				enableApprove = canApprove();
				enableWithdraw = user.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN)
						|| (canWithdraw() && qirForm.isSameSubmittedBy(user));
			} else if (qirForm.isApproved()) {
				if (user.getVendor() != null) {
					enableReply = canReply() && qirForm.isSameVendor(user);
				}
				enableClose = canClose();
			} else if (qirForm.isReplied()) {
				enableActions = canReply() && qirForm.isSameVendor(user);
				enableAgree = canAgree() && qirForm.isSameSubmittedBy(user);
				enableClose = canClose();
				enableReply = canReply() && qirForm.isSameVendor(user);
				enableVendorResponseDate = vendorResponseDueDate != null;
			} else if (qirForm.isAgreed()) {
				enableClose = canClose();
				enableReply = canReply() && qirForm.isSameVendor(user);
				enableVendorResponseDate = vendorResponseDueDate != null;
			}

			enableActions |= enableReply;

			// if vendor - disable recs and remarks
			enableRemarks = !enableReply || !enableActions;
		}
		model.addAttribute("canSubmit", enableSubmit);
		model.addAttribute("canApprove", enableApprove);
		model.addAttribute("canWithdraw", enableWithdraw);
		model.addAttribute("canAgree", enableAgree);
		model.addAttribute("canReply", enableReply);
		model.addAttribute("canClose", enableClose);
		model.addAttribute("userIsVendor", isVendor());
		model.addAttribute("userIsNationalAdmin", isNationalAdmin());
		model.addAttribute("title", buildEditFormTitle(qirForm));
		model.addAttribute("enableRemarks", enableRemarks);
		model.addAttribute("showVendorResponseDueDate", enableVendorResponseDate);
		model.addAttribute("canEditIsDmpIssue", canEditIsDmpIssue(qirForm));
		
		if(qirForm.getDmpQirType() != null){
			model.addAttribute("dmpType", qirForm.getDmpQirType().getLabel());
			model.addAttribute("dmpTypeId", qirForm.getSelectedDmpQirTypeId());
			List<? extends Lookup> terms = terminologyCache.getTerms("gov.va.med.ccht.model.qir.DmpQirType");
			Map<String, String> dmpQirTypes = new LinkedHashMap<String, String>();
			for (Lookup dmpQirType : terms) {
				gov.va.med.ccht.model.qir.DmpQirType t = (gov.va.med.ccht.model.qir.DmpQirType) dmpQirType;
				dmpQirTypes.put(t.getCode().toString(), t.getName());
			}
			qirForm.setDmpQirTypes(dmpQirTypes);
		}
		if (enableVendorResponseDate) {
			String vendorResponseDueDateAsString = DateUtils.format(vendorResponseDueDate, DateUtils.MMDDYYYY);
			model.addAttribute("vendorResponseDueDateAsString", vendorResponseDueDateAsString);
		}

	}

	private QIRForm buildForm(String selectedQIR, String fromPage, boolean readOnly) throws Exception {
		
		QIR qir = qirService.getQIR(new Long(selectedQIR));
		
		QIRForm qirForm = new QIRForm();
		qirForm.setQirStatusType(qir.getQirStatusType());
		
		qirForm.setId(selectedQIR);
		qiRConversionService.convertToQIRForm(qir, qirForm);
		qirForm.setFromPage(fromPage);
		qirForm.setReadOnly(readOnly);
		qirForm.setSubmittedDateAsString(DateUtils.format(qir.getSubmittedDate(), DateUtils.MMDDYYYY));
		
		List<? extends Lookup> terms = terminologyCache.getTerms("gov.va.med.ccht.model.qir.DmpQirType");
		Map<String, String> dmpQirTypes = new LinkedHashMap<String, String>();
		for (Lookup dmpQirType : terms) {
			gov.va.med.ccht.model.qir.DmpQirType t = (gov.va.med.ccht.model.qir.DmpQirType) dmpQirType;
			dmpQirTypes.put(t.getCode().toString(), t.getName());
		}
		qirForm.setDmpQirTypes(dmpQirTypes);

		List<QIRType> qirTypes = terminologyCache.getTerms(QIRType.class);
		Map<String, String> qirTypeMap = new LinkedHashMap<String, String>();
		for (Lookup t : qirTypes) {
			qirTypeMap.put(t.getCode().toString(), t.getName());
		}
		qirForm.setQirTypes(qirTypeMap);

		return qirForm;
	}

	private String buildEditFormTitle(QIRForm qirForm) {
		StringBuilder title = new StringBuilder();
		title.append("TN: ");
		title.append(qirForm.getId());
		title.append(" - ");

		title.append(qirForm.getQirStatusType().getName());
		title.append(" - ");

		title.append(qirForm.getHeadline());
		title.append("; ");

		title.append(getUserName());
		title.append(" ");

		title.append(qirForm.getFacility().getLabel());
		return title.toString();
	}

	/* Helper CRUD functions */

	private boolean canApprove() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.APPROVE_QIR);
	}

	private boolean canCreate() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.CREATE_QIR);
	}

	private boolean canWithdraw() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.WITHDRAW_QIR);
	}

	private boolean canReply() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.REPLY_QIR);
	}

	private boolean canUpdate() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.UPDATE_QIR);
	}

	private boolean canClose() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.CLOSE_QIR);
	}

	private boolean canAgree() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.AGREE_QIR);
	}

	public boolean canEditIsDmpIssue(QIRForm form) {
		return isNationalAdmin() && !form.isClosed() && !form.isWithdrawn();
	}

	// ------- Roles

	public boolean isNationalAdmin() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN);
	}

	private boolean isFacilityAdmin() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.FACILITY_ADMIN);
	}

	private boolean isVisnAdmin() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.VISN_ADMIN);
	}

	public boolean isVendor() {
		return getCurrentUser().isPermissionGranted(CCHTPermissions.VENDOR);
	}
	
	public static User getCurrentUser() {
		return CommonReferenceDataInterceptor.getCurrentUserAsOrNull(User.class);
	}

	public static String getUserName() {
		return CommonReferenceDataInterceptor.getCurrentUserDisplayName();
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		
		Validate.notNull(qirService, "qirService must be set");
		Validate.isTrue(maxCombinedAttachmentLength > 0, "maxCombinedAttachmentLength must be set");
		
		qirService.getAllQIRStatusTypes().forEach(status -> {
			
			final QIRStatusTypeEnum statusEnum = QIRStatusTypeEnum.toEnum(status);
			
			if (statusEnum == QIRStatusTypeEnum.New) {
				qirStatusTypeNew = status;
			} else if (statusEnum == QIRStatusTypeEnum.Approved) {
				qirStatusTypeApproved = status;
			} else if (statusEnum == QIRStatusTypeEnum.Closed) {
				qirStatusTypeClosed = status;
			} else if (statusEnum == QIRStatusTypeEnum.Withdrawn) {
				qirStatusTypeWithdrawn = status;
			} else if (statusEnum == QIRStatusTypeEnum.Replied) {
				qirStatusTypeReplied = status;
			} else if (statusEnum == QIRStatusTypeEnum.Agreed) {
				qirStatusTypeAgreed = status;
			}
			
		});

	}
	
	public void sanitizeInput(QIRForm form, Errors errors) { // we only need to sanitize the fields that are allowed by the initbinder in EditQirController.
		// other fields are auto-generated.
		// "isDmpIssue", "selectedDmpQirTypeId", "dmpQirType", "complaint" , "remarks", "vendorActions", "attachments", "attachmentFiles"
		String field = "";
		
		try {
			
			if(form.getIsDmpIssue() != null) {
				// get an error if isDmpIssue contains invalid characters
				 field = "isDmpIssue";
				form.setIsDmpIssue(ESAPIValidator.validateStringInput(form.getIsDmpIssue(), ESAPIValidationType.IsDmpIssue_WhiteList));
			}
			
			if(form.getFromPage() != null ) {
				field = "fromPage";
				form.setFromPage(ESAPIValidator.validateStringInput(form.getFromPage(), ESAPIValidationType.FromPage_WhiteList));
			}
				
			// selectedDmpQirTypeId  not checked, only strings.
			if(form.getDmpQirType() != null) {
				field = "selectedDmpQirTypeId";
				TermType tt = new TermType();
				tt.setLabel(validateMultilineField(form.getDmpQirType().getLabel(),errors, field));
				tt.setValue(validateMultilineField(form.getDmpQirType().getValue(),errors, field));
				form.setQirType(tt);
			}
			if(form.getComplaint() != null) {
				field = "complaint";
				form.setComplaint(validateMultilineField(form.getComplaint(),errors, field));
			}
			if(form.getRemarks() != null) {
				field = "remarks";
				form.setRemarks(validateMultilineField(form.getRemarks(),errors, field));
			}
			if(form.getVendorActions() != null) {
				field = "vendorActions";
				form.setVendorActions(validateMultilineField(form.getVendorActions(),errors, field));
			}
			
			// attachments and attachmentsFiles do not need to be validated. They are never "opened" in the application and there for their content to not need to be validated.
			
			
		} catch (Exception e) {
			errors.rejectValue(field, "", "Program has encountered an error, please re-enter applicable fields and try submission again.");
		}
	}
	
	private String validateMultilineField(String value, Errors errors, String field){ //throws ServiceException, IOException, DocumentException{
		String[] fieldArray = StringUtils.split(value.toString(), "\n");
		String cleanValue = "";
		String temp = "";
					
		try {
			for(int i = 0; i < fieldArray.length; i++) {
				if(field.equals("selectedDmpQirTypeId")) {
					temp = ESAPIValidator.validateStringInput(fieldArray[i].trim(), ESAPIValidationType.DmpQirType_WhiteList);
				}
				else {
					temp = ESAPIValidator.validateStringInput(fieldArray[i].trim(), ESAPIValidationType.CROSS_SITE_SCRIPTING_PERSISTENT);
					temp = ESAPIValidator.validateStringInput(fieldArray[i].trim(), ESAPIValidationType.CROSS_SITE_SCRIPTING_REFLECTED);
				}
				if(i > 0) {
					cleanValue = cleanValue.concat("\n");
				}
				cleanValue = cleanValue.concat(temp);
			}
		} catch (Exception e) {
			errors.rejectValue(field, "", "Program has encountered an error, please re-enter applicable fields and try submission again.");
		}
		return cleanValue;
	}
}
