package gov.va.med.ccht.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;

import org.springframework.beans.factory.annotation.Autowired;

import gov.va.med.ccht.model.CCHTPermissions;
import gov.va.med.ccht.model.CCHTRoles;
import gov.va.med.ccht.model.User;
import gov.va.med.ccht.model.common.DeviceType;
import gov.va.med.ccht.model.common.Facility;
import gov.va.med.ccht.model.common.Vendor;
import gov.va.med.ccht.model.common.Visn;
import gov.va.med.ccht.model.terminology.InactiveReason;
import gov.va.med.ccht.model.terminology.RegistrationDeniedReason;
import gov.va.med.ccht.model.terminology.RegistrationStatus;
import gov.va.med.ccht.persistent.SecurityDAO;
import gov.va.med.ccht.service.common.SecurityService;
import gov.va.med.ccht.service.common.TerminologyCache;
import gov.va.med.ccht.service.common.TerminologyException;
import gov.va.med.ccht.service.common.facility.FacilityService;
import gov.va.med.ccht.service.common.vendor.VendorService;
import gov.va.med.ccht.service.common.visn.VisnService;
import gov.va.med.ccht.service.qir.QIRService;
import gov.va.med.ccht.service.report.ScheduleConversionService;
import gov.va.med.ccht.service.scheduleJob.ScheduleJobService;
import gov.va.med.ccht.ui.ajax.AjaxRequestHandler;
import gov.va.med.ccht.ui.common.UserNotifier;
import gov.va.med.ccht.ui.interceptor.CommonReferenceDataInterceptor;
import gov.va.med.ccht.ui.model.QIRForm;
import gov.va.med.ccht.ui.model.UserForm;
import gov.va.med.fw.batchprocess.BatchProcessInvoker;
import gov.va.med.fw.batchprocess.BatchProcessService;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.scheduling.SchedulingService;
import gov.va.med.fw.security.SecurityContextHelper;
import gov.va.med.fw.security.SimpleRole;
import gov.va.med.fw.security.UserPrincipal;
import gov.va.med.fw.ui.model.TermType;

public class CchtController implements CCHTRoles {

	public final static String NEGATIVE_ONE = "-1";

	public static final TermType NATIONAL_ADMIN_ROLE = new TermType(NATIONAL_ADMIN,
			NATIONAL_ADMIN);
	public static final TermType VISN_ADMIN_ROLE = new TermType(VISN_ADMIN,
			VISN_ADMIN);
	public static final TermType FACILITY_ADMIN_ROLE = new TermType(FACILITY_ADMIN,
			FACILITY_ADMIN);
	public static final TermType APPLICATION_ADMIN_ROLE = new TermType(APPLICATION_ADMIN,
			APPLICATION_ADMIN);
	public static final TermType MANAGEMENT_ROLE = new TermType(MANAGEMENT,
			MANAGEMENT);
	public static final TermType VENDOR_ROLE = new TermType(VENDOR, VENDOR);
	public static final TermType NAC_POC_ROLE = new TermType(NAC, NAC);
	public static final TermType SYSTEM_ADMIN_ROLE = new TermType(SYSTEM_ADMIN,
			SYSTEM_ADMIN);
	public static final TermType CARE_COORDINATOR_ROLE = new TermType(CARE_COORDINATOR,
			CARE_COORDINATOR);
	public static final TermType PROGRAM_ASSISTANT_ROLE = new TermType(PROGRAM_ASSISTANT,
			PROGRAM_ASSISTANT);
	public static final TermType QIR_ORIGINATOR_ROLE = new TermType(QIR_ORIGINATOR,
			QIR_ORIGINATOR);

	public static final TermType REPORTS_ONLY_ROLE = new TermType(REPORTS_ONLY,
			REPORTS_ONLY); 

	@Autowired
	protected AjaxRequestHandler ajaxRequestHandler;
	@Autowired
	protected BatchProcessService batchProcessService;
	@Autowired
	protected BatchProcessInvoker batchProcessInvoker;
	@Autowired
	protected ScheduleJobService scheduleJobService;
	@Autowired
	protected QIRService qirService = null;
	@Autowired
	protected ReportConversionService reportConversionService;
	@Autowired
	protected ScheduleConversionService scheduleConversionService;
	@Autowired
	protected SchedulingService schedulingService;
	@Autowired
	protected SecurityConversionService securityConversionService;
	@Autowired
	protected SecurityDAO securityDAO;
	@Autowired
	protected SecurityService securityService;
	@Autowired
	protected TerminologyCache terminologyCache;
	@Autowired
	protected TerminologyCacheController terminologyCacheController;
	@Autowired
	protected TerminologyController terminologyController;
	@Autowired
	protected UserNotifier userNotifier;
	@Autowired
	protected VendorService vendorService;
	@Autowired
	protected VisnService visnService;
	@Autowired
	protected FacilityService facilityService;

	protected String basePath = "";
	
	public Map<Integer, Vendor> getVendorsLookUpMap() throws Exception {
		List<Vendor> vendorsList = vendorService.getActiveVendors();
		Map<Integer, Vendor> vendorsMap = new HashMap<Integer, Vendor>();

		for (Vendor vendor : vendorsList) {
			vendorsMap.put(vendor.getId(), vendor);
		}

		return vendorsMap;
	}

	public Map<String, Vendor> getVendors() throws Exception {
		Map<String, Vendor> retVal = null;

		List<Vendor> vendorList = vendorService.getActiveVendors();
		retVal = new LinkedHashMap<String, Vendor>();
		for (Vendor vendor : vendorList) {
			retVal.put(vendor.getCode() , vendor);
		}

		return retVal;
	}
	
	public Map<Long, DeviceType> getDevicesLookUpMap(List<DeviceType> devices) throws Exception {

		// List<Device> devices = getDevices();
		Map<Long, DeviceType> devicesMap = new LinkedHashMap<Long, DeviceType>();

		for (DeviceType device : devices) {
			devicesMap.put(device.getId(), device);
		}

		return devicesMap;
	}
	
	public Map<String, InactiveReason> getInactiveReasons(boolean insertSelect) throws Exception {
		
		Map<String, InactiveReason> retVal = new LinkedHashMap<String, InactiveReason>();
		
		if (insertSelect) {
			InactiveReason reason = new InactiveReason();
			reason.setName("Select");
			retVal.put(NEGATIVE_ONE, reason);
		}
		
		List<InactiveReason> inactiveReasons = terminologyCacheController.getInactivationReasons();
		
		for(InactiveReason item : inactiveReasons) {
			retVal.put(item.getCode(), item);
		}
		
		
		return retVal;
	}

	public boolean isApplicationAdmin(User currentUser) {		
		return currentUser.isPermissionGranted(CCHTPermissions.APPLICATION_ADMIN);		
	}

	public Map<String, RegistrationDeniedReason> getRegistrationDeniedReasons(boolean insertSelect) throws Exception {

		Map<String, RegistrationDeniedReason> retVal =  new LinkedHashMap<String, RegistrationDeniedReason>();


		if(insertSelect) {
			RegistrationDeniedReason val = new RegistrationDeniedReason();
			val.setName("Select");
			retVal.put(NEGATIVE_ONE, val);
		}

		Map<String, RegistrationDeniedReason> regDeniedReasons = terminologyCacheController.getRegistrationDeniedLookUpMap();

		retVal.putAll(regDeniedReasons);
		return retVal;

	}

	public Map<String, Facility> getFacilities() throws Exception {
		Map<String, Facility> retVal = null;

		SortedSet<Facility> list = facilityService.getFacilities();
		retVal = new LinkedHashMap<String, Facility>();
		for (Facility facility : list) {
			retVal.put(facility.getCode(), facility);
		}

		return retVal;
	}


	public Map<String, Vendor> getVendors(boolean insertSelect) throws Exception {
		Map<String, Vendor> retVal = null;

		List<Vendor> vendorList = vendorService.getActiveVendors();
		retVal = new LinkedHashMap<String, Vendor>();
		if (insertSelect) {
			Vendor selectVendor = new Vendor();
			selectVendor.setName("Select Vendor");
			retVal.put(NEGATIVE_ONE, selectVendor);
		}
		for (Vendor vendor : vendorList) {
			retVal.put(vendor.getName(), vendor);
		}

		return retVal;
	}

	public Map<String, RegistrationStatus> getRegistrationStatuses() throws Exception {
		return terminologyCacheController.getRegistrationStatusLookUpMap();
	}

	public Map<String, String> getJobDescriptions() throws Exception {

		return terminologyCacheController.getJobDescriptions();
	}

	public Map<String, String> getUserGroups() throws Exception {
		return terminologyCacheController.getUserGroups();
	}

	public Map<String, String> getTerms(String className) throws TerminologyException {
		Map<String, String> retVal = new LinkedHashMap<String, String>();
		List<? extends Lookup> terms = terminologyCache.getTerms(className);

		for (Lookup term : terms) {
			retVal.put(term.getCode(), term.getName());
		}

		return retVal;
	}

	public boolean canEditIsDmpIssue(QIRForm form) {
		boolean canSet = false;
		UserPrincipal up = SecurityContextHelper.getCurrentUser();

		if (!up.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN)) {
			return canSet;
		}
		if (form.isClosed() || form.isWithdrawn()) {
			return canSet;
		}

		return true;
	}

	public Map<String, Facility> getFacilities(boolean insertSelect) throws Exception {
		Map<String, Facility> retVal = null;

		SortedSet<Facility> list = facilityService.getFacilities();
		retVal = new LinkedHashMap<String, Facility>();

		// new facility
		if (insertSelect) {
			Facility selecFacility = new Facility();
			selecFacility.setName("Select Facility");
			retVal.put(NEGATIVE_ONE, selecFacility);
		}
		// get facility
		for (Facility facility : list) {
			retVal.put(facility.getCode().trim(), facility);
		}

		return retVal;
	}

	public Map<String, Visn> getVisns(boolean insertSelect) throws Exception {
		Map<String, Visn> retVal = null;
		List<Visn> visns = visnService.getVisns();
		retVal = new LinkedHashMap<String, Visn>();

		if (insertSelect) {
			Visn selectVisn = new Visn();
			selectVisn.setName("Select VISN");
			retVal.put(NEGATIVE_ONE, selectVisn);
		}

		for (Visn visn : visns) {
			retVal.put(visn.getCode().trim(), visn);
		}

		return retVal;
	}

	public boolean isNationalAdmin() {
		UserPrincipal up = SecurityContextHelper.getCurrentUser();

		if (up != null) {
			return up.isPermissionGranted(CCHTPermissions.NATIONAL_ADMIN);
		} else {
			return false;
		}
	}

	public Map<String, String> getAvailableRoles() throws Exception {
		Map<String, String> retVal = new LinkedHashMap<String, String>();

		UserForm userForm = getUser(getCurrentUser().getUsername());
		List<TermType> availableRoles = userForm.getAvailableRoles();

		for (TermType role : availableRoles) {
			String roleId = role.getValue();
			String roleName = role.getLabel();
			retVal.put(roleId, roleName);
		}

		// Add assigned roles to available roles 
		List<TermType> assignedRoles = userForm.getRoles();
		for(TermType role : assignedRoles) {
			String roleId = role.getValue();
			String roleName = role.getLabel();
			retVal.put(roleId, roleName);
		}
		return retVal;
	}

	public UserForm getUser(String userName) throws Exception {
		UserPrincipal currentUser = getCurrentUser();
		User user = securityService.getUser(userName);
		UserForm userForm = new UserForm();
		securityConversionService.convert(user, userForm);
		List<TermType> assignedRoles = userForm.getRoles();
		List<TermType> availableRoles = new ArrayList<TermType>(); // getAllRoles();
		List<TermType> allRoles = new ArrayList<TermType>(); // getAllRoles();
		List<SimpleRole> roles;
		if (!currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN)) {
			roles = securityService.getAllDMPRoles();
		} else {
			roles = securityService.getAllSimpleRoles();
		}
		for (SimpleRole role : roles) {
			allRoles.add(new TermType(role.getName(), role.getName()));
		}
		// filter the available roles based on the user role
		if (currentUser.isPermissionGranted(CCHTRoles.NATIONAL_ADMIN)) {
			// no filtering
			availableRoles = allRoles;
		} else {
			User loginUser = (User) currentUser;
			boolean canAddRole = false;
			if (currentUser.isPermissionGranted(CCHTRoles.FACILITY_ADMIN)) {

				if (loginUser.getFacility() != null) {
					if (user.getFacility() != null && 
							(loginUser.getFacility().getId().intValue() == user.getFacility().getId().intValue())) {
						canAddRole = true;
					} else if (user.getSecondaryFacility() != null && 
							(loginUser.getFacility().getId().intValue() == user.getSecondaryFacility().getId().intValue())) {
						canAddRole = true;
					}
				} else if (loginUser.getSecondaryFacility() != null) {
					if (user.getFacility() != null && 
							(loginUser.getSecondaryFacility().getId().intValue() == user.getFacility().getId().intValue())) {
						canAddRole = true;
					} else if (user.getSecondaryFacility() != null && 
							(loginUser.getSecondaryFacility().getId().intValue() == user.getSecondaryFacility().getId().intValue())) {
						canAddRole = true;
					}
				}
				if (canAddRole) {
					availableRoles.add(FACILITY_ADMIN_ROLE);
					availableRoles.add(CARE_COORDINATOR_ROLE);
					availableRoles.add(PROGRAM_ASSISTANT_ROLE);
					availableRoles.add(QIR_ORIGINATOR_ROLE);
					//TODO : Enforcing Reports Only being viewable for Facility Admin
					availableRoles.add(REPORTS_ONLY_ROLE);
				}
			}
			canAddRole = false;
			if (currentUser.isPermissionGranted(CCHTRoles.VISN_ADMIN)) {
				// keep Facility Administrator, Care Coordinator, QIR
				// Originator, Program Support Assistant, Application
				// Administrator
				// if user and current user are in the same VISN, otherwise,
				// remove all
				if (loginUser.getVisn() != null && user.getVisn() != null
						&& (loginUser.getVisn().getId() == user.getVisn().getId())) {
					canAddRole = true;
				}
				if (canAddRole) {
					if (!availableRoles.contains(VISN_ADMIN_ROLE))
						availableRoles.add(VISN_ADMIN_ROLE);
					if (!availableRoles.contains(FACILITY_ADMIN_ROLE))
						availableRoles.add(FACILITY_ADMIN_ROLE);
					if (!availableRoles.contains(CARE_COORDINATOR_ROLE))
						availableRoles.add(CARE_COORDINATOR_ROLE);
					if (!availableRoles.contains(QIR_ORIGINATOR_ROLE))
						availableRoles.add(QIR_ORIGINATOR_ROLE);
					if (!availableRoles.contains(PROGRAM_ASSISTANT_ROLE))
						availableRoles.add(PROGRAM_ASSISTANT_ROLE);

					//TODO : Enforcing Reports Only being viewable for VISN Admin
					if (!availableRoles.contains(REPORTS_ONLY_ROLE))
						availableRoles.add(REPORTS_ONLY_ROLE);
				}
			}
		}

		availableRoles.removeAll(assignedRoles);
		userForm.setAvailableRoles(availableRoles);

		return userForm;
	}



	public static UserPrincipalImpl getCurrentUser() {
		return CommonReferenceDataInterceptor.getCurrentUser();
	}

}
