package gov.va.genisis2;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.rdf.model.Literal;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.log4j.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import gov.va.genisis2.ts.common.dto.TripleDTO;

/**
 * 
 * @author PII
 *
 */
public class CommandClient {
	
	private static final Logger LOGGER = Logger.getLogger(CommandClient.class);
	
	private static final long LIMIT = 1000L;
	private static long last = 0;
	
	public static String subjectQuery = "SELECT DISTINCT ?s { ?s ?p ?o . }";
	public static String predicateQuery = "SELECT DISTINCT ?p { ?s ?p ?o . }";
	public static String objectQuery = "SELECT DISTINCT ?o { ?s ?p ?o . }";
	
	public FusekiClient fusekiClient;
	public PropertyManager propertyManager;
	public String endpoint;
	/**
	 * limit for how many data rows should be returned
	 */
	public int generationLimit;
	public int randomOffset;
	
	public int chanceForCreatingNewTriple;
	
	public CommandClient (PropertyManager prop) {
		fusekiClient = FusekiClient.getInstance();
		propertyManager = prop;
		
		this.chanceForCreatingNewTriple = propertyManager.getPropertyValue
				(CliScriptConstants.CHANCE_NEW_TRIPLE, 10);
		this.randomOffset = propertyManager.getPropertyValue(CliScriptConstants.RANDOM_OFFSET, 50);
		this.generationLimit = propertyManager.getPropertyValue(CliScriptConstants.GENERATION_LIMIT, 1000);
	}
	
	/**
	 * Generate existing triples
	 * @param endpoint
	 */
	public void generateExistingTripleFiles(String endpoint) {
		
		this.endpoint = endpoint;
		//generate subjects file
		generateDistinctQueryFile(subjectQuery, CliScriptConstants.SUBJECT_FILE, "?s");
		
		//generate predicates file
		generateDistinctQueryFile(predicateQuery, CliScriptConstants.PREDICATE_FILE, "?p");
		
		//generate objects file
		generateDistinctQueryFile(objectQuery, CliScriptConstants.OBJECT_FILE, "?o");
	}
	
	/**
	 * 
	 * @param query
	 * @param filename
	 * @param headVar
	 */
	public void generateDistinctQueryFile(String query, String filename, String headVar) {
		QueryExecution qe = null;
		ResultSet resultSet = null;
		String results = "";
		
		Random rand = new Random(); 
		//ME: dont know how large the dataset is...randomly choosing 
		//a number as an offset could possible move over the range of triples...may need to ask for the
		//number of triples first..gonna go with 50 for now
		query+=" OFFSET "+rand.nextInt(randomOffset)+" LIMIT "+generationLimit;
		try {
			qe = QueryExecutionFactory.sparqlService(endpoint, query);

			resultSet = fusekiClient.performSelectQuery(qe);

			if (null != resultSet) {
				while (resultSet.hasNext()) {
					QuerySolution binding = resultSet.nextSolution();
					results += getValueOfLiteralOrUri(binding, headVar)+"\n";
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in execution query", e);
			throw e;
		} finally {
			if (!qe.isClosed()) {
				qe.close();
			}
		}
		
		writeToFile( results, filename );
//		writeToFile( results, new File(filename));
	}
	
	
//	public void writeToFile(String contents, File file) {
//		if (file.getParentFile().mkdir()) {
//			try {
//				file.createNewFile();
//			} catch (IOException e) {
//				LOGGER.error(e.getMessage() );
//			}
//		} 
//		writeToFile( contents, file.getName());
//	}
	
	/**
	 * Will write contents to a file given a file
	 * @param contents
	 * @param filename
	 */
	public void writeToFile(String contents, String filename) {
		try( BufferedWriter writer = new BufferedWriter( new FileWriter( filename ))){
			writer.write(contents);
		} catch (IOException e) {
			LOGGER.error(e.getMessage() );
		}
	}
	
	/**
	 * 
	 * @param triples
	 * @param printToStdOut
	 */
	public void checkIfTriplesExists(List<TripleDTO> triples, boolean printToStdOut, String endpoint) {

		FusekiClient fs = FusekiClient.getInstance();
		List<TripleExistDTO> existsList = new ArrayList<TripleExistDTO>();
		boolean exists = false;

		for (TripleDTO tripleDTO : triples) {
			try {
				exists = fs.performAsk(getAskForm(tripleDTO), endpoint);
			} catch (Exception e) {
				LOGGER.error(e.getMessage());
			}
			TripleExistDTO tripleExist = new TripleExistDTO();
			tripleExist.setAlreadyExists(exists);
			tripleExist.setTriple(tripleDTO);
			existsList.add(tripleExist);
		}

		ObjectMapper mapper = new ObjectMapper();
		mapper.enable(SerializationFeature.INDENT_OUTPUT);

		if (printToStdOut) {
			// print to STDOUT
			try (OutputStream out = new ByteArrayOutputStream()) {

				// Object to JSON in file
				mapper.writeValue(out, existsList);

				final byte[] data = ((ByteArrayOutputStream) out).toByteArray();
				System.out.println(new String(data));

			} catch (IOException e) {
				LOGGER.error(e.getMessage());
			}
		} else {
			// print to a file
			try (OutputStream out = new FileOutputStream(
					new File(CliScriptConstants.EXISTS_OUTPUT_FILENAME))) {

				// Object to JSON in file
				mapper.writeValue(out, existsList);
			} catch (IOException e) {
				LOGGER.error(e.getMessage());
			}
		}
	}
	
	/**
	 * 
	 * @param numberOfTriples
	 */
	public void createTriples(int numberOfTriples) {
		String mvpPrefix = "http://genisis.va.gov/mvp-schema#Definition_";
		
		List<TripleDTO> triples = new ArrayList<TripleDTO>();
		
		//read files into memory..if a file is not found STOP!
		RandomAccessList<String> randomSubjects = new RandomAccessList<String>(
				readLines( CliScriptConstants.SUBJECT_FILE ));
		RandomAccessList<String> randomPredicates = new RandomAccessList<String>(
				readLines ( CliScriptConstants.PREDICATE_FILE ));
		RandomAccessList<String> randomObjects = new RandomAccessList<String>(
				readLines( CliScriptConstants.OBJECT_FILE ));
		
		Random rand = new Random();
		
		for(int i = 0; i < numberOfTriples; i++ ) {
			TripleDTO dto = new TripleDTO();
			if (rand.nextInt(chanceForCreatingNewTriple) == 0) {
				// manually create a new triple
				dto.setS(mvpPrefix + getID());
				//TODO: need a proper list of predicates to draw from
				dto.setP("http://www.w3.org/2000/01/rdf-schema#label");
				dto.setO("random-test-generation");
			} else {
				// find an existing triple
				dto.setS(randomSubjects.pick());
				dto.setP(randomPredicates.pick());
				dto.setO(randomObjects.pick());
			}
			triples.add(dto);
		}
		
		writeJsontoFile(triples, CliScriptConstants.CREATE_OUTPUT_FILENAME);
	}
	
	/** 
	 * Uses jacksons object mapper to write json formatted strings given a list DTO objects 
	 * and a filename
	 * @param triples
	 * @param filename
	 */
	public void writeJsontoFile(List<?> triples, String filename) {
		try (OutputStream out = new FileOutputStream(filename)) {
			ObjectMapper mapper = new ObjectMapper();
			mapper.enable(SerializationFeature.INDENT_OUTPUT);

			// Object to JSON in file
			mapper.writeValue(out, triples);
		} catch (IOException e) {
			LOGGER.error(e.getMessage());
		}
	}
	
	/**
	 * Opens the given file and reads the data into a list of strings
	 * @param filename
	 * @return list of lines
	 */
	public List<String> readLines(String filename) {
		ArrayList<String> array = null;

		try (FileInputStream fs = new FileInputStream(filename);
				InputStreamReader is = new InputStreamReader(fs);
				BufferedReader br = new BufferedReader(is)) {

			array = new ArrayList<String>();
			String line;
			while ((line = br.readLine()) != null)
				array.add(br.readLine());
		} catch (IOException ie) {
			LOGGER.error("Can't read input file : "+filename+". Please run this program with "
					+ "the -generate flag so input files can be generated");
			System.exit(-1);
		}
		return array;
	}
	
	/**
	 *  Generate unique ids
	 *  Taken from: https://stackoverflow.com/questions/18227787/java-unique-10-digit-id
	 * @return
	 */
	public static long getID() {
		  long id = System.currentTimeMillis() % LIMIT;
		  if ( id <= last ) {
		    id = (last + 1) % LIMIT;
		  }
		  return last = id;
		}
	
	/**
	 * Formats a triple into ask form
	 * @param triple
	 * @return
	 */
	private static String getAskForm(TripleDTO triple) {
		if (null == triple) {
			return null;
		} else {
			return "ASK { <" + triple.getS() + "> <" + triple.getP() + "> \"" + triple.getO() + "\" }";
		}
	}
	
	/**
	 * Will retreive the literal or uri given the binding header
	 * @param binding
	 * @param headVar
	 * @return
	 */
	private String getValueOfLiteralOrUri(QuerySolution binding, String headVar) {
		RDFNode literalOrUriNode;
		String valueOfiteralOrUri = null;
		if (null == binding) {
			return null;
		}
		if (binding.contains(headVar) && null != (literalOrUriNode = binding.get(headVar))) {
			org.apache.jena.rdf.model.Resource uriResource;
			if (literalOrUriNode.isLiteral()) {
				Literal literal = literalOrUriNode.asLiteral();
				if (null != literal) {
					valueOfiteralOrUri = literal.toString().trim();
				}
			} else if (literalOrUriNode.isURIResource() && 
					null != (uriResource = literalOrUriNode.asResource())) {
				valueOfiteralOrUri = uriResource.getURI().trim();
			}
		}
		return valueOfiteralOrUri;
	}
}
